# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AsphaltIntelligenceDockWidget
                                 A QGIS plugin
 Raster preparation and mapping tools for EPSG:3857 workflows
 (JPG+JGW, SID, GeoTIFF)
                             -------------------
        begin                : 2026-01-20
        copyright            : (C) 2026 Asphalt Intelligence, LLC
        email                : kate@asphaltintelligence.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os 
from qgis.PyQt.QtWidgets import (
    QDockWidget,
    QWidget,
    QVBoxLayout,
    QPushButton,
    QLabel,
)
from qgis.PyQt.QtGui import QPixmap
from qgis.PyQt.QtCore import pyqtSignal, Qt, QUrl
from qgis.PyQt.QtWidgets import QLabel
from qgis.PyQt.QtGui import QDesktopServices


from .utils.project_setup import start_setup
from .utils.raster_io import load_jpg_jgw_and_convert
from .utils.raster_io import load_sid_and_convert
from .utils.raster_io import reproject_geotiff_to_3857
from .utils.raster_io import add_ai_output_tif


class AsphaltIntelligenceDockWidget(QDockWidget):

    closingPlugin = pyqtSignal()

    def __init__(self, iface, plugin_dir, parent=None):
        super().__init__(parent)
        self.iface = iface
        self.plugin_dir = plugin_dir


        # -----------------------------
        # Central container + layout
        # -----------------------------
        container = QWidget(self)
        layout = QVBoxLayout(container)
        layout.setContentsMargins(12, 12, 12, 12)
        layout.setSpacing(10)

        # -----------------------------
        # Title
        # -----------------------------
        title = QLabel("Asphalt Intelligence")
        title.setAlignment(Qt.AlignCenter)
        title.setStyleSheet(
            "font-size: 18px; font-weight: 700; color: black; margin-bottom: 8px;"
        )
        layout.addWidget(title)

        # -----------------------------
        # Logo (top)
        # -----------------------------
        logo_path = os.path.join(self.plugin_dir, "images", "logo_alpha.png")
        
        logo_label = QLabel()
        logo_label.setAlignment(Qt.AlignCenter)

        if os.path.exists(logo_path):
            pixmap = QPixmap(logo_path)
            pixmap = pixmap.scaledToWidth(180, Qt.SmoothTransformation)
            logo_label.setPixmap(pixmap)
        else:
            logo_label.setText("[Logo not found]")

        layout.addWidget(logo_label)


        # -----------------------------
        # Buttons
        # -----------------------------
        self.btn_start = QPushButton("Start Setup ➜")
        self.btn_add_tif = QPushButton("Add Asphalt Intelligence Output (tif)")
        self.btn_jpg = QPushButton("Load JPG + JGW → Convert to GeoTIFF")
        self.btn_sid = QPushButton("Load SID → Convert to GeoTIFF")
        self.btn_reproject = QPushButton("Reproject GeoTIFF to EPSG:3857")

        buttons = [
            self.btn_start,
            self.btn_add_tif,
            self.btn_jpg,
            self.btn_sid,
            self.btn_reproject,
        ]

        for btn in buttons:
            btn.setMinimumHeight(42)
            layout.addWidget(btn)

        # -----------------------------
        # Brand styling helper
        # -----------------------------
        def style_button(btn, bg_color):
            btn.setStyleSheet(f"""
                QPushButton {{
                    background-color: {bg_color};
                    color: white;
                    border: 1px solid rgba(0, 0, 0, 0.25);
                    border-radius: 4px;
                    font-size: 14px;
                    font-weight: 400;
                 padding: 8px;
                }}
                QPushButton:hover {{
                    background-color: {bg_color};
                    border: 1px solid rgba(0, 0, 0, 0.45);
                }}
                QPushButton:pressed {{
                    background-color: {bg_color};
                    border: 1px solid rgba(0, 0, 0, 0.6);
                }}
            """)


        # Apply brand colors
        style_button(self.btn_start, "#a057d6")      # Primary purple
        style_button(self.btn_add_tif, "#33b6db")    # Blue
        style_button(self.btn_jpg, "#33b6db")        # Blue
        style_button(self.btn_sid, "#33b6db")        # Blue
        style_button(self.btn_reproject, "#33b6db")  # Blue

        link = QLabel(
            '<a href="https://asphaltintelligence.com">asphaltintelligence.com</a>'
        )
        link.setAlignment(Qt.AlignCenter)
        link.setOpenExternalLinks(True)
        link.setStyleSheet(
            "color: #666666; font-size: 14px; margin-top: 6px;"
        )

        layout.addWidget(link)


        
        
        # -----------------------------
        # Legend section (bottom)
        # -----------------------------

        # Add vertical whitespace before legend
        layout.addSpacing(155)

        # Legend title
        legend_title = QLabel("Legend")
        legend_title.setAlignment(Qt.AlignCenter)
        legend_title.setStyleSheet(
            "font-size: 13px; font-weight: 600; color: #333333; margin-bottom: 4px;"
        )
        layout.addWidget(legend_title)

        # Legend image
        legend_path = os.path.join(self.plugin_dir, "images", "legend_alpha.png")

        legend_label = QLabel()
        legend_label.setAlignment(Qt.AlignCenter)

        if os.path.exists(legend_path):
            pixmap = QPixmap(legend_path)
            pixmap = pixmap.scaledToWidth(220, Qt.SmoothTransformation)
            legend_label.setPixmap(pixmap)
        else:
            legend_label.setText("[Legend not found]")

        layout.addWidget(legend_label)


        layout.addStretch()

        # -----------------------------
        # Finalize dock widget
        # -----------------------------
        self.setWidget(container)
        self.setWindowTitle("Asphalt Intelligence Setup")
        self.btn_start.clicked.connect(
            lambda: start_setup(self.iface)
        )
        self.btn_jpg.clicked.connect(
            lambda: load_jpg_jgw_and_convert(self.iface)
        )
        self.btn_sid.clicked.connect(
            lambda: load_sid_and_convert(self.iface)
        )
        self.btn_reproject.clicked.connect(
            lambda: reproject_geotiff_to_3857(self.iface)
        )
        self.btn_add_tif.clicked.connect(
            lambda: add_ai_output_tif(self.iface)
        )





    def closeEvent(self, event):
        self.closingPlugin.emit()
        event.accept()
