"""
/***************************************************************************
 ArrowsDialog
                                 A QGIS plugin
 Draw polygon arrows based on polylines
                             -------------------
        begin                : 2012-03-06
        copyright            : (C) 2012 by Gregoire Piffault
        email                : gregoire.piffault@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os.path

from PyQt4 import QtCore, QtGui
from qgis.gui import QgsEncodingFileDialog
from qgis.core import *

from ui_arrows import Ui_Arrows

# create the dialog for zoom to point
class ArrowsDialog(QtGui.QDialog):
	def __init__(self, plugin):
		QtGui.QDialog.__init__(self)
		# Set up the user interface from Designer.
		self.ui = Ui_Arrows()
		self.ui.setupUi(self)
		# Vars
		self.plugin = plugin
		self.encoding = "CP1252"
		self.outshape = None
		# Connections
		self.ui.inshape.currentIndexChanged.connect(self.check_inshape)
		self.ui.width_field_forward.currentIndexChanged.connect(self.guess_backward)
		self.ui.one_arrow.toggled.connect(self.radio_change)
		self.ui.guess_head_size.clicked.connect(self.guess_head_size)
		self.ui.browse_outshape.clicked.connect(self.browse_outshape)
		# Populate point layer names
		self.ui.inshape.clear()
		self.ui.inshape.addItems([layer.name() for layer in self.plugin.linelayers])
		# Read conf
		self.load_conf()
	
	def load_conf(self):
		if "arrow_inshape_index" in self.plugin.conf and self.plugin.conf["arrow_inshape_index"] < self.ui.inshape.count():
			self.ui.inshape.setCurrentIndex(self.plugin.conf["arrow_inshape_index"])
		if "arrow_one_arrow" in self.plugin.conf and not self.plugin.conf["arrow_one_arrow"]:
			self.ui.two_arrows.toggle()
		if "arrow_width_field" in self.plugin.conf and self.plugin.conf["arrow_width_field"] < self.ui.width_field_forward.count():
			self.ui.width_field.setCurrentIndex(self.plugin.conf["arrow_width_field"])
		if "arrow_double_head" in self.plugin.conf and not self.plugin.conf["arrow_double_head"]:
			self.ui.double_head.setCheckState(QtCore.Qt.Unchecked)
		if "arrow_width_field_forward" in self.plugin.conf and self.plugin.conf["arrow_width_field_forward"] < self.ui.width_field_forward.count():
			self.ui.width_field_forward.setCurrentIndex(self.plugin.conf["arrow_width_field_forward"])
		if "arrow_width_field_backward" in self.plugin.conf and self.plugin.conf["arrow_width_field_backward"] < self.ui.width_field_backward.count():
			self.ui.width_field_backward.setCurrentIndex(self.plugin.conf["arrow_width_field_backward"])
		if "arrow_width_factor" in self.plugin.conf:
			self.ui.width_factor.setText(self.plugin.conf["arrow_width_factor"])
		if "arrow_head_size" in self.plugin.conf:
			self.ui.head_size.setText(self.plugin.conf["arrow_head_size"])
		if "arrow_threshold" in self.plugin.conf:
			self.ui.threshold.setText(self.plugin.conf["arrow_threshold"])
		if "arrow_outshape" in self.plugin.conf:
			self.ui.outshape.setText(self.plugin.conf["arrow_outshape"])
	
	def accept(self):
		# Inputs check
		if self.ui.inshape.currentIndex() == -1:
			QtGui.QMessageBox.warning(self, "Arrows", "No line layer selected")
			return False
		if self.ui.outshape.text() == "":
			QtGui.QMessageBox.warning(self, "Arrows", "No output file specified")
			return False
		if self.ui.head_size.text() == "":
			self.guess_head_size()
		# Write conf
		self.plugin.conf["arrow_inshape_index"] = self.ui.inshape.currentIndex()
		self.plugin.conf["arrow_one_arrow"] = self.ui.one_arrow.isChecked()
		self.plugin.conf["arrow_width_field"] = self.ui.width_field.currentIndex()
		self.plugin.conf["arrow_double_head"] = self.ui.double_head.isChecked()
		self.plugin.conf["arrow_width_field_forward"] = self.ui.width_field_forward.currentIndex()
		self.plugin.conf["arrow_width_field_backward"] = self.ui.width_field_backward.currentIndex()
		self.plugin.conf["arrow_width_factor"] = unicode(self.ui.width_factor.text())
		self.plugin.conf["arrow_head_size"] = unicode(self.ui.head_size.text())
		self.plugin.conf["arrow_threshold"] = unicode(self.ui.threshold.text())
		self.plugin.conf["arrow_outshape"] = unicode(self.ui.outshape.text())
		self.plugin.conf["arrow_outshape_encoding"] = unicode(self.encoding)
		open(unicode(self.ui.outshape.text()) + ".conf", "w").write(repr(self.plugin.conf))
		# Accept
		QtGui.QDialog.accept(self)
	
	def browse_outshape(self):
		fileDialog = QgsEncodingFileDialog(self, "Save file", "", "Shapefiles (*.shp)", "CP1252")
		fileDialog.setDefaultSuffix("shp")
		fileDialog.setFileMode(fileDialog.AnyFile)
		fileDialog.setAcceptMode(fileDialog.AcceptSave)
		if fileDialog.exec_() == fileDialog.Accepted:
			self.ui.outshape.clear()
			self.ui.outshape.setText(fileDialog.selectedFiles()[0])
			self.encoding = fileDialog.encoding()
			conf_path = unicode(fileDialog.selectedFiles()[0]) + ".conf"
			if os.path.isfile(conf_path) and QtGui.QMessageBox.question(self, "Arrows", "Load existing arrows configuration?", QtGui.QMessageBox.Yes, QtGui.QMessageBox.No, QtGui.QMessageBox.NoButton) == QtGui.QMessageBox.Yes:
				conf = eval(open(conf_path).read())
				del conf["arrow_outshape"]
				self.plugin.conf.update(conf)
				self.load_conf()
	
	def check_inshape(self):
		selected_index = self.ui.inshape.currentIndex()
		if selected_index != -1:
			self.selected_linelayer = self.plugin.linelayers[selected_index]
			fields = [field.name() for field in self.selected_linelayer.dataProvider().fields()]
			self.ui.width_field.clear()
			self.ui.width_field.addItems(fields)
			self.ui.width_field_forward.clear()
			self.ui.width_field_forward.addItems(fields)
			self.ui.width_field_backward.clear()
			self.ui.width_field_backward.addItems(fields)
	
	def guess_backward(self):
		guess_field = unicode(self.ui.width_field_forward.currentText()) + "_r"
		guess_index = self.ui.width_field_backward.findText(guess_field)
		if guess_index != -1:
			self.ui.width_field_backward.setCurrentIndex(guess_index)
	
	def radio_change(self):
		if self.ui.one_arrow.isChecked():
			self.ui.width_field.setEnabled(True)
			self.ui.double_head.setEnabled(True)
			self.ui.width_field_forward.setEnabled(False)
			self.ui.width_field_backward.setEnabled(False)
		else:
			self.ui.width_field.setEnabled(False)
			self.ui.double_head.setEnabled(False)
			self.ui.width_field_forward.setEnabled(True)
			self.ui.width_field_backward.setEnabled(True)
	
	def guess_head_size(self):
		# Calculate an appropriate head size based on arrows lenghts
		lengths = [feature.geometry().length() for feature in self.selected_linelayer.dataProvider().getFeatures() if feature.isValid()]
		if len(lengths):
			lengths_mean = sum(lengths) / len(lengths)
			lengths_min = min(lengths)
			head_length = min(lengths_mean / 15, lengths_min / 4)
		else:
			head_length = 1
		self.ui.head_size.setText(str(head_length))
