"""
/***************************************************************************
Arrows
								A QGIS plugin
Draw polygon arrows based on polylines
							-------------------
		begin                : 2012-03-06
		copyright            : (C) 2012 by Gregoire Piffault
		email                : gregoire.piffault@gmail.com
***************************************************************************/

/***************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/
"""

import os.path
# Import the PyQt and QGIS libraries
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
# Initialize Qt resources from file resources.py
import resources
# Import the code for the dialog
from arrowsdialog import ArrowsDialog
# Arrow drawing package
import arrowtrace

class Arrows:

	def __init__(self, iface):
		# Save reference to the QGIS interface
		self.iface = iface
		# Vars
		self.conf = {}

	def initGui(self):
		# Create action that will start plugin configuration
		self.action_set = QAction(QIcon(":/plugins/arrows/arrows_set.png"), "Arrows set", self.iface.mainWindow())
		self.action_rf = QAction(QIcon(":/plugins/arrows/arrows_rf.png"), "Arrows refresh", self.iface.mainWindow())
		self.action_rf.setEnabled(False)
		# Connect actions
		QObject.connect(self.action_set, SIGNAL("triggered()"), self.run_set)
		QObject.connect(self.action_rf, SIGNAL("triggered()"), self.run_rf)
		# Add toolbar button and menu item
		self.iface.addToolBarIcon(self.action_set)
		self.iface.addToolBarIcon(self.action_rf)
		self.iface.addPluginToMenu("&Arrows", self.action_set)
		self.iface.addPluginToMenu("&Arrows", self.action_rf)

	def unload(self):
		# Remove the plugin menu item and icon
		self.iface.removePluginMenu("&Arrows",self.action_set)
		self.iface.removePluginMenu("&Arrows",self.action_rf)
		self.iface.removeToolBarIcon(self.action_set)
		self.iface.removeToolBarIcon(self.action_rf)

	def run_set(self):
		layers = QgsMapLayerRegistry.instance().mapLayers()
		# retrieve line layers
		self.linelayers = [layer for layer in layers.itervalues() if layer.type() == QgsMapLayer.VectorLayer and layer.geometryType() == QGis.Line]
		# create and show the dialog
		dlg = ArrowsDialog(self)
		# show the dialog
		dlg.show()
		result = dlg.exec_()
		# See if OK was pressed
		if result == 1:
			self.action_rf.setEnabled(True)
			# Remove output shape from map if opened
			for opened_id in [id for id, layer in layers.items() if layer.source() == self.conf["arrow_outshape"]]:
				QgsMapLayerRegistry.instance().removeMapLayer(opened_id)
			# Create new layer
			self.draw_arrows()
	
	def run_rf(self):
		# Remove output shape from map if opened
		for opened_id in [id for id, layer in QgsMapLayerRegistry.instance().mapLayers().items() if layer.source() == self.conf["arrow_outshape"]]:
			QgsMapLayerRegistry.instance().removeMapLayer(opened_id)
		# Create new layer
		self.draw_arrows()
	
	def draw_arrows(self):
		# Open polyline layer for reading
		selected_linelayer = self.linelayers[self.conf["arrow_inshape_index"]]
		provider = selected_linelayer.dataProvider()
		# Open outshape for writing
		old_fields = provider.fields()
		if self.conf["arrow_one_arrow"]:
			field_count = len(old_fields)
		else:
			field_count = len(old_fields) / 2 + 1
		new_fields = QgsFields()
		for i in range (field_count):
			new_fields.append( provider.fields()[i] )
		new_fields.append( QgsField("Label_x", QVariant.Double) )
		new_fields.append( QgsField("Label_y", QVariant.Double) )
		new_fields.append( QgsField("Label_ang", QVariant.Double) )
		new_fields.append( QgsField("Label_h", QVariant.String) )
		new_fields.append( QgsField("Label_v", QVariant.String) )
		writer = QgsVectorFileWriter(self.conf["arrow_outshape"], 
									self.conf["arrow_outshape_encoding"], 
									new_fields, 
									QGis.WKBPolygon, 
									selected_linelayer.crs(), 
									"ESRI Shapefile")
		# Calculate an appropriate head size based on arrows lenghts
		lengths = [feature.geometry().length() for feature in provider.getFeatures()]
		lengths_mean = sum(lengths) / len(lengths)
		lengths_min = min(lengths)
		head_length = min(lengths_mean / 15, lengths_min / 4)
		# Retrieve common values
		head_length = float(unicode(self.conf["arrow_head_size"]))
		threshold = float(unicode(self.conf["arrow_threshold"])) * float(unicode(self.conf["arrow_width_factor"]))
		# Process features
		for feature in provider.getFeatures():
			# Retrieve attributes
			attributes = feature.attributes()
			# Retrieve geometry
			line = feature.geometry().asPolyline()			
			# One arrow
			if self.conf["arrow_one_arrow"]:
				width = attributes[self.conf["arrow_width_field"]] * float(unicode(self.conf["arrow_width_factor"]))
				if width >= threshold:
					arc, label = qgs_arrow(line, width, head_length, 0, self.conf["arrow_double_head"])
					# arc.addAttribute(0, attributes[0])
					# arc.addAttribute(1, attributes[1])
					# for i in range(2, field_count):
						# arc.addAttribute(i, QVariant(attributes[i].toFloat()[0] + attributes[i + field_count - 2].toFloat()[0]))
					arc.initAttributes( field_count+5 )
					for i, v in enumerate(attributes[:field_count]):
						arc.setAttribute(i, v)
					# Label fields
					for i in range(3):
						arc.setAttribute(field_count + i, label[i])
					arc.setAttribute(field_count + 3, "center")
					arc.setAttribute(field_count + 4, "half")
					# Write feature
					writer.addFeature(arc)
			# Two arrows
			else:
				width_f = attributes[self.conf["arrow_width_field_forward"]] * float(unicode(self.conf["arrow_width_factor"]))
				width_b = attributes[self.conf["arrow_width_field_backward"]] * float(unicode(self.conf["arrow_width_factor"]))
				if width_f >= threshold and width_b >= threshold:
					offset = 1
				else:
					offset = 0
				if width_f >= threshold:
					arc, label = qgs_arrow(line, width_f, head_length, offset * (width_f / 2. + head_length / 4.), False)
					arc.initAttributes( field_count+5 )
					for i, v in enumerate(attributes[:field_count]):
						arc.setAttribute(i, v)
					# Label fields
					for i in range(3):
						arc.setAttribute(field_count + i, label[i])
					arc.setAttribute(field_count + 3, "center")
					arc.setAttribute(field_count + 4, "half")
					# Write feature
					writer.addFeature(arc)
				
				if width_b >= threshold:
					arc, label = qgs_arrow(line[::-1], width_b, head_length, offset * (width_b / 2. + head_length / 4.), False)
					arc.initAttributes( field_count+5 )
					for i, v in enumerate(attributes[1::-1] + attributes[field_count:]):
						arc.setAttribute(i, v)
					# Label fields
					for i in range(3):
						arc.setAttribute(field_count + i, label[i])
					arc.setAttribute(field_count + 3, "center")
					arc.setAttribute(field_count + 4, "half")
					# Write feature
					writer.addFeature(arc)
		# Write shape to disk
		del writer
		# Add layer to canvas
		new_layer = QgsVectorLayer(self.conf["arrow_outshape"], os.path.basename(unicode(self.conf["arrow_outshape"])).rstrip(".shp"), "ogr")
		QgsMapLayerRegistry.instance().addMapLayer(new_layer)

def qgs_arrow(points, width, head_length, offset=0, double_head=False):
	points, label = arrowtrace.arrow(points, width, head_length, offset, double_head, return_label=True)
	feature = QgsFeature()
	feature.setGeometry(QgsGeometry.fromPolygon([[QgsPoint(point[0], point[1]) for point in points]]))
	return feature, label
