import math
from pointstocircle import pointstocircle

def arrow(points, width, head_length, offset=0, double_head=False, return_label=False):
	"""Takes up to 3 points and return the outline of an arrow with
	given width and offset (offset on the right).
	With 3 distinct points, the arrow follows an arc.
	"""
	po = points[0]
	pm = points[len(points) // 2]
	pd = points[-1]
	circle = pointstocircle(po, pm, pd)
	# Straight arrow
	if circle is None:
		result = arrow_straight(po, pd, width, head_length, offset, double_head)
		if return_label:
			l_angle = math.atan2(pd[1] - po[1], pd[0] - po[0]) * 180 / math.pi
			if len(points) == 2:
				result = [result, ((po[0] + pd[0]) / 2., (po[1] + pd[1]) / 2., l_angle)]
			else:
				result = [result, (pm[0], pm[1], l_angle)]
	# Curved arrow
	else:
		result = arrow_curved(po, pm, pd, circle, width, head_length, offset, double_head)
		if return_label:
			l_angle = math.atan2(pm[1] - circle["y"], pm[0] - circle["x"])
			if l_angle > math.pi:
				l_angle -= math.pi / 2
			else:
				l_angle += math.pi / 2
			result = [result, (pm[0], pm[1],l_angle  * 180 / math.pi)]
	return result

def arrow_straight(po, pd, width, head_length, offset, double_head):
	head_overwidth = head_length / 2.
	po = complex(po[0], po[1])
	pd = complex(pd[0], pd[1])
	dir_vect = (pd - po) / abs(pd - po)
	norm_vect = dir_vect * -1j
	# Offseting po and pd (with an additional half head_length offset from extremities)
	po = po + norm_vect + dir_vect * head_length / 2.
	pd = pd + norm_vect - dir_vect * head_length / 2.
	
	points = [pd]
	points += [pd - dir_vect * head_length - norm_vect * (head_overwidth + width / 2.)]
	points += [pd - dir_vect * head_length - norm_vect * width / 2.]
	if double_head:
		points += [po + dir_vect * head_length - norm_vect * width / 2.]
		points += [po + dir_vect * head_length - norm_vect * (head_overwidth + width / 2.)]
		points += [po]
		points += [po + dir_vect * head_length + norm_vect * (head_overwidth + width / 2.)]
		points += [po + dir_vect * head_length + norm_vect * width / 2.]
	else:
		points += [po - norm_vect * width / 2.]
		points += [po + norm_vect * width / 2.]
	points += [pd - dir_vect * head_length + norm_vect * width / 2.]
	points += [pd - dir_vect * head_length + norm_vect * (head_overwidth + width / 2.)]
	points += [pd]
	points = [point + norm_vect * offset for point in points]
	return [(point.real, point.imag) for point in points]

def circle_point(circle, angle, r_offset=0.0):
	r = circle["r"] + r_offset
	return (circle["x"] + r * math.cos(angle), circle["y"] + r * math.sin(angle))

def chord_angle(circle, length):
	return length / (2 * math.pi * circle["r"]) * 2 * math.pi

def arrow_curved(po, pm, pd, circle, width, head_length, offset, double_head):
	head_overwidth = head_length / 2.
	# Calculate angles
	angle_o = math.atan2(po[1]-circle["y"], po[0]-circle["x"])
	angle_m = math.atan2(pm[1]-circle["y"], pm[0]-circle["x"])
	angle_d = math.atan2(pd[1]-circle["y"], pd[0]-circle["x"])
	# Calculate arc direction
	if (angle_m - angle_o) % (2 * math.pi) < (angle_m - angle_d) % (2 * math.pi):
		direction = 1
	else:
		direction = -1
	# Defining offset circle
	base_circle = circle.copy()
	base_circle["r"] += direction * offset
	# Circle related head dimensions
	head_angle = direction * math.atan(head_length / base_circle["r"])
	head_point_offset = (head_length ** 2 + base_circle["r"] ** 2) ** 0.5 - base_circle["r"]
	# Additional half head_length offset from extremities
	angle_o += head_angle / 2.
	angle_d -= head_angle / 2.
	# Circle segments
	angle_delta = (angle_d - angle_o) % (direction * 2 * math.pi)
	if double_head:
		angle_delta -= 2 * head_angle
	else:
		angle_delta -= head_angle
	min_angle = max(-0.1, min(0.1, head_angle))
	link_count = max(1, int(abs(math.ceil(angle_delta / min_angle))))
	link_angle = angle_delta / link_count
	angles = [angle_d - head_angle - i * link_angle for i in range(link_count + 1)]
	
	# Destination head part 1
	points = [circle_point(base_circle, angle_d, head_point_offset)]
	points += [circle_point(base_circle, angle_d - head_angle, -direction * (head_overwidth + width / 2))]
	# Inner border
	points += [circle_point(base_circle, angle, -direction * width / 2) for angle in angles]
	if double_head:
		# Origin head
		points += [circle_point(base_circle, angle_o + head_angle, -direction * (head_overwidth + width / 2))]
		points += [circle_point(base_circle, angle_o, head_point_offset)]
		points += [circle_point(base_circle, angle_o + head_angle, direction * (head_overwidth + width / 2))]
	# Outter border
	points += [circle_point(base_circle, angle, direction * width / 2) for angle in angles[::-1]]
	# Destination head part 2
	points += [circle_point(base_circle, angle_d - head_angle, direction * (head_overwidth + width / 2))]
	points += [circle_point(base_circle, angle_d, head_point_offset)]
	return points
