# Copyright 2006-2007 by ARPAT-SIRA (www.arpat.toscana.it http://sira.arpat.toscana.it/sira/)
# Licenced under GNU GPL (General Public License) v2
# Developed by Martin Dobias for Faunalia (www.faunalia.it)
# For details send a mail to PFR_SIRA@arpat.toscana.it
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation (www.fsf.org); version 2 of the License, or any later version.
#
#
# Copyright 2006-2007 di ARPAT-SIRA (www.arpat.toscana.it http://sira.arpat.toscana.it/sira/)
# Rilasciato sotto licenza GNU GPL (General Public License) v2
# Sviluppato da Martin Dobias per Faunalia (www.faunalia.it)
# Per ulteriori informazioni inviare una mail a PFR_SIRA@arpat.toscana.it
#
# Questo programma rientra nel software libero; possibile redistribuirlo e/o modificarlo
# sotto i termini della licenza GNU General Public License come pubblicato da
# Free Software Foundation (www.fsf.org); versione 2 o superiore della licenza.
#


from qgis.core import *

"""
sondaggio is a dictionary:
- key - survey id (SOND_ID)
- value - instance of class SurveyInfo:
            - name (SIGLA)
            - quota above sea level in meters (QPC)
            - x,y coordinates of point (stored in QgsPoint)
            - array with instances of StrataInfo (ordered by STRATO_ID)

geo and idro are dictionaries:
- key - id (GEO_ID)
- value - tuple consisting of description and preferred colour

"""

class SurveyInfo:
    
    def __init__(self, id=0, name='', qpc=0, coord=None, strata=None):
        self.id = id
        self.name = name
        self.qpc = qpc
        self.coord = coord
        if strata == None:
            strata = []
        self.strata = strata
    def __repr__(self):
        return "SURVEY(%d, %f)" % (self.id, self.qpc)
        #return "SURVEY(%d, '%s', %f, %s)" % (self.id, unicode(self.name), self.qpc, str(self.strata))

class StrataInfo:
    def __init__(self, id=0, desc='', depthTop=0, depthBed=0, type=None, geoId=None, hydroId=None):
        self.id = id
        self.description = desc
        self.depthTop = depthTop
        self.depthBed = depthBed
        self.type = type
        self.geoId = geoId
        self.hydroId = hydroId
    def __repr__(self):
        return "strata(%d, G%d, H%d, '%s', %s, %f-%f)" % (self.id, self.geoId, self.hydroId, self.description, self.type, self.depthTop, self.depthBed)

def getTestData():
    """ Returns a tuple containing sample data """

    strata = [
        StrataInfo(1,'terreno vegetale argillo sabbioso di colore marrone', 0  , 1.2, 'T', 2, 3),
        StrataInfo(2,'argille sabbiose'                                   , 1.2, 3  , 'C', 2, 3),
        StrataInfo(3,'sabbia di colore giallastro'                        , 3  , 5.8, 'F', 2, 2),
        StrataInfo(4,'successione di ghiaie arenarie di colore grigio ...', 5.8,10.2, 'F', 2, 2),
        StrataInfo(5,'limi sabbiosi debolmente argillosi di colore gia...',10.2,11.5, 'T', 2, 2)
    ]
    surveys = { 90014 : SurveyInfo(90014,'ASA_PM1', 19, strata, QgsPoint(10,10)) }

    geo = { 0: ('substrato pre pleistocene medio', 'grigio'),
            1: ('depositi terrazzati del pleistocene medio', 'rosso'),
            2: ('depositi alluvionali e marino costieri del pleistocene superiore', 'blu')
            }

    hydro = { 1: ('substrato', 'giallo'),
              2: ('acquifero', 'ciano'),
              3: ('copertura', 'verde'),
              4: ('interstrato', 'grigio')
            }

    return (surveys, geo, hydro)

class DataError(Exception):
    """ exception raised on problems accessing data """

    def __init__(self, problem):
        self.problem = problem
    def __str__(self):
        return self.problem

class DataSanityError(Exception):
    """ exception raised when data don't comply with some constraints """
    
    def __init__(self, sond_id, message):
        self.sond_id = sond_id
        self.message = message
    def __str__(self):
        return "Sond_ID %d: %s" % (self.sond_id, self.message)

class SurveyStore:
    def __init__(self, path):
		
        # load geo and idro DBFs
        # they're small - can be read whole and stored in memory
        self.geo = self.loadGeoIdro(path, "geo")
        self.idro = self.loadGeoIdro(path, "idro")

        # load database strato, but don't read it
        self.strato = self.loadDbf(path, "strato")

        # load database of layer sondaggio 
        self.sondaggio = self.loadDbf(path, "sondaggio")
        
        # cache is empty on startup, will be filled in on first query
        self.cacheStratoFids = None

    def loadDbf(self, path, name):
        lyr = QgsVectorLayer(path + "/" + name + ".dbf", name, "ogr")
        if not lyr.isValid():
            raise DataError,  "Couldn't open %s.dbf" % name
        
        # use this encoding to load correctly some characters
        lyr.setProviderEncoding("iso-8859-15")

        return lyr

    def loadGeoIdro(self, path, name):
        """ load geo/idro database to a dictionary """

        lyr = self.loadDbf(path, name)

        items = {}
        f = QgsFeature()

        provider = lyr.dataProvider()
        provider.enableGeometrylessFeatures(True)
        # get all features with all attributes, without geometry
        provider.select(provider.attributeIndexes(), QgsRectangle(), False)
        
        while provider.nextFeature(f):
            att = f.attributeMap()
            id = att[0].toInt()[0]
            items[id] = (unicode(att[1].toString()), unicode(att[2].toString()))

        return items


    def createCacheStratoFids(self):
        """ creates dictionary with sond_id as keys, containing array of feature IDs
            in strato.dbf which are connected with the sond_id.
            this function is called on first query for data """
        
        self.cacheStratoFids = {}
        f = QgsFeature()
        provider = self.strato.dataProvider()
        provider.enableGeometrylessFeatures(True)
        # get only sond_id field from the database
        provider.select([7], QgsRectangle(), False)
        
        while provider.nextFeature(f):
            
            fid = f.id()
            attrMap = f.attributeMap()
            
            sond_id = attrMap[7].toInt()[0]
            
            if self.cacheStratoFids.has_key(sond_id):
                self.cacheStratoFids[sond_id].append(fid)
            else:
                self.cacheStratoFids[sond_id] = [fid]
 	
        
    def getData(self, featureIds):
        """ get data from databases for array of features specified by their IDs  """

        surveys = self._getDataStep1(featureIds)
        surveys = self._getDataStep2(surveys)
        
        self.sanityCheck(surveys)
        
        return surveys
        
        
    def _getDataStep1(self, featureIds):
        """ STEP 1: get data from sondaggio DB """
        
        surveys = {}
        f = QgsFeature()
        provider = self.sondaggio.dataProvider()
        attrs = [0,1,2] # SOND_ID, SIGLA, QPC
        
        for fid in featureIds:
            
            if not provider.featureAtId(fid, f, True, attrs):
                raise DataError, "Couldn't find feature ID %d in sondaggio.shp!" % fid
            
            attrMap = f.attributeMap()

            # get attribute values
            sond_id = attrMap[0].toInt()[0]
            name = unicode(attrMap[1].toString())
            qpc = attrMap[2].toDouble()[0]

            # get coordinates
            coord = f.geometry().asPoint()
            
            # add to array
            surveys[sond_id] = SurveyInfo(sond_id, name, qpc, coord)
            
        return surveys


    def _getDataStep2(self, surveys):
        """ STEP 2: get strata information for every point """
        
        # create the cache if not yet done
        if self.cacheStratoFids == None:
            self.createCacheStratoFids()
        
        f = QgsFeature()
        provider = self.strato.dataProvider()
        attrs = [0,1,2,3,4,5,6,7] # STRATO_ID, DESC, TIPO, GEO_ID, IDRO_ID, DAPC_TETT, DAPC_LETT, SOND_ID
        
        
        for (sond_id, survey) in surveys.iteritems():
            
            if not self.cacheStratoFids.has_key(sond_id):
                raise DataSanityError(sond_id, "No strata information")

            # now fetch all important strata information
            for fid in self.cacheStratoFids[sond_id]:
                
                if not provider.featureAtId(fid, f, False, attrs):
                    raise DataError, "Couldn't find feature ID %d in strata.dbf!" % fid
                
                attrMap = f.attributeMap()
                
                if sond_id != attrMap[7].toInt()[0]:
                    raise DataError, "Incorrectly cached sond_id!"

                # parse attributes
                id = attrMap[0].toInt()[0]
                desc = unicode(attrMap[1].toString())
                tipo = unicode(attrMap[2].toString())
                tett = attrMap[5].toDouble()[0]
                lett = attrMap[6].toDouble()[0]
                
                # geo_id, idro_id can be null
                if attrMap[3].toString().trimmed().isEmpty():
                    geo_id = -1
                else:
                    geo_id = attrMap[3].toInt()[0]
                if attrMap[4].toString().trimmed().isEmpty():
                    idro_id = -1
                else:
                    idro_id = attrMap[4].toInt()[0]

                st = StrataInfo(id, desc, tett, lett, tipo, geo_id, idro_id)

                # add strata information (in right order) 
                insertAt = 0
                for a in survey.strata:
                    if a.id > id:
                        break
                    insertAt += 1
                survey.strata.insert(insertAt, st)
                        
        return surveys
            
    def sanityCheck(self, surveys):
        """ does a sanity check on retreived data """
        
        for (sond_id, survey) in surveys.iteritems():
            
            # check whether there's at least one strata information
            if len(survey.strata) == 0:
                raise DataSanityError(sond_id, "No strata information")
            
            # check whether the depths are valid
            top1 = survey.strata[0].depthTop
            bed1 = survey.strata[0].depthBed
            for strato in survey.strata[1:]:
                # top (n) < top (n+1)
                if top1 > strato.depthTop:
                    raise DataSanityError(sond_id, "Top depth is incorrect (%.2f > %.2f)" % (top1, strato.depthTop))
                # bed (n) < bed (n+1)
                if bed1 > strato.depthBed:
                    raise DataSanityError(sond_id, "Bed depth is incorrect (%.2f > %.2f)" % (bed1, strato.depthBed))
                # bed (n) = top (n+1)
                if bed1 != strato.depthTop:
                    raise DataSanityError(sond_id, "Top and bed depth don't match (%.2f != %.2f)" % (bed1, strato.depthTop))
                
                top1 = strato.depthTop
                bed1 = strato.depthBed
