import logging
from qgis.core import QgsVectorLayer, QgsProject, QgsDataSourceUri
from PyQt5.QtWidgets import QTreeWidgetItem
from PyQt5.QtCore import Qt
import requests
from xml.etree import ElementTree as ET

logger = logging.getLogger(__name__)

def load_wfs_layers(wfs_url, tree_widget):
    logger.info(f"Loading WFS layers from: {wfs_url}")
    
    # Ensure the URL ends with '?'
    if not wfs_url.endswith('?'):
        wfs_url += '?'
    
    # Request WFS GetCapabilities
    capabilities_url = f"{wfs_url}SERVICE=WFS&REQUEST=GetCapabilities&VERSION=2.0.0"
    try:
        response = requests.get(capabilities_url, timeout=10)
        response.raise_for_status()
    except requests.RequestException as e:
        logger.error(f"Failed to get WFS capabilities: {str(e)}")
        return

    # Parse the XML response
    try:
        root = ET.fromstring(response.content)
        ns = {'wfs': 'http://www.opengis.net/wfs/2.0', 'ows': 'http://www.opengis.net/ows/1.1'}
        feature_types = root.findall('.//wfs:FeatureType', ns)

        for feature_type in feature_types:
            name = feature_type.find('wfs:Name', ns).text
            title = feature_type.find('wfs:Title', ns).text if feature_type.find('wfs:Title', ns) is not None else name
            
            item = QTreeWidgetItem([name, title])
            item.setCheckState(0, Qt.Unchecked)
            item.setData(0, Qt.UserRole, wfs_url)
            tree_widget.addTopLevelItem(item)

        logger.info(f"Loaded {len(feature_types)} WFS layers")
    except ET.ParseError as e:
        logger.error(f"Failed to parse WFS capabilities: {str(e)}")
    except Exception as e:
        logger.error(f"Unexpected error while loading WFS layers: {str(e)}")

def add_wfs_layer(layer_name, wfs_url):
    logger.info(f"Adding WFS layer: {layer_name} from {wfs_url}")
    
    uri = QgsDataSourceUri()
    uri.setParam('url', wfs_url)
    uri.setParam('typename', layer_name)
    uri.setParam('version', '2.0.0')
    uri.setParam('srsname', 'EPSG:4326')
    
    layer = QgsVectorLayer(uri.uri(), layer_name, "WFS")
    
    if layer.isValid():
        QgsProject.instance().addMapLayer(layer)
        logger.info(f"Successfully added WFS layer: {layer_name}")
        return True
    else:
        logger.error(f"Failed to add WFS layer: {layer_name}")
        return False