import os
import logging
from qgis.core import QgsProject
from qgis.utils import iface
from PyQt5.QtWidgets import QAction, QMessageBox
from PyQt5.QtGui import QIcon
from PyQt5.QtCore import Qt
from .ui import QGISWebScraperDialog, QGISWebScraperLayerDialog
from .scraper import scrape_web_page
from .wms_utils import load_wms_layers, add_wms_layer
from .wfs_utils import load_wfs_layers, add_wfs_layer

logger = logging.getLogger(__name__)

class QGISWebScraper:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.dialog = None
        self.wms_layer_dialog = None
        self.wfs_layer_dialog = None
        self.layer_data = None

    def initGui(self):
        icon_path = 'resources/sol_de_mayo.png'
        self.action = QAction(QIcon(icon_path), "ArgentinaGeoServices", self.iface.mainWindow())
        self.action.triggered.connect(self.run)
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&ArgentinaGeoServices", self.action)

    def unload(self):
        self.iface.removePluginMenu("&ArgentinaGeoServices", self.action)
        self.iface.removeToolBarIcon(self.action)

    def run(self):
        logger.info("Plugin run method called")
        if self.dialog is None:
            logger.debug("Creating new QGISWebScraperDialog")
            self.dialog = QGISWebScraperDialog()
            self.dialog.importWMSButton.clicked.connect(self.import_selected_wms_layers)
            self.dialog.importWFSButton.clicked.connect(self.import_selected_wfs_layers)
            self.dialog.checkStatusButton.clicked.connect(self.check_services_status)

        logger.info("Scraping web page for layer data")
        self.layer_data = scrape_web_page()
        if self.layer_data is not None:
            logger.info(f"Layer data retrieved. Number of datasets: {len(self.layer_data)}")
            self.dialog.populate_tabs_with_layers(self.layer_data)
        else:
            logger.error("Failed to retrieve layer data")
            QMessageBox.warning(None, "Data Retrieval Error", "Failed to retrieve data from some or all sources.")

        logger.info("Showing dialog")
        self.dialog.show()

    def check_services_status(self):
        logger.info("Checking services status")
        self.dialog.check_selected_services_status()

    def import_selected_wms_layers(self):
        logger.info("Importing selected WMS layers")
        selected_items = self.dialog.get_selected_items('WMS')

        if not selected_items:
            logger.warning("No WMS layers selected")
            QMessageBox.warning(None, "No Selection", "No WMS layers selected.")
            return

        if self.wms_layer_dialog is None:
            logger.debug("Creating new QGISWebScraperLayerDialog for WMS")
            self.wms_layer_dialog = QGISWebScraperLayerDialog("WMS Layers")
            self.wms_layer_dialog.addLayerButton.clicked.connect(self.add_selected_wms_layers)

        self.wms_layer_dialog.layerTreeWidget.clear()
        for item in selected_items:
            wms_link = item.data(0, Qt.UserRole)[0]
            logger.debug(f"Loading WMS layers for link: {wms_link}")
            load_wms_layers(wms_link, self.wms_layer_dialog.layerTreeWidget)

        logger.info("Showing WMS layer dialog")
        self.wms_layer_dialog.show()

    def import_selected_wfs_layers(self):
        logger.info("Importing selected WFS layers")
        selected_items = self.dialog.get_selected_items('WFS')

        if not selected_items:
            logger.warning("No WFS layers selected")
            QMessageBox.warning(None, "No Selection", "No WFS layers selected.")
            return

        if self.wfs_layer_dialog is None:
            logger.debug("Creating new QGISWebScraperLayerDialog for WFS")
            self.wfs_layer_dialog = QGISWebScraperLayerDialog("WFS Layers")
            self.wfs_layer_dialog.addLayerButton.clicked.connect(self.add_selected_wfs_layers)

        self.wfs_layer_dialog.layerTreeWidget.clear()
        for item in selected_items:
            wfs_link = item.data(0, Qt.UserRole)[1]
            logger.debug(f"Loading WFS layers for link: {wfs_link}")
            load_wfs_layers(wfs_link, self.wfs_layer_dialog.layerTreeWidget)

        logger.info("Showing WFS layer dialog")
        self.wfs_layer_dialog.show()

    def add_selected_wms_layers(self):
        logger.info("Adding selected WMS layers")
        selected_items = self.wms_layer_dialog.get_selected_items()

        for item in selected_items:
            layer_name = item.text(0)
            wms_link = item.data(0, Qt.UserRole)
            if not wms_link:
                logger.warning(f"No WMS link found for layer {layer_name}")
                QMessageBox.critical(None, "Error", f"No WMS link found for layer {layer_name}.")
                continue

            logger.debug(f"Adding WMS layer: {layer_name} from {wms_link}")
            add_wms_layer(layer_name, wms_link)

        logger.info("Finished adding WMS layers")

    def add_selected_wfs_layers(self):
        logger.info("Adding selected WFS layers")
        selected_items = self.wfs_layer_dialog.get_selected_items()

        for item in selected_items:
            layer_name = item.text(0)
            wfs_link = item.data(0, Qt.UserRole)
            if not wfs_link:
                logger.warning(f"No WFS link found for layer {layer_name}")
                QMessageBox.critical(None, "Error", f"No WFS link found for layer {layer_name}.")
                continue

            logger.debug(f"Adding WFS layer: {layer_name} from {wfs_link}")
            add_wfs_layer(layer_name, wfs_link)

        logger.info("Finished adding WFS layers")