import logging
from PyQt5 import QtCore, QtGui, QtWidgets
from PyQt5.QtWidgets import QDialog, QTabWidget, QWidget, QVBoxLayout, QHBoxLayout, QTreeWidgetItem
import pandas as pd
from .service_checker import check_service_availability

logger = logging.getLogger(__name__)

class QGISWebScraperDialog(QDialog):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("ArgentinaGeoServices")
        self.setWindowIcon(QtGui.QIcon('resources/sol_de_mayo.png'))
        self.resize(700, 500)

        layout = QVBoxLayout(self)
        self.tabWidget = QTabWidget()
        layout.addWidget(self.tabWidget)

        buttonLayout = QHBoxLayout()
        self.importWMSButton = QtWidgets.QPushButton("Import Selected WMS")
        self.importWFSButton = QtWidgets.QPushButton("Import Selected WFS")
        self.checkStatusButton = QtWidgets.QPushButton("Check Selected Services Status")
        buttonLayout.addWidget(self.importWMSButton)
        buttonLayout.addWidget(self.importWFSButton)
        buttonLayout.addWidget(self.checkStatusButton)
        layout.addLayout(buttonLayout)

    def populate_tabs_with_layers(self, layer_data):
        logger.info("Populating tabs with layers")
        self.tabWidget.clear()
        tab_names = ["Organismos Nacionales", "Organismos Provinciales", "Organismos Municipales", "Universidades", "Empresas"]

        for i, tab_name in enumerate(tab_names):
            logger.debug(f"Processing tab: {tab_name}")
            tab_widget = QWidget()
            tab_layout = QVBoxLayout(tab_widget)
            tree_widget = QtWidgets.QTreeWidget()

            if i in layer_data and layer_data[i] is not None:
                data = layer_data[i]
                logger.debug(f"Data for tab {tab_name}: {data.shape} rows, columns: {data.columns}")
                
                columns_to_display = [col for col in data.columns if col.upper() != 'WCS']
                header_labels = columns_to_display + ["Status"]
                tree_widget.setHeaderLabels(header_labels)

                for _, row in data.iterrows():
                    values = []
                    for col in columns_to_display:
                        if col in row and pd.notna(row[col]):
                            values.append(str(row[col]))
                        else:
                            values.append("No service available" if col in ['WMS', 'WFS'] else "")
                    values.append("")  # For Status column
                    item = QTreeWidgetItem(values)
                    item.setCheckState(0, QtCore.Qt.Unchecked)
                    wms_link = row.get('WMS', '') if pd.notna(row.get('WMS', '')) else ''
                    wfs_link = row.get('WFS', '') if pd.notna(row.get('WFS', '')) else ''
                    item.setData(0, QtCore.Qt.UserRole, (wms_link, wfs_link))
                    tree_widget.addTopLevelItem(item)
                logger.debug(f"Added {tree_widget.topLevelItemCount()} items to tree widget for tab {tab_name}")
            else:
                logger.warning(f"No data available for tab: {tab_name}")
                tree_widget.setHeaderLabels(['Info', 'Status'])
                item = QTreeWidgetItem([f"No data available for {tab_name}.", ""])
                tree_widget.addTopLevelItem(item)

            tab_layout.addWidget(tree_widget)
            self.tabWidget.addTab(tab_widget, tab_name)
        
        logger.info(f"Populated {self.tabWidget.count()} tabs with layers")

    def check_selected_services_status(self):
        logger.info("Checking selected services status")
        for i in range(self.tabWidget.count()):
            tree_widget = self.tabWidget.widget(i).findChild(QtWidgets.QTreeWidget)
            status_column = tree_widget.columnCount() - 1
            for j in range(tree_widget.topLevelItemCount()):
                item = tree_widget.topLevelItem(j)
                if item.checkState(0) == QtCore.Qt.Checked:
                    wms_link, wfs_link = item.data(0, QtCore.Qt.UserRole)
                    if wms_link or wfs_link:
                        logger.debug(f"Checking services status for WMS: {wms_link}, WFS: {wfs_link}")
                        status = check_service_availability(wms_link, wfs_link)
                        item.setText(status_column, status)
                    else:
                        logger.warning(f"No service links found for selected item {j} in tab {i}")
                else:
                    item.setText(status_column, '')

    def get_selected_items(self, service_type):
        logger.info(f"Getting selected {service_type} items")
        selected_items = []
        for i in range(self.tabWidget.count()):
            tree_widget = self.tabWidget.widget(i).findChild(QtWidgets.QTreeWidget)
            for j in range(tree_widget.topLevelItemCount()):
                item = tree_widget.topLevelItem(j)
                if item.checkState(0) == QtCore.Qt.Checked:
                    wms_link, wfs_link = item.data(0, QtCore.Qt.UserRole)
                    if service_type == 'WMS' and wms_link:
                        selected_items.append(item)
                    elif service_type == 'WFS' and wfs_link:
                        selected_items.append(item)
        logger.debug(f"Found {len(selected_items)} selected {service_type} items")
        return selected_items

class QGISWebScraperLayerDialog(QDialog):
    def __init__(self, title):
        super().__init__()
        self.setWindowTitle(title)
        self.resize(400, 300)

        layout = QVBoxLayout(self)
        self.layerTreeWidget = QtWidgets.QTreeWidget()
        self.layerTreeWidget.setHeaderLabels(["Layer Name", "Title"])
        layout.addWidget(self.layerTreeWidget)

        self.addLayerButton = QtWidgets.QPushButton("Add Selected Layers")
        layout.addWidget(self.addLayerButton)

    def get_selected_items(self):
        selected_items = []
        for i in range(self.layerTreeWidget.topLevelItemCount()):
            item = self.layerTreeWidget.topLevelItem(i)
            if item.checkState(0) == QtCore.Qt.Checked:
                selected_items.append(item)
        return selected_items