# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AreaLimiterDialog
                                 A QGIS plugin
 Keeps polygons within a target area
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-10-18
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Jessie Lindsay
        email                : jessie.lindsay42@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
# -*- coding: utf-8 -*-
from qgis.PyQt.QtWidgets import QDockWidget
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtCore import Qt
from qgis.core import (
    QgsProject,
    QgsGeometry,
    QgsWkbTypes,
    QgsVectorLayer,
    QgsPointXY,
    QgsFeature,
    QgsVertexId,
)
from qgis.gui import QgsMapTool, QgsRubberBand, QgsVertexMarker
from qgis.utils import iface

from .arealimiter_dockwidget_base import Ui_AreaLimiterDockWidgetBase


# -------------------------------------------------------------------------
# Custom map tool for vertex editing with area limit
# -------------------------------------------------------------------------
class VertexEditMapTool(QgsMapTool):
    """Custom map tool to edit polygon vertices with area limit."""

    def __init__(self, canvas, layer, feature, target_area_m2, update_area_callback):
        super().__init__(canvas)
        self.canvas = canvas
        self.layer = layer
        self.feature = feature
        self.target_area_m2 = target_area_m2
        self.update_area_callback = update_area_callback
        
        # Visual elements
        self.rubber_band = QgsRubberBand(self.canvas, QgsWkbTypes.PolygonGeometry)
        self.rubber_band.setColor(QColor(255, 0, 0, 100))
        self.rubber_band.setWidth(2)
        
        self.vertex_markers = []
        self.selected_vertex_idx = None
        self.dragging = False
        self.adding_vertex = False
        
        # Get original geometry
        self.original_geom = feature.geometry()
        if self.original_geom.isMultipart():
            self.points = list(self.original_geom.asMultiPolygon()[0][0])
        else:
            self.points = list(self.original_geom.asPolygon()[0])
        
        # Remove duplicate last point if exists
        if len(self.points) > 1 and self.points[0] == self.points[-1]:
            self.points.pop()
        
        self.setup_vertex_markers()
        self.update_rubberband()
        self.update_area_callback(self.calculate_area())

    def setup_vertex_markers(self):
        """Create markers for each vertex."""
        for marker in self.vertex_markers:
            self.canvas.scene().removeItem(marker)
        self.vertex_markers.clear()
        
        for point in self.points:
            marker = QgsVertexMarker(self.canvas)
            marker.setCenter(point)
            marker.setColor(QColor(0, 0, 255))
            marker.setIconSize(10)
            marker.setIconType(QgsVertexMarker.ICON_CIRCLE)
            marker.setPenWidth(2)
            self.vertex_markers.append(marker)

    def canvasPressEvent(self, event):
        if event.button() == Qt.RightButton:
            # Right click to finish
            self.finish_editing()
            return
        
        if event.button() == Qt.LeftButton:
            click_point = self.toMapCoordinates(event.pos())
            
            # First, check if clicking on existing vertex
            min_dist = float('inf')
            closest_idx = None
            vertex_threshold = 10  # pixels
            
            for idx, point in enumerate(self.points):
                vertex_pixel = self.toCanvasCoordinates(point)
                # Calculate distance manually
                dx = vertex_pixel.x() - event.pos().x()
                dy = vertex_pixel.y() - event.pos().y()
                dist = (dx * dx + dy * dy) ** 0.5
                if dist < vertex_threshold and dist < min_dist:
                    min_dist = dist
                    closest_idx = idx
            
            if closest_idx is not None:
                # Clicked on existing vertex - start dragging
                self.selected_vertex_idx = closest_idx
                self.dragging = True
                self.adding_vertex = False
                # Highlight selected vertex
                self.vertex_markers[closest_idx].setColor(QColor(255, 0, 0))
            else:
                # Check if clicking near an edge to add new vertex
                edge_threshold = 15  # pixels
                closest_edge_idx = None
                min_edge_dist = float('inf')
                insert_point = None
                
                for idx in range(len(self.points)):
                    p1 = self.points[idx]
                    p2 = self.points[(idx + 1) % len(self.points)]
                    
                    # Calculate distance from click to edge
                    edge_dist, point_on_edge = self.distance_to_segment(
                        click_point, p1, p2
                    )
                    
                    # Convert to pixel distance for threshold check
                    point_on_edge_pixel = self.toCanvasCoordinates(point_on_edge)
                    dx = point_on_edge_pixel.x() - event.pos().x()
                    dy = point_on_edge_pixel.y() - event.pos().y()
                    pixel_dist = (dx * dx + dy * dy) ** 0.5
                    
                    if pixel_dist < edge_threshold and pixel_dist < min_edge_dist:
                        min_edge_dist = pixel_dist
                        closest_edge_idx = idx
                        insert_point = point_on_edge
                
                if closest_edge_idx is not None and insert_point:
                    # Add new vertex on the edge
                    insert_idx = closest_edge_idx + 1
                    self.points.insert(insert_idx, insert_point)
                    self.setup_vertex_markers()
                    self.update_rubberband()
                    
                    # Start dragging the new vertex
                    self.selected_vertex_idx = insert_idx
                    self.dragging = True
                    self.adding_vertex = True
                    self.vertex_markers[insert_idx].setColor(QColor(0, 255, 0))
                    
                    # Update area
                    self.update_area_callback(self.calculate_area())

    def distance_to_segment(self, point, seg_start, seg_end):
        """
        Calculate the shortest distance from a point to a line segment.
        Returns (distance, closest_point_on_segment).
        """
        # Vector from seg_start to seg_end
        dx = seg_end.x() - seg_start.x()
        dy = seg_end.y() - seg_start.y()
        
        if dx == 0 and dy == 0:
            # Segment is a point
            return point.distance(seg_start), seg_start
        
        # Parameter t: projection of point onto line
        t = ((point.x() - seg_start.x()) * dx + (point.y() - seg_start.y()) * dy) / (dx * dx + dy * dy)
        
        # Clamp t to [0, 1] to stay on segment
        t = max(0, min(1, t))
        
        # Closest point on segment
        closest = QgsPointXY(
            seg_start.x() + t * dx,
            seg_start.y() + t * dy
        )
        
        return point.distance(closest), closest

    def canvasMoveEvent(self, event):
        if self.dragging and self.selected_vertex_idx is not None:
            new_point = self.toMapCoordinates(event.pos())
            
            # Create temporary geometry with moved vertex
            temp_points = self.points.copy()
            temp_points[self.selected_vertex_idx] = new_point
            
            # Check if area would exceed limit
            temp_geom = QgsGeometry.fromPolygonXY([temp_points])
            new_area = temp_geom.area()
            
            if new_area <= self.target_area_m2:
                # Update vertex position
                self.points[self.selected_vertex_idx] = new_point
                self.vertex_markers[self.selected_vertex_idx].setCenter(new_point)
                self.update_rubberband()
                self.update_area_callback(new_area)
            else:
                # Don't allow move - show warning
                self.update_area_callback(self.calculate_area())

    def canvasReleaseEvent(self, event):
        if event.button() == Qt.LeftButton:
            if self.dragging and self.selected_vertex_idx is not None:
                # Check if the final position exceeds area
                final_area = self.calculate_area()
                if final_area > self.target_area_m2:
                    # Area exceeded - remove the vertex if it was newly added
                    if self.adding_vertex:
                        self.points.pop(self.selected_vertex_idx)
                        self.setup_vertex_markers()
                        self.update_rubberband()
                        self.update_area_callback(self.calculate_area())
                        iface.messageBar().pushWarning("AreaLimiter", 
                            "New vertex would exceed target area - removed!")
                
                # Reset vertex color
                if self.selected_vertex_idx < len(self.vertex_markers):
                    if self.adding_vertex:
                        self.vertex_markers[self.selected_vertex_idx].setColor(QColor(0, 0, 255))
                    else:
                        self.vertex_markers[self.selected_vertex_idx].setColor(QColor(0, 0, 255))
                
                self.selected_vertex_idx = None
                self.dragging = False
                self.adding_vertex = False

    def keyPressEvent(self, event):
        if event.key() == Qt.Key_Escape:
            self.cancel_editing()
        elif event.key() in (Qt.Key_Return, Qt.Key_Enter):
            self.finish_editing()

    def calculate_area(self):
        if len(self.points) >= 3:
            return QgsGeometry.fromPolygonXY([self.points]).area()
        return 0

    def update_rubberband(self):
        if self.points:
            geom = QgsGeometry.fromPolygonXY([self.points])
            self.rubber_band.setToGeometry(geom, None)

    def finish_editing(self):
        """Save the edited geometry back to the feature."""
        if len(self.points) >= 3:
            final_area = self.calculate_area()
            if final_area <= self.target_area_m2:
                # Close the polygon
                closed_points = self.points + [self.points[0]]
                new_geom = QgsGeometry.fromPolygonXY([closed_points])
                
                # Update the feature
                self.layer.changeGeometry(self.feature.id(), new_geom)
                iface.messageBar().pushInfo("AreaLimiter", 
                    f"Polygon updated. Area: {final_area:.2f} m²")
            else:
                iface.messageBar().pushWarning("AreaLimiter", 
                    "Polygon exceeds target area!")
        
        self.cleanup()
        self.canvas.unsetMapTool(self)

    def cancel_editing(self):
        """Cancel editing without saving."""
        iface.messageBar().pushInfo("AreaLimiter", "Editing cancelled.")
        self.cleanup()
        self.canvas.unsetMapTool(self)

    def cleanup(self):
        """Remove all visual elements."""
        self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
        for marker in self.vertex_markers:
            self.canvas.scene().removeItem(marker)
        self.vertex_markers.clear()

    def deactivate(self):
        """Called when tool is deactivated."""
        self.cleanup()
        super().deactivate()


# -------------------------------------------------------------------------
# Custom map tool for limited area drawing (NEW polygons)
# -------------------------------------------------------------------------
class LimitedPolygonMapTool(QgsMapTool):
    """Custom map tool to draw NEW polygons with area limit."""

    def __init__(self, canvas, layer, target_area_m2, update_area_callback):
        super().__init__(canvas)
        self.canvas = canvas
        self.layer = layer
        self.target_area_m2 = target_area_m2
        self.update_area_callback = update_area_callback
        self.rubber_band = QgsRubberBand(self.canvas, QgsWkbTypes.PolygonGeometry)
        self.rubber_band.setColor(QColor(255, 0, 0, 100))
        self.rubber_band.setWidth(2)
        self.points = []

    def canvasPressEvent(self, event):
        if event.button() == Qt.RightButton:
            # Right click to finish
            self.finish_polygon()
            return
        
        if event.button() == Qt.LeftButton:
            point = self.toMapCoordinates(event.pos())
            self.points.append(point)
            self.update_rubberband()

            # Check area as we go
            if len(self.points) >= 3:
                geom = QgsGeometry.fromPolygonXY([self.points])
                area = geom.area()
                self.update_area_callback(area)

                # Stop adding if exceeds target
                if area > self.target_area_m2:
                    iface.messageBar().pushWarning("AreaLimiter", 
                        "Cannot exceed target area!")
                    self.points.pop()
                    self.update_rubberband()
                    area = self.calculate_area()
                    self.update_area_callback(area)

    def canvasMoveEvent(self, event):
        if not self.points:
            return
        temp_point = self.toMapCoordinates(event.pos())
        temp_points = self.points + [temp_point]
        geom = QgsGeometry.fromPolygonXY([temp_points])
        self.rubber_band.setToGeometry(geom, None)

        if len(temp_points) >= 3:
            area = geom.area()
            self.update_area_callback(area)

    def keyPressEvent(self, event):
        if event.key() in (Qt.Key_Return, Qt.Key_Enter):
            self.finish_polygon()
        elif event.key() == Qt.Key_Escape:
            self.reset()

    def finish_polygon(self):
        """Finish and save the polygon."""
        if len(self.points) >= 3:
            geom = QgsGeometry.fromPolygonXY([self.points])
            if geom.area() <= self.target_area_m2:
                # Create new feature
                feature = QgsFeature(self.layer.fields())
                feature.setGeometry(geom)
                self.layer.addFeature(feature)
                iface.messageBar().pushInfo("AreaLimiter", "Polygon created.")
                self.reset()
            else:
                iface.messageBar().pushWarning("AreaLimiter", 
                    "Polygon exceeds target area!")
                self.reset()

    def calculate_area(self):
        if len(self.points) >= 3:
            return QgsGeometry.fromPolygonXY([self.points]).area()
        return 0

    def update_rubberband(self):
        if self.points:
            geom = QgsGeometry.fromPolygonXY([self.points])
            self.rubber_band.setToGeometry(geom, None)
        else:
            self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)

    def reset(self):
        self.points = []
        self.rubber_band.reset(QgsWkbTypes.PolygonGeometry)
        self.update_area_callback(0)


# -------------------------------------------------------------------------
# Dock widget logic
# -------------------------------------------------------------------------
class AreaLimiterDockWidget(QDockWidget):
    """Dock widget logic for AreaLimiter plugin."""

    def __init__(self, parent=None):
        super().__init__(parent)

        self.ui = Ui_AreaLimiterDockWidgetBase()
        self.ui.setupUi(self)

        self.layer = None
        self.target_area = 0.0
        self.unit_factor = 1.0
        self.tool = None

        self.populate_layers()
        self.populate_units()

        # Connect signals
        self.ui.comboLayer.currentIndexChanged.connect(self.on_layer_selected_from_combo)
        self.ui.spinTargetArea.valueChanged.connect(self.on_target_changed)
        self.ui.comboUnits.currentIndexChanged.connect(self.on_unit_changed)
        self.ui.btnApply.clicked.connect(self.on_apply_clicked)
        
        # Add button for editing existing polygon
        if hasattr(self.ui, 'btnEditSelected'):
            self.ui.btnEditSelected.clicked.connect(self.on_edit_selected_clicked)

        # Detect QGIS layer changes
        iface.layerTreeView().currentLayerChanged.connect(self.on_active_layer_changed)
        
        # Connect to project signals for layer updates
        QgsProject.instance().layersAdded.connect(self.on_layers_changed)
        QgsProject.instance().layersRemoved.connect(self.on_layers_changed)
        QgsProject.instance().layerWillBeRemoved.connect(self.on_layer_will_be_removed)

    # -------------------------------------------------------------------------
    # Populate combo boxes
    # -------------------------------------------------------------------------
    def populate_layers(self):
        """Fill combo box with polygon vector layers only."""
        self.ui.comboLayer.clear()
        self.layer_lookup = {}

        polygon_layers = [
            layer for layer in QgsProject.instance().mapLayers().values()
            if isinstance(layer, QgsVectorLayer) and 
            layer.geometryType() == QgsWkbTypes.PolygonGeometry
        ]

        if not polygon_layers:
            self.ui.comboLayer.addItem("No polygon layers found")
            self.ui.comboLayer.setEnabled(False)
            return

        self.ui.comboLayer.setEnabled(True)
        for layer in polygon_layers:
            self.ui.comboLayer.addItem(layer.name())
            self.layer_lookup[layer.name()] = layer
            
            # Connect to layer name changes
            try:
                layer.nameChanged.disconnect(self.on_layer_name_changed)
            except:
                pass
            layer.nameChanged.connect(self.on_layer_name_changed)

    def populate_units(self):
        """Fill combo box with area units (m² and hectares)."""
        self.ui.comboUnits.clear()
        self.ui.comboUnits.addItem("m²", 1.0)
        self.ui.comboUnits.addItem("ha", 10000.0)
        self.ui.comboUnits.setCurrentIndex(0)
        self.unit_factor = 1.0

    # -------------------------------------------------------------------------
    # Signal handlers
    # -------------------------------------------------------------------------
    def on_layers_changed(self, layers=None):
        """Triggered when layers are added or removed from the project."""
        current_layer_name = self.ui.comboLayer.currentText()
        self.populate_layers()
        
        # Try to restore previous selection if layer still exists
        idx = self.ui.comboLayer.findText(current_layer_name)
        if idx >= 0:
            self.ui.comboLayer.setCurrentIndex(idx)
        else:
            # If previous layer no longer exists, select first available
            if self.ui.comboLayer.count() > 0:
                self.ui.comboLayer.setCurrentIndex(0)

    def on_layer_will_be_removed(self, layer_id):
        """Triggered when a layer is about to be removed."""
        # If the currently selected layer is being removed, clear the selection
        if self.layer and self.layer.id() == layer_id:
            self.layer = None
            self.ui.labelStatus.setText("⚠️ Selected layer was removed.")

    def on_layer_name_changed(self):
        """Triggered when a layer's name is changed."""
        # Refresh the combo box to show updated names
        current_layer_id = self.layer.id() if self.layer else None
        self.populate_layers()
        
        # Restore selection based on layer ID rather than name
        if current_layer_id:
            for i in range(self.ui.comboLayer.count()):
                layer_name = self.ui.comboLayer.itemText(i)
                layer = self.layer_lookup.get(layer_name)
                if layer and layer.id() == current_layer_id:
                    self.ui.comboLayer.setCurrentIndex(i)
                    break

    def on_active_layer_changed(self, layer):
        """Triggered when the active QGIS layer changes."""
        if isinstance(layer, QgsVectorLayer) and \
           layer.geometryType() == QgsWkbTypes.PolygonGeometry:
            self.layer = layer
            idx = self.ui.comboLayer.findText(layer.name())
            if idx >= 0:
                self.ui.comboLayer.setCurrentIndex(idx)
            self.ui.labelStatus.setText(f"Active layer: {layer.name()}")
        else:
            self.layer = None
            self.ui.labelStatus.setText("⚠️ Please select a polygon layer first.")

    def on_layer_selected_from_combo(self):
        """Triggered when a polygon layer is selected from combo box."""
        index = self.ui.comboLayer.currentIndex()
        if index < 0:
            self.layer = None
            return

        layer_name = self.ui.comboLayer.currentText()
        layer = QgsProject.instance().mapLayersByName(layer_name)
        self.layer = layer[0] if layer else None

        if self.layer:
            self.ui.labelStatus.setText(f"Selected layer: {self.layer.name()}")
        else:
            self.ui.labelStatus.setText("⚠️ Please select a valid polygon layer.")

    def on_unit_changed(self):
        """Switch between m² and ha."""
        self.unit_factor = self.ui.comboUnits.currentData()
        self.update_status(0)

    def on_target_changed(self):
        """Update target area when user changes spin box."""
        self.target_area = self.ui.spinTargetArea.value()
        self.update_status(0)

    def on_apply_clicked(self):
        """Activate area-limited polygon digitizing for NEW polygons."""
        self.layer = self.get_selected_layer()

        if not self.layer:
            self.ui.labelStatus.setText("⚠️ Please select a polygon layer first.")
            return

        if not self.layer.isEditable():
            self.ui.labelStatus.setText("⚠️ Please put the layer in edit mode first.")
            return

        self.target_area = self.ui.spinTargetArea.value() * self.unit_factor
        self.ui.labelStatus.setText(
            f"Draw new polygon. Target: {self.target_area / self.unit_factor:.2f} "
            f"{self.ui.comboUnits.currentText()}. Right-click to finish."
        )

        canvas = iface.mapCanvas()
        self.tool = LimitedPolygonMapTool(canvas, self.layer, self.target_area, 
                                         self.update_status)
        canvas.setMapTool(self.tool)

    def on_edit_selected_clicked(self):
        """Edit vertices of selected polygon with area constraint."""
        self.layer = self.get_selected_layer()

        if not self.layer:
            self.ui.labelStatus.setText("⚠️ Please select a polygon layer first.")
            return

        if not self.layer.isEditable():
            self.ui.labelStatus.setText("⚠️ Please put the layer in edit mode first.")
            return

        # Get selected features
        selected = self.layer.selectedFeatures()
        if not selected:
            self.ui.labelStatus.setText("⚠️ Please select a polygon to edit.")
            return

        if len(selected) > 1:
            self.ui.labelStatus.setText("⚠️ Please select only one polygon.")
            return

        feature = selected[0]
        self.target_area = self.ui.spinTargetArea.value() * self.unit_factor
        
        current_area = feature.geometry().area()
        self.ui.labelStatus.setText(
            f"Editing polygon. Current: {current_area / self.unit_factor:.2f} "
            f"{self.ui.comboUnits.currentText()}. Right-click to finish."
        )

        canvas = iface.mapCanvas()
        self.tool = VertexEditMapTool(canvas, self.layer, feature, self.target_area, 
                                     self.update_status)
        canvas.setMapTool(self.tool)

    # -------------------------------------------------------------------------
    # Helpers
    # -------------------------------------------------------------------------
    def get_selected_layer(self):
        """Return the polygon layer currently selected in the combo box."""
        layer_name = self.ui.comboLayer.currentText()
        layer = self.layer_lookup.get(layer_name)

        if not layer:
            matches = QgsProject.instance().mapLayersByName(layer_name)
            if matches:
                layer = matches[0]

        return layer if isinstance(layer, QgsVectorLayer) else None

    def update_status(self, area_m2):
        """Update the status label with area info."""
        area_in_units = area_m2 / self.unit_factor
        target_in_units = self.target_area / self.unit_factor
        self.ui.labelStatus.setText(
            f"Current: {area_in_units:.2f} {self.ui.comboUnits.currentText()} / "
            f"Target: {target_in_units:.2f} {self.ui.comboUnits.currentText()}"
        )