# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Archiwalna Ortofotomapa
                                 A QGIS plugin
 Pozwala na podczytanie w QGIS archiwalnej ortofotomapy na podstawie usług geoportal.gov.pl
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2020-03-25
        copyright            : (C) 2020 by EnviroSolutions Sp. z o.o.
        email                : office@envirosolutions.pl
        git sha              : $Format:%H$
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 This script initializes the plugin, making it known to QGIS.
"""

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QT_VERSION_STR, QTimer
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QToolBar, QApplication, QWidget, QLabel, QDialog, QComboBox
from qgis.PyQt import uic
from datetime import datetime
from qgis.core import QgsSettings
from .qgis_feed import QgisFeedDialog
# Initialize Qt resources from file resources.py
from .resources import *
from qgis.core import QgsRasterLayer, QgsProject, Qgis, QgsNetworkAccessManager, QgsPointXY, QgsCoordinateReferenceSystem, QgsCoordinateTransform
# Import the code for the DockWidget
from .archiwalna_ortofotomapa_dockwidget import ArchiwalnaOrtofotomapaDockWidget
import os.path
from . import PLUGIN_VERSION as plugin_version
from .utils import MessageUtils, QtUtils
from . import PLUGIN_NAME as plugin_name
from .constants import ORTO_SERVICE_URL, WMS_BASE_PARAMS, WMS_TIME_SUFFIX, POINT_COORDINATES, INITIAL_SCALE, SRS_CODE



class ArchiwalnaOrtofotomapa:
    """QGIS Plugin Implementation."""

    def __init__(self, iface, is_tested=False):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """

        if not is_tested:
            self.settings = QgsSettings() 

            if Qgis.QGIS_VERSION_INT >= 31000:
                from .qgis_feed import QgisFeed
                self.selected_industry = self.settings.value("selected_industry", None)
                show_dialog = self.settings.value("showDialog", True, type=bool)

                if self.selected_industry is None and show_dialog:
                    self.showBranchSelectionDialog()

                select_indust_session = self.settings.value('selected_industry')

                self.feed = QgisFeed(selected_industry=select_indust_session, plugin_name=plugin_name)
                self.feed.initFeed()

        # Save reference to the QGIS interface
        self.iface = iface

        self.project = QgsProject.instance()
        self.nam = QgsNetworkAccessManager.instance()

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        if not is_tested:
            # initialize locale
            locale = QSettings().value('locale/userLocale')[0:2]
            locale_path = os.path.join(
                self.plugin_dir,
                'i18n',
                'ArchiwalnaOrtofotomapa_{}.qm'.format(locale))

            if os.path.exists(locale_path):
                self.translator = QTranslator()
                self.translator.load(locale_path)
                QCoreApplication.installTranslator(self.translator)

            # Declare instance attributes
            self.actions = []
            self.menu = self.tr(u'&EnviroSolutions')
            # TODO: We are going to let the user set this up in a future iteration
            self.toolbar = self.iface.mainWindow().findChild(QToolBar, 'EnviroSolutions')
            if not self.toolbar:
                self.toolbar = self.iface.addToolBar(u'EnviroSolutions')
                self.toolbar.setObjectName(u'EnviroSolutions')

            self.pluginIsActive = False
            
            # Create the dockwidget and keep reference
            self.dockwidget = ArchiwalnaOrtofotomapaDockWidget()
            
            # Connect to slider signals directly (once during initialization)
            self.dockwidget.timeSlider.valueChanged.connect(self.sliderChanged)
            self.dockwidget.timeSlider.sliderReleased.connect(self.sliderReleased)
            self.dockwidget.timeSlider.sliderPressed.connect(self.sliderPressed)
            self.dockwidget.timeSlider.sliderMoved.connect(self.sliderMoved)
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            self.canvas = self.iface.mapCanvas()
        self.orto = None

        # --------------------------------------------------------------------------


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ArchiwalnaOrtofotomapa', message)
    

    def addAction(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):

        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""    
        
        icon_path = ':/plugins/archiwalna_ortofotomapa/icons/archiwalna_logo.svg'
        self.addAction(
            icon_path,
            text=self.tr(u'Archiwalna Ortofotomapa'),
            callback=self.run,
            parent=self.iface.mainWindow())
        MessageUtils.pushLogInfo("Wtyczka aktywna")


    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        # disconnects
        #self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None
        MessageUtils.pushLogInfo("Wtyczka nieaktywna")

        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&EnviroSolutions'),
                action)
            #self.iface.removeToolBarIcon(action)
            self.toolbar.removeAction(action)
        # remove the toolbar
        del self.toolbar

    #--------------------------------------------------------------------------


    def run(self):
        """Run method that loads and starts the plugin"""
        
        # Increase network timeout to 2 minutes (120000 ms)
        self.nam.setTimeout(120000)

        if not self.pluginIsActive:
            self.pluginIsActive = True
            #print "** STARTING ArchiwalnaOrtofotomapa"
            if self.orto:
                self.orto.willBeDeleted.disconnect()
                self.project.removeMapLayer(self.orto)
                self.orto = None
            
            # Set slider to max year every time plugin is launched
            self.dockwidget.setSliderToMaxYear()

            # Version info
            self.dockwidget.setWindowTitle('%s %s' % (plugin_name, plugin_version))
            self.dockwidget.lbl_pluginVersion.setText('%s %s' % (plugin_name, plugin_version))

            # show the dockwidget
            # TODO: fix to allow choice of dock location
            if QtUtils.isCompatibleQtVersion(QT_VERSION_STR, 6):
                dock_location = Qt.DockWidgetArea.LeftDockWidgetArea
            else:
                dock_location = Qt.LeftDockWidgetArea
            
            self.iface.addDockWidget(dock_location, self.dockwidget)
            self.dockwidget.show()
            MessageUtils.pushLogInfo("Dockwidget ustawiony")

            self.orto = QgsRasterLayer(self.makeDataSourceUri(self.dockwidget.timeSlider.value()),
                                       "Ortofotomapa Archiwalna %d" % self.dockwidget.timeSlider.value(),
                                       'wms')
            self.orto.willBeDeleted.connect(self.ortoRemoval)
            if not self.orto.isValid():
                MessageUtils.pushLogWarning("Błąd: Warstwa jest nieprawidłowa (isValid=False). Sprawdź połączenie z WMS.")
                self.orto = None
                return

            self.project.addMapLayer(self.orto)
            
            if self.project.mapLayer(self.orto.id()):
                MessageUtils.pushLogInfo("Ortofotomapa dodana")
            else:
                MessageUtils.pushLogWarning("Błąd: Warstwa nie została dodana do projektu")

            # Zoom to Warsaw after a short delay to ensure layer is fully loaded
            QTimer.singleShot(500, self.zoomToPoint)

        else:
            #reopened
            pass

    def zoomToPoint(self):
        """Zooms the map canvas to Warsaw coordinates."""
        point = QgsPointXY(*POINT_COORDINATES)
        crs_src = QgsCoordinateReferenceSystem("EPSG:" + SRS_CODE)
        crs_dest = self.canvas.mapSettings().destinationCrs()
        transform = QgsCoordinateTransform(crs_src, crs_dest, self.project)
        point_transformed = transform.transform(point)
        
        self.canvas.setCenter(point_transformed)
        self.canvas.zoomScale(INITIAL_SCALE)
        self.canvas.refresh()

    def showBranchSelectionDialog(self):
        self.qgisfeed_dialog = QgisFeedDialog()

        if self.qgisfeed_dialog.exec() == QDialog.Accepted:
            self.selected_branch = self.qgisfeed_dialog.comboBox.currentText()
            
            self.settings.setValue("selected_industry", self.selected_branch)  
            self.settings.setValue("showDialog", False) 

    def ortoRemoval(self):
        """Function to remove the dockwidget and the orto layer when the layer is deleted"""
        
        MessageUtils.pushLogInfo("Ortofotomapa usunięta")
        self.dockwidget.close()
        self.orto = None


    def sliderChanged(self):
        """Function to update the label with the current year"""

        # Update the label with current year
        self.dockwidget.timeLabel.setText(str(self.dockwidget.timeSlider.value()))
        
        if self.orto is None:
            return

        if not self.dockwidget.isSliderPressed:
            self.changeOrtoLayer()

    def sliderMoved(self):
        """Function to do nothing if the slider is moved"""
        pass


    def sliderPressed(self):
        """Function to control the parameter responsible for actions after clicking the slider"""
        self.dockwidget.isSliderPressed = True


    def sliderReleased(self):
        """Function to control the parameter responsible for actions after releasing the slider"""
        self.dockwidget.isSliderPressed = False
        MessageUtils.pushLogInfo("Slider puszczony")
        self.changeOrtoLayer()


    def changeOrtoLayer(self):
        """Function to change the name of the orto layer after changing the year"""

        year = self.dockwidget.timeSlider.value()
        uri = self.makeDataSourceUri(year)
        self.orto.dataProvider().setDataSourceUri(uri)
        self.orto.triggerRepaint()
        # self.orto.dataProvider().reloadData() #QGIS 3.12 and above
        self.orto.setName("Ortofotomapa Archiwalna %d" % year)
        MessageUtils.pushLogInfo("Nazwa i źródło ortofotomapy zmienione")


    def makeDataSourceUri(self, year):
        """Function to edit the orto layer and change the year it is based on"""
        
        return WMS_BASE_PARAMS + ORTO_SERVICE_URL + "?TIME=" + str(year) + WMS_TIME_SUFFIX
