# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ArchesProject
                                 A QGIS plugin
 This plugin links QGIS to an Arches project.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-09-15
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Knowledge Integration
        email                : samuel.scandrett@k-int.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import Qt
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QDir
from qgis.PyQt.QtGui import QIcon, QFontDatabase, QPixmap, QCursor, QTransform
from qgis.PyQt.QtWidgets import QAction, QTableView, QTableWidgetItem, QApplication
from qgis.core import (QgsProject, 
                       QgsVectorLayer, 
                       QgsVectorLayerCache, 
                       QgsWkbTypes,
                       QgsTask,
                       QgsTaskManager,
                       QgsMessageLog,
                       QgsApplication
                       )
from qgis.gui import (QgsAttributeTableView, 
                      QgsAttributeTableModel, 
                      QgsAttributeTableFilterModel,
                      QgsMapLayerComboBox
                      )

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .ui.arches_project_dialog import ArchesProjectDialog

# Import the confirmation dialogs
from .ui.create_resource_confirmation_dialog import CreateResourceConfirmation
from .ui.edit_resource_add_confirmation_dialog import EditResourceAddConfirmation
from .ui.edit_resource_replace_confirmation_dialog import EditResourceReplaceConfirmation

from .core.arches.connection import ArchesConnection, ConnectionProcess
from .core.arches.resources import ArchesResources

from .core.views.stylesheets import PluginStylesheets
from .core.views.logging import enable_logging
from .core.views.components.missing_credentials import missing_credentials

from .core.utils.format_url import format_url
from .core.utils.spinner import triggerSpinner
from .core.utils.qgis_messaging import show_message

import os.path
import sys
import requests
from datetime import datetime
import time

class ArchesProject:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ArchesProject_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Arches Project')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # Confirmation additional dialogs

        ## ARCHES PLUGIN SPECIFIC VARIABLES
        # Cache connection details to prevent firing duplicate connections
        self.arches_connection_cache = {}
        # Store token data to avoid regenerating every connection
        self.arches_token = {}
        self.arches_graphs_list = []
        self.arches_user_info = {}
        # Store selected arches resource
        self.layers = []
        self.arches_selected_resource = {"resourceinstanceid": "",
                                         "nodeid": "",
                                         "tileid": ""
                                        }

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ArchesProject', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(
            self.plugin_dir,
            'icons',
            'arches.png')
        self.add_action(
            icon_path,
            text=self.tr(u'Arches Project'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Arches Project'),
                action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = ArchesProjectDialog()            
            self.dlg_resource_creation = CreateResourceConfirmation()
            self.dlg_edit_resource_add = EditResourceAddConfirmation()
            self.dlg_edit_resource_replace = EditResourceReplaceConfirmation()

            # Setup Arches Stylesheet
            PluginStylesheets(dlg = self.dlg,
                              dlg_resource_creation = self.dlg_resource_creation,
                              dlg_edit_resource_add = self.dlg_edit_resource_add,
                              dlg_edit_resource_replace = self.dlg_edit_resource_replace,
                              on_start = True,
                              plugin_dir = self.plugin_dir)
            # if stylesheet is disabled
            self.dlg.useStylesheetCheckbox.stateChanged.connect(
                lambda: PluginStylesheets(dlg = self.dlg,
                                          dlg_resource_creation = self.dlg_resource_creation,
                                          dlg_edit_resource_add = self.dlg_edit_resource_add,
                                          dlg_edit_resource_replace = self.dlg_edit_resource_replace,
                                          on_start = False,
                                          plugin_dir = self.plugin_dir))

            ## Have everything called in here so multiple connections aren't made when plugin button pressed
            # This way only one connection is made at a time

            # Set tab index to 0 always
            self.dlg.tabWidget.setCurrentIndex(0)
            self.dlg.tabWidget.setTabVisible(1, False)
            self.dlg.tabWidget.setTabVisible(5, False)

            self.dlg.enableLoggingCheckbox.stateChanged.connect(enable_logging)

            # initiate the current selected layer
            self.map_selection()

            # Connection to Arches instance
            self.dlg.btnConnect.clicked.connect(self.arches_connection_save)
            self.dlg.btnLogout.clicked.connect(lambda: ArchesConnection(None, None, None).
                                              connection_reset(hard_reset=True,
                                                                self_obj=self,
                                                                manual_logout=True))

            # Get the map selection and update when changed
            self.iface.mapCanvas().selectionChanged.connect(self.map_selection)

            ## Set "Create resource" to false to begin with and only update once Arches connection made
            self.dlg.createResModelSelect.setEnabled(False)
            self.dlg.createResFeatureSelect.setEnabled(False)
            self.dlg.addNewRes.setEnabled(False)

            # to run when layer is changed in create resource and edit resource tabs
            self.dlg.hidePostgresLayers.setChecked(True)
            self.dlg.createResFeatureSelect.highlighted.connect(lambda: self.update_map_layers(checkbox=self.dlg.hidePostgresLayers))
            self.dlg.editResSelectFeatures.highlighted.connect(lambda: self.update_map_layers(checkbox=self.dlg.hidePostgresLayers))

            self.dlg.hidePostgresLayers.stateChanged.connect(lambda: self.show_hide_psql_layers(combobox1=self.dlg.createResFeatureSelect,
                                                                                                combobox2=self.dlg.editResSelectFeatures))

            # click add button - should bring up new dialog for confirmation
            self.dlg.addNewRes.clicked.connect(self.create_resource)

            ## Set "Edit Resource" to false to begin with
            self.dlg.selectedResUUID.setText("Connect to your Arches instance to edit resources.")
            self.dlg.addEditRes.setEnabled(False)
            self.dlg.replaceEditRes.setEnabled(False)
            self.dlg.editResSelectFeatures.setEnabled(False)
            self.dlg.selectedResAttributeTable.setEnabled(False)

            self.dlg.addEditRes.clicked.connect(lambda: self.edit_resource(replace=False))
            self.dlg.replaceEditRes.clicked.connect(lambda: self.edit_resource(replace=True))

            # Hide multiple geometry node selection by default
            self.dlg.geometryNodeSelectFrame.hide()

            # Check if selected graph has multiple geometry nodes
            self.dlg.createResModelSelect.currentIndexChanged.connect(self.multiple_geometry_node_check)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass

    def map_selection(self):
        """
        Get the Arches Resource from the map
        """

        active_layer = self.iface.activeLayer()
        canvas = self.iface.mapCanvas()

        # If plugin is opened before QGIS project opened/setup selectedFeatures is None
        try:
            features = active_layer.selectedFeatures()
        except AttributeError:
            features = None

        print("\nmap selection has been fired because selection changed")
        print("layer:",active_layer, "features:",features)

        if features:

            if len(features) > 1:
                print("Select one feature")
                self.dlg.selectedResAttributeTable.setRowCount(0)
                if self.arches_token:
                    self.dlg.selectedResUUID.setText("Multiple features selected, select one feature to proceed.")
                else:
                    self.dlg.selectedResUUID.setText("Connect to your Arches instance to edit resources.")
                return

            elif len(features) == 0:
                print("No feature selected")
                self.dlg.selectedResAttributeTable.setRowCount(0)
                if self.arches_token:
                    self.dlg.selectedResUUID.setText("Select a feature to proceed.")
                    self.dlg.addEditRes.setEnabled(False)
                    self.dlg.replaceEditRes.setEnabled(False)
                else:
                    self.dlg.selectedResUUID.setText("Connect to your Arches instance to edit resources.")
                return

            else:
                print("FEATURE SELECTED")
                for f in features:
                    if "resourceinstanceid" in f.attributeMap():

                        # Initialise attribute table in the plugin window if the geom is recognised as an Arches res
                        # if initialised when arches_token exists then would have to click off and back on to recognise
                        no_rows = len(f.attributes())
                        no_cols = 2
                        self.dlg.selectedResAttributeTable.setRowCount(no_rows)
                        self.dlg.selectedResAttributeTable.setColumnCount(no_cols)

                        # Fill table with attributes
                        for i, (k, v) in enumerate(f.attributeMap().items()):
                            feat = QTableWidgetItem(str(k))
                            val = QTableWidgetItem(str(v))
                            self.dlg.selectedResAttributeTable.setItem(i, 0, feat)
                            self.dlg.selectedResAttributeTable.setItem(i, 1, val)
                            self.dlg.selectedResAttributeTable.setRowHeight(i, 5)
                            # Store current resource info
                            if k == "resourceinstanceid":
                                self.arches_selected_resource["resourceinstanceid"] = v
                            elif k == "nodeid":
                                self.arches_selected_resource["nodeid"] = v
                            elif k == "tileid":
                                self.arches_selected_resource["tileid"] = v

                        self.dlg.selectedResAttributeTable.setHorizontalHeaderLabels([u'Feature',u'Values'])
                        self.dlg.selectedResAttributeTable.resizeColumnsToContents()

                        # if the token exists then enable the UI elements
                        if self.arches_token:
                            resource_string = "Resource: %s" % (f['resourceinstanceid'])
                            self.dlg.selectedResUUID.setText(resource_string)
                            self.dlg.addEditRes.setEnabled(True)
                            self.dlg.replaceEditRes.setEnabled(True)

                            # Save resource instance details once selected
                        else:
                            self.dlg.selectedResUUID.setText("Connect to your Arches instance to edit resources.")
                            self.dlg.addEditRes.setEnabled(False)
                            self.dlg.replaceEditRes.setEnabled(False)

                    else: 
                        if self.arches_token:
                            self.dlg.selectedResUUID.setText("The feature selected is not an Arches resource.")
                        else:
                            self.dlg.selectedResUUID.setText("Connect to your Arches instance to edit resources.")

    def update_map_layers(self, checkbox):
        """
        Function to update new vector layers dynamically
        """

        if checkbox.isChecked():
            all_current_layers = [l for l in QgsProject.instance().mapLayers().values() if l.type() == QgsVectorLayer.VectorLayer if str(l.dataProvider().name()) != "postgres"] 

        elif not checkbox.isChecked():
            all_current_layers = [l for l in QgsProject.instance().mapLayers().values() if l.type() == QgsVectorLayer.VectorLayer]

        if self.layers != all_current_layers:
            self.layers = all_current_layers

    def show_hide_psql_layers(self, combobox1, combobox2):
        """
        Reflect change made by checkbox to show or hide PSQL layers from self.layers
        """
        # TODO: Not sure I like the way this works but it works

        def change_both_comboboxes(c):
            c.blockSignals(True)
            c.clear()
            c.addItems([layer.name() for layer in self.layers])
            c.blockSignals(False)

        if self.dlg.hidePostgresLayers.isChecked():
            self.layers = [l for l in QgsProject.instance().mapLayers().values() if l.type() == QgsVectorLayer.VectorLayer if str(l.dataProvider().name()) != "postgres"]
            change_both_comboboxes(combobox1)
            change_both_comboboxes(combobox2)

        elif not self.dlg.hidePostgresLayers.isChecked():
            self.layers = [l for l in QgsProject.instance().mapLayers().values() if l.type() == QgsVectorLayer.VectorLayer]
            change_both_comboboxes(combobox1)
            change_both_comboboxes(combobox2)

    def multiple_geometry_node_check(self):
        selectedGraphIndex = self.dlg.createResModelSelect.currentIndex()
        selectedGraph = self.arches_graphs_list[selectedGraphIndex]

        self.geometry_nodes = []
        self.dlg.geometryNodeSelect.setEnabled(False)
        self.dlg.geometryNodeSelectFrame.hide()

        if selectedGraph:
            if selectedGraph["multiple_geometry_nodes"] == True:               
                for k,v in selectedGraph["geometry_node_data"].items():
                    self.geometry_nodes.append({"node_id": k, "nodegroup_id": v["nodegroup_id"], "name": v["name"]})
                self.dlg.geometryNodeSelect.setEnabled(True)
                self.dlg.geometryNodeSelect.clear()
                self.dlg.geometryNodeSelect.addItems([n["name"] for n in self.geometry_nodes])
                self.dlg.geometryNodeSelectFrame.show()

    def create_resource(self):
        """Create Resource dialog and functionality"""

        arches_create_resource = ArchesResources(nodeid=None, # filled by selectedNode
                                                 tileid=None,
                                                 arches_token=self.arches_token, 
                                                 arches_graphs_list=self.arches_graphs_list,
                                                 layers=self.layers,
                                                 arches_user_info=self.arches_user_info,
                                                 geometry_nodes=self.geometry_nodes)
        arches_create_resource.create_resource(dlg=self.dlg,
                                               dlg_resource_creation=self.dlg_resource_creation,
                                               iface=self.iface)

    def edit_resource(self, replace):
        """Save geometries to existing resource - either replace or add"""

        arches_edit_resource = ArchesResources(nodeid=self.arches_selected_resource["nodeid"],
                                               tileid=self.arches_selected_resource["tileid"],
                                               arches_token=self.arches_token,
                                               arches_graphs_list=self.arches_graphs_list,
                                               layers=self.layers,
                                               arches_user_info=self.arches_user_info,
                                               geometry_nodes=self.geometry_nodes)
        arches_edit_resource.edit_resource(replace=replace,
                                           arches_selected_resource=self.arches_selected_resource,                                            
                                           dlg=self.dlg,
                                           dlg_edit_resource_replace=self.dlg_edit_resource_replace, 
                                           dlg_edit_resource_add=self.dlg_edit_resource_add,
                                           iface=self.iface)

    def arches_connection_save(self):
        """
        Connection to Arches project server
        """
        self.error_msg = ""
        connection_information = {
            "URL": {
                "value": self.dlg.archesServerInput.text().strip(),
                "input": self.dlg.archesServerInput,
                "label": self.dlg.archesServerLabel,
                }, 
            "username": {
                "value": self.dlg.usernameInput.text().strip(),
                "input": self.dlg.usernameInput,
                "label": self.dlg.usernameLabel
                }, 
            "password": {
                "value": self.dlg.passwordInput.text().strip(),
                "input": self.dlg.passwordInput,
                "label": self.dlg.passwordLabel
                }
        }

        is_valid_input=True
        missing_inputs = []
        for k,v in connection_information.items():
            if not v["value"]:
                is_valid_input=False
                missing_credentials(v["input"], "True")
                missing_credentials(v["label"], "True")
                missing_inputs.append(k)
            else:
                missing_credentials(v["label"], "False")
                missing_credentials(v["input"], "False")

        if missing_inputs:
            if len(missing_inputs) > 1:
                self.error_msg = f"Login missing values for {', '.join(missing_inputs[:-1])} and {missing_inputs[-1]}."
            else:
                self.error_msg = f"Login missing {missing_inputs[0]}."
            self.dlg.loginErrorMessageLabel.setText(self.error_msg)
            self.dlg.loginErrorMessageFrame.show()
            self.dlg.loginErrorMessageLabel.show()
        else:
            self.dlg.loginErrorMessageLabel.setText("")
            self.dlg.loginErrorMessageFrame.hide()
            self.dlg.loginErrorMessageLabel.hide()

        if is_valid_input == True:
            formatted_url = format_url(self.dlg.archesServerInput.text())

            # Adding arches connection to task queue
            self.arches_connection = ConnectionProcess(url=formatted_url,
                                                    username=connection_information["username"]["value"], 
                                                    password=connection_information["password"]["value"],
                                                    archesproject=self)
            QgsApplication.taskManager().addTask(self.arches_connection)

            #24 Tasks must be assigned to self, otherwise finished() won't run
            # https://github.com/qgis/QGIS/issues/59464#issuecomment-2640165772

            spinner = triggerSpinner(arches_obj=self)
            spinner.start_spinner()

            # A log message (or print) is required for the task to be run.
            # It is an existing QGIS issue https://github.com/qgis/QGIS/issues/37655
            QgsMessageLog.logMessage("Connection task started")
