# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Arc3D
                                 A QGIS plugin
 Tool to convert 2D lines to 3D Arcs
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-08-17
        copyright            : (C) 2024 by Shai Sussman
        email                : shai.sussman@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Shai Sussman'
__date__ = '2024-08-17'
__copyright__ = '(C) 2024 by Shai Sussman'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink, QgsCoordinateTransform, QgsCoordinateReferenceSystem, QgsProject, QgsProcessingParameterField, QgsVectorLayer)
from qgis.PyQt.QtGui import QIcon

from .arc_3d_methods import generate_3d_polyline_from_geometry

class Arc3DAlgorithmLayerParameter(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    SEGMENT_SLIDER_field = 'SEGMENT_SLIDER_field'
    Y_ANGLE_field = 'Y_ANGLE_field'
    Z_SCALE_field = 'Z_SCALE_field'
    INPUT = 'INPUT'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input layer'),
                [QgsProcessing.TypeVectorAnyGeometry]
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.SEGMENT_SLIDER_field,
                "select the field with the number of segments",
                "",
                self.INPUT
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.Y_ANGLE_field,
                "select the field with the Y-angle",
                "",
                self.INPUT
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.Z_SCALE_field,
                "select the field with the Z-scale",
                "",
                self.INPUT
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Output layer')
            )
        )
        self.crs_3d = 3857

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)
        
        segments_field = self.parameterAsString(parameters, self.SEGMENT_SLIDER_field, context)
        y_angle_field = self.parameterAsString(parameters, self.Y_ANGLE_field, context)
        z_scale_field = self.parameterAsString(parameters, self.Z_SCALE_field, context)
        layer_3d = QgsVectorLayer(f"LineStringZ?crs=EPSG:{self.crs_3d}", 'temp', "memory")
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                context, source.fields(), layer_3d.wkbType(), source.sourceCrs())

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / source.featureCount() if source.featureCount() else 0
        features = source.getFeatures()
        transform_to_3857 = QgsCoordinateTransform(
                QgsCoordinateReferenceSystem(source.sourceCrs().authid()),
                QgsCoordinateReferenceSystem(self.crs_3d),
                QgsProject.instance(),
            )
        transform_from_3857 = QgsCoordinateTransform(
                QgsCoordinateReferenceSystem(self.crs_3d),
                QgsCoordinateReferenceSystem(source.sourceCrs().authid()),
                QgsProject.instance(),
            )
        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            segments = feature[segments_field]
            y_angle = feature[y_angle_field]
            z_scale = feature[z_scale_field]
            if feedback.isCanceled():
                break
            if '3857' not in source.sourceCrs().authid():
                geometry_ = feature.geometry()
                geometry_.transform(transform_to_3857)
                
                feature_3d_polyline = generate_3d_polyline_from_geometry(geometry_, segments, y_angle, z_scale)
                feature_3d_polyline.transform(transform_from_3857)
            else:
                feature_3d_polyline = generate_3d_polyline_from_geometry(feature.geometry(), segments, y_angle, z_scale)
            feature.setGeometry(feature_3d_polyline)
            # Add a feature in the sink
            sink.addFeature(feature, QgsFeatureSink.FastInsert)

            # Update the progress bar
            feedback.setProgress(int(current * total))
        print(source.sourceCrs())
        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {self.OUTPUT: dest_id}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Create 3D arcs with layer parameters'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return '3D Arcs'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return Arc3DAlgorithmLayerParameter()
    
    def icon(self):
        """Return the icon."""
        return QIcon('icon.png')


    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("""The script converts 2D lines to 3D arcs \n
                       Select the 2d line from the Input lyaer \n
                       to control the smoothness of the curve select the number of segments filed \n
                       to control the angle of the curve select the Y-angle field \n
                       to control the scewness of the curve select the Z-scale field\n
                       """)