# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ToolBoxDialog
                                 A QGIS plugin
 This plugin is a suite of tools for the ImPact Analysis
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2020-07-27
        git sha              : $Format:%H$
        copyright            : (C) ANYWAYS BV
        email                : info@anyways.eu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import json
import os

import time
from qgis.PyQt import (QtWidgets, uic)
from qgis.core import *
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication

from .impact import fod_api, impact_api, transform_layer_to_WGS84, extract_valid_geometries, routing_api, \
    feature_histogram, layer_as_geojson_features, previous_state_tracker, create_layer_from_file, \
    extract_coordinates_array, default_layer_styling

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'ImPact_toolbox_dialog_base.ui'))


class ToolBoxDialog(QtWidgets.QDialog, FORM_CLASS):
    """
    The main dialog. All button actions get linked here with the logic in 'impact'
    """

    def __init__(self, iface, profile_descriptions, profile_keys, parent=None):
        """Constructor."""
        super(ToolBoxDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect

       # QCoreApplication.installTranslator("")
        self.setupUi(self)

        self.iface = iface
        self.impact_api = None
        self.layer_styling = default_layer_styling.default_layer_styling()
        self.impact_api = impact_api.impact_api(None)
        state_tracker = previous_state_tracker.previous_state_tracker(QgsProject.instance())
        self.state_tracker = state_tracker

        # A dict with profile-names and aliases --> description
        self.profile_descriptions = profile_descriptions
        # A sorted list of profile keys (without aliases)
        self.profile_keys = profile_keys

        self.user_settings = QgsSettings()
        self.api_key_field.setText(self.user_settings.value("anyways.eu/impact/api_key"))
        self.current_routeplanning_task = None
        self.save_settings_button.clicked.connect(self.save_setting)
        self.save_setting()  # Fetch impact instances etc...

        # Get project path
        prjpath = QgsProject.instance().fileName()
        if (not os.path.isdir(prjpath)):
            prjpath = os.path.dirname(prjpath)
        self.project_directory.setFilePath(prjpath)
        self.project_directory.setEnabled(False)
        self.path = prjpath

        # Set routing api options
        self.scenario_picker.addItem(self.tr("Plan with a recent version of OpenStreetMap"))
        self.profile_picker.addItems(self.profile_keys)

        # Set layer filters

        self.departure_layer_picker.setFilters(QgsMapLayerProxyModel.PointLayer)
        self.arrival_layer_picker.setFilters(QgsMapLayerProxyModel.PointLayer)

        self.movement_pairs_layer_picker.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.zero_situation_picker.setFilters(QgsMapLayerProxyModel.LineLayer)
        self.new_situation_picker.setFilters(QgsMapLayerProxyModel.LineLayer)

        self.home_locations.setFilters(QgsMapLayerProxyModel.PolygonLayer)
        self.work_locations.setFilters(QgsMapLayerProxyModel.PolygonLayer)

        # Attach button listeners
        self.query_movement_pairs_button.clicked.connect(self.query_movement_pairs)
        self.perform_routeplanning_button.clicked.connect(self.run_routeplanning)
        self.calculate_traffic_shift_button.clicked.connect(self.calculate_traffic_diff)
        self.impact_instance_selector.currentIndexChanged.connect(self.update_scenario_picker)
        self.save_area_outline.clicked.connect(self.save_outline_as_layer)
        self.save_impact_url_button.clicked.connect(self.save_impact_url)
        # Attach mode check for FOD
        self.include_cyclists.clicked.connect(self.check_fod_modes)
        self.include_car.clicked.connect(self.check_fod_modes)
        self.include_train.clicked.connect(self.check_fod_modes)
        self.include_public_transport.clicked.connect(self.check_fod_modes)
        self.check_fod_modes()

        # At last: update and set the profile explanations
        self.profile_picker.currentIndexChanged.connect(self.update_profile_explanation)
        self.scenario_picker.currentIndexChanged.connect(self.update_profile_picker)
        self.update_scenario_picker()
        self.update_profile_explanation()

        # Keep track of the last selected state of qcomboboxes
        state_tracker.init_and_connect("home_locations", self.home_locations)
        state_tracker.init_and_connect("work_locations", self.work_locations)
        state_tracker.init_and_connect("impact_instance_selector", self.impact_instance_selector)
        state_tracker.init_and_connect("scenario_picker", self.scenario_picker, self.impact_instance_selector)
        state_tracker.init_and_connect("departure_layer_picker", self.departure_layer_picker)
        state_tracker.init_and_connect("arrival_layer_picker", self.arrival_layer_picker)
        state_tracker.init_and_connect("movement_pairs_layer_picker", self.movement_pairs_layer_picker)
        state_tracker.init_and_connect("profile_picker", self.profile_picker, self.scenario_picker)
        state_tracker.init_and_connect("zero_situation_picker", self.zero_situation_picker)
        state_tracker.init_and_connect("new_situation_picker", self.new_situation_picker)
        state_tracker.init_and_connect_textfield("impact_url", self.impact_url_textfield)
        # disable components for the next version which require auth
        self.remove_auth_components()

        self.save_impact_url()  # TODO this should not be needed when login works
        self.log("version 2022-01-26 14:19")

    def remove_auth_components(self):
        self.label.hide()
        self.api_key_field.hide()
        self.save_settings_button.hide()
        self.label_2.hide()
        self.impact_instance_selector.hide()
        self.save_area_outline.hide()

    def update_instance_picker(self, projects):
        self.log("Got " + str(len(projects)) + " projects!")
        if (len(projects) == 0):
            self.error_user(self.tr("Could not load any project file"))
            return

        clean_parts = map(lambda p: p["clientId"] + "/" + p["id"], projects)
        for clean_part in clean_parts:
            self.impact_instance_selector.addItem(clean_part)

    def save_outline_as_layer(self):
        instance = self.impact_instance_selector.currentText()

        def withGeoJsonFeature(obj):
            filename = self.path + "/" + "outline_" + instance.replace("/", "_") + ".geojson"
            f = open(filename, "w+")
            json.dump(
                {
                    "type": "FeatureCollection",
                    "features": [
                        {
                            "properties": {
                                "outline": instance
                            },
                            "geometry": obj
                        }]
                }
                , f)
            f.close()
            qgsLayer = create_layer_from_file(self.iface, filename)
            self.layer_styling.style_impact_outline(qgsLayer)
            pass

        self.impact_api.get_outline(instance, withGeoJsonFeature)

    def save_setting(self):
        self.log("Saving settings")
        api_key = self.api_key_field.text()
        self.user_settings.setValue("anyways.eu/impact/api_key", api_key)

        if api_key is not None and api_key != "":
            self.user_settings.setValue("anyways.eu/impact/api_key", "")
            self.impact_api = impact_api.impact_api(None)
            self.impact_api.load_available_projects(self.update_instance_picker, lambda e: self.error_user(
                "Could not fetch projects with this token: " + e))

    def save_impact_url(self):
        self.log("Saving impact url")
        url = self.impact_url_textfield.text()
        url = impact_api.extract_instance_name(url)
        self.impact_url_textfield.setText(url)

        self.impact_instance_selector.addItem(url)
        self.impact_instance_selector.setCurrentIndex(self.impact_instance_selector.count() - 1)
        self.update_scenario_picker()

    def query_movement_pairs(self):
        """
        The button to load movement pairs between two areas has been clicked. Lets load them!
        :return: 
        """

        self.query_movement_pairs_button.setEnabled(False)

        from_features = extract_valid_geometries(self.iface, transform_layer_to_WGS84(
            self.home_locations.currentLayer()))

        to_features = extract_valid_geometries(self.iface, transform_layer_to_WGS84(
            self.work_locations.currentLayer()))

        mode = []
        if self.include_cyclists.isChecked():
            mode.append('Bicycle')

        if self.include_car.isChecked():
            mode.append('Car')

        if self.include_train.isChecked():
            mode.append('Train')

        if self.include_public_transport.isChecked():
            mode.append('BusOrTram')

        if len(mode) == 0:
            self.error_user("Select at least one transportation mode")
            return

        fod = fod_api.fod_api()

        def withData(d):
            if len(d['features']) == 0:
                self.error_user(
                    "Nothing found in the selected area. Make sure you loaded a polygon that is big enough and is located in Flanders",
                    30)
                self.query_movement_pairs_button.setEnabled(True)
                return

            # write FOD movements in a JSON file
            timestr = time.strftime("%Y%m%d_%H%M%S")
            filename = self.path + "/" + "fod_movement_pairs_" + str.join("_", mode) + "_" + timestr + ".geojson"
            self.log("Writing movement pairs to " + filename)
            f = open(filename, "w+")
            json.dump(d, f)
            f.close()

            lyr = QgsVectorLayer(filename, "FOD Movement pairs " + str.join(" ", mode) + " " + timestr, "ogr")
            QgsProject.instance().addMapLayer(lyr)

            self.query_movement_pairs_button.setEnabled(True)

        def onError(err):
            self.error_user(self.tr("Could not query the FOD-API: ") + err)
            self.query_movement_pairs_button.setEnabled(True)

        d = fod.request(from_features, to_features, withData, onError, mode)

    def check_fod_modes(self):
        mode = []
        if self.include_cyclists.isChecked():
            mode.append('Bicycle')

        if self.include_car.isChecked():
            mode.append('Car')

        if self.include_train.isChecked():
            mode.append('Train')

        if self.include_public_transport.isChecked():
            mode.append('BusOrTram')

        if len(mode) == 0:
            self.query_movement_pairs_button.setEnabled(False)
            self.query_movement_pairs_button.setText(self.tr("Please select at least one mode to query movement pairs"))
        else:
            self.query_movement_pairs_button.setEnabled(True)
            self.query_movement_pairs_button.setText(self.tr("Query movement pairs"))


    def createHistLayer(self, features, name, profile, scenario_index):
        if len(features) > 0:
            histogram = feature_histogram.feature_histogram(features)
            geojson = histogram.to_geojson()
            filename = self.path + "/" + name + ".geojson"
            f = open(filename, "w+")
            f.write(json.dumps(geojson))
            f.close()

            lyr = QgsVectorLayer(filename, name, "ogr")
            QgsProject.instance().addMapLayer(lyr)
            self.layer_styling.style_routeplanning_layer(lyr, profile, scenario_index)


    def createFailLayer(self, failed_linestrings, name, profile, scenario_index):
        if len(failed_linestrings) > 0:
            self.error_user(
                self.tr("Not every requested route could be calculated; ") + str(
                    len(failed_linestrings)) + self.tr(" routes failed. A layer with failed requests has been created"))
            histogram = feature_histogram.feature_histogram(failed_linestrings)
            geojson = histogram.to_geojson()
            timestr = time.strftime("%Y%m%d_%H%M%S")
            filename = self.path + "/" + name + ".geojson"
            f = open(filename, "w+")
            f.write(json.dumps(geojson))
            f.close()

            lyr = QgsVectorLayer(filename, name, "ogr")
            QgsProject.instance().addMapLayer(lyr)
            self.layer_styling.style_routeplanning_layer(lyr, "FAILED", scenario_index)    


    def perform_many_to_many_routeplanning(self, routing_api_obj, profile, from_coors, to_coors, scenario, scenario_index, with_routes_callback, with_failed_features_callback, prep_feature_at = None):
        """
        
        :param routing_api_obj: 
        :param profile: 
        :param from_coors: 
        :param to_coors: 
        :param scenario: 
        :param scenario_index: 
        :param with_routes_callback: 
        :param with_failed_features_callback: 
        :param prep_feature_at: (i: number, j: number, feature: geojson) => void. This is used e.g. to set a count on featuers at a certain position in the returned matrix
        :return: 
        """
        
        
        def routeplanning_many_to_many_done(routes):
            # routes: featureCollection[][]
            features = []
            failed_linestrings = []
        
            from_index = 0
            self.log("Routeplanning finished and JSON parsed; inspecting the routes now")
            for route_list in routes["routes"]:
                to_index = 0
                for route in route_list:
        
                    if "error" in route:
                        err_msg = route["error_message"]
                        self.log("Route failed because of "+err_msg)
                        fromC = list(reversed(from_coors[from_index]))
                        toC = list(reversed(to_coors[to_index]))
                        # Something went wrong here
                        failed_linestrings.append({
                            "type": "Feature",
                            "properties": {"error_message": err_msg,
                                           "guid": str(from_coors[from_index]) + "," + str(
                                               to_coors[to_index])},
                            "geometry": {
                                "type": "LineString",
                                "coordinates": [ fromC, toC ]
                            }
                        })
                    else:
                        for feature in route["features"]:
                            if feature["geometry"]["type"] == "Point":
                                continue
        
                            routing_api_obj.patch_feature(feature)
                            if prep_feature_at is not None:
                                prep_feature_at(from_index, to_index, feature)
        
                            features.append(feature)
        
                    to_index = to_index + 1
                from_index = from_index + 1

            self.log("First parsing or routeplanned routes finished, calling callbacks")

            with_failed_features_callback(failed_linestrings)
            with_routes_callback(features)
            self.log("Routeplanning callbacks have run callbacks")
    
            self.perform_routeplanning_button.setEnabled(True)
            self.perform_routeplanning_button.setText(self.tr("Perform routeplanning"))

        self.log("Requesting routes, isImpact? " + str(routing_api_obj.is_impact_backend))
        
        def onError(msg):
            self.error_user(msg)
            with_routes_callback(None)
        
        try:
            routing_api_obj.request_all_routes(
                from_coors, to_coors,
                profile, routeplanning_many_to_many_done, self.error_user)
        except Exception as e:
            self.log("ERROR: "+repr(e))
            self.error_user(self.tr("Planning routes failed: ")+str(e))
            

        return


    def run_routeplanning(self):
        self.perform_routeplanning_button.setEnabled(False)
        self.perform_routeplanning_button.setText(self.tr("Planning routes, please stand by..."))

        key = self.api_key_field.text()
        profile = self.profile_picker.currentText()
        scenario_index = self.scenario_picker.currentIndex()
        routing_api_obj = None
        scenario = "live"
        if (scenario_index == 0):
            # Plan against the routing api
            routing_api_obj = routing_api.routing_api(key)
        else:
            instance = self.impact_instance_selector.currentText()
            label = self.scenario_picker.currentText()
            index = label[1 + label.index(" "):]
            scenario = label[:label.index(" ")].replace("/", "_") + index
            instance_url = self.impact_api.routing_url_for_instance(instance, index)
            self.log("Initing routeplanning against " + instance_url)
            routing_api_obj = routing_api.routing_api(key, instance_url, True, self.api_key_field.text())

        features = None
        source_index = self.toolbox_origin_destination_or_movement.currentIndex()
        from_coordinate = None
        to_coordinates = None
        timestr = time.strftime("%Y%m%d_%H%M%S")

        name = "Routeplanned_hist_" + profile + "_" + scenario.replace("/", "_") + "_" + timestr


        # Which input sources do we have to use?
        if source_index == 0:
            
            # We have to do a matrix call, resulting in n*m features
            
            from_layer = self.departure_layer_picker.currentLayer()
            to_layer = self.arrival_layer_picker.currentLayer()

            from_coordinates = extract_valid_geometries(self.iface, transform_layer_to_WGS84(from_layer))
            to_coordinates = extract_valid_geometries(self.iface, transform_layer_to_WGS84(to_layer))


            from_coors = extract_coordinates_array(from_coordinates, True)
            to_coors = extract_coordinates_array(to_coordinates, True)
            
            def add_count(i, j, feature):
                try:
                    # If the origin point had had a 'count'-attribute, this is applied onto the feature
                    feature['properties']['count'] = from_coordinates[i].attribute('count')
                except:
                    pass

            def with_routes_callback(features):
                self.perform_routeplanning_button.setEnabled(True)
                self.perform_routeplanning_button.setText(self.tr("Perform routeplanning again"))
                
                # If there is a count on the departure coordinate, this count is copied to the correspondig feature
                
                self.createHistLayer(features, name, profile, scenario_index)
              

            def with_failed(failed):
                self.createFailLayer(failed, name, profile, scenario_index)

            self.perform_many_to_many_routeplanning(routing_api_obj, profile, from_coors, to_coors, scenario, scenario_index, with_routes_callback, with_failed, add_count)
        else:
            
            # We have to calculate 'N' classical routes
            
            
            line_layer = self.movement_pairs_layer_picker.currentLayer()
            line_features = extract_valid_geometries(self.iface, transform_layer_to_WGS84(line_layer))
            name = "Routeplanned_hist_" + profile + "_" + scenario.replace("/", "_") + "_" + timestr
            name_failed = "Failed_" + profile + "_" + scenario.replace("/", "_") + "_" + timestr



            # UP next: we have a whole bunch of lines, which might have a common departure- or endpoint, so we merge those together

            grouped_per_departure_coordinate = {}
            grouped_per_arrival_coordinate = {}

            # Keeps track of the counts: { "departure"  {"arrival" --> count}}
            counts = dict()

            for line_feature in line_features:
                line = line_feature.geometry().asPolyline();

                if len(line) != 2:
                    self.log(
                        "Some lines in " + line_layer.name() + " have intermediate points. This is not supported and might result in bugs. Only keeping the origin and the destination")
                    while len(line) > 2:
                        del line[1]
                
                departure = [line[0].y(), line[0].x()]
                arrival = [line[1].y(), line[1].x()]

                departure_str = str(departure)
                arr_str = str(arrival)
                
                try:
                    count = line_feature.attribute('count')
                    if departure_str not in counts:
                        counts[departure_str ] = dict()
                    counts[departure_str][arr_str] = count
                except:
                    # KeyError: key count not found...
                    pass
                
                if departure_str in grouped_per_departure_coordinate:
                    grouped_per_departure_coordinate[departure_str].append(arrival)
                else:
                    grouped_per_departure_coordinate[departure_str] = [arrival]

                if arr_str in grouped_per_arrival_coordinate:
                    grouped_per_arrival_coordinate[arr_str].append(departure)
                else:
                    grouped_per_arrival_coordinate[arr_str] = [departure]

            # We can now pick _either_ of them calculate it. Every key in a dict means one network call, so we pick the one with the least keys
            # What still needs to be done is appended in this list typed as: [ ([departurePoints], [arrivalPoints]) ]    
            toDo = []
            
            if len(grouped_per_departure_coordinate.keys()) < len(grouped_per_arrival_coordinate.keys()):
                # We work with one departure coordinate --> many arrivals
                target_count = len(grouped_per_departure_coordinate.keys())
                for (departure, arrivals) in grouped_per_departure_coordinate.items():
                    departure = json.loads(departure) # TOtally cheating 
                    toDo.append(([departure], arrivals))
            else:
                target_count = len(grouped_per_arrival_coordinate.keys())
                for (arrival, departures) in grouped_per_arrival_coordinate.items():
                    arrival = json.loads(arrival)
                    toDo.append((departures, [arrival]))


            # Allright: this is another bit of cheating.
            # Requesting everythin at once would crash QGIS
            # So, instead, we run the routeplanning. The callback for this routeplanning will gather the results in 'results' and trigger of a new routeplanning
            results = list()
            failed = list()

            def append_failed(failed_features):
                failed.extend(failed_features)
                self.log("Extended the failed list with "+str(len(failed_features))+" up to "+str(len(failed)))
            
            def register_result_and_run_next(features):
                if features is not None:
                    results.extend(features)
                self.perform_routeplanning_button.setText("Performing routeplanning, "+str(len(toDo))+" left...")
    
                # self.createHistLayer(features , name, profile, scenario_index)
                if len(toDo) == 0:
                    self.perform_routeplanning_button.setEnabled(True)
                    self.perform_routeplanning_button.setText(self.tr("Perform routeplanning again"))
                    self.createHistLayer(results , name, profile, scenario_index)
                    if (len(failed) > 0):
                        self.log("Creating a fail-layer with "+str(len(failed)))
                        self.createFailLayer(failed, name_failed, profile, scenario_index)
                else:
                    (departures, arrivals) = toDo.pop()
                    def add_count(i, j, feature):
                        dep_str = str(departures[i])
                        arr_str = str(arrivals[i])
                        if dep_str not in counts:
                            return
                        if arr_str not in counts[dep_str]:
                            return
                        feature['properties']['count'] = counts[dep_str][arr_str]
                        
                    self.perform_many_to_many_routeplanning(routing_api_obj, profile, departures, arrivals, scenario, scenario_index, register_result_and_run_next, append_failed, add_count)

            register_result_and_run_next(None)


    def calculate_traffic_diff(self):
        zero_layer = self.zero_situation_picker.currentLayer()
        new_layer = self.new_situation_picker.currentLayer()

        if zero_layer == new_layer:
            self.error_user(self.tr("The selected layers are the same - not much traffic shift to calculate"))
            return

        geo_features = layer_as_geojson_features(self.iface, zero_layer)
        zero_hist = feature_histogram.feature_histogram(geo_features)
        new_hist = feature_histogram.feature_histogram(layer_as_geojson_features(self.iface, new_layer))

        diff = new_hist.subtract(zero_hist)

        geojson = diff.to_geojson()
        name = "traffic_shift_between_" + zero_layer.name() + "_and_" + new_layer.name()
        filename = self.path + "/" + name.replace("/", "_") + ".geojson"
        f = open(filename, "w+")
        f.write(json.dumps(geojson))
        f.close()

        lyr = QgsVectorLayer(filename, name, "ogr")
        QgsProject.instance().addMapLayer(lyr)
        self.layer_styling.style_traffic_shift_layer(lyr, diff.natural_boundaries(5))

        lyr.setLabelsEnabled(True)
        lyr.triggerRepaint()

        pass

    def update_scenario_picker(self):
        instance_name = self.impact_instance_selector.currentText()
        self.log("Current instance name: " + instance_name)

        def withScenarioList(scenarios):
            self.log("Found scenarios " + ", ".join(scenarios))

            picker = self.scenario_picker
            self.state_tracker.pause_loading()

            picker.clear()

            picker.addItem(self.tr("Plan with a recent version of OpenStreetMap"))

            for item in scenarios:
                key = instance_name + " " + item
                if picker.findText(key) < 0:
                    # Item hasn't been added previously
                    picker.addItem(key)
            self.state_tracker.resume_loading()

        found_instances = self.impact_api.detect_instances(instance_name, withScenarioList)

    def update_profile_picker(self):
        """
        Should be called whenever the scenario is changed
        Makes sure the profile picker only show appropriate entries
        :return: 
        """

        self.state_tracker.pause_loading()
        self.profile_picker.clear()

        index = self.scenario_picker.currentIndex()
        if (index == 0):
            # This is the live API
            self.log("Setting profiles: " + ",".join(self.profile_keys))
            self.profile_picker.addItems(self.profile_keys)
        else:
            # An impact instance
            text = self.scenario_picker.currentText()
            self.profile_picker.addItems(impact_api.SUPPORTED_PROFILES)
        self.state_tracker.resume_loading()

    def update_profile_explanation(self):
        """
        Called whenever the explanation for the profile should be updated
        :return: 
        """

        current_profile = self.profile_picker.currentText()
        explanations = self.profile_descriptions
        if (explanations is None):
            return
        explanation = self.tr("Select a profile above. What the profile does will be shown here...")
        if current_profile in explanations:
            explanation = explanations[current_profile]
        self.profile_explanation.setText(explanation)

    def log(self, msg):
        QgsMessageLog.logMessage(msg, 'ImPact Toolbox', level=Qgis.Info)

    def warn(self, msg):
        QgsMessageLog.logMessage(msg, 'ImPact Toolbox', level=Qgis.Warning)

    def warn_user(self, msg):
        iface.messageBar().pushMessage('ImPact_toolbox Warning', msg, level=Qgis.Warning)

    def error_user(self, msg, duration=60):
        self.iface.messageBar().pushMessage(u'ImPact_toolbox Error', msg, level=Qgis.Critical, duration=duration)
