# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Anomalies2Contours
                                 A QGIS plugin
 This plugin converts statistical anomalies into contours based on custom parameters.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-07-28
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Marhes S. & Hudak V.
        email                : sergemarhes@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from PyQt5.QtGui import QPixmap, QImage, QColor, QPainter
from PyQt5.QtCore import QRectF
from PyQt5.QtWidgets import QMessageBox
from qgis.PyQt.QtWidgets import QAction, QGraphicsScene, QDialogButtonBox, QFileDialog
from qgis.core import QgsProject, QgsRasterLayer, QgsVectorLayer, QgsApplication, QgsGeometry
from PyQt5.QtCore import Qt
from .anomalies2contours_dialog import Anomalies2ContoursDialog
import os, tempfile, shutil
from osgeo import gdal, ogr, osr
import numpy as np
import matplotlib.pyplot as plt
from io import BytesIO

class Anomalies2Contours:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.raster_path = None
        self.raster_temp_dir = None
        self.shp_path = None
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'Anomalies2Contours_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Anomalies to contours')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = True
        self._running = False

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('Anomalies2Contours', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToRasterMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        plugin_dir = os.path.dirname(__file__)
        icon_path = os.path.join(plugin_dir, 'icon.png')
        
        self.add_action(
            icon_path,
            text=self.tr(u'Сonverts statistical anomalies into contours'),
            callback=self.run,
            parent=self.iface.mainWindow())


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginRasterMenu(
                self.tr(u'&Anomalies to contours'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        self.dlg = Anomalies2ContoursDialog()

        if self.first_start:
            try:
                self.dlg.applyButton.clicked.disconnect()
            except Exception:
                pass

        # show the dialog
        self.dlg.on_raster_changed(0)
        self.dlg.show()
        self.dlg.raise_()
        self.dlg.activateWindow()
        layer = self.dlg.mMapLayerComboBox.currentLayer()
        if layer is None:
            QMessageBox.warning(self.iface.mainWindow(), "Увага", "Спочатку додайте растровий шар.")
            return  # Зупиняє виконання функції
        # ... решта коду, яка виконується лише якщо шар вибраний
        self.dlg.doubleSpinBox.valueChanged.connect(self.on_doubleSpinBox)
        self.dlg.pushButton_2.clicked.connect(self.on_pushButton_2)
        self.dlg.pushButton.clicked.connect(self.download_histogram)
        self.dlg.button_box.button(QDialogButtonBox.Apply).clicked.connect(self.on_apply_button_clicked, Qt.UniqueConnection)
        self.first_start = False
        # Викликаємо цикл подій діалогу
        result = self.dlg.exec_()
        # Якщо натиснута кнопка OK
        if result:
            pass


    def on_doubleSpinBox(self):
        self.check_threshold()

    def on_pushButton_2(self):
        if self.first_start:
            self.on_doubleSpinBox()    
            self.show_histogram_in_graphicsview()
        else:
            self.show_histogram_in_graphicsview()

    def get_raster_band(self):
        band = self.dlg.mRasterBandComboBox.currentData()
        if band is None:
            band = 1  # За замовчуванням, якщо нічого не вибрано
        return band
    
    def download_histogram(self):
        scene = self.dlg.graphicsView.scene()
        if scene is not None:
            rect = scene.sceneRect()
            image = QImage(int(rect.width()), int(rect.height()), QImage.Format_ARGB32)
            image.fill(QColor('white').rgba())
            painter = QPainter(image)
            scene.render(painter)
            painter.end()
            # Відкрити діалог збереження файлу
            file_path, _ = QFileDialog.getSaveFileName(
                None,
                "Зберегти графік",
                "Гістограма.png",
                "PNG файли (*.png);;JPEG файли (*.jpg);;PDF файли (*.pdf);;Усі файли (*)"
            )        
        
            if file_path:
                image.save(file_path)

    def set_histogram_range(self, left, right):
        self.dlg.doubleSpinBox_2.setMinimum(left)
        self.dlg.doubleSpinBox_2.setMaximum(right)
        self.dlg.doubleSpinBox_3.setMinimum(left)
        self.dlg.doubleSpinBox_3.setMaximum(right)
    
    def set_histogram_range_values(self, left, right):
        self.dlg.doubleSpinBox_2.setValue(left)
        self.dlg.doubleSpinBox_3.setValue(right)

    def on_apply_button_clicked(self):
        self.raster_creation()
        self.shp_path = self.vector_creation()

        vector_qgis_layer = QgsVectorLayer(self.shp_path, "vectorized", "ogr")
        QgsProject.instance().addMapLayer(vector_qgis_layer)
    
    def show_histogram_in_graphicsview(self):
        raster = self.dlg.mMapLayerComboBox.currentLayer()
        raster_url = raster.dataProvider().dataSourceUri()
        ds = gdal.Open(raster_url)
        band = self.get_raster_band()
        raster_array = np.array(ds.GetRasterBand(band).ReadAsArray())
        raster_array = raster_array[np.isfinite(raster_array)]
        
        # 4. Побудувати лінійний графік розподілу
        min_val = np.min(raster_array)
        max_val = np.max(raster_array)
        self.set_histogram_range(min_val, max_val)
        # Дуже велика кількість бінів — щоб була майже безперервна лінія
        bins = np.linspace(min_val, max_val, 512)
        counts, edges = np.histogram(raster_array, bins=bins)
        centers = (edges[:-1] + edges[1:]) / 2
    
        # 5. Графік
        fig, ax = plt.subplots(figsize=(6, 4), dpi=120)
        ax.plot(centers, counts, color='gray', linewidth=1)
        
        threshold_left = self.dlg.doubleSpinBox_2.value()
        threshold_right = self.dlg.doubleSpinBox_3.value()
        
        # Виділення лівого хвоста
        ax.axvspan(centers.min(), threshold_left, color='blue', alpha=0.3, label='Below threshold')
        # Виділення правого хвоста
        ax.axvspan(threshold_right, centers.max(), color='red', alpha=0.3, label='Above threshold')
        
        ax.set_title("Raster histogram", fontsize=12)
        ax.set_xlabel("Value", fontsize=10)
        ax.set_ylabel("Frequency", fontsize=10)
        ax.grid(True, linestyle='--', alpha=0.3)
        ax.legend()
    
        buf = BytesIO()
        plt.tight_layout()
        plt.savefig(buf, format='png')
        plt.close(fig)
        buf.seek(0)
    
        # Створюємо QPixmap з буфера
        image = QImage()
        image.loadFromData(buf.getvalue())
        pixmap = QPixmap.fromImage(image)
    
        # Додаємо у GraphicsView
        scene = QGraphicsScene()
        scene.addPixmap(pixmap)
        scene.setSceneRect(QRectF(pixmap.rect()))  # встановити розмір сцени = розміру зображення
        self.dlg.graphicsView.setScene(scene)
        self.dlg.graphicsView.fitInView(scene.sceneRect(), Qt.KeepAspectRatio)
    
    def check_threshold(self):
        #перевірка критеріїв за умовою порогу
        if self.dlg.radioButton_4.isChecked() is False:
            # Отримуємо дані растра
            raster = self.dlg.mMapLayerComboBox.currentLayer()
            raster_url = raster.dataProvider().dataSourceUri()
            ds = gdal.Open(raster_url)
            band = self.get_raster_band()
            raster_array = np.array(ds.GetRasterBand(band).ReadAsArray())
            mean = np.mean(raster_array)  # Середнє значення
            std_dev = np.std(raster_array)  # Стандартне відхилення
            threshold = self.dlg.doubleSpinBox.value()
            mask_array = np.where( abs((raster_array - mean) / std_dev) > threshold, 1, 0)
            # Вибрати лише ті значення, які відповідають масці, не є нулем і менші за mean
            masked_left_values = raster_array[(mask_array == 1) & (raster_array != 0) & (raster_array < mean)]
            masked_right_values = raster_array[(mask_array == 1) & (raster_array != 0) & (raster_array > mean)]
            if masked_left_values.size > 0:
                left_max_value = masked_left_values.max()
            else:
                left_max_value = np.min(raster_array)  # або np.nan, або інше значення за замовчуванням
            if masked_right_values.size > 0:
                right_min_value = masked_right_values.min()
            else:
                right_min_value = np.max(raster_array)  # або np.nan, або інше значення за замовчуванням
            self.set_histogram_range_values(left_max_value, right_min_value)
    
    def  raster_creation(self):
        if self.raster_temp_dir is not None:
            if os.path.exists(self.raster_temp_dir):
                shutil.rmtree(self.raster_temp_dir)  # Видаляє папку разом з усіма файлами
                print(f"Тимчасова папка видалена: {self.raster_temp_dir}")
        # Отримуємо дані растра
        raster = self.dlg.mMapLayerComboBox.currentLayer()
        raster_url = raster.dataProvider().dataSourceUri()
        extent = raster.extent()
        ds = gdal.Open(raster_url)
        wkt = ds.GetProjection()
        band = self.get_raster_band()
        raster_array = np.array(ds.GetRasterBand(band).ReadAsArray())
        width, height = raster.width(), raster.height()
        
        self.check_threshold()
        
        threshold_left = self.dlg.doubleSpinBox_2.value()
        threshold_right = self.dlg.doubleSpinBox_3.value()
        
        # Створення маски
        mask_array = np.where((raster_array < threshold_left) | (raster_array > threshold_right), 1, 0)

        # Створюємо тимчасовий файл для растру
        raster_temp_dir = tempfile.mkdtemp()
        temp_tif = tempfile.NamedTemporaryFile(dir=raster_temp_dir,suffix=".tif", delete=False)
        self.raster_path = temp_tif.name
        temp_tif.close()  # Закриваємо, щоб GDAL міг його використовувати

        # Визначаємо тип даних та створюємо GDAL датасет
        driver = gdal.GetDriverByName('GTiff')
        dataset = driver.Create(self.raster_path, width, height, 1, gdal.GDT_Float32)

        # Встановлюємо геоприв’язку
        dataset.SetGeoTransform([
            extent.xMinimum(),
            (extent.xMaximum() - extent.xMinimum()) / width, 0,
            extent.yMaximum(), 0,
            (extent.yMinimum() - extent.yMaximum()) / height
        ])

        dataset.SetProjection(wkt)

        # Записуємо масив у растр
        band = dataset.GetRasterBand(1)
        band.WriteArray(mask_array)
        band.SetNoDataValue(0)

        # Закриваємо датасет
        dataset.FlushCache()
        dataset = None

        print(f"Тимчасовий растр створено: {self.raster_path}")
        # Додаємо растр у QGIS
        raster_layer = QgsRasterLayer(self.raster_path, "Temporary Raster", "gdal")
        QgsProject.instance().addMapLayer(raster_layer)

        # Видалення тимчасової папки після закриття QGIS
        def remove_temp_file():
            if os.path.exists(raster_temp_dir):
                shutil.rmtree(raster_temp_dir)  # Видаляє папку разом з усіма файлами
                print(f"Тимчасова папка видалена: {raster_temp_dir}")

        # Додаємо функцію видалення при закритті QGIS
        QgsApplication.instance().aboutToQuit.connect(remove_temp_file)
        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            if isinstance(layer, QgsRasterLayer) and layer.source() == self.raster_path:
                QgsProject.instance().removeMapLayer(layer)

    def vector_creation(self):
        raster = self.dlg.mMapLayerComboBox.currentLayer()
        raster_url = raster.dataProvider().dataSourceUri()
        ds = gdal.Open(raster_url)
        mask_ds = gdal.Open(self.raster_path)
        mask_array = mask_ds.GetRasterBand(1)
        band = ds.GetRasterBand(1)
        band_array = band.ReadAsArray()
        wkt = ds.GetProjection()
    
        temp_dir = tempfile.mkdtemp()
        shp_path = os.path.join(temp_dir, "temp_vectorized.shp")
    
        drv = ogr.GetDriverByName("ESRI Shapefile")
        vector_ds = drv.CreateDataSource(shp_path)
        srs = osr.SpatialReference()
        srs.ImportFromWkt(wkt)
        vector_layer = vector_ds.CreateLayer("polygons", srs, ogr.wkbPolygon)
    
        # Поля
        vector_layer.CreateField(ogr.FieldDefn("id", ogr.OFTInteger))
        fields = [("min", ogr.OFTReal), ("max", ogr.OFTReal), ("mean", ogr.OFTReal),
                  ("stddev", ogr.OFTReal), ("sum", ogr.OFTReal), ("range", ogr.OFTReal),
                  ("count", ogr.OFTInteger), ("area", ogr.OFTReal)]
        for name, type_ in fields:
            field = ogr.FieldDefn(name, type_)
            field.SetWidth(10)
            field.SetPrecision(3)
            vector_layer.CreateField(field)
    
        # Полігонізація
        gdal.Polygonize(band, mask_array, vector_layer, -1, [], callback=None)
        vector_layer.ResetReading()
        # Обробка полігонів: фільтрація, статистика, оновлення атрибутів
        fids_to_delete = []
        stats_list = []
    
        for feature in vector_layer:
            fid = feature.GetFID()
            geom = feature.GetGeometryRef()
            # if geom is None:
            #     fids_to_delete.append(fid)
            #     continue
        
            # Обчислення площі
            area_val = QgsGeometry.fromWkt(geom.ExportToWkt()).area()
        
            # Створення тимчасового шару з одним полігоном (без атрибутів)
            mem_driver = ogr.GetDriverByName('Memory')
            mem_ds = mem_driver.CreateDataSource('memData')
            mem_layer = mem_ds.CreateLayer('single', srs, ogr.wkbPolygon)
        
            out_feature = ogr.Feature(mem_layer.GetLayerDefn())
            out_feature.SetGeometry(geom.Clone())
            mem_layer.CreateFeature(out_feature)
            
            # Растеризація
            mem_drv = gdal.GetDriverByName("MEM")
            mask_ds = mem_drv.Create("", band.XSize, band.YSize, 1, gdal.GDT_Byte)
            mask_ds.SetProjection(ds.GetProjection())
            mask_ds.SetGeoTransform(ds.GetGeoTransform())
            gdal.RasterizeLayer(mask_ds, [1], mem_layer, burn_values=[1])
        
            mask = mask_ds.GetRasterBand(1).ReadAsArray()
            values = band_array[mask == 1]
        
            if values.size <= 1:
                fids_to_delete.append(fid)
                continue
            min_val = float(np.min(values))
            max_val = float(np.max(values))
            mean_val = float(np.mean(values))
            sum_val = float(np.sum(values))
            stddev_val = float(np.std(values))
            range_val = max_val - min_val
            count_val = int(values.size)
            stats_list.append((area_val, fid, min_val, max_val, mean_val, stddev_val, sum_val, range_val, count_val))
            
            
        vector_layer.ResetReading()
        stats_list.sort(key=lambda x: x[0], reverse=True)
        vector_layer.ResetReading()
        
        # === 5. Присвоєння статистики та ID полігонам ===
        for idx, (area_val, fid, min_val, max_val, mean_val, stddev_val, sum_val, range_val, count_val) in enumerate(stats_list, start=1):
            feature = vector_layer.GetFeature(fid)
            feature.SetField("id", idx)
            feature.SetField("min", min_val)
            feature.SetField("max", max_val)
            feature.SetField("mean", mean_val)
            feature.SetField("stddev", stddev_val)
            feature.SetField("sum", sum_val)
            feature.SetField("range", range_val)
            feature.SetField("area", area_val)
            feature.SetField("count", count_val)
            vector_layer.SetFeature(feature)
        
        # Видалити зайві полігони
        for fid in fids_to_delete:
            vector_layer.DeleteFeature(fid)

        vector_ds.FlushCache()
        vector_ds = None  # Зберегти зміни

        # Тимчасове очищення
        def remove_temp_file():
            if os.path.exists(temp_dir):
                shutil.rmtree(temp_dir)
    
        QgsApplication.instance().aboutToQuit.connect(remove_temp_file)
    
        # Видалення попереднього шару, якщо є
        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            if isinstance(layer, QgsVectorLayer) and layer.source() == self.shp_path:
                QgsProject.instance().removeMapLayer(layer)
        return shp_path