# -*- coding: utf-8 -*-

"""
/***************************************************************************
 AMERTA
                                 A QGIS plugin
 Analisis Multi-kriteria Embung dan Rencana Tata Air
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-09-18
        copyright            : (C) 2025 by Badan Riset dan Inovasi Nasional
        email                : sitaranisafitri@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Sitarani Safitri, Orbita Roswintiarti, Okta Fajar Saputra, Galdita Aruba Chulafak, Gatot Nugroho, Wismu Sunarmodo, Kusumaning Ayu Dyah Sukowati, Hana Listi Fitriana'
__date__ = '2025-09-18'
__copyright__ = '(C) 2025 by Badan Riset dan Inovasi Nasional'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os, re, glob, processing
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing, QgsProcessingAlgorithm, QgsProcessingException,
    QgsProcessingParameterFile, QgsProcessingParameterString,
    QgsProcessingParameterNumber, QgsProcessingParameterRasterDestination,
    QgsProcessingUtils
)
import os
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QUrl

class ChirpsAnnualSumAlgorithm(QgsProcessingAlgorithm):
    INPUT_DIR   = 'INPUT_DIR'
    YEAR        = 'YEAR'
    FILE_PREFIX = 'FILE_PREFIX'
    FILE_SUFFIX = 'FILE_SUFFIX'
    OUTPUT      = 'OUTPUT'

    def tr(self, s: str) -> str:
        return QCoreApplication.translate('Processing', s)

    def createInstance(self):
        return ChirpsAnnualSumAlgorithm()

    def name(self):
        return 'b_chirps_annual_sum'

    def displayName(self):
        return self.tr('CHIRPS – Annual Rainfall (Sum)')

    def groupId(self):
        return 'A. RainPCA'

    def group(self):
        return self.tr(self.groupId())
    
    def icon(self):
        return QIcon(os.path.join(os.path.dirname(__file__), 'rainpca.png'))

    def shortHelpString(self):
        return self.tr("""\
        🇮🇩 ID Modul ini menghitung curah hujan tahunan (penjumlahan 12 bulan) dari data CHIRPS dalam satu folder.

        Alur pakai:
        1) Folder containing CHIRPS TIFFs: arahkan ke folder yang berisi GeoTIFF CHIRPS bulanan (01–12) untuk tahun yang sama.
        2) Year (YYYY): isi tahun target (mis. 2024).
        3) Filename prefix / suffix (opsional): isi bagian nama berkas sebelum dan sesudah penanda bulan untuk mempermudah pencocokan. Contoh pola nama: `chirps-v2.0-2024-01_jawa.tif` → prefix=`chirps-v2.0-`, suffix=`_jawa.tif`.
        4) Run → algoritma akan mencari 12 TIFF bulanan yang cocok, lalu menjumlahkan nilainya menjadi raster tahunan dalam satuan yang sama dengan input CHIRPS (mm).
        5) Annual Rainfall: simpan keluaran GeoTIFF (atau biarkan sementara).

        ──────────────
                
        🌍 EN This module computes annual rainfall (sum of 12 months) from monthly CHIRPS rasters stored in a folder.

        Usage:
        1) Folder containing CHIRPS TIFFs: point to the directory containing monthly CHIRPS GeoTIFFs (months 01–12) for the same year.
        2) Year (YYYY): set the target year (e.g., 2024).
        3) Filename prefix / suffix (optional): parts of the filename before & after the month token to help matching. Example pattern: `chirps-v2.0-2024-01_jawa.tif` → prefix=`chirps-v2.0-`, suffix=`_jawa.tif`.
        4) Run → the algorithm locates the 12 monthly TIFFs and sums them into an annual raster in the same units as CHIRPS (mm).
        5) Annual Rainfall: choose an output path (or keep a temporary layer).""")

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT_DIR, self.tr('Folder containing CHIRPS TIFFs'),
                behavior=QgsProcessingParameterFile.Folder
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.YEAR, self.tr('Year (YYYY)'),
                type=QgsProcessingParameterNumber.Integer, defaultValue=2024
            )
        )
        self.addParameter(
            QgsProcessingParameterString(
                self.FILE_PREFIX, self.tr('Filename prefix (optional)'),
                defaultValue='chirps-v2.0.'
            )
        )
        self.addParameter(
            QgsProcessingParameterString(
                self.FILE_SUFFIX, self.tr('Filename suffix (optional, e.g., _jawa.tif)'),
                defaultValue='_jawa.tif'
            )
        )
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT, self.tr('Annual Rainfall')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        folder      = self.parameterAsFile(parameters, self.INPUT_DIR, context)
        year        = int(self.parameterAsInt(parameters, self.YEAR, context))
        prefix      = self.parameterAsString(parameters, self.FILE_PREFIX, context) or ''
        suffix      = self.parameterAsString(parameters, self.FILE_SUFFIX, context) or '.tif'
        out_raster  = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)

        if not os.path.isdir(folder):
            raise QgsProcessingException(self.tr('Folder is not valid.'))

        # Pilih file: <prefix><YYYY>.<MM><suffix>
        pat = re.compile(r'^' + re.escape(prefix) + rf'{year}\.(\d{{2}})' + re.escape(suffix) + r'$')
        monthly = []
        for path in sorted(glob.glob(os.path.join(folder, '*.tif'))):
            m = pat.match(os.path.basename(path))
            if m:
                monthly.append((int(m.group(1)), path))

        if not monthly:
            raise QgsProcessingException(self.tr(f'No TIFF files were found for year {year} matching the given pattern.'))

        monthly.sort(key=lambda t: t[0])
        src_files = [p for _, p in monthly]
        feedback.pushInfo(self.tr(f'Selected rasters (year {year}): {len(src_files)} file(s).'))

        # --- Mark -9999 as NoData via per-file VRT (no duplicate -of flag!) ---
        fixed_files = []
        for i, src in enumerate(src_files, 1):
            vrt_path = QgsProcessingUtils.generateTempFilename(f'fix_nodata_{i:02d}.vrt')
            processing.run(
                'gdal:translate',
                {
                    'INPUT': src,
                    'TARGET_CRS': None,
                    'NODATA': -9999.0,          # treat -9999 as NoData
                    'COPY_SUBDATASETS': False,
                    'OPTIONS': '',               # keep empty
                    'EXTRA': '',                 # DO NOT force "-of VRT" here
                    'DATA_TYPE': 0,
                    'OUTPUT': vrt_path           # .vrt extension → QGIS sets -of VRT
                },
                context=context, feedback=feedback, is_child_algorithm=True
            )
            fixed_files.append(vrt_path)

        ref = fixed_files[0]
        tmp_sum = QgsProcessingUtils.generateTempFilename('annual_sum_tmp.tif')

        # --- SUM with NoData propagation ---
        try:
            processing.run(
                'qgis:cellstatistics',
                {
                    'INPUT': fixed_files,
                    'STATISTIC': 0,          # 0 = Sum
                    'IGNORE_NODATA': False,  # any NoData in stack → output NoData
                    'REFERENCE_LAYER': ref,
                    'OUTPUT': tmp_sum
                },
                context=context, feedback=feedback, is_child_algorithm=True
            )
        except Exception as e1:
            feedback.pushInfo(self.tr(f'Fallback ke GRASS r.series (sum) karena: {e1}'))
            processing.run(
                'grass7:r.series',
                {
                    'input': fixed_files,
                    'method': 'sum',
                    'range': None,
                    'quantile': 50,
                    'weights': '',
                    'output': tmp_sum,
                    'GRASS_REGION_PARAMETER': ref,
                    'GRASS_REGION_CELLSIZE_PARAMETER': 0,
                    'GRASS_RASTER_FORMAT_OPT': '',
                    'GRASS_RASTER_FORMAT_META': ''
                },
                context=context, feedback=feedback, is_child_algorithm=True
            )

        # --- Final translate: set NoData, compress, write to destination ---
        processing.run(
            'gdal:translate',
            {
                'INPUT': tmp_sum,
                'TARGET_CRS': None,
                'NODATA': -9999.0,
                'COPY_SUBDATASETS': False,
                'OPTIONS': 'COMPRESS=LZW',
                'EXTRA': '',
                'DATA_TYPE': 0,
                'OUTPUT': out_raster
            },
            context=context, feedback=feedback, is_child_algorithm=True
        )

        return {self.OUTPUT: out_raster}
