# -*- coding: utf-8 -*-

"""
/***************************************************************************
 AMERTA
                                 A QGIS plugin
 Analisis Multi-kriteria Embung dan Rencana Tata Air
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-09-18
        copyright            : (C) 2025 by Badan Riset dan Inovasi Nasional
        email                : sitaranisafitri@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Sitarani Safitri, Orbita Roswintiarti, Okta Fajar Saputra, Galdita Aruba Chulafak, Gatot Nugroho, Wismu Sunarmodo, Kusumaning Ayu Dyah Sukowati, Hana Listi Fitriana'
__date__ = '2025-09-18'
__copyright__ = '(C) 2025 by Badan Riset dan Inovasi Nasional'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import math
import os
from contextlib import contextmanager

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsProcessing, QgsProcessingAlgorithm, QgsProcessingException,
    QgsProcessingParameterRasterLayer, QgsProcessingParameterVectorLayer,
    QgsProcessingParameterEnum, QgsProcessingParameterNumber,
    QgsProcessingParameterVectorDestination, QgsProcessingParameterBoolean,
    QgsCoordinateReferenceSystem, QgsField, QgsVectorLayer, QgsProcessingUtils,
    QgsRasterLayer, QgsPointXY
)
import processing
import os
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QUrl

@contextmanager
def edit(layer):
    layer.startEditing()
    try:
        yield
    except Exception:
        layer.rollBack()
        raise
    else:
        layer.commitChanges()


class GridRainAlgorithm(QgsProcessingAlgorithm):
    RASTER_EIG   = 'RASTER_EIG'
    RASTER_RAIN  = 'RASTER_RAIN'
    AOI          = 'AOI'
    GRID_SIZE    = 'GRID_SIZE'
    CUSTOM_M     = 'CUSTOM_M'
    BUILD_INDEX  = 'BUILD_INDEX'
    FILL_GAPS    = 'FILL_GAPS'
    OUTPUT       = 'OUTPUT'

    GRID_CHOICES = ['30″ (≈ 925 m)', '5″ (≈ 150 m)', 'Custom (meter)']

    def tr(self, s): return QCoreApplication.translate('Processing', s)
    def name(self): return 'grid_rain_ier'
    def displayName(self): return self.tr('Grid Rainfall')
    def groupId(self): return 'C. MCDA Factors for Retention Ponds'
    def group(self): return self.tr('C. MCDA Factors for Retention Ponds')
    def icon(self):
        return QIcon(os.path.join(os.path.dirname(__file__), 'preanalysis.png'))
    def shortHelpString(self):
        return self.tr("""\
🇮🇩 ID  Modul ini membuat grid dan menghitung faktor hujan per sel: rata-rata hujan tahunan (mrain) dan variabilitas dari Eigenvector PCA (eigen). Nilai tersebut dikonversi menjadi skor sv (variabilitas) dan sa (hujan tahunan), lalu digabungkan menjadi S_Rain untuk pra-MCDA analisis lokasi embung.

Alur kerja:
1) Siapkan raster Annual Rainfall (mm/yr) dan raster PCA Eigenvector (PC1) yang diolah di modul RainPCA.
2) Siapkan data spasial Area of Interest (AOI), misalnya batas administrasi atau wilayah sungai.
3) Tentukan resolusi grid dari dropdown atau custom dalam satuan meter (30″/5″/Custom).
4) Masukkan skor berdasarkan nilai annual rainfall (sa) dan eigenvector (sv) ke dalam grid sesuai AOI.
5) Hitung S_Rain berdasarkan kombinasi sa dan sv.

Keluaran:
• Grid Rainfall berisi kolom: id, mrain, eigen, sa, sv, S_Rain.

──────────────

🌍 EN  This module builds a grid and computes per-cell rainfall factors: annual rainfall mean (mrain) and variability from PCA Eigenvector (eigen). These are converted to sv (variability) and sa (annual rainfall) scores, then combined into S_Rain for the pre-MCDA stage of small-reservoir siting.

Workflow:
1) Prepare the Annual Rainfall raster (mm/yr) and PCA Eigenvector raster (PC1) produced by the RainPCA module.
2) Prepare the Area of Interest (AOI), e.g., an administrative boundary or river-basin boundary.
3) Choose the grid resolution from the dropdown or use a custom value in meters (30″/5″/Custom).
4) Populate scores into the grid within the AOI: annual rainfall score (sa) and eigenvector-based variability score (sv).
5) Compute S_Rain from the combination of sa and sv.

Output:
• Grid Rainfall with fields: id, mrain, eigen, sa, sv, S_Rain.""")
        
    def createInstance(self): return GridRainAlgorithm()

    # ---------- Parameters ----------
    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterRasterLayer(self.RASTER_EIG,  self.tr('Eigenvector PCA (PC1)')))
        self.addParameter(QgsProcessingParameterRasterLayer(self.RASTER_RAIN, self.tr('Annual Rainfall')))
        self.addParameter(QgsProcessingParameterVectorLayer(self.AOI, self.tr('Area of Interest (AOI)'),
                                                           types=[QgsProcessing.TypeVectorPolygon]))
        self.addParameter(QgsProcessingParameterEnum(self.GRID_SIZE, self.tr('Grid Resolution'),
                                                     self.GRID_CHOICES, defaultValue=0))
        self.addParameter(QgsProcessingParameterNumber(self.CUSTOM_M, self.tr('Grid Resolution (meter) for "Custom (meter)"'),
                                                       type=QgsProcessingParameterNumber.Double,
                                                       defaultValue=1.0, minValue=0.0))
        self.addParameter(QgsProcessingParameterBoolean(self.BUILD_INDEX, self.tr('Create spatial index on output'), defaultValue=True))
        self.addParameter(QgsProcessingParameterBoolean(self.FILL_GAPS, self.tr('Fill NoData'), defaultValue=True))
        self.addParameter(QgsProcessingParameterVectorDestination(self.OUTPUT, self.tr('Grid Rainfall')))

    # ---------- Helpers ----------
    def _as_layer(self, ref, context):
        try:
            if hasattr(ref, 'extent'): return ref
            lyr = QgsProcessingUtils.mapLayerFromString(ref, context)
            if lyr is not None: return lyr
            lyr = QgsVectorLayer(ref, 'layer', 'ogr')
            if lyr is not None and lyr.isValid(): return lyr
        except Exception: pass
        raise QgsProcessingException(self.tr('Failed to load layer from reference: {}').format(ref))

    def _to_wgs84(self, layer, context, feedback):
        res = processing.run('native:reprojectlayer',
            {'INPUT': layer, 'TARGET_CRS': QgsCoordinateReferenceSystem('EPSG:4326'),
             'OPERATION':'', 'OUTPUT':'TEMPORARY_OUTPUT'},
            context=context, feedback=feedback, is_child_algorithm=True)
        return res['OUTPUT']

    def _warp_to_wgs84_res(self, raster, xres_deg, yres_deg, context, feedback):
        res = processing.run('gdal:warpreproject',
            {'INPUT': raster, 'SOURCE_CRS': None, 'TARGET_CRS': QgsCoordinateReferenceSystem('EPSG:4326'),
             'RESAMPLING': 1, 'NODATA': None, 'TARGET_RESOLUTION': None,
             'X_RES': float(xres_deg), 'Y_RES': float(yres_deg), 'MULTITHREADING': True,
             'DATA_TYPE': 0, 'TARGET_EXTENT': None, 'TARGET_EXTENT_CRS': None, 'OPTIONS': '',
             'EXTRA':'', 'OUTPUT':'TEMPORARY_OUTPUT'},
            context=context, feedback=feedback, is_child_algorithm=True)
        return res['OUTPUT']

    def _fill_nodata(self, in_raster, distance, context, feedback):
        if distance <= 0:
            return in_raster
        res = processing.run('gdal:fillnodata',
            {'INPUT': in_raster, 'BAND': 1, 'DISTANCE': int(distance), 'ITERATIONS': 0,
             'NO_MASK': True, 'MASK_LAYER': None, 'MASK_VALID_DATA': False,
             'USE_MULTITHREADING': True, 'EXTRA':'', 'OUTPUT':'TEMPORARY_OUTPUT'},
            context=context, feedback=feedback, is_child_algorithm=True)
        return res['OUTPUT']

    def _extent_str(self, layer_obj):
        ext = layer_obj.extent()
        return f'{ext.xMinimum()},{ext.xMaximum()},{ext.yMinimum()},{ext.yMaximum()} [EPSG:4326]'

    def _deg_spacing_from_option(self, option_idx, custom_m, aoi_wgs_layer_obj):
        if option_idx == 0: return 30.0/3600.0, 30.0/3600.0
        if option_idx == 1: return 5.0/3600.0, 5.0/3600.0
        c = aoi_wgs_layer_obj.extent().center()
        mean_lat_rad = math.radians(c.y())
        deg_lat = float(custom_m) / 111320.0
        coslat = max(1e-6, math.cos(mean_lat_rad))
        deg_lon = float(custom_m) / (111320.0 * coslat)
        return deg_lon, deg_lat

    def _assign_stable_id_global(self, vlayer, dx_deg, dy_deg, lon0=-180.0, lat0=-90.0, numeric=True):
        prov = vlayer.dataProvider()
        if vlayer.fields().indexOf('id') == -1:
            prov.addAttributes([QgsField('id', QVariant.Int if numeric else QVariant.String)])
            vlayer.updateFields()
        idx_id = vlayer.fields().indexOf('id')
        arcsec_x = int(round(dx_deg * 3600.0 + 1e-9))
        with edit(vlayer):
            for ft in vlayer.getFeatures():
                c = ft.geometry().centroid().asPoint()
                ix = int((c.x() - lon0) // dx_deg)
                iy = int((c.y() - lat0) // dy_deg)
                gid = int(iy) * 1_000_000 + int(ix) if numeric else f"G{arcsec_x}s_{iy:06d}_{ix:06d}"
                vlayer.changeAttributeValue(ft.id(), idx_id, gid)
        return vlayer

    def _keep_only_fields(self, vlayer, keep_names):
        fields = vlayer.fields()
        to_delete = [i for i, f in enumerate(fields) if f.name() not in keep_names]
        if to_delete:
            with edit(vlayer):
                vlayer.dataProvider().deleteAttributes(to_delete)
                vlayer.updateFields()

    # ---- Zonal helpers ----
    def _try_zonal(self, algo_id, params, context, feedback):
        try:
            res = processing.run(algo_id, params, context=context,
                                 feedback=feedback, is_child_algorithm=True)
            return res.get('OUTPUT', res.get('INPUT', None))
        except Exception as e:
            feedback.pushInfo(f'{algo_id} gagal: {e}')
            return None

    def _run_zonal_pair(self, grid_layer, rain_ras, eigen_ras, context, feedback):
        # mrain (Mean)
        mean_layer = None
        for algo in ('native:zonalstatisticsfb', 'qgis:zonalstatistics'):
            params = {'INPUT': grid_layer, 'INPUT_RASTER': rain_ras, 'RASTER': rain_ras,
                      'RASTER_BAND':1, 'COLUMN_PREFIX':'mrain_', 'STATISTICS':[2],
                      'OUTPUT':'TEMPORARY_OUTPUT'}
            if algo == 'qgis:zonalstatistics': params.pop('INPUT_RASTER', None)
            mean_layer = self._try_zonal(algo, params, context, feedback)
            if mean_layer: break
        if mean_layer is None:
            raise QgsProcessingException(self.tr('Failed to compute zonal mean for rainfall.'))

        # eigen (StdDev)
        std_layer = None
        for algo in ('native:zonalstatisticsfb', 'qgis:zonalstatistics'):
            for code in (6,5):
                params = {'INPUT': mean_layer, 'INPUT_RASTER': eigen_ras, 'RASTER':eigen_ras,
                          'RASTER_BAND':1, 'COLUMN_PREFIX':'eigen_', 'STATISTICS':[code],
                          'OUTPUT':'TEMPORARY_OUTPUT'}
                if algo == 'qgis:zonalstatistics': params.pop('INPUT_RASTER', None)
                std_layer = self._try_zonal(algo, params, context, feedback)
                if std_layer: break
            if std_layer: break
        if std_layer is None:
            raise QgsProcessingException(self.tr('Failed to compute zonal standard deviation for eigenvector.'))

        v = self._as_layer(std_layer, context)
        mrain_field = None; eigen_field = None
        for f in v.fields():
            nm = f.name().lower()
            if nm.startswith('mrain_') and 'mean' in nm: mrain_field = f.name()
            if nm.startswith('eigen_') and any(k in nm for k in ('stdev','std','dev')): eigen_field = f.name()
        if not mrain_field:
            cand = [f.name() for f in v.fields() if f.name().lower().startswith('mrain_')]
            if len(cand)==1: mrain_field = cand[0]
        if not eigen_field:
            cand = [f.name() for f in v.fields() if f.name().lower().startswith('eigen_')]
            if len(cand)==1: eigen_field = cand[0]

        null_count = 0
        if mrain_field and eigen_field:
            im = v.fields().indexOf(mrain_field)
            ie = v.fields().indexOf(eigen_field)
            for ft in v.getFeatures():
                if ft[im] is None or ft[ie] is None:
                    null_count += 1
        return v, mrain_field, eigen_field, null_count

    def _rerun_zonal_for_nulls(self, vlayer, rain_ras, eigen_ras, context, feedback):
        expr = '"mrain" IS NULL OR "eigen" IS NULL'
        sub = processing.run(
            'native:extractbyexpression',
            {'INPUT': vlayer, 'EXPRESSION': expr, 'OUTPUT': 'TEMPORARY_OUTPUT'},
            context=context, feedback=feedback, is_child_algorithm=True
        )['OUTPUT']
        sub_layer = self._as_layer(sub, context)
        if sub_layer.featureCount() == 0:
            return vlayer

        pfx_mr = 'rzmr_'; pfx_eg = 'rzeg_'
        z1 = processing.run(
            'native:zonalstatisticsfb',
            {'INPUT': sub_layer, 'INPUT_RASTER': rain_ras, 'RASTER_BAND': 1,
             'COLUMN_PREFIX': pfx_mr, 'STATISTICS': [2], 'OUTPUT': 'TEMPORARY_OUTPUT'},
            context=context, feedback=feedback, is_child_algorithm=True
        )['OUTPUT']
        try:
            z2 = processing.run(
                'native:zonalstatisticsfb',
                {'INPUT': z1, 'INPUT_RASTER': eigen_ras, 'RASTER_BAND': 1,
                 'COLUMN_PREFIX': pfx_eg, 'STATISTICS': [6], 'OUTPUT': 'TEMPORARY_OUTPUT'},
                context=context, feedback=feedback, is_child_algorithm=True
            )['OUTPUT']
        except Exception:
            z2 = processing.run(
                'native:zonalstatisticsfb',
                {'INPUT': z1, 'INPUT_RASTER': eigen_ras, 'RASTER_BAND': 1,
                 'COLUMN_PREFIX': pfx_eg, 'STATISTICS': [5], 'OUTPUT': 'TEMPORARY_OUTPUT'},
                context=context, feedback=feedback, is_child_algorithm=True
            )['OUTPUT']

        zl = self._as_layer(z2, context)
        mr_col = next((f.name() for f in zl.fields() if f.name().lower().startswith(pfx_mr) and 'mean' in f.name().lower()), None)
        eg_col = next((f.name() for f in zl.fields() if f.name().lower().startswith(pfx_eg) and any(k in f.name().lower() for k in ('stdev','std','dev'))), None)
        if not mr_col or not eg_col:
            feedback.pushInfo(self.tr('Re-run zonal: result columns not found.'))
            return vlayer

        if vlayer.fields().indexOf('id') == -1 or zl.fields().indexOf('id') == -1:
            feedback.pushInfo(self.tr('Re-run zonal: “id” column missing; skipping.'))
            return vlayer

        joined = processing.run(
            'native:joinattributestable',
            {'INPUT': vlayer, 'FIELD': 'id',
             'INPUT_2': zl, 'FIELD_2': 'id',
             'FIELDS_TO_COPY': [mr_col, eg_col], 'METHOD': 1, 'DISCARD_NONMATCHING': True,
             'PREFIX': '_rz_', 'OUTPUT': 'TEMPORARY_OUTPUT'},
            context=context, feedback=feedback, is_child_algorithm=True
        )['OUTPUT']

        j = self._as_layer(joined, context)
        idx_mr = j.fields().indexOf('mrain')
        idx_ei = j.fields().indexOf('eigen')
        idx_rz_mr = j.fields().indexOf('_rz_' + mr_col)
        idx_rz_ei = j.fields().indexOf('_rz_' + eg_col)

        with edit(j):
            for ft in j.getFeatures():
                if ft[idx_mr] is None and idx_rz_mr != -1 and ft[idx_rz_mr] is not None:
                    j.changeAttributeValue(ft.id(), idx_mr, float(ft[idx_rz_mr]))
                if ft[idx_ei] is None and idx_rz_ei != -1 and ft[idx_rz_ei] is not None:
                    j.changeAttributeValue(ft.id(), idx_ei, float(ft[idx_rz_ei]))
            # bersihkan kolom join sementara
            to_drop = []
            for nm in (idx_rz_mr, idx_rz_ei):
                if nm != -1: to_drop.append(nm)
            if to_drop:
                j.dataProvider().deleteAttributes(sorted(to_drop, reverse=True))
                j.updateFields()
        return j

    def _fill_nulls_by_sampling(self, poly_layer, rain_path, eigen_path,
                                 field_mrain='mrain', field_eigen='eigen', feedback=None):
        r_rain  = QgsRasterLayer(rain_path, 'rain')
        r_eigen = QgsRasterLayer(eigen_path, 'eigen')
        if not r_rain.isValid() or not r_eigen.isValid():
            raise QgsProcessingException(self.tr('Raster for sampling is invalid.'))
        idx_mr = poly_layer.fields().indexOf(field_mrain)
        idx_ei = poly_layer.fields().indexOf(field_eigen)
        prov_r = r_rain.dataProvider()
        prov_e = r_eigen.dataProvider()

        updated = 0
        with edit(poly_layer):
            for ft in poly_layer.getFeatures():
                if ft[idx_mr] is None:
                    cpt = ft.geometry().centroid().asPoint()
                    rv, okr = prov_r.sample(QgsPointXY(cpt), 1)
                    if okr and rv is not None:
                        try:
                            fv = float(rv)
                            if not math.isnan(fv):
                                poly_layer.changeAttributeValue(ft.id(), idx_mr, fv); updated += 1
                        except Exception:
                            pass
                if ft[idx_ei] is None:
                    cpt = ft.geometry().centroid().asPoint()
                    ev, oke = prov_e.sample(QgsPointXY(cpt), 1)
                    if oke and ev is not None:
                        try:
                            fv = float(ev)
                            if not math.isnan(fv):
                                poly_layer.changeAttributeValue(ft.id(), idx_ei, fv); updated += 1
                        except Exception:
                            pass
        if feedback: feedback.pushInfo(self.tr(f'Centroid sampling filled {updated} value(s).'))
        return poly_layer

    # ---------- Core ----------
    def processAlgorithm(self, parameters, context, feedback):
        eigen_ras = self.parameterAsRasterLayer(parameters, self.RASTER_EIG, context)
        rain_ras  = self.parameterAsRasterLayer(parameters, self.RASTER_RAIN, context)
        aoi_vec   = self.parameterAsVectorLayer(parameters, self.AOI, context)
        grid_idx  = int(self.parameterAsEnum(parameters, self.GRID_SIZE, context))
        custom_m  = float(self.parameterAsDouble(parameters, self.CUSTOM_M, context))
        make_idx  = bool(self.parameterAsBoolean(parameters, self.BUILD_INDEX, context))
        do_fill   = bool(self.parameterAsBoolean(parameters, self.FILL_GAPS, context))
        out_path  = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)

        if eigen_ras is None or rain_ras is None:
            raise QgsProcessingException(self.tr('Raster input is invalid.'))
        if aoi_vec is None:
            raise QgsProcessingException(self.tr('AOI layer is invalid.'))

        # AOI → WGS84
        aoi_wgs_ref = self._to_wgs84(aoi_vec, context, feedback)
        aoi_wgs = self._as_layer(aoi_wgs_ref, context)

        # Resolusi grid (deg)
        xdeg, ydeg = self._deg_spacing_from_option(grid_idx, custom_m, aoi_wgs)
        feedback.pushInfo(self.tr(f'Grid resolution (deg): dX={xdeg}, dY={ydeg}'))

        # Raster → WGS84 + resolusi target (bilinear)
        eigen_wgs0 = self._warp_to_wgs84_res(eigen_ras, xdeg, ydeg, context, feedback)
        rain_wgs0  = self._warp_to_wgs84_res(rain_ras,  xdeg, ydeg, context, feedback)

        # Grid pada extent AOI
        grid = processing.run('native:creategrid',
            {'TYPE':2, 'EXTENT': self._extent_str(aoi_wgs),
             'HSPACING': xdeg, 'VSPACING': ydeg, 'HOVERLAY':0.0, 'VOVERLAY':0.0,
             'CRS': QgsCoordinateReferenceSystem('EPSG:4326'), 'OUTPUT':'TEMPORARY_OUTPUT'},
            context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        try:
            processing.run('native:createspatialindex', {'INPUT': grid},
                           context=context, feedback=feedback, is_child_algorithm=True)
        except Exception:
            pass

        grid_aoi = processing.run('qgis:extractbylocation',
            {'INPUT': grid, 'PREDICATE':[0], 'INTERSECT': aoi_wgs, 'OUTPUT':'TEMPORARY_OUTPUT'},
            context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        # ID stabil global
        grid_aoi_layer = self._as_layer(grid_aoi, context)
        grid_aoi_layer = self._assign_stable_id_global(grid_aoi_layer, dx_deg=xdeg, dy_deg=ydeg,
                                                       lon0=-180.0, lat0=-90.0, numeric=True)
        grid_aoi = processing.run('native:savefeatures',
            {'INPUT': grid_aoi_layer, 'OUTPUT':'TEMPORARY_OUTPUT'},
            context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        # ==== FILL NODATA ADAPTIF + ZONAL LOOP ====
        eigen_wgs = eigen_wgs0
        rain_wgs  = rain_wgs0
        if do_fill:
            dist = 10; cap = 320
        else:
            dist = 0;  cap = 0

        last_null = None
        while True:
            if do_fill and dist > 0:
                feedback.pushInfo(self.tr(f'Fill NoData search distance = {dist} px'))
                eigen_wgs = self._fill_nodata(eigen_wgs0, dist, context, feedback)
                rain_wgs  = self._fill_nodata(rain_wgs0,  dist, context, feedback)

            v_zonal, mrain_field, eigen_field, null_count = self._run_zonal_pair(
                grid_aoi, rain_wgs, eigen_wgs, context, feedback
            )
            feedback.pushInfo(self.tr(f'Zonal: {null_count} feature(s) still NULL'))

            if not do_fill or null_count == 0:
                vlayer = v_zonal; break
            if last_null is not None and null_count >= last_null and dist >= cap:
                feedback.pushInfo(self.tr('Fill-NoData cap reached & no improvement — proceeding to fallback.'))
                vlayer = v_zonal; break
            if dist >= cap and null_count > 0:
                vlayer = v_zonal; break
            last_null = null_count
            dist = min(cap, dist * 2)

        # Tambah kolom final
        fields = vlayer.fields()
        add_defs = []
        for name, typ in (('mrain',QVariant.Double),('eigen',QVariant.Double),
                          ('sv',QVariant.Int),('sa',QVariant.Int),('S_Rain',QVariant.Int)):
            if fields.indexOf(name)==-1: add_defs.append(QgsField(name, typ))
        if add_defs:
            vlayer.dataProvider().addAttributes(add_defs); vlayer.updateFields()

        # Salin hasil zonal → mrain & eigen
        mrain_field = next((f.name() for f in vlayer.fields()
                            if f.name().lower().startswith('mrain_') and 'mean' in f.name().lower()), None) or 'mrain_mean'
        eigen_field = next((f.name() for f in vlayer.fields()
                            if f.name().lower().startswith('eigen_') and any(k in f.name().lower() for k in ('stdev','std','dev'))), None)
        if not eigen_field:
            cand = [f.name() for f in vlayer.fields() if f.name().lower().startswith('eigen_')]
            if len(cand)==1: eigen_field = cand[0]
        if not mrain_field or not eigen_field:
            raise QgsProcessingException(self.tr('Could not find the expected zonal result columns.'))

        idx_mrain_src = vlayer.fields().indexOf(mrain_field)
        idx_eigen_src = vlayer.fields().indexOf(eigen_field)
        idx_mrain = vlayer.fields().indexOf('mrain'); idx_eigen = vlayer.fields().indexOf('eigen')
        idx_sv = vlayer.fields().indexOf('sv'); idx_sa = vlayer.fields().indexOf('sa'); idx_sr = vlayer.fields().indexOf('S_Rain')

        with edit(vlayer):
            for f in vlayer.getFeatures():
                vlayer.changeAttributeValue(f.id(), idx_mrain, f[idx_mrain_src])
                vlayer.changeAttributeValue(f.id(), idx_eigen, f[idx_eigen_src])

        # Re-run zonal utk NULL → join prefix unik
        if any((ft[idx_mrain] is None or ft[idx_eigen] is None) for ft in vlayer.getFeatures()):
            vlayer = self._rerun_zonal_for_nulls(vlayer, rain_wgs, eigen_wgs, context, feedback)
            fields = vlayer.fields()
            idx_mrain = fields.indexOf('mrain'); idx_eigen = fields.indexOf('eigen')
            idx_sv = fields.indexOf('sv'); idx_sa = fields.indexOf('sa'); idx_sr = fields.indexOf('S_Rain')

        # Fallback centroid sampling
        if any((ft[idx_mrain] is None or ft[idx_eigen] is None) for ft in vlayer.getFeatures()):
            vlayer = self._fill_nulls_by_sampling(vlayer, rain_wgs, eigen_wgs, feedback=feedback)
            fields = vlayer.fields()
            idx_mrain = fields.indexOf('mrain'); idx_eigen = fields.indexOf('eigen')
            idx_sv = fields.indexOf('sv'); idx_sa = fields.indexOf('sa'); idx_sr = fields.indexOf('S_Rain')

        # eigen yang masih NULL → 0.0
        with edit(vlayer):
            for ft in vlayer.getFeatures():
                if ft[idx_eigen] is None and ft[idx_mrain] is not None:
                    vlayer.changeAttributeValue(ft.id(), idx_eigen, 0.0)

        # Normalisasi eigen → skor sv; hujan → skor sa; S_Rain
        min_e = float('inf'); max_e = float('-inf')
        for f in vlayer.getFeatures():
            ve = f[idx_eigen]
            if ve is not None:
                v=float(ve); min_e=min(min_e,v); max_e=max(max_e,v)
        if not math.isfinite(min_e) or not math.isfinite(max_e) or abs(max_e-min_e)<1e-12:
            min_e,max_e = 0.0,1.0

        def _score_variability_from_ne(ne):
            if ne is None: return None
            if ne < 0.2: return 5
            elif ne < 0.4: return 4
            elif ne < 0.6: return 3
            elif ne < 0.8: return 2
            else: return 1

        def _score_rain(mm):
            if mm is None: return None
            if mm <= 500: return 1
            elif mm <= 1500: return 2
            elif mm <= 3000: return 3
            elif mm <= 4500: return 4
            else: return 5

        with edit(vlayer):
            for f in vlayer.getFeatures():
                ve = f[idx_eigen]
                ne = None if ve is None else (float(ve)-min_e)/(max_e-min_e)
                sv = _score_variability_from_ne(ne)
                sa = _score_rain(f[idx_mrain])
                if sv is None or sa is None:
                    sr = None
                else:
                    s = 0.7*sv + 0.3*sa
                    sr = int(max(1, min(5, math.floor(s + 0.5))))
                vlayer.changeAttributeValue(f.id(), idx_sv, sv)
                vlayer.changeAttributeValue(f.id(), idx_sa, sa)
                vlayer.changeAttributeValue(f.id(), idx_sr, sr)

        # keep only final fields
        self._keep_only_fields(vlayer, keep_names=['id','mrain','eigen','sv','sa','S_Rain'])

        # save & index
        try:
            if isinstance(out_path,str) and out_path and out_path!='TEMPORARY_OUTPUT':
                base = out_path.split('|')[0]
                if base.lower().endswith('.gpkg') and os.path.exists(base):
                    os.remove(base)
        except Exception:
            pass

        saved = processing.run('native:savefeatures', {'INPUT': vlayer, 'OUTPUT': out_path},
                               context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
        if self.parameterAsBoolean(parameters, self.BUILD_INDEX, context):
            try:
                processing.run('native:createspatialindex', {'INPUT': saved},
                               context=context, feedback=feedback, is_child_algorithm=True)
            except Exception:
                pass
        return {self.OUTPUT: saved}
