# -*- coding: utf-8 -*-

"""
/***************************************************************************
 AMERTA
                                 A QGIS plugin
 Analisis Multi-kriteria Embung dan Rencana Tata Air
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-09-18
        copyright            : (C) 2025 by Badan Riset dan Inovasi Nasional
        email                : sitaranisafitri@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Sitarani Safitri, Orbita Roswintiarti, Okta Fajar Saputra, Galdita Aruba Chulafak, Gatot Nugroho, Wismu Sunarmodo, Kusumaning Ayu Dyah Sukowati, Hana Listi Fitriana'
__date__ = '2025-09-18'
__copyright__ = '(C) 2025 by Badan Riset dan Inovasi Nasional'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import math, os
from contextlib import contextmanager
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsProcessing, QgsProcessingAlgorithm, QgsProcessingException,
    QgsProcessingParameterVectorLayer, QgsProcessingParameterEnum,
    QgsProcessingParameterNumber, QgsProcessingParameterBoolean,
    QgsProcessingParameterVectorDestination, QgsCoordinateReferenceSystem,
    QgsField, QgsVectorLayer, QgsProcessingUtils
)
import processing
import os
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QUrl

@contextmanager
def edit(layer):
    layer.startEditing()
    try:
        yield
    except Exception:
        layer.rollBack(); raise
    else:
        layer.commitChanges()


class GridJTAlgorithm(QgsProcessingAlgorithm):
    AOI         = 'AOI'
    JENIS_TNH   = 'JENIS_TNH'
    GRID_SIZE   = 'GRID_SIZE'
    CUSTOM_M    = 'CUSTOM_M'
    BUILD_INDEX = 'BUILD_INDEX'
    OUTPUT      = 'OUTPUT'

    GRID_CHOICES = ['30″ (≈ 925 m)', '5″ (≈ 150 m)', 'Custom (meter)']

    def tr(self, s): return QCoreApplication.translate('Processing', s)
    def name(self): return 'd_grid_soil_type_score'
    def displayName(self): return self.tr('Grid Soil Type')
    def groupId(self): return 'C. MCDA Factors for Retention Ponds'
    def group(self): return self.tr('C. MCDA Factors for Retention Ponds')
    def icon(self):
        return QIcon(os.path.join(os.path.dirname(__file__), 'preanalysis.png'))
    def shortHelpString(self):
        return self.tr("""\
🇮🇩 ID  Modul ini membuat grid dan menetapkan kelas Jenis Tanah (JTNH) dominan per grid, lalu memberi skor S_JTNH.

Alur kerja:
1) Siapkan Soil Type yang sudah distandarisasi pada modul Soil Type Standardization (JTNH).
2) Siapkan data spasial Area of Interest (AOI), contoh batas administrasi atau batas wilayah sungai.
3) Tentukan resolusi grid dari dropdown atau custom dalam satuan meter (30″/5″/Custom).
4) Memasukkan JTNH ke dalam grid sesuai AOI.
5) Mengisi skor S_JTNH berdasarkan kategori JTNH.

Keluaran:
• Grid Soil Type berisi kolom: id, JTNH, S_JTNH.

──────────────

🌍 EN  This module creates a grid, assigns the dominant Soil Type (JTNH) per grid, then computes S_JTNH scores.

Workflow:
1) Prepare standardized Soil Type using the Soil Type Standardization (JTNH) module.
2) Prepare AOI spatial data (e.g., administrative boundary or river basin boundary).
3) Choose grid resolution from the dropdown or use a custom value in meters (30″/5″/Custom).
4) Populate JTNH into the grid constrained by the AOI.
5) Assign S_JTNH scores based on JTNH categories.

Output:
• Grid Soil Type with fields: id, JTNH, S_JTNH.""")

    def createInstance(self): return GridJTAlgorithm()

    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterVectorLayer(self.JENIS_TNH, self.tr('Soil Type (must contain a "JTNH" field)'), types=[QgsProcessing.TypeVectorPolygon]))
        self.addParameter(QgsProcessingParameterVectorLayer(self.AOI, self.tr('Area of Interest (AOI)'), types=[QgsProcessing.TypeVectorPolygon]))
        self.addParameter(QgsProcessingParameterEnum(self.GRID_SIZE, self.tr('Grid Resolution'), self.GRID_CHOICES, defaultValue=0))
        self.addParameter(QgsProcessingParameterNumber(self.CUSTOM_M, self.tr('Grid Resolution (meter) for "Custom (meter)"'), type=QgsProcessingParameterNumber.Double, defaultValue=1.0, minValue=0.0))
        self.addParameter(QgsProcessingParameterBoolean(self.BUILD_INDEX, self.tr('Create spatial index on outputs'), defaultValue=True))
        self.addParameter(QgsProcessingParameterVectorDestination(self.OUTPUT, self.tr('Grid Soil Type')))

    # -------------- helpers --------------
    def _as_layer(self, ref, context):
        try:
            if hasattr(ref, 'extent'): return ref
            lyr = QgsProcessingUtils.mapLayerFromString(ref, context)
            if lyr is not None: return lyr
            lyr = QgsVectorLayer(ref, 'layer', 'ogr')
            if lyr is not None and lyr.isValid(): return lyr
        except Exception:
            pass
        raise QgsProcessingException(self.tr('Failed to load layer: {}').format(ref))

    def _mk_spatial_index(self, vlayer, context, feedback):
        try:
            processing.run('native:createspatialindex', {'INPUT': vlayer}, context=context, feedback=feedback, is_child_algorithm=True)
        except Exception:
            pass
        return vlayer

    def _fix_geoms(self, vlayer, context, feedback):
        try:
            return processing.run('native:fixgeometries', {'INPUT': vlayer, 'OUTPUT':'TEMPORARY_OUTPUT'}, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
        except Exception:
            return vlayer

    def _dissolve_all(self, vlayer, context, feedback):
        try:
            return processing.run('native:dissolve', {'INPUT': vlayer, 'FIELD':[], 'SEPARATE_DISJOINT': False, 'OUTPUT':'TEMPORARY_OUTPUT'}, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
        except Exception:
            return vlayer

    def _to_wgs84(self, layer, context, feedback):
        res = processing.run('native:reprojectlayer', {'INPUT': layer, 'TARGET_CRS': QgsCoordinateReferenceSystem('EPSG:4326'), 'OPERATION':'', 'OUTPUT':'TEMPORARY_OUTPUT'}, context=context, feedback=feedback, is_child_algorithm=True)
        return res['OUTPUT']

    def _extent_str(self, layer_obj):
        e = layer_obj.extent()
        return f'{e.xMinimum()},{e.xMaximum()},{e.yMinimum()},{e.yMaximum()} [EPSG:4326]'

    def _deg_spacing_from_option(self, option_idx, custom_m, aoi_wgs_layer_obj):
        if option_idx == 0: return 30.0/3600.0, 30.0/3600.0
        if option_idx == 1: return 5.0/3600.0, 5.0/3600.0
        c = aoi_wgs_layer_obj.extent().center()
        mean_lat_rad = math.radians(c.y())
        deg_lat = float(custom_m) / 111320.0
        coslat = max(1e-6, math.cos(mean_lat_rad))
        deg_lon = float(custom_m) / (111320.0 * coslat)
        return deg_lon, deg_lat

    def _assign_stable_id_global(self, vlayer, dx_deg, dy_deg, lon0=-180.0, lat0=-90.0, numeric=True):
        prov = vlayer.dataProvider()
        if vlayer.fields().indexOf('id') == -1:
            prov.addAttributes([QgsField('id', QVariant.Int if numeric else QVariant.String)])
            vlayer.updateFields()
        idx_id = vlayer.fields().indexOf('id')
        arcsec_x = int(round(dx_deg * 3600.0 + 1e-9))
        with edit(vlayer):
            for ft in vlayer.getFeatures():
                c = ft.geometry().centroid().asPoint()
                ix = int((c.x() - lon0) // dx_deg)
                iy = int((c.y() - lat0) // dy_deg)
                gid = int(iy) * 1_000_000 + int(ix) if numeric else f"G{arcsec_x}s_{iy:06d}_{ix:06d}"
                vlayer.changeAttributeValue(ft.id(), idx_id, gid)
        return vlayer

    def _utm_crs_from_aoi(self, aoi_wgs_layer_obj):
        c = aoi_wgs_layer_obj.extent().center()
        lon, lat = c.x(), c.y()
        zone = int((lon + 180) / 6) + 1
        epsg = (32700 if lat < 0 else 32600) + zone
        return QgsCoordinateReferenceSystem(f'EPSG:{epsg}')

    def _keep_only_fields(self, vlayer, keep_names):
        fields = vlayer.fields()
        drop = [i for i, f in enumerate(fields) if f.name() not in keep_names]
        if drop:
            with edit(vlayer):
                vlayer.dataProvider().deleteAttributes(sorted(drop, reverse=True))
                vlayer.updateFields()

    # -------------- core --------------
    def processAlgorithm(self, parameters, context, feedback):
        tanah_vec = self.parameterAsVectorLayer(parameters, self.JENIS_TNH, context)
        aoi_vec   = self.parameterAsVectorLayer(parameters, self.AOI, context)
        grid_idx  = int(self.parameterAsEnum(parameters, self.GRID_SIZE, context))
        custom_m  = float(self.parameterAsDouble(parameters, self.CUSTOM_M, context))
        make_idx  = bool(self.parameterAsBoolean(parameters, self.BUILD_INDEX, context))
        out_path  = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)

        if tanah_vec is None or not tanah_vec.isValid():
            raise QgsProcessingException(self.tr('Soil type layer is invalid.'))
        if aoi_vec is None or not aoi_vec.isValid():
            raise QgsProcessingException(self.tr('AOI layer is invalid.'))
        if tanah_vec.fields().indexOf('JTNH') == -1:
            raise QgsProcessingException(self.tr('Soil type layer does not contain a "JTNH" field.'))

        # AOI → WGS84 → fix → dissolve → index
        aoi_wgs = self._to_wgs84(aoi_vec, context, feedback)
        aoi_wgs = self._fix_geoms(aoi_wgs, context, feedback)
        aoi_wgs = self._dissolve_all(aoi_wgs, context, feedback)
        aoi_wgs = self._as_layer(aoi_wgs, context)
        self._mk_spatial_index(aoi_wgs, context, feedback)

        # grid resolution (deg)
        xdeg, ydeg = self._deg_spacing_from_option(grid_idx, custom_m, aoi_wgs)
        feedback.pushInfo(self.tr(f'Grid resolution (deg): dX={xdeg}, dY={ydeg}'))

        # Grid → index → subset AOI
        grid = processing.run('native:creategrid', {
            'TYPE':2, 'EXTENT': self._extent_str(aoi_wgs), 'HSPACING': xdeg, 'VSPACING': ydeg,
            'HOVERLAY':0.0, 'VOVERLAY':0.0, 'CRS': QgsCoordinateReferenceSystem('EPSG:4326'), 'OUTPUT':'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
        self._mk_spatial_index(grid, context, feedback)

        grid_aoi = processing.run('qgis:extractbylocation', {
            'INPUT': grid, 'PREDICATE':[0], 'INTERSECT': aoi_wgs, 'OUTPUT':'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        grid_aoi_layer = self._as_layer(grid_aoi, context)
        grid_aoi_layer = self._assign_stable_id_global(grid_aoi_layer, dx_deg=xdeg, dy_deg=ydeg, lon0=-180.0, lat0=-90.0, numeric=True)
        self._mk_spatial_index(grid_aoi_layer, context, feedback)
        grid_aoi = processing.run('native:savefeatures', {'INPUT': grid_aoi_layer, 'OUTPUT':'TEMPORARY_OUTPUT'}, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        # Soil → WGS84 → fix → index
        tanah_wgs = self._to_wgs84(tanah_vec, context, feedback)
        tanah_wgs = self._fix_geoms(tanah_wgs, context, feedback)
        self._mk_spatial_index(tanah_wgs, context, feedback)

        # Intersect grid with soil (carry id & JTNH)
        inters = processing.run('native:intersection', {
            'INPUT': grid_aoi, 'OVERLAY': tanah_wgs, 'INPUT_FIELDS':[], 'OVERLAY_FIELDS':['JTNH'],
            'OVERLAY_FIELDS_PREFIX':'', 'GRID_SNAP':'NO_GRID_SNAP', 'OUTPUT':'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        # Dissolve by (id, JTNH)
        diss = processing.run('native:dissolve', {
            'INPUT': inters, 'FIELD':['id','JTNH'], 'SEPARATE_DISJOINT': False, 'OUTPUT':'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        # Compute area (m²) in UTM (per-AOI zone)
        utm = self._utm_crs_from_aoi(self._as_layer(aoi_wgs, context))
        diss_utm = processing.run('native:reprojectlayer', {
            'INPUT': diss, 'TARGET_CRS': utm, 'OPERATION':'', 'OUTPUT':'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        with_lm2 = processing.run('qgis:fieldcalculator', {
            'INPUT': diss_utm, 'FIELD_NAME':'LM2', 'FIELD_TYPE':0, 'FIELD_LENGTH':20, 'FIELD_PRECISION':3,
            'NEW_FIELD': True, 'FORMULA':'$area', 'OUTPUT':'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        # Max LM2 per id → winners
        stats = processing.run('qgis:statisticsbycategories', {
            'INPUT': with_lm2, 'CATEGORIES_FIELD_NAME': ['id'], 'VALUES_FIELD_NAME': 'LM2', 'STATISTICS': [5],
            'OUTPUT': 'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
        with_max = processing.run('native:joinattributestable', {
            'INPUT': with_lm2, 'FIELD':'id', 'INPUT_2': stats, 'FIELD_2':'id', 'FIELDS_TO_COPY':['max'],
            'METHOD':1, 'DISCARD_NONMATCHING': True, 'PREFIX':'', 'OUTPUT':'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
        winners = processing.run('native:extractbyexpression', {
            'INPUT': with_max, 'EXPRESSION':'"LM2" = "max"', 'OUTPUT':'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        # Join majority JTNH back to grid id (keep non-matching rows)
        joined = processing.run('native:joinattributestable', {
            'INPUT': grid_aoi, 'FIELD':'id', 'INPUT_2': winners, 'FIELD_2':'id', 'FIELDS_TO_COPY':['JTNH'],
            'METHOD':1, 'DISCARD_NONMATCHING': False, 'PREFIX':'', 'OUTPUT':'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        # -------- Fill NULL strictly by attribute using joinbynearest (1:1 via min distance then min $id tie-break) --------
        has_val = processing.run('native:extractbyexpression', {
            'INPUT': joined, 'EXPRESSION': '"JTNH" IS NOT NULL', 'OUTPUT': 'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
        no_val = processing.run('native:extractbyexpression', {
            'INPUT': joined, 'EXPRESSION': '"JTNH" IS NULL', 'OUTPUT': 'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        merged = joined
        if self._as_layer(no_val, context).featureCount() > 0 and self._as_layer(has_val, context).featureCount() > 0:
            # (A) nearest from already-filled grid cells (copy JTNH and distance)
            nn1 = processing.run('native:joinbynearest', {
                'INPUT': no_val,
                'INPUT_2': has_val,
                'FIELDS_TO_COPY': ['JTNH'],
                'DISCARD_NONMATCHING': False,
                'PREFIX': 'nn_',
                'NEIGHBORS': 5,   # allow ties, we'll resolve deterministically
                'MAX_DISTANCE': 0.0,
                'OUTPUT': 'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

            # Find min distance per id
            dist_min1 = processing.run('qgis:statisticsbycategories', {
                'INPUT': nn1,
                'CATEGORIES_FIELD_NAME': ['id'],
                'VALUES_FIELD_NAME': 'distance',
                'STATISTICS': [3],  # 3 = Min
                'OUTPUT': 'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
            nn1_join = processing.run('native:joinattributestable', {
                'INPUT': nn1, 'FIELD':'id', 'INPUT_2': dist_min1, 'FIELD_2':'id', 'FIELDS_TO_COPY':['min'],
                'METHOD':1, 'DISCARD_NONMATCHING': True, 'PREFIX':'', 'OUTPUT':'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
            nn1_min = processing.run('native:extractbyexpression', {
                'INPUT': nn1_join, 'EXPRESSION':'"distance" = "min"', 'OUTPUT':'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

            # Tie-break if multiple rows remain (same min distance): keep smallest $id per grid id
            pickid1 = processing.run('native:aggregate', {
                'INPUT': nn1_min,
                'GROUP_BY': 'id',
                'AGGREGATES': [
                    {'aggregate': 'min', 'delimiter': ',', 'input': 'id', 'length': 20, 'name': 'id', 'precision': 0, 'type': 4},
                    {'aggregate': 'min', 'delimiter': ',', 'input': '$id', 'length': 20, 'name': 'pick_rowid', 'precision': 0, 'type': 4}
                ],
                'OUTPUT': 'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
            nn1_min2 = processing.run('native:joinattributestable', {
                'INPUT': nn1_min, 'FIELD':'id', 'INPUT_2': pickid1, 'FIELD_2':'id', 'FIELDS_TO_COPY':['pick_rowid'],
                'METHOD':1, 'DISCARD_NONMATCHING': True, 'PREFIX':'', 'OUTPUT':'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
            nn1_final = processing.run('native:extractbyexpression', {
                'INPUT': nn1_min2, 'EXPRESSION':'$id = "pick_rowid"', 'OUTPUT':'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

            # Fill JTNH from nearest
            no_val_filled = processing.run('native:joinattributestable', {
                'INPUT': no_val, 'FIELD': 'id', 'INPUT_2': nn1_final, 'FIELD_2': 'id', 'FIELDS_TO_COPY': ['nn_JTNH'],
                'METHOD': 1, 'DISCARD_NONMATCHING': False, 'PREFIX': '', 'OUTPUT': 'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
            no_val_filled = processing.run('qgis:fieldcalculator', {
                'INPUT': no_val_filled, 'FIELD_NAME': 'JTNH', 'FIELD_TYPE': 2, 'FIELD_LENGTH': 100, 'FIELD_PRECISION': 0,
                'NEW_FIELD': False, 'FORMULA': 'coalesce("JTNH","nn_JTNH")', 'OUTPUT': 'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

            merged1 = processing.run('native:mergevectorlayers', {
                'LAYERS': [has_val, no_val_filled], 'CRS': None, 'OUTPUT': 'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

            # (B) fallback: still NULL? do the same using winners (polygons)
            remain_null = processing.run('native:extractbyexpression', {
                'INPUT': merged1, 'EXPRESSION': '"JTNH" IS NULL', 'OUTPUT': 'TEMPORARY_OUTPUT'
            }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
            if self._as_layer(remain_null, context).featureCount() > 0:
                winners_wgs = processing.run('native:reprojectlayer', {
                    'INPUT': winners, 'TARGET_CRS': QgsCoordinateReferenceSystem('EPSG:4326'), 'OPERATION': '', 'OUTPUT': 'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
                nn2 = processing.run('native:joinbynearest', {
                    'INPUT': remain_null, 'INPUT_2': winners_wgs, 'FIELDS_TO_COPY': ['JTNH'], 'DISCARD_NONMATCHING': False,
                    'PREFIX': 'nn_', 'NEIGHBORS': 5, 'MAX_DISTANCE': 0.0, 'OUTPUT': 'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
                dist_min2 = processing.run('qgis:statisticsbycategories', {
                    'INPUT': nn2, 'CATEGORIES_FIELD_NAME': ['id'], 'VALUES_FIELD_NAME': 'distance', 'STATISTICS': [3],
                    'OUTPUT': 'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
                nn2_join = processing.run('native:joinattributestable', {
                    'INPUT': nn2, 'FIELD':'id', 'INPUT_2': dist_min2, 'FIELD_2':'id', 'FIELDS_TO_COPY':['min'], 'METHOD':1,
                    'DISCARD_NONMATCHING': True, 'PREFIX':'', 'OUTPUT':'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
                nn2_min = processing.run('native:extractbyexpression', {
                    'INPUT': nn2_join, 'EXPRESSION':'"distance" = "min"', 'OUTPUT':'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
                pickid2 = processing.run('native:aggregate', {
                    'INPUT': nn2_min, 'GROUP_BY': 'id', 'AGGREGATES': [
                        {'aggregate': 'min', 'delimiter': ',', 'input': 'id', 'length': 20, 'name': 'id', 'precision': 0, 'type': 4},
                        {'aggregate': 'min', 'delimiter': ',', 'input': '$id', 'length': 20, 'name': 'pick_rowid', 'precision': 0, 'type': 4}
                    ], 'OUTPUT': 'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
                nn2_min2 = processing.run('native:joinattributestable', {
                    'INPUT': nn2_min, 'FIELD':'id', 'INPUT_2': pickid2, 'FIELD_2':'id', 'FIELDS_TO_COPY':['pick_rowid'],
                    'METHOD':1, 'DISCARD_NONMATCHING': True, 'PREFIX':'', 'OUTPUT':'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
                nn2_final = processing.run('native:extractbyexpression', {
                    'INPUT': nn2_min2, 'EXPRESSION':'$id = "pick_rowid"', 'OUTPUT':'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

                non_null_after_first = processing.run('native:extractbyexpression', {
                    'INPUT': merged1, 'EXPRESSION': '"JTNH" IS NOT NULL', 'OUTPUT': 'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
                filled_2 = processing.run('native:joinattributestable', {
                    'INPUT': remain_null, 'FIELD': 'id', 'INPUT_2': nn2_final, 'FIELD_2': 'id', 'FIELDS_TO_COPY': ['nn_JTNH'],
                    'METHOD': 1, 'DISCARD_NONMATCHING': False, 'PREFIX': '', 'OUTPUT': 'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
                filled_2 = processing.run('qgis:fieldcalculator', {
                    'INPUT': filled_2, 'FIELD_NAME': 'JTNH', 'FIELD_TYPE': 2, 'FIELD_LENGTH': 100, 'FIELD_PRECISION': 0,
                    'NEW_FIELD': False, 'FORMULA': 'coalesce("JTNH","nn_JTNH")', 'OUTPUT': 'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
                merged = processing.run('native:mergevectorlayers', {
                    'LAYERS': [non_null_after_first, filled_2], 'CRS': None, 'OUTPUT': 'TEMPORARY_OUTPUT'
                }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
            else:
                merged = merged1
        # -------- end fill --------

        # Score S_JTNH
        case_expr = '''
CASE
  WHEN "JTNH" IN ('Aluvial','Aluvial (Fluvisol)','Fluvisol') THEN 2
  WHEN "JTNH" = 'Andosol' THEN 1
  WHEN "JTNH" = 'Regosol' THEN 2
  WHEN "JTNH" = 'Oksisol (Oxisol)' THEN 3
  WHEN "JTNH" IN ('Litosol','Litosol (Leptosol)','Leptosol') THEN 2
  WHEN "JTNH" IN ('Planosol Hidromorf','Planosol Hidromorf (Planosol)','Planosol') THEN 5
  WHEN "JTNH" IN ('Grumosol','Grumosol (Vertisol)','Vertisol') THEN 4
  WHEN "JTNH" IN ('Latosol','Latosol (Ferrasol)','Ferrasol') THEN 3
  WHEN "JTNH" IN ('Mediteran','Mediteran (Terra rossa)','Terra rossa') THEN 3
  WHEN "JTNH" IN ('Rendzina','Rensina','Rendzina (Rensina)') THEN 4
  WHEN "JTNH" IN ('Gleisol','Gleysol','Gleisol (Gleysol)') THEN 5
  WHEN "JTNH" = 'Kambisol' THEN 3
  WHEN "JTNH" IN ('Podsolik','Podsolik (Acrisol/Alisol)','Acrisol','Alisol') THEN 5
  WHEN "JTNH" IN ('Podzol','Podsol','Podzol (Podsol)') THEN 5
  WHEN "JTNH" IN ('Organosol','Gambut','Organosol/Gambut','Organosol/Gambut (Histosol)','Histosol') THEN 5
  WHEN "JTNH" IN ('Nitosol','Nitisol','Nitosol (Nitisol)') THEN 2
  WHEN "JTNH" IN ('Arenosol','Pasiran','Arenosol (Pasiran)') THEN 1
  WHEN "JTNH" = 'Plinthosol' THEN 5
  WHEN "JTNH" IN ('Solonetz','Sodik','Solonetz (Sodik)') THEN 4
  WHEN "JTNH" IN ('Solonchak','Salin','Solonchak (Salin)') THEN 4
  WHEN "JTNH" IN ('Waduk','Danau','Situ','Waduk/Danau/Situ') THEN 5
  ELSE NULL
END
'''
        with_score = processing.run('qgis:fieldcalculator', {
            'INPUT': merged, 'FIELD_NAME':'S_JTNH', 'FIELD_TYPE':1, 'FIELD_LENGTH':10, 'FIELD_PRECISION':0,
            'NEW_FIELD': True, 'FORMULA': case_expr, 'OUTPUT':'TEMPORARY_OUTPUT'
        }, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']

        v = self._as_layer(with_score, context)
        self._keep_only_fields(v, ['id','JTNH','S_JTNH'])

        try:
            if isinstance(out_path, str) and out_path and out_path != 'TEMPORARY_OUTPUT':
                base = out_path.split('|')[0]
                if base.lower().endswith('.gpkg') and os.path.exists(base):
                    os.remove(base)
        except Exception:
            pass

        saved = processing.run('native:savefeatures', {'INPUT': v, 'OUTPUT': out_path}, context=context, feedback=feedback, is_child_algorithm=True)['OUTPUT']
        if make_idx:
            self._mk_spatial_index(saved, context, feedback)

        return {self.OUTPUT: saved}
