# -*- coding: utf-8 -*-

"""
/***************************************************************************
 AMERTA
                                 A QGIS plugin
 Analisis Multi-kriteria Embung dan Rencana Tata Air
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-09-18
        copyright            : (C) 2025 by Badan Riset dan Inovasi Nasional
        email                : sitaranisafitri@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Sitarani Safitri, Orbita Roswintiarti, Okta Fajar Saputra, Galdita Aruba Chulafak, Gatot Nugroho, Wismu Sunarmodo, Kusumaning Ayu Dyah Sukowati, Hana Listi Fitriana'
__date__ = '2025-09-18'
__copyright__ = '(C) 2025 by Badan Riset dan Inovasi Nasional'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing,
    QgsFeatureSink,
    QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterBoolean,
    QgsProcessingException,
    QgsVectorLayer,
)
import processing
import os
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QUrl

class AMERTA_MCDAGridAlgorithm(QgsProcessingAlgorithm):
    """MCDA Grid (Rain+TWI+PL+JTNH) — join by field 'id' (fixed)"""

    # Param keys
    P_TWI = 'TWI'
    P_RAIN = 'RAIN'
    P_PL = 'PL'
    P_JTNH = 'JTNH'
    P_BUILD_INDEX = 'BUILD_INDEX'
    P_OUTPUT = 'OUTPUT'

    # Fixed join field
    KEY_FIELD = 'id'

    # Expected columns
    COL_S_TWI = 'S_TWI'
    COL_S_RAIN = 'S_Rain'
    COL_S_PL = 'S_PL'
    COL_S_JTNH = 'S_JTNH'
    COL_MCDA = 'MCDA'
    COL_K_MCDA = 'K_MCDA'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    # UI
    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterFeatureSource(
            self.P_PL, self.tr('Grid Land Cover/Land Use (must contain "id" and "{}")'.format(self.COL_S_PL)),
            [QgsProcessing.TypeVectorPolygon, QgsProcessing.TypeVectorAnyGeometry]
        ))
        self.addParameter(QgsProcessingParameterFeatureSource(
            self.P_RAIN, self.tr('Grid Rainfall (must contain "id" and "{}")'.format(self.COL_S_RAIN)),
            [QgsProcessing.TypeVectorPolygon, QgsProcessing.TypeVectorAnyGeometry]
        ))
        self.addParameter(QgsProcessingParameterFeatureSource(
            self.P_JTNH, self.tr('Grid Soil Type (must contain "id" and "{}")'.format(self.COL_S_JTNH)),
            [QgsProcessing.TypeVectorPolygon, QgsProcessing.TypeVectorAnyGeometry]
        ))
        self.addParameter(QgsProcessingParameterFeatureSource(
            self.P_TWI, self.tr('Grid TWI (must contain "id" and "{}")'.format(self.COL_S_TWI)),
            [QgsProcessing.TypeVectorPolygon]
        ))

        self.addParameter(QgsProcessingParameterBoolean(
            self.P_BUILD_INDEX, self.tr('Build spatial/attribute index & fix geometries (faster/safer)'),
            defaultValue=True
        ))

        self.addParameter(QgsProcessingParameterFeatureSink(
            self.P_OUTPUT, self.tr('Retention Ponds Site Suitability Model')
        ))

    # Core
    def processAlgorithm(self, parameters, context, feedback):
        # Ambil sebagai VECTOR LAYER
        lyr_twi: QgsVectorLayer = self.parameterAsVectorLayer(parameters, self.P_TWI, context)
        lyr_rain: QgsVectorLayer = self.parameterAsVectorLayer(parameters, self.P_RAIN, context)
        lyr_pl: QgsVectorLayer = self.parameterAsVectorLayer(parameters, self.P_PL, context)
        lyr_jtnh: QgsVectorLayer = self.parameterAsVectorLayer(parameters, self.P_JTNH, context)

        build_index = self.parameterAsBoolean(parameters, self.P_BUILD_INDEX, context)

        # Validasi layer & kolom skor + 'id'
        def _fields(layer):
            return [f.name() for f in layer.fields()]

        if not isinstance(lyr_twi, QgsVectorLayer) or not lyr_twi.isValid():
            raise QgsProcessingException(self.tr('Grid TWI layer tidak valid.'))
        checks = [
            (lyr_twi, [self.KEY_FIELD, self.COL_S_TWI], 'Grid TWI'),
            (lyr_rain, [self.KEY_FIELD, self.COL_S_RAIN], 'Grid Rainfall'),
            (lyr_pl, [self.KEY_FIELD, self.COL_S_PL], 'Grid Land Cover/Land Use'),
            (lyr_jtnh, [self.KEY_FIELD, self.COL_S_JTNH], 'Grid Soil Type')
        ]
        for lyr, required_cols, label in checks:
            if not isinstance(lyr, QgsVectorLayer) or not lyr.isValid():
                raise QgsProcessingException(self.tr(f'{label} layer tidak valid.'))
            fields = _fields(lyr)
            for col in required_cols:
                if col not in fields:
                    raise QgsProcessingException(self.tr(f'Layer {label} wajib memiliki kolom "{col}".'))

        # Utility: fix geometries + index
        def fix_and_index(vlayer: QgsVectorLayer, step_label: str) -> QgsVectorLayer:
            if not build_index:
                return vlayer
            out = vlayer
            try:
                res = processing.run(
                    'native:fixgeometries',
                    {'INPUT': vlayer, 'OUTPUT': 'memory:'},
                    context=context, feedback=feedback
                )
                out = res['OUTPUT']
            except Exception as e:
                feedback.reportError(self.tr(f'fixgeometries gagal untuk {step_label}: {e}; memakai layer asli.'))
            try:
                processing.run('native:createspatialindex', {'INPUT': out}, context=context, feedback=feedback)
            except Exception as e:
                feedback.reportError(self.tr(f'createspatialindex gagal untuk {step_label}: {e}'))
            # Attribute index pada kolom 'id' bila ada
            if self.KEY_FIELD in [f.name() for f in out.fields()]:
                try:
                    processing.run('native:createattributeindex',
                                   {'INPUT': out, 'FIELD': self.KEY_FIELD},
                                   context=context, feedback=feedback)
                except Exception as e:
                    feedback.reportError(self.tr(f'createattributeindex gagal untuk {step_label}: {e}'))
            return out

        feedback.pushInfo(self.tr('Memproses fix geometries & index (jika dipilih)…'))
        base = fix_and_index(lyr_twi, 'Grid TWI')
        rain = fix_and_index(lyr_rain, 'Grid Rainfall')
        pl = fix_and_index(lyr_pl, 'Grid Land Cover/Land Use')
        jtnh = fix_and_index(lyr_jtnh, 'Grid Soil Type')

        # Join 1: Rain → TWI
        feedback.pushInfo(self.tr('Join atribut: Rain → TWI (by "id")…'))
        joined1 = processing.run('native:joinattributestable', {
            'INPUT': base,
            'FIELD': self.KEY_FIELD,
            'INPUT_2': rain,
            'FIELD_2': self.KEY_FIELD,
            'FIELDS_TO_COPY': [self.COL_S_RAIN],
            'METHOD': 1,
            'DISCARD_NONMATCHING': False,
            'PREFIX': '',
            'OUTPUT': 'memory:'
        }, context=context, feedback=feedback)['OUTPUT']

        # Join 2: PL → hasil(1)
        feedback.pushInfo(self.tr('Join atribut: PL → hasil(1) (by "id")…'))
        joined2 = processing.run('native:joinattributestable', {
            'INPUT': joined1,
            'FIELD': self.KEY_FIELD,
            'INPUT_2': pl,
            'FIELD_2': self.KEY_FIELD,
            'FIELDS_TO_COPY': [self.COL_S_PL],
            'METHOD': 1,
            'DISCARD_NONMATCHING': False,
            'PREFIX': '',
            'OUTPUT': 'memory:'
        }, context=context, feedback=feedback)['OUTPUT']

        # Join 3: JTNH → hasil(2)
        feedback.pushInfo(self.tr('Join atribut: JTNH → hasil(2) (by "id")…'))
        joined3 = processing.run('native:joinattributestable', {
            'INPUT': joined2,
            'FIELD': self.KEY_FIELD,
            'INPUT_2': jtnh,
            'FIELD_2': self.KEY_FIELD,
            'FIELDS_TO_COPY': [self.COL_S_JTNH],
            'METHOD': 1,
            'DISCARD_NONMATCHING': False,
            'PREFIX': '',
            'OUTPUT': 'memory:'
        }, context=context, feedback=feedback)['OUTPUT']

        # MCDA
        feedback.pushInfo(self.tr(f'Menghitung kolom "{self.COL_MCDA}"…'))
        expr_mcda = (
            f'0.30*coalesce("{self.COL_S_RAIN}",0) + '
            f'0.30*coalesce("{self.COL_S_TWI}",0) + '
            f'0.20*coalesce("{self.COL_S_PL}",0) + '
            f'0.20*coalesce("{self.COL_S_JTNH}",0)'
        )
        with_mcda = processing.run('native:fieldcalculator', {
            'INPUT': joined3,
            'FIELD_NAME': self.COL_MCDA,
            'FIELD_TYPE': 0,   # Float
            'FIELD_LENGTH': 20,
            'FIELD_PRECISION': 6,
            'NEW_FIELD': True,
            'FORMULA': expr_mcda,
            'OUTPUT': 'memory:'
        }, context=context, feedback=feedback)['OUTPUT']

        # K_MCDA → memory
        feedback.pushInfo(self.tr(f'Menentukan klasifikasi "{self.COL_K_MCDA}"…'))
        expr_k = (
            f'CASE '
            f'WHEN "{self.COL_MCDA}" < 2.5 THEN \'Tidak sesuai\' '
            f'WHEN "{self.COL_MCDA}" < 3.5 THEN \'Cukup sesuai\' '
            f'WHEN "{self.COL_MCDA}" < 4.5 THEN \'Sesuai\' '
            f'ELSE \'Sangat sesuai\' END'
        )
        with_k = processing.run('native:fieldcalculator', {
            'INPUT': with_mcda,
            'FIELD_NAME': self.COL_K_MCDA,
            'FIELD_TYPE': 2,      # String
            'FIELD_LENGTH': 30,
            'FIELD_PRECISION': 0,
            'NEW_FIELD': True,
            'FORMULA': expr_k,
            'OUTPUT': 'memory:'
        }, context=context, feedback=feedback)['OUTPUT']

        # === COPY KE SINK OUTPUT (stabil) ===
        fields = with_k.fields()
        wkb = with_k.wkbType()
        crs = with_k.sourceCrs()

        sink, dest_id = self.parameterAsSink(parameters, self.P_OUTPUT, context, fields, wkb, crs)
        if sink is None:
            raise QgsProcessingException(self.tr('Gagal membuat OUTPUT sink.'))

        for f in with_k.getFeatures():
            sink.addFeature(f, QgsFeatureSink.FastInsert)

        return {self.P_OUTPUT: dest_id}

    # Metadata
    def name(self): return 'mcdagrid_rain_twi_pl_jtnh'
    def displayName(self): return self.tr('Retention Ponds Site Suitability Model')
    def groupId(self): return 'D. MCDA Analysis Result'
    def group(self): return self.tr(self.groupId())
    def createInstance(self): return AMERTA_MCDAGridAlgorithm()
    def icon(self):
        return QIcon(os.path.join(os.path.dirname(__file__), 'site.png'))
    def shortHelpString(self):
        return self.tr("""\
🇮🇩 ID  Modul ini menggabungkan empat grid skor (TWI, Curah Hujan, Penutup Lahan, Jenis Tanah) berdasarkan kolom 'id', lalu menghitung skor MCDA dan klasifikasinya.

Alur kerja:
1) Siapkan empat grid: Grid TWI (wajib kolom: id, S_TWI), Grid Rainfall (wajib kolom: id, S_Rain), Grid Land Cover/Land Use (wajib kolom: id, S_PL), dan Grid Soil Type (wajib kolom: id, S_JTNH).
2) (Opsional) Perbaiki geometri & buat indeks (lebih aman/cepat).
3) Join atribut berdasarkan 'id' dan hitung MCDA = 0.30*S_Rain + 0.30*S_TWI + 0.20*S_PL + 0.20*S_JTNH.
4) Tentukan kelas K_MCDA berdasarkan nilai MCDA.

Keluaran:
• Grid MCDA berisi kolom: id, S_TWI, S_Rain, S_PL, S_JTNH, MCDA, K_MCDA.

──────────────

🌍 EN  This module merges four score grids (TWI, Rainfall, Land Cover/Land Use, Soil Type) based on 'id' field, then computes the MCDA score and its class.

Workflow:
1) Prepare the four grids with required fields: TWI grid (id, S_TWI), Rainfall grid (id, S_Rain), Land Cover/Land Use grid (id, S_PL), dan Soil Type grid (id, S_JTNH).
2) (Optional) Fix geometries & build indexes (safer/faster).
3) Join attributes by 'id' and compute MCDA = 0.30*S_Rain + 0.30*S_TWI + 0.20*S_PL + 0.20*S_JTNH.
4) Classify K_MCDA based on MCDA value.

Output:
• MCDA grid with fields: id, S_TWI, S_Rain, S_PL, S_JTNH, MCDA, K_MCDA.""")

