# -*- coding: utf-8 -*-

"""
/***************************************************************************
 AMERTA
                                 A QGIS plugin
 Analisis Multi-kriteria Embung dan Rencana Tata Air
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-09-18
        copyright            : (C) 2025 by Badan Riset dan Inovasi Nasional
        email                : sitaranisafitri@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Sitarani Safitri, Orbita Roswintiarti, Okta Fajar Saputra, Galdita Aruba Chulafak, Gatot Nugroho, Wismu Sunarmodo, Kusumaning Ayu Dyah Sukowati, Hana Listi Fitriana'
__date__ = '2025-09-18'
__copyright__ = '(C) 2025 by Badan Riset dan Inovasi Nasional'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

import os, re, glob, processing
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing, QgsProcessingAlgorithm, QgsProcessingException,
    QgsProcessingParameterFile, QgsProcessingParameterString,
    QgsProcessingParameterNumber, QgsProcessingParameterRasterDestination,
    QgsProcessingUtils
)
import os
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QUrl

class ChirpsMeanAlgorithm(QgsProcessingAlgorithm):
    INPUT_DIR   = 'INPUT_DIR'
    START_YEAR  = 'START_YEAR'
    END_YEAR    = 'END_YEAR'
    FILE_PREFIX = 'FILE_PREFIX'
    FILE_SUFFIX = 'FILE_SUFFIX'
    OUTPUT      = 'OUTPUT'

    def tr(self, s: str) -> str:
        return QCoreApplication.translate('Processing', s)

    def createInstance(self):
        return ChirpsMeanAlgorithm()

    def name(self):
        return 'a_chirps_mean_strict_nodata'

    def displayName(self):
        return self.tr('CHIRPS – Mean Rainfall')

    def groupId(self):
        return 'A. RainPCA'

    def group(self):
        return self.tr(self.groupId())

    def icon(self):
        return QIcon(os.path.join(os.path.dirname(__file__), 'rainpca.png'))
    
    def shortHelpString(self):
        return self.tr("""\
    🇮🇩 ID Modul ini menghitung rata-rata curah hujan multi-tahun dari berkas-berkas CHIRPS bulanan dalam satu folder.

    Alur pakai:
    1) Folder containing CHIRPS TIFFs: arahkan ke folder yang berisi GeoTIFF CHIRPS bulanan (01–12) untuk setiap tahun di rentang A–B.
    2) Start year (A) dan End year (B): tentukan periode analisis (mis. 1981–2024).
    3) Filename prefix / suffix (opsional): bagian nama berkas sebelum & sesudah penanda bulan untuk mempermudah pencocokan. Contoh pola: `chirps-v2.0-2010-01_jawa.tif` → prefix=`chirps-v2.0-`, suffix=`_jawa.tif`.
    4) Klik Run → algoritma mencari 12 TIFF bulanan per tahun, menjumlahkannya menjadi tahunan, lalu menghitung rata-rata tahunan lintas tahun A–B.
    5) Mean Rainfall: simpan raster keluaran (atau biarkan sementara).

    ──────────────
            
    🌍 EN This module computes the multi-year mean rainfall from monthly CHIRPS rasters in a folder.

    Usage:
    1) Folder containing CHIRPS TIFFs: point to the directory containing monthly CHIRPS GeoTIFFs (months 01–12) for each year within A–B.
    2) Start year (A) and End year (B): set the analysis period (e.g., 1981–2024).
    3) Filename prefix / suffix (optional): parts of filenames before & after the month token to help matching. Example: `chirps-v2.0-2010-01_jawa.tif` → prefix=`chirps-v2.0-`, suffix=`_jawa.tif`.
    4) Click Run → the algorithm finds the 12 monthly TIFFs per year, sums them to annual totals, then computes the mean annual rainfall across A–B.
    5) Mean Rainfall: choose an output path (or keep a temporary layer).""")

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFile(
                self.INPUT_DIR, self.tr('Folder containing CHIRPS TIFFs'),
                behavior=QgsProcessingParameterFile.Folder
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.START_YEAR, self.tr('Start year (A)'),
                type=QgsProcessingParameterNumber.Integer, defaultValue=1981
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.END_YEAR, self.tr('End year (B)'),
                type=QgsProcessingParameterNumber.Integer, defaultValue=2024
            )
        )
        self.addParameter(
            QgsProcessingParameterString(
                self.FILE_PREFIX, self.tr('Filename prefix (opsional)'),
                defaultValue='chirps-v2.0.'
            )
        )
        self.addParameter(
            QgsProcessingParameterString(
                self.FILE_SUFFIX, self.tr('Filename suffix (opsional, e.g., _jawa.tif)'),
                defaultValue='_jawa.tif'
            )
        )
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT, self.tr('Mean Rainfall')
            )
        )

    def _select_files(self, folder, y0, y1, prefix, suffix):
        # pola: prefix + YYYY + '.' + MM + suffix
        pat = re.compile(r'^' + re.escape(prefix) + r'(\d{4})\.(\d{2})' + re.escape(suffix) + r'$')
        items = []
        for path in sorted(glob.glob(os.path.join(folder, '*.tif'))):
            name = os.path.basename(path)
            m = pat.match(name)
            if not m:
                continue
            yy = int(m.group(1))
            if y0 <= yy <= y1:
                mm = int(m.group(2))
                items.append((yy, mm, path))
        items.sort(key=lambda t: (t[0], t[1]))
        return [p for _, _, p in items]

    def _make_vrt_with_nodata(self, src, idx, context, feedback):
        vrt_path = QgsProcessingUtils.generateTempFilename(f'fix_nodata_{idx:04d}.vrt')
        processing.run(
            'gdal:translate',
            {
                'INPUT': src,
                'TARGET_CRS': None,
                'NODATA': -9999.0,          # set -9999 as NoData for this source
                'COPY_SUBDATASETS': False,
                'OPTIONS': '',               # leave empty (avoid duplicate -of)
                'EXTRA': '',
                'DATA_TYPE': 0,
                'OUTPUT': vrt_path           # .vrt → GDAL will use -of VRT automatically
            },
            context=context, feedback=feedback, is_child_algorithm=True
        )
        return vrt_path

    def processAlgorithm(self, parameters, context, feedback):
        folder      = self.parameterAsFile(parameters, self.INPUT_DIR, context)
        y0          = int(self.parameterAsInt(parameters, self.START_YEAR, context))
        y1          = int(self.parameterAsInt(parameters, self.END_YEAR, context))
        prefix      = self.parameterAsString(parameters, self.FILE_PREFIX, context) or ''
        suffix      = self.parameterAsString(parameters, self.FILE_SUFFIX, context) or '.tif'
        out_raster  = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)

        if not os.path.isdir(folder):
            raise QgsProcessingException(self.tr('Folder is not valid.'))
        if y1 < y0:
            raise QgsProcessingException(self.tr('END_YEAR must be ≥ START_YEAR.'))

        src_files = self._select_files(folder, y0, y1, prefix, suffix)
        if not src_files:
            raise QgsProcessingException(self.tr('No files match the selected year range and filename pattern.'))

        feedback.pushInfo(self.tr(f'Selected rasters: {len(src_files)}'))

        # Bungkus semua source jadi VRT dengan NoData=-9999
        fixed = []
        for i, p in enumerate(src_files, 1):
            fixed.append(self._make_vrt_with_nodata(p, i, context, feedback))

        ref = fixed[0]
        tmp_mean = QgsProcessingUtils.generateTempFilename('mean_tmp.tif')

        # 1) Utama: qgis:cellstatistics → MEAN dengan IGNORE_NODATA=False (propagate NoData)
        try:
            processing.run(
                'qgis:cellstatistics',
                {
                    'INPUT': fixed,
                    'STATISTIC': 2,           # 2 = mean
                    'IGNORE_NODATA': False,   # any NoData in stack → output NoData
                    'REFERENCE_LAYER': ref,
                    'OUTPUT': tmp_mean
                },
                context=context, feedback=feedback, is_child_algorithm=True
            )
        except Exception as e1:
            feedback.pushInfo(self.tr(f'Fallback to GRASS r.series (average+count) because: {e1}'))
            # a) mean
            mean_out = QgsProcessingUtils.generateTempFilename('mean_grass.tif')
            processing.run(
                'grass7:r.series',
                {
                    'input': fixed,
                    'method': 'average',
                    'range': None,
                    'quantile': 50,
                    'weights': '',
                    'output': mean_out,
                    'GRASS_REGION_PARAMETER': ref,
                    'GRASS_REGION_CELLSIZE_PARAMETER': 0,
                    'GRASS_RASTER_FORMAT_OPT': '',
                    'GRASS_RASTER_FORMAT_META': ''
                },
                context=context, feedback=feedback, is_child_algorithm=True
            )
            # b) count valid
            count_out = QgsProcessingUtils.generateTempFilename('count_grass.tif')
            processing.run(
                'grass7:r.series',
                {
                    'input': fixed,
                    'method': 'count',
                    'range': None,
                    'quantile': 50,
                    'weights': '',
                    'output': count_out,
                    'GRASS_REGION_PARAMETER': ref,
                    'GRASS_REGION_CELLSIZE_PARAMETER': 0,
                    'GRASS_RASTER_FORMAT_OPT': '',
                    'GRASS_RASTER_FORMAT_META': ''
                },
                context=context, feedback=feedback, is_child_algorithm=True
            )
            # c) Mask: jika count < Nlayers → NoData
            N = len(fixed)
            tmp_masked = QgsProcessingUtils.generateTempFilename('mean_masked.tif')
            processing.run(
                'qgis:rastercalculator',
                {
                    'EXPRESSION': f"({{A}}) * ({{B}} = {N}) + (-9999) * ({{B}} < {N})",
                    'LAYERS': [mean_out, count_out],
                    'CRS': None,
                    'EXTENT': None,
                    'CELLSIZE': 0,
                    'OUTPUT': tmp_masked
                },
                context=context, feedback=feedback, is_child_algorithm=True
            )
            tmp_mean = tmp_masked

        # 2) Final translate: set NoData=-9999 + compress
        processing.run(
            'gdal:translate',
            {
                'INPUT': tmp_mean,
                'TARGET_CRS': None,
                'NODATA': -9999.0,
                'COPY_SUBDATASETS': False,
                'OPTIONS': 'COMPRESS=LZW',
                'EXTRA': '',
                'DATA_TYPE': 0,
                'OUTPUT': out_raster
            },
            context=context, feedback=feedback, is_child_algorithm=True
        )

        return {self.OUTPUT: out_raster}
