# -*- coding: utf-8 -*-

"""
/***************************************************************************
 AMERTA
                                 A QGIS plugin
 Analisis Multi-kriteria Embung dan Rencana Tata Air
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-09-18
        copyright            : (C) 2025 by Badan Riset dan Inovasi Nasional
        email                : sitaranisafitri@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Sitarani Safitri, Orbita Roswintiarti, Okta Fajar Saputra, Galdita Aruba Chulafak, Gatot Nugroho, Wismu Sunarmodo, Kusumaning Ayu Dyah Sukowati, Hana Listi Fitriana'
__date__ = '2025-09-18'
__copyright__ = '(C) 2025 by Badan Riset dan Inovasi Nasional'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import (
    QCoreApplication, QVariant, Qt, QObject, pyqtSlot, QMetaObject
)
from qgis.PyQt.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, QPushButton,
    QTableWidget, QTableWidgetItem, QDialogButtonBox, QComboBox, QFileDialog
)
from qgis.core import (
    QgsProcessing, QgsProcessingAlgorithm,
    QgsProcessingParameterFeatureSource, QgsProcessingParameterField,
    QgsProcessingParameterBoolean, QgsProcessingParameterFeatureSink,
    QgsFields, QgsField, QgsFeatureSink, QgsFeature, QgsProcessingException,
    QgsProcessingUtils
)
import processing  # for native:fixgeometries
import os, json, re, unicodedata
import os
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QUrl


def _tr(s): return QCoreApplication.translate('Processing', s)

# Daftar kelas Jenis Tanah standar (sesuai list yang kamu berikan)
SOIL_STANDAR = [
    "Aluvial (Fluvisol)",
    "Andosol",
    "Arenosol (Pasiran)",
    "Gleisol (Gleysol)",
    "Grumosol (Vertisol)",
    "Kambisol",
    "Latosol (Ferrasol)",
    "Litosol (Leptosol)",
    "Mediteran (Terra rossa)",
    "Nitosol (Nitisol)",
    "Oksisol (Oxisol)",
    "Organosol/Gambut (Histosol)",
    "Planosol Hidromorf (Planosol)",
    "Plinthosol",
    "Podsolik (Acrisol/Alisol)",
    "Podzol (Podsol)",
    "Regosol",
    "Rendzina (Rensina)",
    "Solonchak (Salin)",
    "Solonetz (Sodik)",
    "Waduk/Danau/Situ"
]

def _norm(s: str) -> str:
    """Normalisasi untuk pencocokan mapping."""
    if s is None: return ''
    s = str(s)
    s = unicodedata.normalize('NFKD', s)
    s = ''.join(ch for ch in s if not unicodedata.combining(ch))
    s = s.strip().lower()
    s = re.sub(r'[^a-z0-9 ]+', ' ', s)
    s = re.sub(r'\s+', ' ', s)
    return s


# ---------------- Dialog ----------------
class MappingDialog(QDialog):
    def __init__(self, unique_values, parent=None):
        super().__init__(parent)
        self.setWindowTitle(_tr('Standardized Soil Type Mapping (JTNH)'))
        self.resize(860, 560)

        layout = QVBoxLayout(self)

        # Top bar: Save/Load + status
        top = QHBoxLayout()
        self.btnLoad = QPushButton("Load JSON…")
        self.btnSave = QPushButton("Save JSON…")
        self.lblInfo = QLabel("")
        self.lblInfo.setStyleSheet("color: #666;")
        top.addWidget(self.btnLoad)
        top.addWidget(self.btnSave)
        top.addStretch(1)
        top.addWidget(self.lblInfo)
        layout.addLayout(top)

        # Table
        self.table = QTableWidget(len(unique_values), 2, self)
        self.table.setHorizontalHeaderLabels([
            _tr('Soil Type'),
            _tr('Standard Soil Type (JTNH)')
        ])
        self.table.horizontalHeader().setStretchLastSection(True)
        layout.addWidget(self.table, 1)

        for r, src in enumerate(unique_values):
            item = QTableWidgetItem('' if src is None else str(src).strip())
            item.setFlags(item.flags() & ~Qt.ItemIsEditable)
            self.table.setItem(r, 0, item)

            combo = QComboBox(self.table)
            combo.setEditable(False)
            combo.addItem('')                # kosong = skip
            combo.addItems(SOIL_STANDAR)     # fixed options
            self.table.setCellWidget(r, 1, combo)

        # Buttons
        btns = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel, self)
        btns.accepted.connect(self.accept)
        btns.rejected.connect(self.reject)
        layout.addWidget(btns)

        # Signals
        self.btnLoad.clicked.connect(self._load_json)
        self.btnSave.clicked.connect(self._save_json)

    # ---- Save/Load helpers ----
    def _current_mapping_norm(self):
        """dict: _norm(src) -> JTNH or None"""
        mp = {}
        for r in range(self.table.rowCount()):
            src_raw = self.table.item(r, 0).text().strip()
            val = self.table.cellWidget(r, 1).currentText().strip()
            mp[_norm(src_raw)] = (val if val else None)
        return mp

    def _apply_mapping_norm(self, mapping_norm):
        """prefill combos dari mapping (keys normalized)"""
        filled = 0
        for r in range(self.table.rowCount()):
            src_raw = self.table.item(r, 0).text().strip()
            key = _norm(src_raw)
            val = mapping_norm.get(key)
            if val and val in SOIL_STANDAR:
                combo = self.table.cellWidget(r, 1)
                combo.setCurrentText(val)
                filled += 1
        self.lblInfo.setText(_tr(f"Loaded: {filled} rows filled."))

    def _load_json(self):
        path, _ = QFileDialog.getOpenFileName(self, _tr("Select JSON mapping file"), "", "JSON (*.json)")
        if not path: return
        try:
            with open(path, "r", encoding="utf-8") as f:
                data = json.load(f)
            mapping_norm = data.get("mapping_norm", data if isinstance(data, dict) else {})
            mapping_norm = {str(k): (v if (v in SOIL_STANDAR) else None) for k, v in mapping_norm.items()}
            self._apply_mapping_norm(mapping_norm)
            self.lblInfo.setText(_tr(f"Loaded: {os.path.basename(path)}"))
        except Exception as e:
            self.lblInfo.setText(_tr(f"Failed to load: {e}"))

    def _save_json(self):
        path, _ = QFileDialog.getSaveFileName(self, _tr("Save JSON mapping"), "mapping_jtnh.json", "JSON (*.json)")
        if not path: return
        try:
            payload = {
                "type": "jtnh_mapping",
                "version": 1,
                "mapping_norm": self._current_mapping_norm()
            }
            with open(path, "w", encoding="utf-8") as f:
                json.dump(payload, f, ensure_ascii=False, indent=2)
            self.lblInfo.setText(_tr(f"Saved: {os.path.basename(path)}"))
        except Exception as e:
            self.lblInfo.setText(_tr(f"Failed to save: {e}"))

    def mapping_dict(self):
        """Return dict normalized: _norm(src_text) -> mapped JTNH (or None)"""
        return self._current_mapping_norm()


# Helper agar dialog dipanggil di GUI thread (hindari freeze)
class _DialogRunner(QObject):
    def __init__(self, uniques):
        super().__init__()
        self.uniques = uniques
        self.mapping = {}
        self.ok = False

    @pyqtSlot()
    def run(self):
        dlg = MappingDialog(self.uniques, parent=None)
        if dlg.exec_() == QDialog.Accepted:
            self.mapping = dlg.mapping_dict()
            self.ok = True
        else:
            self.ok = False

def _ask_mapping_on_main_thread(unique_values):
    runner = _DialogRunner(unique_values)
    runner.moveToThread(QCoreApplication.instance().thread())
    QMetaObject.invokeMethod(runner, "run", Qt.BlockingQueuedConnection)
    if not runner.ok:
        raise QgsProcessingException(_tr('Canceled by user.'))
    return runner.mapping


# ---------------- Algorithm ----------------
class StandardSoilClass(QgsProcessingAlgorithm):
    P_LAYER    = 'P_LAYER'
    P_FIELD    = 'P_FIELD'
    P_SHOWUI   = 'P_SHOWUI'
    P_ONLY_NULL= 'P_ONLY_NULL'
    P_OUTPUT   = 'P_OUTPUT'

    def tr(self, s): return QCoreApplication.translate('Processing', s)
    def name(self): return 'standarisasi_jenistanah_interaktif'
    def displayName(self): return self.tr('Soil Type Standardization (JTNH)')
    def groupId(self): return 'B. Preprocessing'
    def group(self): return self.tr(self.groupId())
    def createInstance(self): return StandardSoilClass()
    def icon(self):
        return QIcon(os.path.join(os.path.dirname(__file__), 'preprocessing.png'))

    def shortHelpString(self):
        return self.tr("""\
    🇮🇩 ID Modul men-standarisasi Jenis Tanah dari kolom sumber ke daftar standar (JTNH).
    Di awal proses, input otomatis difiksasi dengan native:fixgeometries agar tidak gagal karena geometri tidak valid.

    Alur:
    1) Pilih layer & kolom sumber Jenis Tanah → Run.
    2) Dialog mapping muncul (bisa Load/Save JSON).
    3) Centang "Hanya isi yang JTNH masih NULL" bila tidak ingin overwrite.
    4) Output berisi field 'JTNH' terstandar.
    
    ──────────────
    
    🌍 EN This tool standardizes soil-type values from a source field into a fixed list of standard classes (JTNH).
    At the start, the input is automatically repaired with native:fixgeometries to avoid failures from invalid geometries.

    Workflow:
    1) Select the layer and the source soil-type field, then click Run.
    2) An interactive mapping dialog opens: left = unique source values, right = standard soil-class dropdown.
    3) Use Load JSON to prefill mappings or Save JSON to reuse them later.
    4) Check "Only fill when JTNH is NULL" to keep existing non-NULL values.
    5) The output layer contains a standardized 'JTNH' field (the input layer is not modified).
    """)

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.P_LAYER, self.tr('Input Layer'),
                [QgsProcessing.TypeVectorAnyGeometry]
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                self.P_FIELD, self.tr('Soil Type field (source)'),
                parentLayerParameterName=self.P_LAYER,
                type=QgsProcessingParameterField.String
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.P_SHOWUI, self.tr('Show interactive mapping dialog'), defaultValue=True
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.P_ONLY_NULL, self.tr('Only fill when JTNH is NULL'), defaultValue=True
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.P_OUTPUT, self.tr('JTNH (standard)')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        # ---------- STEP 0: FIX GEOMETRIES DI AWAL ----------
        feedback.pushInfo(self.tr('Fixing input geometries (native:fixgeometries)…'))
        fx = processing.run(
            'native:fixgeometries',
            {'INPUT': parameters[self.P_LAYER], 'OUTPUT': 'memory:'},
            context=context, feedback=feedback, is_child_algorithm=True
        )

        # Hasil bisa berupa layer object atau string layer-id (QGIS 3.40)
        out_any = fx['OUTPUT']
        if isinstance(out_any, str):
            src_layer = QgsProcessingUtils.mapLayerFromString(out_any, context)
        else:
            src_layer = out_any

        if src_layer is None:
            raise QgsProcessingException(self.tr('Failed to retrieve the layer from fix geometries.'))

        fld = self.parameterAsString(parameters, self.P_FIELD, context)
        showui = self.parameterAsBool(parameters, self.P_SHOWUI, context)
        only_null = self.parameterAsBool(parameters, self.P_ONLY_NULL, context)

        if fld not in [f.name() for f in src_layer.fields()]:
            raise QgsProcessingException(self.tr('Source field not found on the layer (after fix geometries).'))

        # ---------- STEP 1: kumpulkan nilai unik ----------
        uniques_display, seen_norm = [], set()
        for f in src_layer.getFeatures():
            raw = '' if f[fld] is None else str(f[fld]).strip()
            key = _norm(raw)
            if key not in seen_norm:
                seen_norm.add(key)
                uniques_display.append(raw)

        # ---------- STEP 2: dialog → mapping ----------
        mapping_norm = {}
        if showui:
            feedback.pushInfo(self.tr('Opening JTNH mapping dialog…'))
            mapping_norm = _ask_mapping_on_main_thread(uniques_display)
        if not mapping_norm:
            mapping_norm = {}

        # ---------- STEP 3: siapkan field output ----------
        out_fields = QgsFields(src_layer.fields())
        low_out = [f.name().lower() for f in out_fields]
        if 'jtnh' in low_out:
            idx_out = low_out.index('jtnh')
        else:
            out_fields.append(QgsField('JTNH', QVariant.String, '', 80, 0))
            idx_out = len(out_fields) - 1

        # index JTNH pada source (jika ada)
        low_src = [f.name().lower() for f in src_layer.fields()]
        idx_src = low_src.index('jtnh') if 'jtnh' in low_src else -1

        sink, out_id = self.parameterAsSink(
            parameters, self.P_OUTPUT, context,
            out_fields, src_layer.wkbType(), src_layer.sourceCrs()
        )

        # ---------- STEP 4: tulis output ----------
        total = max(1, src_layer.featureCount())
        changed = 0

        for i, feat in enumerate(src_layer.getFeatures()):
            if feedback.isCanceled(): break

            # nilai existing JTNH di source (jika ada)
            existing = None
            if idx_src >= 0:
                existing = feat.attributes()[idx_src]
                if isinstance(existing, str):
                    existing = existing.strip()
                if existing == '':
                    existing = None

            # calon nilai baru dari mapping
            raw = '' if feat[fld] is None else str(feat[fld]).strip()
            mapped = mapping_norm.get(_norm(raw))

            # terapkan aturan only_null
            if only_null and (existing is not None):
                new_val = existing
            else:
                new_val = mapped if (mapped is not None) else existing

            attrs = list(feat.attributes())
            if len(attrs) < len(out_fields):
                attrs += [None] * (len(out_fields) - len(attrs))

            # hitung perubahan
            if (idx_src < 0 and new_val is not None) or (idx_src >= 0 and new_val != existing):
                changed += 1

            attrs[idx_out] = new_val

            nf = QgsFeature(out_fields)
            nf.setGeometry(feat.geometry())
            nf.setAttributes(attrs)
            sink.addFeature(nf, QgsFeatureSink.FastInsert)

            if i % 1000 == 0:
                feedback.setProgress(int(100*i/total))

        feedback.pushInfo(self.tr(f"Rows updated/filled: {changed}"))
        return { self.P_OUTPUT: out_id }
