# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AlgoMapsPlugin
                                 A QGIS plugin
 Plugin Algolytics do standaryzacji danych adresowych i geokodowania
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-09-05
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Algolytics Technologies
        email                : info@algolytics.pl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QTableWidgetItem, QComboBox, QMessageBox

from qgis.core import (
    QgsApplication,
    Qgis,
    QgsMessageLog,
    QgsProject,
    QgsRectangle, QgsPointXY, QgsCoordinateReferenceSystem, QgsCoordinateTransform,
    QgsField, QgsFeature, QgsVectorLayer,
    QgsGeometry
)

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .algomaps_qgis_dockwidget import AlgoMapsPluginDockWidget
import os
import sys

import json

CONFIG_PATH = 'dq_config.json'
DEBUG_MODE = False  # Verbose messages


def find_python():
    def _log_python_path(ppath, tag='-'):
        if DEBUG_MODE:
            QgsMessageLog().logMessage(message=f'[{tag}] Found python executable: {ppath}',
                                       tag='AlgoMaps',
                                       level=Qgis.MessageLevel.Info)
        pass

    import platform

    if platform.system() != "Windows":  # -> 'Linux' or 'Darwin' (MacOS)
        if 'python' in sys.executable:
            _log_python_path(sys.executable, tag='UNIX')
            return sys.executable

    if platform.system() == "Darwin":
        if DEBUG_MODE:
            py_exe = os.path.join(os.path.dirname(sys.executable), 'bin', 'python3')
            _log_python_path(py_exe, tag='MacOS')
            return py_exe

    if platform.system() == 'Linux':
        ...

    if platform.system() == "Windows":
        for path in sys.path:  # TODO: check correctness
            assumed_path = os.path.join(path, "python.exe")
            if os.path.isfile(assumed_path):
                _log_python_path(assumed_path, tag='Windows')
                return assumed_path

    return None


def _get_package_manager_command():
    import platform

    if platform.system() == 'Linux':
        os_info = platform.freedesktop_os_release().get('ID', 'linux')
        if os_info in ('ubuntu', 'debian', 'linuxmint', 'kubuntu', 'xubuntu', 'lubuntu', 'pop', 'peppermint', 'mx'):
            return ['apt', 'install']
        if os_info in ('fedora', 'centos', 'rhel', 'fedoraremixforwsl'):
            return ['yum', 'install']
        if os_info in ('arch', 'manjaro'):
            return ['pacman', '-S']
        if os_info in ('opensuse', 'suse'):
            return ['zypper', 'install']
        if os_info in ('gentoo', 'funtoo', 'chromeos'):
            return ['emerge']

    if platform.system() == 'Darwin':
        return ['brew', 'install']


def version_is_lower(version: str, to_compare: tuple):
    """
    A simple function to compare module versions
    works fine with simple examples like 1.3.10 or 2.0.3a but may produce false results with versions lik
    0.1rc1 (-> 0.11) or 0.0.3beta2 (-> 0.0.32)
    """
    import re
    ver_str = re.sub(r'[^0-9.]', '', version).split(".")  # Good enough for us
    ver_tuple = tuple(map(int, ver_str))

    return ver_tuple < to_compare


class AlgoMapsPlugin:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.batch_header_type = None
        self.batch_quote = None
        self.batch_sep = None
        self.include_gus = None
        self.include_teryt = None
        self.include_buildinfo = None
        self.include_financial = None
        self.flags = {}
        self.iface = iface
        self.canvas = self.iface.mapCanvas()
        self.batch_combo_widgets = []  # List of column-role comboBoxes for Batch processing
        self.csv_path = None
        self.csv_path_output = None
        self.taskManager = QgsApplication.taskManager()

        # Initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        self.config_path = os.path.join(self.plugin_dir, CONFIG_PATH)

        # Initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'AlgoMapsPlugin_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&AlgoMaps')

        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'AlgoMapsPlugin')
        self.toolbar.setObjectName(u'AlgoMapsPlugin')

        self.pluginIsActive = False
        self.dockwidget = None
        self.qproj = None

        if Qgis.versionInt() > 33800:
            from qgis.PyQt.QtCore import QMetaType
            self._field_string_type = QMetaType.QString
            self._field_int_type = QMetaType.Int
            self._field_double_type = QMetaType.Double
        else:
            from qgis.PyQt.QtCore import QVariant
            self._field_string_type = QVariant.String
            self._field_int_type = QVariant.Int
            self._field_double_type = QVariant.Double

        # Read config file
        try:

            with open(self.config_path) as f:
                conf = json.load(f)

            self.dq_user = conf.get("dq_user")
            self.dq_token = conf.get("dq_token")
            self.api_key = conf.get("api_key")

            self.default_chk_teryt = conf.get("default_chk_teryt")
            self.default_chk_gus = conf.get("default_chk_gus")
            self.default_chk_buildinfo = conf.get("default_chk_buildinfo")
            self.default_chk_financial = conf.get("default_chk_financial")

        except Exception as e:
            iface.messageBar().pushMessage(self.tr(u'AlgoMaps'),
                                           self.tr(u'Cannot read config file, check details in "Log Messages" tab.'),
                                           level=Qgis.MessageLevel.Critical)
            QgsMessageLog().logMessage(repr(e), tag='AlgoMaps', level=Qgis.MessageLevel.Critical)

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('AlgoMapsPlugin', message)

    def install_module(self, module_name, upgrade=False, force_external=False, curr_depth=0):
        if curr_depth > 3:  # Recursion failsafe
            return

        import subprocess

        if DEBUG_MODE:
            QgsMessageLog().logMessage(message=f'Install {module_name}',
                                       tag='AlgoMaps',
                                       level=Qgis.MessageLevel.Info)

        arg_list = [find_python(), '-m', 'pip', 'install', module_name]
        if upgrade:
            arg_list.insert(4, '--upgrade')
        if force_external:
            arg_list.insert(4, '--break-system-packages')

        if DEBUG_MODE:
            QgsMessageLog().logMessage(message=' '.join(arg_list),
                                       tag='AlgoMaps',
                                       level=Qgis.MessageLevel.Info)

        result = subprocess.run(arg_list,
                                stdout=subprocess.PIPE,
                                stderr=subprocess.STDOUT)

        return_code = result.returncode

        if DEBUG_MODE:
            QgsMessageLog().logMessage(message=f'Return code: {return_code}',
                                       tag='AlgoMaps',
                                       level=Qgis.MessageLevel.Info)

        if return_code == 1:
            pip_message = result.stdout.decode(encoding="utf-8").lower()

            if 'no module named pip' in pip_message:

                if DEBUG_MODE:
                    QgsMessageLog().logMessage(message=f'Trying to install pip...',
                                               tag='AlgoMaps',
                                               level=Qgis.MessageLevel.Info)

                # Install pip via official get-pip.py script
                # print('Try to install pip via official get-pip.py script')

                d_script = "import urllib.request; import os; urllib.request.urlretrieve('https://bootstrap.pypa.io/get-pip.py', 'algomaps_get-pip.py')"
                try:
                    download_pip_python = subprocess.run(
                        [sys.executable,
                         '-c',
                         d_script
                         ]
                    )
                    install_pip_res = subprocess.run([find_python(), 'algomaps_get-pip.py'])
                except:
                    QgsMessageLog().logMessage(message=f'Could not install pip to install modules. Try to execute '
                                                       f'the troubleshooting steps (README.md) or contact the '
                                                       f'developer team (e.g. post an issue on GitHub). You can use'
                                                       f' the plugin but Batch functionality will not work.',
                                               tag='AlgoMaps',
                                               level=Qgis.MessageLevel.Critical)
                    # print('ERRORS! download pip and install get-pip.py')
                    return

                # Check if pip has been installed
                if download_pip_python.returncode == 0 and install_pip_res.returncode == 0:
                    subprocess.run(['rm', '~/algomaps_get-pip.py'])  # cleanup
                    # print('OK, installed pip')
                else:
                    # Try to install pip with apt/other package manager
                    # print('Try to install pip with apt/other package manager')
                    package_manager_cmd = _get_package_manager_command()
                    args_list = [*package_manager_cmd,
                                 'python3-pip',
                                 'python3-setuptools',  # for zypper
                                 'python3-wheel']  # for yum and zypper
                    if package_manager_cmd[0] == 'pacman':
                        args_list = [*package_manager_cmd, 'python-pip']

                    apt_res = subprocess.run(args_list)
                    # print(args_list)
                    if apt_res.returncode != 0:
                        QgsMessageLog().logMessage(message=f'Could not install pip to install modules. Try to execute '
                                                           f'the troubleshooting steps (README.md) or contact the '
                                                           f'developer team (e.g. post an issue on GitHub). You can use'
                                                           f' the plugin but Batch functionality will not work.',
                                                   tag='AlgoMaps',
                                                   level=Qgis.MessageLevel.Critical)
                        return

                # Try reinstalling the Python module
                self.install_module(module_name, upgrade=upgrade, curr_depth=curr_depth + 1)

            if 'externally managed environment' in pip_message or 'externally-managed-environment' in pip_message:
                if force_external:  # We tried to force install before and it didn't work
                    QgsMessageLog().logMessage(message=f'Could not use pip to install modules. Batch will not work!',
                                               tag='AlgoMaps',
                                               level=Qgis.MessageLevel.Critical)
                    return

                # Try force install module to system-wise Python
                msg_box = QMessageBox()
                msg_box.setIcon(QMessageBox.Question)
                msg_box.setWindowTitle(f"Instalacja biblioteki {module_name.split('>')[0]}")  # nazwa bez nr wersji
                msg_box.setText("Twoja instalacja Python jest zarządzana zewnętrznie (externally-managed). Instalacja "
                                "dodatkowych bibliotek może w rzadkich przypadkach spowodować niekompatybilność innch "
                                f"modułów. Czy na pewno chcesz zainstalować bibliotekę {module_name.split('>')[0]}?")
                msg_box.setStandardButtons(QMessageBox.Ok | QMessageBox.Cancel)

                ret_value = msg_box.exec()

                if ret_value == QMessageBox.Ok:
                    self.install_module(module_name, upgrade=upgrade, force_external=True, curr_depth=curr_depth + 1)
                return

        if return_code != 0:
            QgsMessageLog().logMessage(message=f'Unknown error code. Stdout: {result.stdout.decode(encoding="utf-8")}',
                                       tag='AlgoMaps',
                                       level=Qgis.MessageLevel.Warning)

        return return_code

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/algomaps_qgis/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'AlgoMaps - standaryzacja i geokodowanie adresów'),
            callback=self.run,
            parent=self.iface.mainWindow())

        self.qproj = QgsProject.instance()

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&AlgoMaps'),
                action)
            self.iface.removeToolBarIcon(action)
        # Remove the toolbar
        del self.toolbar

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget is None:
                # Check imports
                try:
                    import pandas as pd
                    import dq

                    # Check Pandas version
                    version = pd.__version__
                    if version_is_lower(version, (1, 3, 0)):
                        raise ModuleNotFoundError

                except ModuleNotFoundError as e:
                    msg_box = QMessageBox()
                    msg_box.setIcon(QMessageBox.Question)
                    msg_box.setWindowTitle("Instalacja bibliotek Python")
                    msg_box.setText("Czy chcesz zainstalować dodatkowe bilioteki Python? \n - pandas \n - dq-client\n\n"
                                    "Są one potrzebne do poprawnego działania funkcji Batch (przetwarzanie wsadowe).\n"
                                    "Bez ich instalacji nie będzie można przetwarzać plików CSV.")
                    msg_box.setStandardButtons(QMessageBox.Ok | QMessageBox.Cancel)

                    ret_value = msg_box.exec()
                    self._msg_install_clicked(ret_value)

                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = AlgoMapsPluginDockWidget()

                # Fill the `config.json` values into UI
                self.populate_dq_api_settings_ui()

                # Set default checkboxes values
                self.populate_checkbox_settings_ui()

                # Connect the buttons
                self.dockwidget.btn_settings_save.clicked.connect(self.save_settings)
                self.dockwidget.btn_settings_reset.clicked.connect(self.reset_settings)

                self.dockwidget.btn_geocode_general.clicked.connect(self.clicked_geocode_general)
                self.dockwidget.btn_geocode_details.clicked.connect(self.clicked_geocode_details)

                self.dockwidget.btn_batch_process.clicked.connect(self.clicked_batch_process)

                # Conenct settings checkboxes
                self.dockwidget.chk_teryt.stateChanged.connect(self.settings_chkbox_changed)
                self.dockwidget.chk_gus.stateChanged.connect(self.settings_chkbox_changed)
                self.dockwidget.chk_buildinfo.stateChanged.connect(self.settings_chkbox_changed)
                self.dockwidget.chk_financial.stateChanged.connect(self.settings_chkbox_changed)

                # Batch UI
                self.dockwidget.progress_batch.setVisible(False)
                self.dockwidget.btn_cancel_batch.setVisible(False)
                self.dockwidget.tableWidget_batch.setVisible(False)
                self.dockwidget.group_csv_info.setVisible(False)
                self.dockwidget.group_batch.setVisible(False)
                self.dockwidget.file_batch_load.fileChanged.connect(self.file_batch_load_changed)
                self.dockwidget.file_batch_save.fileChanged.connect(self.file_batch_save_changed)
                self.dockwidget.txt_sep.textEdited.connect(self.txt_sep_changed)
                self.dockwidget.txt_quotechar.textEdited.connect(self.txt_quotechar_changed)

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            # show the dockwidget
            # TODO: fix to allow choice of dock location
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()

    def populate_dq_api_settings_ui(self):
        self.dockwidget.txt_dq_user.setText(self.dq_user)
        self.dockwidget.txt_dq_token.setText(self.dq_token)
        self.dockwidget.txt_api_key.setText(self.api_key)

    def populate_checkbox_settings_ui(self):
        self.dockwidget.chk_financial.setChecked(self.default_chk_financial)
        self.dockwidget.chk_teryt.setChecked(self.default_chk_teryt)
        self.dockwidget.chk_gus.setChecked(self.default_chk_gus)
        self.dockwidget.chk_buildinfo.setChecked(self.default_chk_buildinfo)
        if self.dockwidget.chk_financial.isChecked():
            self.dockwidget.chk_gus.setEnabled(False)

    def reset_settings(self):
        # Set previous DQ/API data
        self.populate_dq_api_settings_ui()

        # Set default checkbox values
        self.populate_checkbox_settings_ui()

        if DEBUG_MODE:
            QgsMessageLog().logMessage(message="Reset ustawień.", tag='AlgoMaps', level=Qgis.MessageLevel.Info)

    def save_settings(self):
        try:
            new_settings = {
                # Save new DQ/API data
                "dq_user": self.dockwidget.txt_dq_user.text(),
                "dq_token": self.dockwidget.txt_dq_token.text(),
                "api_key": self.dockwidget.txt_api_key.text(),
                # Save the checkbox values
                "default_chk_teryt": self.dockwidget.chk_teryt.isChecked(),
                "default_chk_gus": self.dockwidget.chk_gus.isChecked(),
                "default_chk_buildinfo": self.dockwidget.chk_buildinfo.isChecked(),
                "default_chk_financial": self.dockwidget.chk_financial.isChecked(),
            }

            with open(self.config_path, 'w') as f:
                json.dump(new_settings, f)

            self.dq_user = new_settings['dq_user']
            self.dq_token = new_settings['dq_token']
            self.api_key = new_settings['api_key']

            self.default_chk_teryt = new_settings["default_chk_teryt"]
            self.default_chk_gus = new_settings["default_chk_gus"]
            self.default_chk_buildinfo = new_settings["default_chk_buildinfo"]
            self.default_chk_financial = new_settings["default_chk_financial"]

            if DEBUG_MODE:
                QgsMessageLog().logMessage(message="Zapisano ustawienia", tag='AlgoMaps', level=Qgis.MessageLevel.Success)

            self.iface.messageBar().pushMessage(self.tr(u'AlgoMaps'),
                                                self.tr(u'Zapisano ustawienia'),
                                                level=Qgis.MessageLevel.Success)

        except:
            self.iface.messageBar().pushMessage(self.tr(u'AlgoMaps'),
                                                self.tr(u'Zapis ustawień nie powiódł się'),
                                                level=Qgis.MessageLevel.Warning)

    def clicked_geocode_general(self):
        if DEBUG_MODE:
            QgsMessageLog().logMessage(message="Geokoduj (jedno pole adresowe)", tag='AlgoMaps', level=Qgis.MessageLevel.Info)

        dane_ogolne = self.dockwidget.txt_generaldata.text()

        # API request
        req_data = {
            "generalData": dane_ogolne
        }
        result_json = self.send_single_algomaps_request(req_data,
                                                        self.include_teryt,
                                                        self.include_gus,
                                                        self.include_buildinfo,
                                                        self.include_financial)
        if result_json is None:
            return

        self.dockwidget.txt_outputstand.setText(
            json.dumps(result_json, indent=4, ensure_ascii=False).encode('utf8').decode())

        if 'latitude' in result_json and 'longitude' in result_json:
            self.add_response_to_map(result_json, dane_ogolne, self.include_teryt,
                                     self.include_gus, self.include_buildinfo, self.include_financial)
        else:
            self.iface.messageBar().pushMessage(self.tr(u'AlgoMaps'),
                                                self.tr(u'Brak geokodowania dla podanego adresu'),
                                                level=Qgis.MessageLevel.Warning)

    def clicked_geocode_details(self):
        if DEBUG_MODE:
            QgsMessageLog().logMessage(message="Geokoduj (dane szczegółowe)", tag='AlgoMaps', level=Qgis.MessageLevel.Info)

        w = self.dockwidget.txt_voivodeship.text()
        p = self.dockwidget.txt_county.text()
        g = self.dockwidget.txt_commune.text()
        m = self.dockwidget.txt_city.text()
        k = self.dockwidget.txt_postal.text()
        u = self.dockwidget.txt_street.text()
        n = self.dockwidget.txt_houseno.text()
        l = self.dockwidget.txt_flatno.text()

        # API request
        req_data = {
            "voivodeshipName": w,
            "countyName": p,
            "communeName": g,
            "cityName": m,
            "postalCode": k,
            "streetName": u,
            "streetNumber": n,
            "apartmentNumber": l
        }

        result_json = self.send_single_algomaps_request(req_data,
                                                        self.include_teryt,
                                                        self.include_gus,
                                                        self.include_buildinfo,
                                                        self.include_financial)

        if result_json is None:  # Error
            return

        self.dockwidget.txt_outputstand.setText(
            json.dumps(result_json, indent=4, ensure_ascii=False).encode('utf8').decode())

        if 'latitude' in result_json and 'longitude' in result_json:
            input_text = f'{w}|{p}|{g}|{m}|{k}|{u}|{n}|{l}'
            self.add_response_to_map(result_json, input_text, self.include_teryt,
                                     self.include_gus, self.include_buildinfo, self.include_financial)
        else:
            self.iface.messageBar().pushMessage(self.tr(u'AlgoMaps'),
                                                self.tr(u'Brak geokodowania dla podanego adresu'),
                                                level=Qgis.MessageLevel.Warning)

    def send_single_algomaps_request(self, req_data, teryt=False, gus=False, buildinfo=False, financial=False):
        import requests

        active_modules = ["ADDRESSES"] if not financial else ["ADDRESSES", "FINANCES"]
        gus = gus if not financial else True  # If using financial data, we need GUS identifiers
        input_json = {
            "inputRows": [req_data],
            "processParameters": {
                "activeModules": active_modules,
                "includeBuildingsInfo": buildinfo,
                "includeSymbolicNames": teryt,
                "includeDiagnosticInfo": True,
                "includeGeographicCoordinates": True,
                "includeGusZones": gus
            }
        }
        if DEBUG_MODE:
            QgsMessageLog().logMessage(message='SEND: ' + repr(input_json), tag='AlgoMaps', level=Qgis.MessageLevel.Info)

        headers = {
            'Content-Type': 'application/json',
            'Cache-Control': 'no-cache',
            'Ocp-Apim-Subscription-Key': self.api_key
        }

        response = requests.post('https://api.algolytics.pl/dqo/api/v1/rows',
                                 json=input_json,
                                 headers=headers)
        if response.status_code == 200:
            return response.json()[0]
        else:
            self.iface.messageBar().pushMessage(self.tr(u'AlgoMaps'),
                                                self.tr(u'Nie udało się pobrać danych. Sprawdź poprawność klucza API w '
                                                        u'Ustawieniach'),
                                                level=Qgis.MessageLevel.Critical)
            return None

    def add_response_to_map(self, result_json, input_data=None, teryt=False, gus=False, buildinfo=False,
                            financial=False):
        default_fields_names = ["inputData", "voivodeshipName", "countyName", "communeName", "postalCode",
                                "cityName", "cityDistrictName", "streetAttribute", "streetName",
                                "streetNameMajorPart", "streetNameMinorPart", "streetNumber", "apartmentNumber",
                                "addressId", "numberOfApartments", "numberOfJuridicalPersons",
                                "latitude", "longitude", "statusMatch", "statusGeocoding", "statusOther"]

        default_fields = []

        for field in default_fields_names:
            if field in ["latitude", "longitude"]:
                field_type = self._field_double_type
            elif field in ["numberOfApartments", "numberOfJuridicalPersons"]:
                field_type = self._field_int_type
            else:
                field_type = self._field_string_type
            default_fields.append(self._define_field(field, field_type, result_json))

        lat = result_json.get('latitude')
        lon = result_json.get('longitude')
        status = result_json.get('statuses')

        status_match, status_geocode, status_other = self._parse_statuses(status)

        default_definitions = [QgsField(x[0], x[1]) for x in default_fields]
        default_values = dict([[x[0], x[2]] for x in default_fields])  # Field-value map
        default_values['inputData'] = input_data
        default_values['statusMatch'] = status_match
        default_values['statusGeocoding'] = status_geocode
        default_values['statusOther'] = status_other

        # Additional fields (checkboxes selected)
        additional_fields = []

        if teryt:
            additional_fields.append(self._define_field("communeSymbol", self._field_string_type, result_json))
            additional_fields.append(self._define_field("communeTypeSymbol", self._field_int_type, result_json))
            additional_fields.append(self._define_field("communeTypeName", self._field_string_type, result_json))
            additional_fields.append(self._define_field("citySymbol", self._field_string_type, result_json))
            additional_fields.append(self._define_field("cityDistrictSymbol", self._field_string_type, result_json))
            additional_fields.append(self._define_field("streetSymbol", self._field_string_type, result_json))

        if gus:
            additional_fields.append(
                self._define_field("statisticalRegionSymbol", self._field_string_type, result_json))
            additional_fields.append(self._define_field("censusCircuitSymbol", self._field_string_type, result_json))

        if buildinfo:
            # Integer fields
            for field in ["numberOfInhabitedApartmentsByGUS", "numberOfInhabitedApartmentsByPESEL",
                          "numberOfInhabitantsByGUS", "numberOfInhabitantsByPESEL",
                          "numberOfMen",
                          "numberOfMenBetween_0_4_yearsOld", "numberOfMenBetween_5_9_yearsOld",
                          "numberOfMenBetween_10_14_yearsOld", "numberOfMenBetween_15_19_yearsOld",
                          "numberOfMenBetween_20_24_yearsOld", "numberOfMenBetween_25_29_yearsOld",
                          "numberOfMenBetween_30_34_yearsOld", "numberOfMenBetween_35_39_yearsOld",
                          "numberOfMenBetween_40_44_yearsOld", "numberOfMenBetween_45_49_yearsOld",
                          "numberOfMenBetween_50_54_yearsOld", "numberOfMenBetween_55_59_yearsOld",
                          "numberOfMenBetween_60_64_yearsOld", "numberOfMenOver_65_yearsOld",
                          "numberOfWomen",
                          "numberOfWomenBetween_0_4_yearsOld", "numberOfWomenBetween_5_9_yearsOld",
                          "numberOfWomenBetween_10_14_yearsOld", "numberOfWomenBetween_15_19_yearsOld",
                          "numberOfWomenBetween_20_24_yearsOld", "numberOfWomenBetween_25_29_yearsOld",
                          "numberOfWomenBetween_30_34_yearsOld", "numberOfWomenBetween_35_39_yearsOld",
                          "numberOfWomenBetween_40_44_yearsOld", "numberOfWomenBetween_45_49_yearsOld",
                          "numberOfWomenBetween_50_54_yearsOld", "numberOfWomenBetween_55_59_yearsOld",
                          "numberOfWomenBetween_60_64_yearsOld", "numberOfWomenOver_65_yearsOld",
                          "numberOfMicroEntrepreneurs"]:
                additional_fields.append(self._define_field(field, self._field_int_type, result_json))
            # String field
            additional_fields.append(self._define_field("taxationAuthority", self._field_string_type, result_json))

        if financial:
            # Double precision fields
            for field in ["individualClientFraudStatistic", "individualClientFraudScore",
                          "individualClientDefaultStatistic", "individualClientDefaultScore",
                          "businessClientFraudStatistic", "businessClientFraudScore",
                          "businessClientDefaultStatistic", "businessClientDefaultScore",
                          "entrepreneurFraudStatistic", "entrepreneurFraudScore",
                          "entrepreneurDefaultStatistic", "entrepreneurDefaultScore",
                          "averageIncome", "incomePercentile5", "incomePercentile25", "incomePercentile50",
                          "incomePercentile75", "incomePercentile95"]:
                additional_fields.append(self._define_field(field, self._field_double_type, result_json))

        additional_values = dict([[x[0], x[2]] for x in additional_fields])  # Field-value map
        additional_definitions = [QgsField(x[0], x[1]) for x in
                                  additional_fields]  # Field definitions for data provider

        # Add to map
        layer_name = "AlgoMaps standaryzacja i geokodowanie"
        layer_find = QgsProject.instance().mapLayersByName(layer_name)
        if len(layer_find) == 0:
            # Create layer if not exists
            vl = QgsVectorLayer("Point?crs=epsg:4326", layer_name, "memory")
            self.qproj.addMapLayer(vl)

            pr = vl.dataProvider()

            pr.addAttributes([*default_definitions,
                              *additional_definitions])
            vl.updateFields()
        else:
            vl = layer_find[0]
            pr = vl.dataProvider()

        # Create feature
        f = QgsFeature()
        f.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(lon, lat)))

        # Set feature fields as layer fields
        f.setFields(vl.fields())

        # Populate fields values
        for key, value in default_values.items():
            f.setAttribute(key, value)

        cnt_invalid = 0
        for key, value in additional_values.items():
            if key in vl.fields().names():
                f.setAttribute(key, value)
            else:
                cnt_invalid += 1
                if DEBUG_MODE:
                    QgsMessageLog().logMessage(message=f'Can\'t add {key}', tag='AlgoMaps', level=Qgis.MessageLevel.Warning)

        if cnt_invalid > 0:
            self.iface.messageBar().pushMessage(self.tr(u'AlgoMaps'),
                                                self.tr(
                                                    f'Nie udało się wypełnić {cnt_invalid} atrybutów. Rozważ usunięcie/zmianę nazwy warstwy, aby poprawnie utworzyć obiekt ze wszystkimi żądanymi atrybutami.'),
                                                level=Qgis.MessageLevel.Warning)

        pr.addFeature(f)

        vl.updateFields()
        vl.updateExtents()

        # Recenter the map
        self.recenter_at_xy(lon, lat)

    def _define_field(self, field_name, field_type, result_json):
        if field_type == 'str' or field_type == 'string':
            field_type = self._field_string_type
        if field_type == 'int':
            field_type = self._field_int_type
        if field_type == 'double' or field_type == 'float':
            field_type = self._field_double_type
        return [field_name, field_type, result_json.get(field_name)]

    def recenter_at_xy(self, lon, lat, srs=4326):
        original_rect = QgsRectangle(QgsPointXY(lon, lat), QgsPointXY(lon, lat))

        # We need to transform the point lat/lon to map's CRS
        source_crs = QgsCoordinateReferenceSystem(f"EPSG:{srs}")
        dest_crs = self.qproj.crs()

        transform_context = QgsProject.instance().transformContext()
        coordinate_transform = QgsCoordinateTransform(source_crs, dest_crs, transform_context)
        transformed_rect = coordinate_transform.transformBoundingBox(original_rect)

        # Center map at the point
        self.canvas.setExtent(transformed_rect)
        self.canvas.refresh()

    def settings_chkbox_changed(self, i):
        self.include_teryt = True if self.dockwidget.chk_teryt.isChecked() else False
        self.include_gus = True if self.dockwidget.chk_gus.isChecked() else False
        self.include_buildinfo = True if self.dockwidget.chk_buildinfo.isChecked() else False
        if self.dockwidget.chk_financial.isChecked():
            self.include_financial = True
            self.include_gus = True
            self.dockwidget.chk_gus.setChecked(True)
            self.dockwidget.chk_gus.setEnabled(False)
        else:
            self.dockwidget.chk_gus.setEnabled(True)
            self.include_financial = False

        if DEBUG_MODE:
            include_txt = str([self.include_teryt, self.include_gus, self.include_buildinfo, self.include_financial])
            QgsMessageLog().logMessage(message='Checkbox state: [TERYT, GUS, BUILDINFO, FINANCIAL]', tag='AlgoMaps',
                                       level=Qgis.MessageLevel.Info)
            QgsMessageLog().logMessage(include_txt, tag='AlgoMaps', level=Qgis.MessageLevel.Info)

        self.flags = {
            'gus': self.include_gus,
            'teryt': self.include_teryt,
            'buildinfo': self.include_buildinfo,
            'financial': self.include_financial
        }

    @staticmethod
    def _parse_statuses(status):

        # Split status into three separate strings (match, geocode, others)
        import re
        matches = re.findall(r'(<dopasowanie: [^>]+>)|(<geokodowanie: [^>]+>)|(<[^>]+>)',
                             status.strip())

        status_dop = None
        status_geo = None
        status_other = []

        for match in matches:
            if match[0]:
                status_dop = match[0]
            elif match[1]:
                status_geo = match[1]
            elif match[2]:
                status_other.append(match[2])

        # Concatenate all "other" matches into a single string
        status_other = ''.join(status_other) if status_other else None

        return status_dop, status_geo, status_other

    def file_batch_load_changed(self):
        from .csv_utils import identify_header, identify_delimiter_and_quotechar

        if DEBUG_MODE:
            QgsMessageLog().logMessage(message='BATCH FILE PATH CHANGED', tag='AlgoMaps', level=Qgis.MessageLevel.Info)

        try:
            self.csv_path = self.dockwidget.file_batch_load.filePath()

            if not self.csv_path:  # Empty fileWidget path
                return

            # Infer the csv separator and quoting char
            sep, quotechar = identify_delimiter_and_quotechar(self.csv_path)
            if DEBUG_MODE:
                QgsMessageLog().logMessage(message=f'Separator: {sep} / Quotechar: {quotechar}', tag='AlgoMaps', level=Qgis.MessageLevel.Info)

            self.batch_sep = str(sep) if sep else ','
            self.batch_quote = str(quotechar) if quotechar else '"'
            self.batch_header_type = identify_header(self.csv_path, sep=sep)
            self.dockwidget.txt_sep.setText(self.batch_sep)
            self.dockwidget.txt_quotechar.setText(self.batch_quote)

            self.show_csv_table(self.csv_path, sep=self.batch_sep, header_type=self.batch_header_type, quotechar=self.batch_quote)

        except Exception as e:
            raise
            # QgsMessageLog().logMessage(str(e), tag='AlgoMaps', level=Qgis.MessageLevel.Warning)
            # pass

    def clicked_batch_process(self):
        column_roles = [cb.currentText() for cb in self.batch_combo_widgets]

        required_cols = ['DANE_OGOLNE', 'KOD_POCZTOWY', 'MIEJSCOWOSC']
        any_value = any(item in required_cols for item in column_roles)

        # Check 1: At least one of `DANE_OGOLNE`, `KOD_POCZTOWY` or `MIEJSCOWOSC` is present
        if not any_value:
            msg_box = QMessageBox()
            msg_box.setIcon(QMessageBox.Warning)
            msg_box.setWindowTitle(f"Brak wymaganych kolumn")  # nazwa bez nr wersji
            msg_box.setText("Musisz wybrać co najmniej jedną z kolumn: (DANE_OGOLNE, KOD_POCZTOWY, MIEJSCOWOSC)")
            msg_box.setStandardButtons(QMessageBox.Ok)
            msg_box.exec()
            return

        # Check 2: Duplicates
        duplicates = []
        for role in column_roles:
            if column_roles.count(role) > 1 and role not in duplicates and role not in ('PRZEPISZ', 'POMIN'):
                duplicates.append(role)

        if len(duplicates) > 0:
            msg_box = QMessageBox()
            msg_box.setIcon(QMessageBox.Warning)
            msg_box.setWindowTitle(f"Zduplikowane kolumny")  # nazwa bez nr wersji
            msg_box.setText(f"Kolumny (z wyjątkiem POMIN i PRZEPISZ) nie mogą się powtarzać! \n\nDuplikaty: {duplicates}")
            msg_box.setStandardButtons(QMessageBox.Ok)
            msg_box.exec()
            return

        # TODO: check if save as csv and filepath not empty

        if DEBUG_MODE:
            QgsMessageLog().logMessage(message='CLICKED PROCESS', tag='AlgoMaps', level=Qgis.MessageLevel.Info)
            QgsMessageLog().logMessage(message=f'COLUMN ROLES:\n{column_roles}', tag='AlgoMaps', level=Qgis.MessageLevel.Info)

        try:
            from .BatchGeocoder import BatchGeocoder
        except Exception as e:
            self.iface.messageBar().pushMessage(self.tr(u'AlgoMaps'),
                                                self.tr(
                                                    u'Cannot initialize batch processing - error in code. Contact the AlgoMaps developers'),
                                                level=Qgis.MessageLevel.Critical)
            if DEBUG_MODE:
                QgsMessageLog().logMessage(str(e), tag='AlgoMaps', level=Qgis.MessageLevel.Warning)
            return

        geocoder = BatchGeocoder(csv_path=self.csv_path,
                                 column_roles=column_roles,
                                 iface=self.iface,
                                 dock_handle=self.dockwidget,
                                 qproj=self.qproj,
                                 dq_user=self.dq_user,
                                 dq_token=self.dq_token,
                                 flags=self.flags,
                                 add_to_map=self.dockwidget.chk_add_map.isChecked(),
                                 save_csv_path=self.csv_path_output,
                                 header_type=self.batch_header_type,
                                 sep=self.batch_sep,
                                 quote=self.batch_quote)
        self.taskManager.addTask(geocoder)
        self.dockwidget.progress_batch.setVisible(True)
        self.dockwidget.btn_cancel_batch.setVisible(True)
        self.dockwidget.btn_batch_process.setEnabled(False)
        self.dockwidget.btn_batch_process.setText('Przetwarzanie...')
        self.dockwidget.txt_output_batch.setText('Przetwarzanie zadania może zająć od kilku do nawet kilkudziesięciu minut')

    def file_batch_save_changed(self):
        self.csv_path_output = self.dockwidget.file_batch_save.filePath()
        if not self.csv_path_output:
            self.dockwidget.chk_save_csv.setChecked(False)
        else:
            self.dockwidget.chk_save_csv.setChecked(True)

    def txt_sep_changed(self):
        new_sep = self.dockwidget.txt_sep.text()
        if DEBUG_MODE:
            QgsMessageLog().logMessage(message=f'CHANGED SEP: {new_sep}', tag='AlgoMaps', level=Qgis.MessageLevel.Info)
        if not new_sep:  # Empty lineEdit
            return
        if new_sep == '\\':
            return
        if new_sep == r'\t':
            new_sep = '\t'
        self.batch_sep = new_sep
        self.show_csv_table(self.csv_path, self.batch_sep, self.batch_header_type, self.batch_quote)

    def txt_quotechar_changed(self):
        new_quote = self.dockwidget.txt_quotechar.text()
        if DEBUG_MODE:
            QgsMessageLog().logMessage(message=f'CHANGED QUOTECHAR: {new_quote}', tag='AlgoMaps', level=Qgis.MessageLevel.Info)
        if not new_quote:  # Empty lineEdit
            return
        if len(new_quote) > 1:
            return
        self.batch_quote = new_quote
        self.show_csv_table(self.csv_path, self.batch_sep, self.batch_header_type, self.batch_quote)

    def show_csv_table(self, csv_path, sep=',', header_type=None, quotechar='"'):
        if DEBUG_MODE:
            QgsMessageLog().logMessage(message=f'SHOW TABLE', tag='AlgoMaps', level=Qgis.MessageLevel.Info)
            QgsMessageLog().logMessage(message=f'{csv_path} {sep} {header_type} {quotechar}', tag='AlgoMaps', level=Qgis.MessageLevel.Info)

        #  Read the first 4 rows (to examine the columns and set the DQ parameters)
        import pandas as pd
        from .csv_utils import get_file_line_count

        self.dockwidget.tableWidget_batch.clear()
        self.dockwidget.tableWidget_batch.setColumnCount(0)
        self.dockwidget.tableWidget_batch.setRowCount(0)
        self.batch_combo_widgets = []

        try:
            df = pd.read_csv(csv_path, sep=sep, header=header_type, quotechar=quotechar, nrows=4, escapechar='\\',
                             engine='python')
        except:
            return

        # Add record count to UI
        line_count = get_file_line_count(csv_path, header_type)
        self.dockwidget.lbl_records.setText(f'Rekordów: {str(line_count)}')

        # Add columns and rows
        row_count, col_count = df.shape
        [self.dockwidget.tableWidget_batch.insertColumn(0) for _ in range(col_count)]
        [self.dockwidget.tableWidget_batch.insertRow(0) for _ in range(row_count + 1)]  # One more for comboBoxes
        self.dockwidget.tableWidget_batch.setHorizontalHeaderLabels([str(col) for col in df.columns])

        # Fill the table with DataFrame values
        for i, row in enumerate(df.itertuples()):
            for k in range(col_count):
                self.dockwidget.tableWidget_batch.setItem(i + 1, k,
                                                          QTableWidgetItem(str(row[k + 1])))  # k=0 is index

        # Add column roles for DQ
        for k in range(col_count):
            new_role_combobox = QComboBox()
            role_item_list = ['PRZEPISZ', 'POMIN',
                              'ID_REKORDU',
                              'DANE_OGOLNE',
                              'KOD_POCZTOWY', 'MIEJSCOWOSC', 'ULICA_NUMER_DOMU_I_MIESZKANIA', 'ULICA', 'NUMER_DOMU',
                              'NUMER_MIESZKANIA', 'NUMER_DOMU_I_MIESZKANIA', 'WOJEWODZTWO', 'POWIAT', 'GMINA'
                              ]
            new_role_combobox.insertItems(0, role_item_list)
            new_role_combobox.insertSeparator(4)
            new_role_combobox.insertSeparator(3)
            new_role_combobox.insertSeparator(2)
            self.batch_combo_widgets.append(new_role_combobox)
            self.dockwidget.tableWidget_batch.setCellWidget(0, k, new_role_combobox)

        # Show the table
        self.dockwidget.tableWidget_batch.setVisible(True)
        self.dockwidget.group_csv_info.setVisible(True)
        self.dockwidget.group_batch.setVisible(True)

    def _msg_install_clicked(self, val):
        if val == QMessageBox.Ok:

            if DEBUG_MODE:
                QgsMessageLog().logMessage(message='User confirmed modules install',
                                         tag='AlgoMaps',
                                         level=Qgis.MessageLevel.Info)

            # Check pandas import
            try:
                import pandas as pd

                # Check Pandas version
                version = pd.__version__
                if version_is_lower(version, (1, 3, 0)):
                    raise ModuleNotFoundError

            except ModuleNotFoundError as e:

                QgsMessageLog().logMessage(message='Installing pandas...', level=Qgis.MessageLevel.Info)
                ret_code = self.install_module('pandas>=1.3.0', upgrade=True)
                if ret_code == 0:
                    self.iface.messageBar().pushMessage(self.tr(u'AlgoMaps'),
                                                        self.tr(
                                                            u'New module `pandas` installed. Restart QGIS to use '
                                                            u'AlgoMaps plugin'),
                                                        level=Qgis.MessageLevel.Warning)

            # Check dq-client import
            try:
                import dq

            except ModuleNotFoundError as e:

                QgsMessageLog().logMessage(message='Installing dq-client...', level=Qgis.MessageLevel.Info)
                ret_code_dq = self.install_module('dq-client', upgrade=False)
                QgsMessageLog().logMessage(message='Installing requests...', level=Qgis.MessageLevel.Info)
                ret_code_r = self.install_module('requests', upgrade=True)

                if ret_code_dq == 0 and ret_code_r == 0:
                    self.iface.messageBar().pushMessage(self.tr(u'AlgoMaps'),
                                                        self.tr(
                                                            u'New module `dq-client` installed. Restart QGIS to use '
                                                            u'AlgoMaps plugin'),
                                                        level=Qgis.MessageLevel.Warning)
