# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AfpolGIS
                                 A QGIS plugin
 Fetch and load geospatial data from ODK, Onadata, Kobo, ES World, GTS and DHIS
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-01-09
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Kipchirchir Cheroigin
        email                : kcheroigin@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import logging
import json
import math
import os
import time
import re
import requests
import csv
from requests.auth import HTTPBasicAuth

from PyQt5 import *
from qgis.core import (
    Qgis,
    QgsProject,
    QgsVectorLayer,
    QgsRasterLayer,
    QgsFeature,
    QgsGeometry,
    QgsField,
    QgsMessageLog,
    QgsCoordinateReferenceSystem,
)
from PyQt5.QtWidgets import *
from PyQt5.QtCore import *
from PyQt5.QtGui import *
from .afpolgis_dialog import AfpolGISDialog
from datetime import datetime, timezone

from .request_threads import (
    OnaRequestThread,
    FetchOnaFormsThread,
    FetchOnaGeoFieldsThread,
)
from .utils import polis_indicators_dict


# Configure logging
logging.basicConfig(
    filename="fetch_data.log",  # Log file name
    filemode="a",  # Append mode
    format="%(asctime)s - %(levelname)s - %(message)s",
    level=logging.DEBUG,  # Log level
)


class AfpolGIS(QObject):
    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
                     which provides the hook by which you can manipulate the QGIS
                     application at run time.
        :type iface: QgsInterface
        """
        QObject.__init__(self)
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.actions = []
        self.menu = self.tr("&AfpolGIS Data Connector")
        self.toolbar = self.iface.addToolBar("AfpolGIS")
        self.toolbar.setObjectName("AfpolGIS")
        self.features = []

        # Json data var
        self.json_data = list()
        self.odk_json_data = list()
        self.kobo_json_data = list()
        self.gts_json_data = list()
        self.es_json_data = list()
        self.dhis_json_data = list()
        self.polis_json_data = list()

        self.new_features = []
        self.form_versions = None
        self.current_form_version = None
        self.curr_geo_field = None
        self.geo_fields = set()
        self.geo_fields_dict = dict()
        self.geo_types = ["geopoint", "geoshape", "geotrace"]
        self.vlayer = dict()
        self.vlayers = dict()
        self.odk_forms_to_projects_map = dict()

        # Initializing the dialog and other components
        self.dlg = AfpolGISDialog(option="GetODK")
        self.dlg.tabWidget.setCurrentIndex(0)

        self.dlg.ComboDhisAdminLevels.addItems(
            ["Level 1", "Level 2", "Level 3", "Level 4", "Level 5"]
        )

        self.dlg.ComboDhisCategory.addItems(["Programs", "DataSets"])

        self.dlg.comboDhisPeriod.addItems(
            [
                "TODAY",
                "YESTERDAY",
                "LAST_3_DAYS",
                "LAST_7_DAYS",
                "LAST_14_DAYS",
                "THIS_MONTH",
                "LAST_MONTH",
                "LAST_3_MONTHS",
                "LAST_6_MONTHS",
                "LAST_12_MONTHS",
                "THIS_BIMONTH",
                "LAST_BIMONTH",
                "THIS_QUARTER",
                "LAST_QUARTER",
                "LAST_4_QUARTERS",
                "THIS_SIX_MONTH",
                "LAST_SIX_MONTH",
                "LAST_2_SIXMONTHS",
                "THIS_YEAR",
                "LAST_YEAR",
                "LAST_5_YEARS",
                "THIS_FINANCIAL_YEAR",
                "LAST_FINANCIAL_YEAR",
                "LAST_5_FINANCIAL_YEARS",
            ]
        )

        self.polis_indicators_dict = {
            "AFP_DOSE_0": "AFP 0 dose percentage for 6M-59M",
            "AFP_DOSE_1_to_2": "AFP 1 to 2 doses percentage for 6M-59M",
            "AFP_DOSE_3PLUS": "AFP 3+ doses percentage for 6M-59M",
            "AFP_COUNT": "AFP cases",
            "cVDPV_COUNT": "Circulating VDPV case count (all Serotypes)",
            "cVDPV_COUNT_REPORTING": "Circulating VDPV case count (all serotypes) - Reporting",
            "SURVINDCAT": "Combined Surveillance Indicators category",
            "ENV_CVDPV_COUNT": "Environmental sample circulating VDPV",
            "ENV_COUNT": "Environmental samples count",
            "ENV_WPV_COUNT": "Environmental Wild samples",
            "NPAFP_COUNT": "Non polio AFP cases (under 15Y)",
            "NPAFP_RATE_NOPENDING": "Non polio AFP rate (Pending excluded)",
            "NPAFP_RATE": "Non polio AFP rate (Pending included)",
            "NPAFP_DOSE_0_to_2": "NPAFP 0 - 2 dose percentage for 6M-59M",
            "NPAFP_DOSE_0": "NPAFP 0 dose percentage for 6M-59M",
            "NPAFP_DOSE_1_to_2": "NPAFP 1 - 2 dose percentage for 6M-59M",
            "NPAFP_DOSE_3PLUS": "NPAFP 3+ dose percentage for 6M-59M",
            "FUP_INSA_CASES_PERCENT": "Percent of 60-day follow-up cases with inadequate stool",
            "UNCLASS_CASES_PERCENT": "Percent of cases not classified",
            "NPAFP_SA_WithStoolCond": "Percent of cases w/ adeq stools specimens (condition+timeliness)",
            "NPAFP_SA_GoodStoolCond": "Percent of cases w/ adeq stools specimens (good condition+timeliness)",
            "NPAFP_SA": "Percent of cases with two specimens within 14 days of onset",
            "ENV_cVDPV_COUNT_REPORTING": "Reported circulating VDPV environmental samples count (all serotypes)",
            "ENV_WPV_COUNT_REPORTING": "Reported Wild environmental samples count",
            "SIA_BOPV": "SIA bOPV campaigns",
            "SIA_MOPV": "SIA mOPV campaigns",
            "SIA_LASTCASE_COUNT": "SIA planned since last case",
            "SIA_TOPV": "SIA tOPV campaigns",
            "SIA_OPVTOT": "Total SIA campaigns",
            "WPV_COUNT": "Wild poliovirus case count",
            "WPV_COUNT_REPORTING": "Wild poliovirus case count - Reporting",
        }

        # Set initial domains
        self.dlg.onadata_api_url.setText("api.whonghub.org")
        self.dlg.kobo_api_url.setText("kf.kobotoolbox.org")
        self.dlg.odk_api_url.setText("aap-odk-sinp.cen-nouvelle-aquitaine.dev")
        self.dlg.es_api_url.setText("es.world")
        self.dlg.gts_api_url.setText("gts.health")
        self.dlg.dhis_api_url.setText("dhis-minsante-cm.org")
        self.dlg.polis_api_url.setText("extranet.who.int/polis")

        # Set default ES API Version
        self.dlg.esAPIVersion.setText("4.4")

        # disable forms dropdown
        self.dlg.comboODKForms.setEnabled(False)
        self.dlg.comboKoboForms.setEnabled(False)

        # ES topography dropdown
        self.dlg.combESTopology.addItems(["Sites", "Labs"])

        # self.dlg.comboProviders.addItems(["OnaData", "GetODK", "KoboToolbox", "GTS"])

        # obscure the password field
        # self.dlg.password.setEchoMode(QLineEdit.Password)
        # self.dlg.mLineEdit.mousePressEvent.connect(self.toggle_password_visibility)

        # self.dlg.horizontalSlider.setMinimum(0)  # Minimum value
        # self.dlg.horizontalSlider.setMaximum(10000)  # Maximum value
        # self.dlg.horizontalSlider.setValue(1000)  # default value
        # self.dlg.slider_label.setText(str(1000))

        # date range controls

        # ONA
        self.dlg.onaDateTimeFrom.setEnabled(False)
        self.dlg.onaDateTimeTo.setEnabled(False)

        # ODK
        self.dlg.ODKDateTimeFrom.setEnabled(False)
        self.dlg.ODKDateTimeTo.setEnabled(False)

        # Kobo
        self.dlg.KoboDateTimeFrom.setEnabled(False)
        self.dlg.KoboDateTimeTo.setEnabled(False)

        # clear version and geo field dropdown
        self.dlg.comboOnaForms.setEnabled(False)
        self.dlg.comboOnaGeoFields.setEnabled(False)
        self.dlg.comboODKGeoFields.setEnabled(False)

        # Connect the "OK" button to the fetch_button_clicked method
        self.dlg.onaOkButton.clicked.connect(self.fetch_button_clicked)
        self.dlg.onaCancelButton.clicked.connect(self.cancel_button_clicked)
        self.dlg.btnRemoveAll.clicked.connect(self.clear_logs)

        # Connect the ODK "OK" button to the fetch_odk_form_data_clicked function
        self.dlg.odkOkButton.clicked.connect(self.fetch_odk_form_data_clicked)
        self.dlg.odkCancelButton.clicked.connect(self.odk_cancel_button_clicked)

        # Connect the ODK "OK" button to the fetch_kobo_form_data_clicked function
        self.dlg.koboOkButton.clicked.connect(self.fetch_kobo_form_data_clicked)
        self.dlg.koboCancelButton.clicked.connect(self.kobo_cancel_button_clicked)

        self.dlg.btnFetchOnaForms.clicked.connect(self.fetch_ona_forms_handler)

        # POLIS action handlers
        self.dlg.btnFetchPolis.clicked.connect(self.fetch_polis_virus_handler)

        self.dlg.comboPolisArea.currentIndexChanged.connect(self.fetch_polis_admin_one)
        self.dlg.comboPolisCountry.currentIndexChanged.connect(
            self.fetch_polis_admin_two
        )
        self.dlg.comboPolisProvince.currentIndexChanged.connect(
            self.fetch_polis_admin_three
        )
        self.dlg.comboPolisVizCategory.currentIndexChanged.connect(
            self.polis_viz_category_handler
        )

        # Connect the "OK" button in the POLIS Tab to the corresponding fetch handler
        self.dlg.polisOkButton.clicked.connect(self.fetch_polis_data_clicked)

        self.dlg.afroBoundOkBtn.clicked.connect(self.add_afro_shapefile_layer)

        # self.dlg.comboPolisArea.checkedItemsChanged.connect(self.on_checked_items_changed)

        # Connect the "OK" button in the ODK Tab to the corresponding fetch handler
        self.dlg.btnFetchODKForms.clicked.connect(self.fetch_odk_forms_handler)

        # Connect the "OK" button in the Kobo Tab to the corresponding fetch handler
        self.dlg.btnFetchKoboForms.clicked.connect(self.fetch_kobo_assets_handler)

        # Connect the "Connect" button in the GTS Tab to the corresponding fetch handler
        self.dlg.btnFetchGTSTables.clicked.connect(self.fetch_gts_indicators_handler)

        # Connect the "Connect" button in the DHIS Tab to the corresponding fetch handler
        self.dlg.btnFetchDhisCategory.clicked.connect(
            self.fetch_dhis_selected_category_handler
        )

        self.dlg.ComboDhisCategory.currentIndexChanged.connect(
            self.fetch_dhis_selected_category_handler
        )

        # Connect the "OK" button in the DHIS tab to the corresponding handler
        self.dlg.dhisOkButton.clicked.connect(self.fetch_dhis_indicator_data_handler)

        # Connect page size slider on change
        # self.dlg.horizontalSlider.valueChanged.connect(self.update_slider_value_label)

        # Connect calendar widget
        self.dlg.onaDateTimeFrom.dateChanged.connect(self.on_from_date_changed)
        self.dlg.onaDateTimeTo.dateChanged.connect(self.on_to_date_changed)

        # Connect the GTS OK button to handler
        self.dlg.gtsOkButton.clicked.connect(
            self.fetch_gts_tracking_rounds_data_handler
        )

        # Connect the GTS Cancel button to handler
        self.dlg.gtsCancelButton.clicked.connect(self.handle_gts_cancel_btn)

        # Connect form dropdown on change
        self.dlg.comboOnaForms.currentIndexChanged.connect(
            self.fetch_ona_form_geo_fields
        )

        # Connect form version dropdown on change
        self.dlg.comboOnaGeoFields.currentIndexChanged.connect(
            self.on_combo_box_geo_fields_change
        )

        # Connect ODK forms dropdown on change
        self.dlg.comboODKForms.currentIndexChanged.connect(
            self.on_odk_forms_combo_box_change
        )

        # Connect Kobo forms dropdown on change
        self.dlg.comboKoboForms.currentIndexChanged.connect(
            self.on_combo_box_kobo_forms_change
        )

        # Connect ES topography dropdown on change
        self.dlg.esOkButton.clicked.connect(self.fetch_es_data_clicked)

        self.dlg.combESTopology.currentIndexChanged.connect(
            self.on_es_combo_topography_change
        )

        # Connect GTS table names dropdown on change
        self.dlg.comboGTSTableTypes.currentIndexChanged.connect(
            self.on_gts_tables_combo_box_change
        )

        # Connect GTS field activities dropdown on change
        self.dlg.comboGTSFieldActivities.currentIndexChanged.connect(
            self.on_gts_field_activity_change
        )

        # Connect GTS track rounds on change
        self.dlg.comboGTSTrackingRounds.currentIndexChanged.connect(
            self.on_gts_tracking_rounds_on_change
        )

        # Connect DHIS Levels action
        self.dlg.ComboDhisAdminLevels.currentIndexChanged.connect(
            self.on_dhis_combo_admin_level_change
        )

        self.dlg.comboDhisPeriod.currentIndexChanged.connect(
            self.on_dhis_combo_period_change
        )

        # DHIS Org units on change
        # self.dlg.comboDhisOrgUnits.currentIndexChanged.connect(
        #     self.on_dhis_org_units_change
        # )

        # DHIS Indicator groups on change
        self.dlg.comboDhisProgramsOrDataSets.currentIndexChanged.connect(
            self.dhis_indicator_groups_on_change
        )

        # DHIS DataSets on change
        # self.dlg.comboDhisDataSets.currentIndexChanged.connect(
        #     self.on_dhis_datasets_change
        # )

        # DHIS Indicators on change
        self.dlg.comboDhisIndicators.currentIndexChanged.connect(
            self.on_dhis_indicators_change
        )

        self.dlg.onaOkButton.setEnabled(False)
        self.dlg.odkOkButton.setEnabled(False)
        self.dlg.koboOkButton.setEnabled(False)
        self.dlg.gtsOkButton.setEnabled(False)
        self.dlg.dhisOkButton.setEnabled(False)

        # CSV Export functionality
        self.dlg.onaDownloadCSV.setEnabled(False)
        self.dlg.odkDownloadCSV.setEnabled(False)
        self.dlg.koboDownloadCSV.setEnabled(False)

        self.dlg.gtsDownloadCSV.setEnabled(False)
        self.dlg.esDownloadCSV.setEnabled(False)
        self.dlg.dhisDownloadCSV.setEnabled(False)
        self.dlg.polisDownloadCSV.setEnabled(False)

        self.dlg.onaDownloadCSV.clicked.connect(
            lambda: self.download_csv(self.json_data)
        )
        self.dlg.odkDownloadCSV.clicked.connect(
            lambda: self.download_csv(self.odk_json_data)
        )
        self.dlg.koboDownloadCSV.clicked.connect(
            lambda: self.download_csv(self.kobo_json_data)
        )

        self.dlg.gtsDownloadCSV.clicked.connect(
            lambda: self.download_csv(self.gts_json_data)
        )
        self.dlg.esDownloadCSV.clicked.connect(
            lambda: self.download_csv(self.es_json_data)
        )
        self.dlg.dhisDownloadCSV.clicked.connect(
            lambda: self.download_csv(self.dhis_json_data)
        )

        self.dlg.polisDownloadCSV.clicked.connect(
            lambda: self.download_csv(self.polis_json_data)
        )

        # password visibility
        self.showPassword = False

        # data sync
        self.timer = QTimer()

        # Connect the timer to the data-fetching function
        self.timer.timeout.connect(self.fetch_data_async)

        # ONA sync

        self.ona_sync_timer = QTimer()
        self.ona_sync_timer.timeout.connect(self.ona_fetch_data_sync_enabled)

        # ODK Data sync
        self.odk_sync_timer = QTimer()
        self.odk_sync_timer.timeout.connect(self.on_odk_data_sync_enabled)

        # Kobo Data sync
        self.kobo_sync_timer = QTimer()
        self.kobo_sync_timer.timeout.connect(self.on_kobo_data_sync_enabled)

        # date fields
        self.from_date = None
        self.to_date = None

        # misc
        self.page_size = 20
        self.data_count = 0
        self.fields = None
        self.excluded_types = [
            "deviceid",
            "end",
            "imei",
            "instanceID",
            "phonenumber",
            "simserial",
            "start",
            "subscriberid",
            "today",
            "uuid",
            "_media_all_received",
            "group",
            "repeat",
            "geopoint",
            "gps",
            "geoshape",
            "geotrace",
            "osm",
            "start-geopoint",
            "note",
        ]

        self.afro_admin_boundaries = {
            "Countries": "https://github.com/KipSigei/who-global-boundaries/raw/refs/heads/main/GLOBAL_ADM0",
            "States": "https://github.com/KipSigei/who-global-boundaries/raw/refs/heads/main/GLOBAL_ADM1",
            "Districts": "https://github.com/KipSigei/who-global-boundaries/raw/refs/heads/main/GLOBAL_ADM2",
        }

        for k, v in self.afro_admin_boundaries.items():
            self.dlg.comboAfroAdmins.addItem(k, v)

        html_content = """
        <h2>AfPoLGIS Data Connector</h2>
        <p>
            This software has been developed by the WHO AFRO GIS Team. It is designed to extract, transform and load data from OnaData, ODK, KoboToolbox, ES World, GTS and DHIS then adds as a layer on QGIS. It also has the capability of filtering data by date ranges as well as synchronizing real time data from ODK, OnaData and KoboToolbox. Note that it could take a while to load large datasets depending on the performance of the API Servers.
        </p>
        <h3>Purpose and Features</h3>
        <ul>
            <li>Access and integrate spatial data from multiple external platforms.</li>
            <li>Simplify spatial data workflows for African administrative and policy geography.</li>
            <li>Tools for selecting and visualizing administrative boundaries, catchments, and other geospatial datasets.</li>
        </ul>
        <h3>Author Information</h3>
        <p>
            <strong>Developer:</strong> Kipchirchir Cheroigin<br>
            <strong>Contact:</strong> <a href="mailto:kcheroigin@gmail.com">kcheroigin@gmail.com</a>
        </p>
        <h3>License</h3>
        <p>
            This plugin is released under the GNU v3.0 License.<br>
            For details, refer to the <a href="https://github.com/KipSigei/afpolgis-data-connector/blob/main/LICENSE" target="_blank">LICENSE</a> file in the repository.
        </p>
        <h3>Version Information</h3>
        <p><strong>Version:</strong> 2.2.0<br><strong>Release Date:</strong> 15/01/2025</p>
        <h3>Supported Data Sources</h3>
        <ul>
            <li>ODK</li>
            <li>OnaData</li>
            <li>Kobo</li>
            <li>ES World</li>
            <li>GTS</li>
            <li>DHIS</li>
        </ul>
        <h3>How to Use</h3>
        <p>
            Follow these steps to connect to external data sources and manage data sync:
        </p>
        <ol>
            <li>
                <strong>Authentication:</strong>  
                Input the API Base Domain of choice, your username, and password in the <em>Authentication</em> section. Click <strong>Connect</strong> to establish a connection with the selected data source.
            </li>
            <li>
                <strong>Dataset/Form Selection:</strong>  
                Depending on the data source, Use the <em>Select Form</em> or <em>Select Category</em> dropdown to choose a specific form or dataset.
            </li>
            <li>
                <strong>Sync Options:</strong>  
                To Configure synchronization preferences for ODK, OnaData or Kobo, including:
                <ul>
                    <li><strong>Date Range From:</strong> Specify the start date for data synchronization.</li>
                    <li><strong>Date Range To:</strong> Specify the end date for data synchronization.</li>
                    <li><strong>Sync Interval:</strong> Set the time interval for periodic data syncs.</li>
                    <li><strong>Page Size:</strong> Adjust the number of records fetched per API call.</li>
                </ul>
            </li>
            <li>
                Click <strong>OK</strong> to start the sync process or <strong>Cancel</strong> to abort.
            </li>
        </ol>
        <p>
            Monitor the sync progress through the progress bar at the bottom of the interface or by accessing the Logs tab.
        </p>
        <h3>Repository and Documentation</h3>
        <p>
            <a href="https://github.com/KipSigei/afpolgis-data-connector" target="_blank">GitHub Repository</a><br>
            For issues or feature requests, open an issue in the repository.
        </p>

        <h3>Acknowledgments</h3>
        <p>
            The development of the AfPoLGIS Data Connector was made possible through the collaborative efforts of various individuals and teams. Special thanks to <b> Derrick Demeveng </b> for his immense support and guidance during the design process, without which we could not have had an easier time in the development process. Special thanks also go to the following team for their guidance towards the improvement of the plugin:
        </p>
        <ul>
            <li>Touray Kebba</li>
            <li>John Kipterer</li>
            <li>Mike Mwanza</li>
            <li>Caroline Gathenji</li>
            <li>David Collins Owuor</li>
        </ul>
        <p>
            Your contributions have helped shape this tool into a valuable resource for integrating geospatial data. Thank you!
        </p>

        """

        self.dlg.about_text.setText(html_content)

        # Initialize QThreadPool for managing worker threads
        self.thread_pool = QThreadPool.globalInstance()

        self.is_interrupted = False

        self.asset_from_date = None

        # Add progress bar to the dialog
        # self.dlg.progress_bar.setValue(0)

        # init worker
        self.worker = None

        self.directory = "./geojson_data"  # Replace with your target directory
        # Create the directory if it doesn't exist
        os.makedirs(self.directory, exist_ok=True)

    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        :param message: String for translation.
        :type message: str, QString
        :returns: Translated string.
        :rtype: QString
        """
        return QCoreApplication.translate("FetchDataPlugin", message)

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = os.path.join(self.plugin_dir, "icon.png")
        self.add_action(
            icon_path,
            text=self.tr("AfpolGIS Data Connector"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

    def add_action(
        self,
        icon_path,
        text,
        callback,
        parent=None,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        enabled=True,
    ):
        """Add a toolbar icon to the toolbar."""
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        if status_tip:
            action.setStatusTip(status_tip)
        if whats_this:
            action.setWhatsThis(whats_this)
        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)
        action.triggered.connect(callback)
        action.setEnabled(enabled)
        self.toolbar.addAction(action)
        self.iface.addPluginToMenu(self.menu, action)
        self.actions.append(action)
        return action

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&AfpolGIS Data Connector"), action)
            self.iface.removeToolBarIcon(action)
        del self.toolbar

    def add_basemap(self):
        # Define the basemap URL (OpenStreetMap in this example)
        basemap_url = "type=xyz&url=https://tile.openstreetmap.org/{z}/{x}/{y}.png"
        layer_name = "OpenStreetMap"

        # Create a raster layer
        basemap_layer = QgsRasterLayer(basemap_url, layer_name, "wms")

        existing_layer = None
        for layer in QgsProject.instance().mapLayers().values():
            if layer.name() == basemap_layer.name():
                existing_layer = True
                break

        # Check if the layer is valid
        if not basemap_layer.isValid():
            print("Failed to load the basemap layer!")
            return

        if not existing_layer:
            # Add the layer to the project
            QgsProject.instance().addMapLayer(basemap_layer)

            curr_layer = QgsProject.instance().mapLayersByName(f"{layer_name}")[0]

            # Access the active map canvas
            canvas = self.iface.mapCanvas()

            # Set the extent of the canvas to the basemap layer's extent
            canvas.setExtent(curr_layer.extent())

            canvas.refresh()

            print(f"{layer_name} basemap added and zoomed to native resolution.")

    def run(self):
        """Run method that performs all the real work."""
        # Show the dialog
        self.dlg.show()
        # self.add_basemap()

    def fetch_data_async(
        self, api_url, formID, username, password, geo_field, page_size, directory
    ):
        """Start the data-fetching timer."""
        value = self.dlg.mQgsDoubleSpinBox.value()
        self.timer.setInterval(int(value) * 1000)
        self.timer.start()
        self.dlg.app_logs.appendPlainText(f"Start of Data Sync...\n")
        self.json_data = []
        self.worker = self.fetch_and_save_data(
            api_url,
            formID,
            username,
            password,
            geo_field,
            page_size,
            directory,
        )
        self.thread_pool.start(self.worker)
        self.dlg.app_logs.appendPlainText(f"Done")

    def download_csv(self, data):
        if data:
            # Open file dialog to select save location
            output_file, _ = QFileDialog.getSaveFileName(
                None, "Save CSV File", "", "CSV Files (*.csv)"
            )
            if not output_file:
                QMessageBox.warning(None, "Cancelled", "No file selected.")
                return

            if not output_file.endswith(".csv"):
                output_file += ".csv"

            # Write to CSV
            with open(output_file, "w", newline="", encoding="utf-8") as csvfile:
                headers = data[0].keys()
                writer = csv.writer(csvfile)
                writer.writerow(headers)

                for row in data:
                    writer.writerow(row.values())

            QgsMessageLog.logMessage(
                f"CSV successfully saved to {output_file}", "CSV Download"
            )
            QMessageBox.information(None, "Success", f"CSV File saved to {output_file}")

    def download_file(self, url, save_path):
        try:
            response = requests.get(url)
            response.raise_for_status()
            with open(save_path, "wb") as file:
                file.write(response.content)
            return True
        except requests.RequestException as e:
            QMessageBox.critical(
                None, "Download Error", f"Failed to download {url}: {e}"
            )
            return False

    def add_afro_shapefile_layer(self):
        self.dlg.afroBoundOkBtn.setEnabled(False)
        curr_admin_name = self.dlg.comboAfroAdmins.currentText()
        curr_admin_base_url = self.dlg.comboAfroAdmins.currentData()

        local_dir = os.path.join(os.path.expanduser("~"), "qgis_temp")
        os.makedirs(local_dir, exist_ok=True)

        extensions = [".shp", ".shx", ".dbf", ".prj"]  # Required shapefile components
        local_files = {}

        for ext in extensions:
            url = curr_admin_base_url + ext
            local_path = os.path.join(local_dir, f"AFRO_{curr_admin_name}" + ext)
            self.dlg.afroProgressBar.setRange(0, 0)
            self.dlg.afroProgressBar.setTextVisible(False)
            self.dlg.afroProgressBar.repaint()
            if self.download_file(url, local_path):
                self.dlg.afroProgressBar.setRange(0, 100)
                self.dlg.afroBoundOkBtn.setEnabled(True)
                self.dlg.afroProgressBar.setTextVisible(True)
                local_files[ext] = local_path

        if all(ext in local_files for ext in [".shp", ".shx", ".dbf"]):
            layer = QgsVectorLayer(
                local_files[".shp"], f"AFRO_{curr_admin_name}", "ogr"
            )
            if layer.isValid():
                if ".prj" in local_files:
                    crs = QgsCoordinateReferenceSystem()
                    with open(local_files[".prj"], "r") as prj_file:
                        crs.createFromWkt(prj_file.read())

                    layer.setCrs(crs)
                QgsProject.instance().addMapLayer(layer)
                QMessageBox.information(
                    None, "Success", "Shapefile added successfully!"
                )
            else:
                QMessageBox.critical(None, "Error", "Failed to load the shapefile.")
        else:
            QMessageBox.critical(
                None, "Error", "One or more required shapefile components are missing."
            )

    def providers_map(self):
        return {
            "Onadata": "api.whonghub.org",
            "GetODK": "aap-odk-sinp.cen-nouvelle-aquitaine.dev",
            "Kobo": "kf.kobotoolbox.org",
            "GTS": "gts.health",
        }

    def stop_fetching(self):
        """Stop the data-fetching timer."""
        self.timer.stop()
        self.tr("Stopped fetching data.")
        self.dlg.app_logs.appendPlainText(f"Stopped fetching data.")

    def stop_data_fetching(self):
        if hasattr(self, "worker") and self.worker and self.worker.isRunning():
            self.worker.stop()
            self.is_interrupted = True
            self.worker.quit()
            self.worker.wait(500)  # Wait for the thread to finish
            self.iface.messageBar().pushMessage(
                "Notice", "Data fetching cancelled.", level=Qgis.Warning
            )

    def update_slider_value_label(self, value):
        self.dlg.slider_label.setText(
            str(value)
        )  # Update the label with the current slider value

    def on_from_date_changed(self, value):
        date_str = value.toString("yyyy-MM-dd")
        self.from_date = date_str
        self.dlg.onaDownloadCSV.setEnabled(False)
        self.dlg.onaDownloadCSV.repaint()

    def on_to_date_changed(self, value):
        date_str = value.toString("yyyy-MM-dd")
        self.to_date = date_str
        self.dlg.onaDownloadCSV.setEnabled(False)
        self.dlg.onaDownloadCSV.repaint()

    def polis_from_date_changed(self, value):
        date_str = value.toString("yyyy-MM-dd")

    def cancel_button_clicked(self):
        self.reset_inputs()
        # self.dlg.close()

    def odk_cancel_button_clicked(self):
        self.reset_odk_inputs()

    def kobo_cancel_button_clicked(self):
        self.reset_kobo_inputs()

    def clear_logs(self):
        self.dlg.app_logs.clear()

    def on_checked_items_changed(self):
        """Handles the 'Select All' option behavior and retrieves data."""
        selected_items = self.dlg.comboPolisArea.checkedItems()
        all_items = [
            self.dlg.comboPolisArea.itemText(i)
            for i in range(self.dlg.comboPolisArea.count())
        ]
        data_map = {
            self.dlg.comboPolisArea.itemText(i): self.dlg.comboPolisArea.itemData(
                i, Qt.UserRole
            )
            for i in range(self.dlg.comboPolisArea.count())
        }

        self.dlg.app_logs.appendPlainText(f"Print selected items - {selected_items}")
        self.dlg.app_logs.appendPlainText(f"Print all items - {all_items}")

        if "Select All" in selected_items:
            # Select all items
            self.dlg.comboPolisArea.setCheckedItems(all_items)
        elif not any(
            item in selected_items for item in all_items if item != "Select All"
        ):
            # If nothing else is selected, uncheck everything
            for i in range(self.dlg.comboPolisArea.count()):
                self.dlg.comboPolisArea.setItemCheckState(i, False)  # Uncheck item
        else:
            self.dlg.comboPolisArea.setCheckedItems(selected_items)

        # Retrieve selected items with data
        selected_data = [data_map[item] for item in selected_items if item in data_map]
        # print("Selected Data:", selected_data)

    def fetch_polis_admin_one(self):
        self.dlg.comboPolisCountry.clear()
        api_url = self.dlg.polis_api_url.text()
        api_token = self.dlg.polisAuthToken.text()
        current_area_id = self.dlg.comboPolisArea.currentData()
        # self.dlg.app_logs.appendPlainText(f"The selected data - {current_data}")

        if current_area_id and api_token:
            self.dlg.btnFetchPolis.setEnabled(False)
            self.dlg.btnFetchPolis.repaint()
            self.dlg.comboPolisCountry.setEnabled(False)

            headers = {"Authorization-Token": api_token}
            params = {
                "$filter": f"FK_GeolevelId eq 2 and FK_ParentId eq {current_area_id}"
            }

            admin_url = f"https://{api_url}/api/v2/Geography"

            try:
                response = self.fetch_with_retries(
                    admin_url, params=params, headers=headers
                )

                if response.status_code == 200:
                    res = response.json()
                    data = res.get("value")

                    if data:
                        self.dlg.comboPolisCountry.addItem("Show All", "Show All")
                        for index, datum in enumerate(data):
                            label = datum.get("DisplayName")
                            admin_id = datum.get("Id")

                            self.dlg.comboPolisCountry.addItem(label, admin_id)
                        self.dlg.comboPolisCountry.setEnabled(True)
                        self.dlg.btnFetchPolis.setEnabled(True)
                    else:
                        self.iface.messageBar().pushMessage(
                            "Notice",
                            "No Data Found",
                            level=Qgis.Warning,
                            duration=10,
                        )
                        self.dlg.btnFetchPolis.setEnabled(True)
                else:
                    self.iface.messageBar().pushMessage(
                        "Error",
                        f"Error fetching data: {response.status_code}",
                        level=Qgis.Critical,
                        duration=10,
                    )
                    self.dlg.btnFetchPolis.setEnabled(True)
            except Exception as e:
                self.iface.messageBar().pushMessage(
                    "Error",
                    str(e),
                    level=Qgis.Critical,
                    duration=10,
                )
                self.dlg.btnFetchPolis.setEnabled(True)

    def fetch_polis_admin_two(self):
        self.dlg.comboPolisProvince.clear()
        api_url = self.dlg.polis_api_url.text()
        api_token = self.dlg.polisAuthToken.text()
        current_area_id = self.dlg.comboPolisCountry.currentData()

        if current_area_id and current_area_id != "Show All" and api_token:
            self.dlg.btnFetchPolis.setEnabled(False)
            self.dlg.btnFetchPolis.repaint()
            self.dlg.comboPolisProvince.setEnabled(False)

            headers = {"Authorization-Token": api_token}
            params = {
                "$filter": f"FK_GeolevelId eq 3 and FK_ParentId eq {current_area_id}"
            }

            admin_url = f"https://{api_url}/api/v2/Geography"

            try:
                response = self.fetch_with_retries(
                    admin_url, params=params, headers=headers
                )

                if response.status_code == 200:
                    res = response.json()
                    data = res.get("value")

                    if data:
                        self.dlg.comboPolisProvince.addItem("Show All", "Show All")
                        for datum in data:
                            label = datum.get("DisplayName")
                            admin_id = datum.get("Id")

                            self.dlg.comboPolisProvince.addItem(label, admin_id)
                        self.dlg.btnFetchPolis.setEnabled(True)
                        self.dlg.comboPolisProvince.setEnabled(True)
                    else:
                        self.iface.messageBar().pushMessage(
                            "Notice",
                            "No Data Found",
                            level=Qgis.Warning,
                            duration=10,
                        )
                        self.dlg.btnFetchPolis.setEnabled(True)
                else:
                    self.iface.messageBar().pushMessage(
                        "Error",
                        f"Error fetching data: {response.status_code}",
                        level=Qgis.Critical,
                        duration=10,
                    )
                    self.dlg.btnFetchPolis.setEnabled(True)
            except Exception as e:
                self.iface.messageBar().pushMessage(
                    "Error",
                    str(e),
                    level=Qgis.Critical,
                    duration=10,
                )
                self.dlg.btnFetchPolis.setEnabled(True)

    def fetch_polis_admin_three(self):
        self.dlg.comboPolisDistrict.clear()
        api_url = self.dlg.polis_api_url.text()
        api_token = self.dlg.polisAuthToken.text()
        current_area_id = self.dlg.comboPolisProvince.currentData()

        if current_area_id and current_area_id != "Show All" and api_token:
            self.dlg.btnFetchPolis.setEnabled(False)
            self.dlg.btnFetchPolis.repaint()
            self.dlg.comboPolisDistrict.setEnabled(False)

            headers = {"Authorization-Token": api_token}
            params = {
                "$filter": f"FK_GeolevelId eq 4 and FK_ParentId eq {current_area_id}"
            }

            admin_url = f"https://{api_url}/api/v2/Geography"

            try:
                response = self.fetch_with_retries(
                    admin_url, params=params, headers=headers
                )

                if response.status_code == 200:
                    res = response.json()
                    data = res.get("value")

                    if data:
                        self.dlg.comboPolisDistrict.addItem("Show All", "Show All")
                        for datum in data:
                            label = datum.get("DisplayName")
                            admin_id = datum.get("Id")

                            self.dlg.comboPolisDistrict.addItem(label, admin_id)
                        self.dlg.btnFetchPolis.setEnabled(True)
                        self.dlg.comboPolisDistrict.setEnabled(True)
                    else:
                        self.iface.messageBar().pushMessage(
                            "Notice",
                            "No Data Found",
                            level=Qgis.Warning,
                            duration=10,
                        )
                        self.dlg.btnFetchPolis.setEnabled(True)
                else:
                    self.iface.messageBar().pushMessage(
                        "Error",
                        f"Error fetching data: {response.status_code}",
                        level=Qgis.Critical,
                        duration=10,
                    )
                    self.dlg.btnFetchPolis.setEnabled(True)
            except Exception as e:
                self.iface.messageBar().pushMessage(
                    "Error",
                    str(e),
                    level=Qgis.Critical,
                    duration=10,
                )
                self.dlg.btnFetchPolis.setEnabled(True)

    def get_currently_selected_data(self):
        """Retrieves data for currently selected (checked) items."""
        selected_texts = (
            self.dlg.comboPolisArea.checkedItems()
        )  # Get selected item labels
        selected_data = []

        for i in range(self.dlg.comboPolisArea.count()):
            item_text = self.dlg.comboPolisArea.itemText(i)
            if item_text in selected_texts:
                data = self.dlg.comboPolisArea.itemData(
                    i, Qt.UserRole
                )  # Retrieve associated data
                selected_data.append(data)

        return selected_data

    def fetch_polis_virus_handler(self):
        self.dlg.comboPolisArea.clear()
        self.dlg.comboPolisCountry.clear()
        self.dlg.comboPolisProvince.clear()
        self.dlg.comboPolisDistrict.clear()

        self.dlg.comboPolisIndicators.clear()
        self.dlg.comboPolisVizCategory.clear()
        self.dlg.comboPolisVirusType.clear()

        self.polis_reset_saved_data()

        self.polis_areas_dict = {
            "AFRO": 4,
            "AMRO": 1,
            "EURO": 2,
            "SEARO": 8,
            "EMRO": 9,
            "WPRO": 10,
        }

        index = 1

        for key, value in self.polis_areas_dict.items():
            self.dlg.comboPolisArea.addItem(key, value)
        self.dlg.comboPolisArea.setEnabled(True)

        self.dlg.comboPolisVizCategory.addItems(["Indicators", "Virus", "Cases"])

        for key, value in self.polis_indicators_dict.items():
            self.dlg.comboPolisIndicators.addItem(value, key)

        self.polis_virus_types_dict = {
            "WILD 1": 2,
            "WILD 2": 5,
            "WILD 3": 8,
            "VDPV 1": 1,
            "VDPV 2": 4,
            "VDPV 3": 7,
            "VACCINE 1": 3,
            "VACCINE 2": 6,
            "VACCINE 3": 9,
            "NPEV": 13,
        }

        self.dlg.comboPolisVirusType.addItem("Show All", "Show All")

        for key, val in self.polis_virus_types_dict.items():
            self.dlg.comboPolisVirusType.addItem(key, val)

    def polis_viz_category_handler(self):
        current_viz = self.dlg.comboPolisVizCategory.currentText()
        if current_viz == "Indicators":
            self.dlg.comboPolisIndicators.setEnabled(True)
            self.dlg.comboPolisVirusType.setEnabled(False)
        else:
            self.dlg.comboPolisIndicators.setEnabled(False)
            self.dlg.comboPolisVirusType.setEnabled(True)

    def fetch_polis_data_clicked(self):
        self.polis_reset_saved_data()
        api_url = self.dlg.polis_api_url.text()
        api_token = self.dlg.polisAuthToken.text()
        viz_category = self.dlg.comboPolisVizCategory.currentText()
        selected_area = self.dlg.comboPolisArea.currentText()
        selected_virus_type = "".join(
            self.dlg.comboPolisVirusType.currentText().split(" ")
        )
        selected_virus_type_id = self.dlg.comboPolisVirusType.currentData()
        selected_indicator_val = self.dlg.comboPolisIndicators.currentData()
        layer_label_str = ""
        layer_sub_label_str = ""

        # Admins
        selected_country_id = self.dlg.comboPolisCountry.currentData()
        selected_province_id = self.dlg.comboPolisProvince.currentData()
        selected_district_id = self.dlg.comboPolisDistrict.currentData()

        selected_country_text = ""
        selected_province_text = ""
        selected_district_text = ""

        polis_from_date = self.dlg.polisDateFrom.date()
        polis_to_date = self.dlg.polisDateTo.date()

        from_dt = datetime(
            polis_from_date.year(),
            polis_from_date.month(),
            polis_from_date.day(),
            0,
            0,
            0,
        )  # 12:00 AM
        to_dt = datetime(
            polis_to_date.year(), polis_to_date.month(), polis_to_date.day(), 23, 59, 59
        )

        polis_from_timestamp = from_dt.strftime("%Y-%m-%dT%H:%M:%S")
        polis_to_timestamp = to_dt.strftime("%Y-%m-%dT%H:%M:%S")

        polis_from_timestamp_str = from_dt.strftime("%Y-%m-%d")
        polis_to_timestamp_str = to_dt.strftime("%Y-%m-%d")

        curr_url = None
        filter_query = None

        if viz_category == "Indicators":
            ind_val = f"('{selected_indicator_val}')"
            curr_url = f"https://{api_url}/api/v2/IndicatorValue{ind_val}"
            filter_query = f"LastCalculationDateTime ge DateTime'{polis_from_timestamp}' and LastCalculationDateTime le DateTime'{polis_to_timestamp}'"
            layer_sub_label_str += ind_val

            if selected_country_id and selected_country_id != "Show All":
                selected_country_text = self.dlg.comboPolisCountry.currentText()
                layer_label_str += f"{selected_country_text}"
                filter_query += f"and Admin0Id eq {int(selected_country_id)}"
                if selected_province_id and selected_province_id != "Show All":
                    selected_province_text = self.dlg.comboPolisProvince.currentText()
                    layer_label_str = f"_{selected_province_text}"
                    filter_query += f"and Admin1Id eq {int(selected_province_id)}"
                    if selected_district_id and selected_district_id != "Show All":
                        selected_district_text = (
                            self.dlg.comboPolisDistrict.currentText()
                        )
                        layer_label_str = f"_{selected_district_text}"
                        filter_query += f"and Admin2Id eq {int(selected_district_id)}"
            else:
                layer_label_str += selected_area
                filter_query += f"and WHORegion eq '{selected_area}'"

        elif viz_category == "Cases":
            virus_type_query = ""
            curr_url = f"https://{api_url}/api/v2/Case"

            parsed_virus_type = selected_virus_type
            if "VDPV" in selected_virus_type:
                parsed_virus_type = f"c{selected_virus_type}"

            layer_sub_label_str += selected_virus_type

            virus_type_query_str = None

            if selected_virus_type != "ShowAll":
                virus_type_query = f"PolioVirusTypes eq '{parsed_virus_type}'"
                virus_type_query_str = virus_type_query
            else:
                virus_keys_list = list()

                virus_keys = [
                    "".join(k.split(" ")) for k in self.polis_virus_types_dict.keys()
                ]

                for x in virus_keys:
                    if "VDPV" in x:
                        virus_keys_list.append(f"c{x}")
                    else:
                        virus_keys_list.append(x)

                virus_type_query = " or ".join(
                    [f"PolioVirusTypes eq '{v}'" for v in virus_keys_list]
                )
                virus_type_query_str = f"({virus_type_query})"

            filter_query = f"CaseDate ge DateTime'{polis_from_timestamp}' and CaseDate le DateTime'{polis_to_timestamp}' and {virus_type_query_str}"
            self.dlg.app_logs.appendPlainText(f"print filter query - {filter_query}")

            if selected_country_id and selected_country_id != "Show All":
                selected_country_text = self.dlg.comboPolisCountry.currentText()
                layer_label_str += f"{selected_country_text}"
                filter_query += f" and Admin0Id eq {int(selected_country_id)}"
                if selected_province_id and selected_province_id != "Show All":
                    selected_province_text = self.dlg.comboPolisProvince.currentText()
                    layer_label_str = f"_{selected_province_text}"
                    filter_query += f" and Admin1Id eq {int(selected_province_id)}"
                    if selected_district_id and selected_district_id != "Show All":
                        selected_district_text = (
                            self.dlg.comboPolisDistrict.currentText()
                        )
                        layer_label_str = f"_{selected_district_text}"
                        filter_query += f" and Admin2Id eq {int(selected_district_id)}"
            else:
                layer_label_str += selected_area
                filter_query += f" and WHORegion eq '{selected_area}'"

        else:
            curr_url = f"https://{api_url}/api/v2/Virus"
            virus_type_query = ""
            virus_type_query_str = None

            if selected_virus_type != "ShowAll":
                virus_type_query = f"VirusTypeCode eq '{selected_virus_type}'"
                virus_type_query_str = virus_type_query
            else:
                virus_keys_list = list()

                virus_keys = [
                    "".join(k.split(" ")) for k in self.polis_virus_types_dict.keys()
                ]

                virus_type_query = " or ".join(
                    [f"VirusTypeCode eq '{v}'" for v in virus_keys]
                )
                virus_type_query_str = f"({virus_type_query})"

            filter_query = f"CreatedDate ge DateTime'{polis_from_timestamp}' and CreatedDate le DateTime'{polis_to_timestamp}' and {virus_type_query_str}"
            layer_sub_label_str += selected_virus_type

            if selected_country_id and selected_country_id != "Show All":
                selected_country_text = self.dlg.comboPolisCountry.currentText()
                layer_label_str += f"{selected_country_text}"
                filter_query += f" and Admin0Id eq {int(selected_country_id)}"
                if selected_province_id and selected_province_id != "Show All":
                    selected_province_text = self.dlg.comboPolisProvince.currentText()
                    layer_label_str = f"_{selected_province_text}"
                    filter_query += f" and Admin1Id eq {int(selected_province_id)}"
                    if selected_district_id and selected_district_id != "Show All":
                        selected_district_text = (
                            self.dlg.comboPolisDistrict.currentText()
                        )
                        layer_label_str = f"_{selected_district_text}"
                        filter_query += f" and Admin2Id eq {int(selected_district_id)}"
            else:
                layer_label_str += f"{selected_area}"
                filter_query += f" and WHORegion eq '{selected_area}'"

        if api_token and selected_country_id:
            self.dlg.btnFetchPolis.setEnabled(False)
            self.dlg.btnFetchPolis.repaint()

            headers = {"Authorization-Token": api_token}

            params = {
                "$top": 10000 if viz_category in ["Cases", "Virus"] else 1000,
                "$filter": filter_query,
            }

            # Fetch GeoJSON for active level
            feature_collection = {
                "type": "FeatureCollection",
                "features": [],
            }

            self.dlg.polisProgressBar.setValue(20)

            try:
                response = self.fetch_with_retries(
                    curr_url, headers=headers, params=params
                )

                combined_data = []

                if response.status_code == 200:
                    data = response.json()
                    self.dlg.polisProgressBar.setValue(100)
                    result = data.get("value")
                    if result:
                        for datum in result:
                            combined_data.append(datum)
                            latitude = datum.get("Latitude")
                            longitude = datum.get("Longitude")

                            if latitude and longitude:
                                geometry = {
                                    "type": "Point",
                                    "coordinates": [float(longitude), float(latitude)],
                                }

                                feature = {
                                    "type": "Feature",
                                    "geometry": geometry,
                                    "properties": datum,
                                }

                                feature_collection["features"].append(feature)
                            else:
                                continue
                    else:
                        self.iface.messageBar().pushMessage(
                            "Notice",
                            "No Data Found",
                            level=Qgis.Warning,
                            duration=10,
                        )
                        self.dlg.polisProgressBar.setValue(0)
                else:
                    self.iface.messageBar().pushMessage(
                        "Error",
                        f"Error fetching data: {response.status_code}",
                        level=Qgis.Critical,
                        duration=10,
                    )
                    self.dlg.btnFetchPolis.setEnabled(True)
                    self.dlg.polisProgressBar.setValue(0)

                if (
                    feature_collection["features"]
                    and len(feature_collection["features"]) > 0
                ):
                    self.dlg.btnFetchPolis.setEnabled(True)
                    self.load_data_to_qgis(
                        feature_collection,
                        layer_label_str,
                        f"{viz_category}_{layer_sub_label_str}_{polis_from_timestamp_str}_to_{polis_to_timestamp_str}",
                    )
                    self.dlg.polisProgressBar.setValue(0)
                    self.polis_json_data = combined_data
                    self.dlg.polisDownloadCSV.setEnabled(True)

                elif len(feature_collection["features"]) == 0 and combined_data:
                    self.polis_json_data = combined_data
                    self.dlg.polisDownloadCSV.setEnabled(True)
                    self.iface.messageBar().pushMessage(
                        "Notice",
                        f"Selected Indicator Has No Geographic Data, Kindly select option to Export data as CSV",
                        level=Qgis.Warning,
                        duration=10,
                    )
                    self.dlg.btnFetchPolis.setEnabled(True)
                    self.dlg.polisProgressBar.setValue(0)
                else:
                    self.dlg.app_logs.appendPlainText(
                        "No Available Geometry or Data to Display"
                    )
                    self.iface.messageBar().pushMessage(
                        "Notice",
                        f"No Available Geometry or Data to Display",
                        level=Qgis.Warning,
                        duration=10,
                    )
                    self.dlg.btnFetchPolis.setEnabled(True)
                    self.dlg.polisProgressBar.setValue(0)
            except Exception as e:
                self.iface.messageBar().pushMessage(
                    "Error",
                    str(e),
                    level=Qgis.Warning,
                    duration=10,
                )

    def fetch_dhis_indicator_data_handler(self):
        """
        Fetch DHIS Indicator Data
        """
        # disable buttun during fetch
        self.dlg.dhisOkButton.setEnabled(False)
        self.dlg.dhisOkButton.repaint()

        api_url = self.dlg.dhis_api_url.text()
        username = self.dlg.dhis_username.text()
        password = self.dlg.dhisMLineEdit.text()
        auth = HTTPBasicAuth(username, password)

        selected_period = self.dlg.comboDhisPeriod.currentText()
        curr_indicator_id = self.dlg.comboDhisIndicators.currentData()
        curr_indicator_text = self.dlg.comboDhisIndicators.currentText()
        adm_level = self.dlg.ComboDhisAdminLevels.currentText()
        cleaned_adm_lvl = adm_level.split(" ")[-1]

        geo_data = None

        # Fetch GeoJSON for active level
        feature_collection = {
            "type": "FeatureCollection",
            "features": [],
        }

        geo_url = f"https://{api_url}/api/geoFeatures"
        geo_params = [
            ("ou", f"ou:LEVEL-{cleaned_adm_lvl}"),
            ("displayProperty", "NAME"),
        ]

        geo_response = self.fetch_with_retries(geo_url, auth, geo_params)

        if geo_response.status_code == 200:
            geo_data = geo_response.json()
        else:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Error fetching Geometry: {geo_response.status_code}",
                level=Qgis.Critical,
                duration=10,
            )
            self.dlg.dhisOkButton.setEnabled(True)
            self.dlg.dhisProgressBar.setValue(0)

        if geo_data and curr_indicator_id:
            params = [
                ("dimension", f"dx:{curr_indicator_id}"),
                ("dimension", f"ou:LEVEL-{cleaned_adm_lvl}"),
                ("dimension", f"pe:{selected_period}"),
            ]

            url = f"https://{api_url}/api/analytics.json"

            self.dlg.dhisProgressBar.setValue(20)

            response = self.fetch_with_retries(url, auth, params)

            if response.status_code == 200:
                data = response.json()
                if data:
                    self.dlg.dhisProgressBar.setValue(100)
                    rows = data.get("rows")
                    metadata = data.get("metaData")
                    meta_items = metadata.get("items")
                    if rows:
                        cleaned_data = dict()
                        for row in rows:
                            if not cleaned_data.get(row[1]):
                                cleaned_data[row[1]] = {
                                    "Org ID": row[1],
                                    "Org Unit": meta_items.get(row[1]).get("name"),
                                    "Indicator": meta_items.get(row[0]).get("name"),
                                    "Period": meta_items.get(row[2]).get("name"),
                                    "Value": float(row[3]),
                                }
                            else:
                                cleaned_data[row[1]]["Value"] += float(row[3])
                                cleaned_data[row[1]]["Period"] = (
                                    cleaned_data[row[1]]["Period"]
                                    + ","
                                    + meta_items.get(row[2]).get("name")
                                )

                        # get single geometry
                        if cleaned_data:
                            for datum in cleaned_data.values():
                                single_geom_obj = next(
                                    (
                                        geom
                                        for geom in geo_data
                                        if geom.get("id") == datum.get("Org ID")
                                    ),
                                    None,
                                )
                                if single_geom_obj:
                                    coordinates = json.loads(single_geom_obj.get("co"))
                                    geom_type = "Polygon"
                                    if len(coordinates) > 1:
                                        geom_type = "Point"

                                    geometry = {
                                        "type": geom_type,
                                        "coordinates": coordinates,
                                    }

                                    feature = {
                                        "type": "Feature",
                                        "geometry": geometry,
                                        "properties": datum,
                                    }

                                    feature_collection["features"].append(feature)

                            if (
                                feature_collection["features"]
                                and len(feature_collection["features"]) > 0
                            ):
                                self.dhis_json_data = [
                                    feature.get("properties")
                                    for feature in feature_collection["features"]
                                ]
                                self.dlg.dhisDownloadCSV.setEnabled(True)

                                cleaned_indicator_text = "_".join(
                                    curr_indicator_text.split(" ")
                                )
                                self.load_data_to_qgis(
                                    feature_collection,
                                    cleaned_indicator_text,
                                    f"LEVEL_{cleaned_adm_lvl}_{selected_period}",
                                )
                            else:
                                self.dlg.app_logs.appendPlainText(
                                    "No Available Geometry to Display"
                                )
                                self.iface.messageBar().pushMessage(
                                    "Notice",
                                    f"No Available Geometry to Display",
                                    level=Qgis.Warning,
                                    duration=10,
                                )
                                self.dlg.dhisOkButton.setEnabled(True)

                            self.dlg.dhisOkButton.setEnabled(True)
                            self.dlg.dhisProgressBar.setValue(0)
                    else:
                        self.iface.messageBar().pushMessage(
                            "Notice",
                            "No Data Found for Selected Indicator",
                            level=Qgis.Warning,
                            duration=10,
                        )
                        self.dlg.dhisOkButton.setEnabled(True)
                        self.dlg.dhisProgressBar.setValue(0)
            else:
                self.iface.messageBar().pushMessage(
                    "Error",
                    f"Error fetching data: {response.status_code}",
                    level=Qgis.Critical,
                    duration=10,
                )
                self.dlg.dhisOkButton.setEnabled(True)
                self.dlg.dhisProgressBar.setValue(0)
        else:
            self.iface.messageBar().pushMessage(
                "Notice",
                f"Indicator Not Set",
                level=Qgis.Warning,
                duration=10,
            )
            self.dlg.dhisOkButton.setEnabled(True)
            self.dlg.dhisProgressBar.setValue(0)

    def fetch_dhis_org_units_handler(self):
        api_url = self.dlg.dhis_api_url.text()
        username = self.dlg.dhis_username.text()
        password = self.dlg.dhisMLineEdit.text()
        self.fetch_dhis_org_units(api_url, username, password)

    def on_dhis_indicators_change(self):
        self.dhis_reset_saved_data()
        indicator_data = self.dlg.comboDhisIndicators.currentData()
        if indicator_data:
            self.dlg.dhisOkButton.setEnabled(True)

    def on_dhis_datasets_change(self):
        api_url = self.dlg.dhis_api_url.text()
        username = self.dlg.dhis_username.text()
        password = self.dlg.dhisMLineEdit.text()

        auth = HTTPBasicAuth(username, password)
        dataset_data = self.dlg.comboDhisDataSets.currentData()
        self.dlg.comboDhisIndicators.clear()
        dataset_id = None

        if dataset_data:
            dataset_id = dataset_data.get("dataset_id")

            if dataset_id:
                params = {"fields": "name,id,indicators[id,name]"}

                url = f"https://{api_url}/api/dataSets/{dataset_id}"

                response = self.fetch_with_retries(url, auth, params)
                if response.status_code == 200:
                    data = response.json()
                    if data:
                        indicators = data.get("indicators")
                        if indicators:
                            for indicator in indicators:
                                indicator_name = indicator.get("name")
                                indicator_id = indicator.get("id")
                                self.dlg.comboDhisIndicators.addItem(
                                    indicator_name, {"indicator_id": indicator_id}
                                )
                        else:
                            self.dlg.app_logs.appendPlainText(
                                f"No Available Indicators for selected Dataset {dataset_id}"
                            )
                            self.iface.messageBar().pushMessage(
                                "Notice",
                                "No Available Indicators for Selected Dataset",
                                level=Qgis.Warning,
                                duration=10,
                            )
                else:
                    self.iface.messageBar().pushMessage(
                        "Error",
                        f"Error fetching data: Status Code {response.status_code}",
                        level=Qgis.Critical,
                        duration=10,
                    )

    def on_dhis_org_units_change(self):
        org_units_data = self.dlg.comboDhisOrgUnits.currentData()
        if org_units_data:
            curr_org_datasets = org_units_data.get("dataSets")
            if curr_org_datasets:
                for dataset in curr_org_datasets:
                    self.dlg.comboDhisDataSets.addItem(
                        dataset.get("name"), {"dataset_id": dataset.get("id")}
                    )

    def fetch_dhis_org_units(self, api_url, username, password):
        auth = HTTPBasicAuth(username, password)
        adm_level = self.dlg.ComboDhisAdminLevels.currentText()
        cleaned_adm_lvl = adm_level.split(" ")[-1]

        self.dlg.comboDhisOrgUnits.clear()
        self.dlg.comboDhisDataSets.clear()
        self.dlg.comboDhisIndicators.clear()

        self.dlg.btnFetchDhisCategory.setEnabled(False)
        self.dlg.dhisOkButton.setEnabled(False)
        self.dlg.dhisOkButton.repaint()

        self.dlg.btnFetchDhisCategory.setText("Connecting...")
        self.dlg.btnFetchDhisCategory.repaint()

        page = 1

        feature_collection = {
            "type": "FeatureCollection",
            "features": [],
        }

        url = f"https://{api_url}/api/organisationUnits"
        hasData = True

        while hasData:
            params = [
                (
                    "fields",
                    "id,name,lastUpdated,dimensionItemType,shortName,displayName,children[id,name],dataSets[id,name],geometry",
                ),
                ("filter", f"level:eq:{cleaned_adm_lvl}"),
                ("filter", "children:gte:0"),
                ("page", page),
                ("pageSize", 1000),
            ]

            response = self.fetch_with_retries(url, auth, params)

            if response.status_code == 200:
                data = response.json()
                result = data.get("organisationUnits")

                # handle progress
                pager = data.get("pager")
                total_pages = (
                    pager.get("total") + pager.get("pageSize") - 1
                ) // pager.get("pageSize")
                progress = (
                    (int(page) / int(total_pages)) * 100
                    if int(total_pages) > 1
                    else 100
                )
                self.dlg.dhisProgressBar.setValue(math.ceil(progress) * 2)
                self.dlg.dhisProgressBar.repaint()

                if result:
                    for datum in result:
                        org_id = datum.get("id")
                        org_name = datum.get("name")
                        org_datasets = datum.get("dataSets")
                        self.dlg.comboDhisOrgUnits.addItem(
                            org_name, {"id": org_id, "dataSets": org_datasets}
                        )
                        if datum.get("geometry"):
                            geometry = datum.get("geometry")
                            feature_collection["features"].append(
                                {
                                    "type": "Feature",
                                    "geometry": geometry,
                                    "properties": {
                                        "name": datum.get("name"),
                                        "lastUpdated": datum.get("lastUpdated"),
                                        "dimensionItemType": datum.get(
                                            "dimensionItemType"
                                        ),
                                        "shortName": datum.get("shortName"),
                                        "displayName": datum.get("displayName"),
                                    },
                                }
                            )
                elif not pager.get("nextPage"):
                    hasData = False
                    self.dlg.dhisProgressBar.setValue(0)
                    self.dlg.btnFetchDhisCategory.setEnabled(True)
                    self.dlg.btnFetchDhisCategory.setText("Connect")
                    self.dlg.btnFetchDhisCategory.repaint()
                else:
                    self.dlg.dhisProgressBar.setValue(0)
                    self.iface.messageBar().pushMessage(
                        "Notice", "No Data Found", level=Qgis.Warning
                    )
                    self.dlg.btnFetchDhisCategory.setEnabled(True)
                    self.dlg.btnFetchDhisCategory.setText("Connect")
                    self.dlg.btnFetchDhisCategory.repaint()
                    break
            else:
                self.dlg.dhisProgressBar.setValue(0)
                self.iface.messageBar().pushMessage(
                    "Error",
                    f"Error fetching data: {response.status_code}",
                    level=Qgis.Critical,
                )
                self.dlg.btnFetchDhisCategory.setEnabled(True)
                self.dlg.btnFetchDhisCategory.setText("Connect")
                self.dlg.btnFetchDhisCategory.repaint()
                break

            page += 1

        if feature_collection["features"] and len(feature_collection["features"]) > 0:
            self.load_data_to_qgis(
                feature_collection, "dhis", f"level_{cleaned_adm_lvl}"
            )
        else:
            self.dlg.app_logs.appendPlainText("No Available Geometry to Display")
            self.iface.messageBar().pushMessage(
                "Notice",
                f"No Available Geometry to Display",
                level=Qgis.Warning,
                duration=10,
            )
            self.dlg.dhisOkButton.setEnabled(True)

    def fetch_dhis_selected_category_handler(self):
        api_url = self.dlg.dhis_api_url.text()
        username = self.dlg.dhis_username.text()
        password = self.dlg.dhisMLineEdit.text()
        self.dhis_reset_saved_data()
        self.fetch_dhis_selected_category(api_url, username, password)

    def fetch_dhis_selected_category(self, api_url, username, password):
        auth = HTTPBasicAuth(username, password)
        self.dlg.comboDhisProgramsOrDataSets.clear()
        self.dlg.comboDhisIndicators.clear()

        category_text = self.dlg.ComboDhisCategory.currentText()
        indicator_text = None

        # extract datasets or programs text
        if category_text.lower().strip() == "datasets":
            category_text = "dataSets"
            indicator_text = "indicators"
        elif category_text.lower().strip() == "programs":
            category_text = "programs"
            indicator_text = "programIndicators"

        self.dlg.btnFetchDhisCategory.setEnabled(False)
        self.dlg.dhisOkButton.setEnabled(False)
        self.dlg.dhisOkButton.repaint()

        page = 1

        url = f"https://{api_url}/api/{category_text}"
        hasData = True

        while hasData:
            params = [
                ("fields", f"id,name,{indicator_text}[id,name]"),
                ("page", page),
                ("pageSize", 1000),
            ]

            response = self.fetch_with_retries(url, auth, params)

            if response.status_code == 200:
                data = response.json()
                result = data.get(category_text)

                pager = data.get("pager")
                total_pages = (
                    pager.get("total") + pager.get("pageSize") - 1
                ) // pager.get("pageSize")
                progress = (
                    (int(page) / int(total_pages)) * 100
                    if int(total_pages) > 1
                    else 100
                )
                self.dlg.dhisProgressBar.setValue(math.ceil(progress))
                self.dlg.dhisProgressBar.repaint()

                if result:
                    for datum in result:
                        category_name = datum.get("name")
                        category_id = datum.get("id")
                        curr_indicators = datum.get(indicator_text)

                        self.dlg.comboDhisProgramsOrDataSets.addItem(
                            category_name,
                            {
                                "category_id": category_id,
                                "curr_indicators": curr_indicators,
                            },
                        )

                elif not pager.get("nextPage"):
                    hasData = False
                    self.dlg.dhisProgressBar.setValue(0)
                    self.dlg.btnFetchDhisCategory.setEnabled(True)
                    self.dlg.btnFetchDhisCategory.setText("Connect")
                    self.dlg.btnFetchDhisCategory.repaint()

                else:
                    hasData = False
                    self.dlg.dhisProgressBar.setValue(0)
                    self.iface.messageBar().pushMessage(
                        "Notice", "No Data Found", level=Qgis.Warning
                    )
                    self.dlg.btnFetchDhisCategory.setEnabled(True)
                    self.dlg.btnFetchDhisCategory.setText("Connect")
                    self.dlg.btnFetchDhisCategory.repaint()
            else:
                hasData = False
                self.dlg.dhisProgressBar.setValue(0)
                self.iface.messageBar().pushMessage(
                    "Error",
                    f"Error fetching data: {response.status_code}",
                    level=Qgis.Critical,
                )
                self.dlg.btnFetchDhisCategory.setEnabled(True)
                self.dlg.btnFetchDhisCategory.setText("Connect")
                self.dlg.btnFetchDhisCategory.repaint()

            page += 1

    def on_dhis_combo_period_change(self):
        self.dhis_reset_saved_data()

    def on_dhis_combo_admin_level_change(self):
        self.dhis_reset_saved_data()

    def dhis_indicator_groups_on_change(self):
        self.dhis_reset_saved_data()
        self.dlg.comboDhisIndicators.clear()
        indicators_data = self.dlg.comboDhisProgramsOrDataSets.currentData()
        if indicators_data:
            indicators = indicators_data.get("curr_indicators")
            if indicators and isinstance(indicators, list):
                for indicator in indicators:
                    indicator_name = indicator.get("name")
                    indicator_id = indicator.get("id")
                    self.dlg.comboDhisIndicators.addItem(indicator_name, indicator_id)

    def fetch_gts_tracking_rounds_data_handler(self):
        api_url = self.dlg.gts_api_url.text()
        username = self.dlg.gts_username.text()
        password = self.dlg.gtsMLineEdit.text()

        # Disable OK button
        self.dlg.gtsOkButton.setEnabled(False)
        self.dlg.gtsOkButton.repaint()

        selected_tracking_round = self.dlg.comboGTSTrackingRounds.currentData()

        tracking_round_data = self.dlg.comboGTSFieldActivities.currentData()

        if selected_tracking_round and tracking_round_data:
            field_activity_text = self.dlg.comboGTSFieldActivities.currentText()
            cleaned_field_act_text = "_".join(field_activity_text.split(" "))

            single_tracking_url = selected_tracking_round.get("url")
            single_round_name = selected_tracking_round.get("round_name")

            url = f"https://{api_url}/fastapi/odata/v1/{single_tracking_url}"
            auth = HTTPBasicAuth(username, password)

            feature_collection = {
                "type": "FeatureCollection",
                "features": [],
            }

            params = {"$top": 50000, "$skip": 0}

            hasData = True
            page = 0

            while hasData:
                page += 1
                max_pages = 100

                progress = (int(page) / max_pages) * 100 if int(max_pages) > 1 else 100
                self.dlg.gtsProgressBar.setValue(math.ceil(progress))
                self.dlg.gtsProgressBar.repaint()

                self.dlg.gtsOkButton.setEnabled(False)
                response = self.fetch_with_retries(url, auth, params)
                if response.status_code == 200:
                    data = response.json()
                    data_list = data.get("value")

                    if data_list:
                        for datum in data_list:
                            long = (
                                datum.get("X") or datum.get("Lon") or datum.get("Long")
                            )
                            lat = datum.get("Y") or datum.get("Lat")

                            if lat and long:
                                geometry = {
                                    "type": "Point",
                                    "coordinates": [float(long), float(lat)],
                                }
                                # try:
                                #     del datum["geometry"]
                                # except ValueError:
                                #     pass
                                feature_collection["features"].append(
                                    {
                                        "type": "Feature",
                                        "geometry": geometry,
                                        "properties": datum,
                                    }
                                )
                    else:
                        hasData = False
                        self.dlg.gtsProgressBar.setValue(0)
                        self.dlg.gtsOkButton.setEnabled(True)
                else:
                    hasData = False
                    self.dlg.gtsProgressBar.setValue(0)
                    self.iface.messageBar().pushMessage(
                        "Error",
                        f"Error fetching data: {response.status_code}",
                        level=Qgis.Critical,
                        duration=10,
                    )
                    self.dlg.gtsOkButton.setEnabled(True)

                params["$skip"] += params["$top"]

            if (
                feature_collection["features"]
                and len(feature_collection["features"]) > 0
            ):
                self.gts_json_data = [
                    feature.get("properties")
                    for feature in feature_collection["features"]
                ]
                self.dlg.gtsDownloadCSV.setEnabled(True)

                self.dlg.gtsProgressBar.setValue(100)
                self.load_data_to_qgis(
                    feature_collection,
                    f"gts_{cleaned_field_act_text}",
                    "_".join(single_round_name.split(" ")),
                )
                self.dlg.gtsProgressBar.setValue(0)
            else:
                self.dlg.app_logs.appendPlainText(
                    f"No available Geo Data for Selected Tracking Round"
                )
                self.iface.messageBar().pushMessage(
                    "Notice",
                    f"No available Geo Data for Selected Tracking Round",
                    level=Qgis.Warning,
                    duration=10,
                )
                self.dlg.gtsOkButton.setEnabled(True)
                self.dlg.gtsOkButton.repaint()

                self.dlg.gtsProgressBar.setValue(0)

    def handle_gts_cancel_btn(self):
        self.dlg.gtsProgressBar.setValue(0)
        self.dlg.gtsOkButton.setEnabled(True)

    def on_gts_tracking_rounds_on_change(self):
        self.dlg.gtsOkButton.setEnabled(True)
        self.gts_reset_saved_data()

    def on_gts_field_activity_change(self):
        field_activity_data = self.dlg.comboGTSFieldActivities.currentData()
        self.dlg.comboGTSTrackingRounds.clear()
        self.gts_reset_saved_data()

        if field_activity_data:
            for datum in field_activity_data:
                url = datum.get("url")
                round_name = datum.get("round_name")
                self.dlg.comboGTSTrackingRounds.addItem(
                    round_name, {"url": url, "round_name": round_name}
                )

    def fetch_gts_tables_data(self, api_url, username, password, tables_url):
        auth = HTTPBasicAuth(username, password)

        self.dlg.comboGTSTableTypes.setEnabled(False)

        gts_field_activities = dict()
        url = f"https://{api_url}/fastapi/odata/v1/{tables_url}"
        self.dlg.gtsProgressBar.setValue(50)
        response = self.fetch_with_retries(url, auth)
        if response.status_code == 200:
            data = response.json()
            data_list = data.get("value")
            if data_list:
                self.dlg.gtsProgressBar.setValue(100)
                for datum in data_list:
                    if tables_url == "track_table_names":
                        table_name = datum.get("table_name")
                        tracking_round_id = datum.get("tracking_round_id")
                        field_activity_name = datum.get("field_activity_name")
                        tracking_round_name = datum.get("tracking_round_name")
                        tracking_round_nb_tracks = datum.get("tracking_round_nb_tracks")

                        if field_activity_name:
                            if not gts_field_activities.get(field_activity_name):
                                gts_field_activities[field_activity_name] = []
                                gts_field_activities[field_activity_name].append(
                                    {
                                        "round_name": tracking_round_name,
                                        "url": f"track/{tracking_round_id}",
                                    }
                                )
                            else:
                                gts_field_activities[field_activity_name].append(
                                    {
                                        "round_name": tracking_round_name,
                                        "url": f"track/{tracking_round_id}",
                                    }
                                )

                    if tables_url == "odk_table_names":
                        table_name = datum.get("table_name")
                        form_id = datum.get("form_id")
                        tracking_round_id = datum.get("tracking_round_id")
                        tracking_round_name = datum.get("tracking_round_name")
                        field_activity_name = datum.get("field_activity_name")
                        if field_activity_name:
                            if not gts_field_activities.get(field_activity_name):
                                gts_field_activities[field_activity_name] = []
                                gts_field_activities[field_activity_name].append(
                                    {
                                        "round_name": tracking_round_name,
                                        "url": f"odk/{tracking_round_id}_{form_id}",
                                    }
                                )
                            else:
                                gts_field_activities[field_activity_name].append(
                                    {
                                        "round_name": tracking_round_name,
                                        "url": f"odk/{tracking_round_id}_{form_id}",
                                    }
                                )

                    if tables_url == "indicator_table_names":
                        table_name = datum.get("table_name")
                        tracking_round_id = datum.get("tracking_round_id")
                        indicator_level = datum.get("indicator_level")
                        if "level" in indicator_level:
                            indicator_level = indicator_level.split("_")[-1]
                        elif "targeted" in indicator_level:
                            indicator_level = "ta"

                        field_activity_name = datum.get("field_activity_name")
                        tracking_round_name = datum.get("tracking_round_name")

                        if field_activity_name:
                            if not gts_field_activities.get(field_activity_name):
                                gts_field_activities[field_activity_name] = []
                                gts_field_activities[field_activity_name].append(
                                    {
                                        "round_name": tracking_round_name,
                                        "url": f"indicator/{tracking_round_id}_{indicator_level}",
                                    }
                                )
                            else:
                                gts_field_activities[field_activity_name].append(
                                    {
                                        "round_name": tracking_round_name,
                                        "url": f"indicator/{tracking_round_id}_{indicator_level}",
                                    }
                                )

                for key, val in gts_field_activities.items():
                    self.dlg.comboGTSFieldActivities.addItem(key, val)

                self.dlg.comboGTSFieldActivities.setEnabled(True)
                self.dlg.gtsProgressBar.setValue(0)
                self.dlg.comboGTSTableTypes.setEnabled(True)
            else:
                self.dlg.comboGTSTableTypes.setEnabled(True)
                self.dlg.gtsProgressBar.setValue(0)
                self.iface.messageBar().pushMessage(
                    "Notice", "No Data Found", level=Qgis.Warning
                )
        else:
            self.dlg.comboGTSTableTypes.setEnabled(True)
            self.dlg.gtsProgressBar.setValue(0)
            self.iface.messageBar().pushMessage(
                "Error",
                f"Error fetching data: {response.status_code}",
                level=Qgis.Critical,
            )

    def on_gts_tables_combo_box_change(self):
        api_url = self.dlg.gts_api_url.text()
        username = self.dlg.gts_username.text()
        password = self.dlg.gtsMLineEdit.text()
        tables_url = self.dlg.comboGTSTableTypes.currentData()

        # clear table names dropdown
        self.dlg.comboGTSFieldActivities.clear()
        self.gts_reset_saved_data()

        # Fetch data for each GTS Indicator
        self.fetch_gts_tables_data(api_url, username, password, tables_url)

    def fetch_gts_indicators_handler(self):
        api_url = self.dlg.gts_api_url.text()
        username = self.dlg.gts_username.text()
        password = self.dlg.gtsMLineEdit.text()
        self.reset_gts_inputs()
        self.gts_reset_saved_data()
        self.fetch_gts_indicators(api_url, username, password)

    def update_gts_ui_components(self, text="Connect"):
        self.dlg.comboGTSTableTypes.setEnabled(True)
        self.dlg.btnFetchGTSTables.setText(text)
        self.dlg.btnFetchGTSTables.setEnabled(True)
        self.dlg.btnFetchGTSTables.repaint()
        time.sleep(0.2)

    def fetch_gts_indicators(self, api_url, username, password):
        auth = HTTPBasicAuth(username, password)
        self.dlg.comboGTSTableTypes.clear()
        self.dlg.btnFetchGTSTables.setEnabled(False)
        self.dlg.btnFetchGTSTables.setText("Connecting...")
        self.dlg.btnFetchGTSTables.repaint()
        field_activities_set = set()
        time.sleep(0.2)

        url = f"https://{api_url}/fastapi/odata/v1/"

        response = self.fetch_with_retries(url, auth)
        if response.status_code == 200:
            data = response.json()
            if data:
                data_list = data.get("value")
                if data_list:
                    gts_tables = [
                        datum
                        for datum in data_list
                        if "table_names" in datum.get("name")
                    ]

                    for table in gts_tables:
                        table_name = table.get("name")
                        table_url = table.get("url")
                        self.dlg.comboGTSTableTypes.addItem(table_name, table_url)

                    self.update_gts_ui_components()
                else:
                    self.update_gts_ui_components()
                    self.iface.messageBar().pushMessage(
                        "Notice", "No Data Found", level=Qgis.Warning
                    )
            else:
                self.update_gts_ui_components()
                self.iface.messageBar().pushMessage(
                    "Notice", "No Data Found", level=Qgis.Warning
                )
        else:
            self.update_gts_ui_components()
            self.iface.messageBar().pushMessage(
                "Error",
                f"Error fetching data: {response.status_code}",
                level=Qgis.Critical,
            )

        return

    def fetch_kobo_form_data_clicked(self):
        api_url = self.dlg.kobo_api_url.text()
        selected_form = self.dlg.comboKoboForms.currentData()

        # disable OK button
        self.dlg.koboOkButton.setEnabled(False)
        self.dlg.koboPorgressBar.setValue(0)
        self.dlg.koboOkButton.repaint()

        # reset saved data
        self.kobo_reset_saved_data()

        asset_id = None
        if selected_form:
            asset_id = selected_form.get("asset_uid")

        username = self.dlg.kobo_username.text()
        password = self.dlg.koboMLineEdit.text()
        geo_field = self.dlg.comboKoboGeoFields.currentText()
        kobo_sync_interval = int(self.dlg.koboSyncInterval.value())

        kobo_from_date = self.dlg.KoboDateTimeFrom.date()
        kobo_to_date = self.dlg.KoboDateTimeTo.date()

        from_dt = datetime(
            kobo_from_date.year(), kobo_from_date.month(), kobo_from_date.day(), 0, 0, 0
        )  # 12:00 AM
        to_dt = datetime(
            kobo_to_date.year(), kobo_to_date.month(), kobo_to_date.day(), 23, 59, 59
        )

        # Convert datetime to timestamp string
        kobo_from_timestamp = from_dt.strftime("%Y-%m-%dT%H:%M:%S.%fZ")[
            :-3
        ]  # Adjust to match original format
        kobo_to_timestamp = to_dt.strftime("%Y-%m-%dT%H:%M:%S")

        if asset_id:
            self.fetch_and_save_kobo_data(
                api_url,
                username,
                password,
                asset_id,
                geo_field,
                kobo_from_timestamp,
                kobo_to_timestamp,
            )

            if kobo_sync_interval > 0:
                self.kobo_sync_timer.start(kobo_sync_interval * 1000)

    def on_kobo_data_sync_enabled(self):
        api_url = self.dlg.kobo_api_url.text()
        selected_form = self.dlg.comboKoboForms.currentData()
        username = self.dlg.kobo_username.text()
        password = self.dlg.koboMLineEdit.text()
        geo_field = self.dlg.comboKoboGeoFields.currentText()

        asset_id = None
        asset_name = None

        if selected_form:
            asset_name = selected_form.get("asset_name")
            asset_id = selected_form.get("asset_uid")

        # Fetch latest date fields
        self.fetch_kobo_date_range_fields(api_url, username, password, asset_id)

        # Extract date fields from time widget
        kobo_from_date = self.dlg.KoboDateTimeFrom.date()
        kobo_to_date = self.dlg.KoboDateTimeTo.date()

        from_dt = datetime(
            kobo_from_date.year(), kobo_from_date.month(), kobo_from_date.day(), 0, 0, 0
        )  # 12:00 AM
        to_dt = datetime(
            kobo_to_date.year(), kobo_to_date.month(), kobo_to_date.day(), 23, 59, 59
        )

        # Convert datetime to timestamp string
        kobo_from_timestamp = from_dt.strftime("%Y-%m-%dT%H:%M:%S.%fZ")[
            :-3
        ]  # Adjust to match original format
        kobo_to_timestamp = to_dt.strftime("%Y-%m-%dT%H:%M:%S")

        if asset_id and asset_name:
            cleaned_asset_name = "".join(asset_name.split(" "))

            if hasattr(self, "vlayers"):
                if self.vlayers.get(f"{cleaned_asset_name}_{geo_field}"):
                    self.vlayers[f"{cleaned_asset_name}_{geo_field}"]["syncData"] = True

            self.fetch_and_save_kobo_data(
                api_url,
                username,
                password,
                asset_id,
                geo_field,
                kobo_from_timestamp,
                kobo_to_timestamp,
            )

    def fetch_and_save_kobo_data(
        self, api_url, username, password, asset_id, geo_field, from_date, to_date
    ):
        auth = HTTPBasicAuth(username, password)
        selected_form = self.dlg.comboKoboForms.currentData()
        page_size = int(self.dlg.koboPageSize.value())
        asset_name = selected_form.get("asset_name")
        self.dlg.koboOkButton.setEnabled(False)

        self.dlg.koboOkButton.repaint()
        time.sleep(0.5)

        sort_param = json.dumps({"_submission_time": -1})

        params = {
            "sort": sort_param,
            "limit": page_size,
            "start": 0,
        }

        if from_date and to_date:
            params["query"] = json.dumps(
                {"_submission_time": {"$gte": from_date, "$lte": to_date}}
            )

        feature_collection = {
            "type": "FeatureCollection",
            "features": [],
        }

        url = f"https://{api_url}/api/v2/assets/{asset_id}/data.json"
        hasData = True

        while hasData:
            response = self.fetch_with_retries(url, auth, params=params)
            if response.status_code == 200:
                data = response.json()
                data_list = data.get("results")
                if data_list:
                    self.dlg.koboPorgressBar.setValue(100)
                    for datum in data_list:
                        flattened_datum = self.flatten_dict(datum)
                        self.kobo_json_data.append(flattened_datum)
                        self.get_geo_data(
                            flattened_datum, geo_field, feature_collection
                        )

                elif not data.get("next"):
                    hasData = False
                    self.dlg.koboOkButton.setEnabled(True)
                else:
                    hasData = False
                    self.dlg.koboOkButton.setEnabled(True)
                    break
            else:
                hasData = False
                self.iface.messageBar().pushMessage(
                    "Error",
                    f"Error fetching data: {response.status_code}",
                    level=Qgis.Critical,
                )
                self.dlg.koboOkButton.setEnabled(True)
                break

            params["start"] += params["limit"]

        # count = len(feature_collection["features"])
        # self.dlg.app_logs.appendPlainText(f"Features count: {count}")

        if self.kobo_json_data:
            self.dlg.koboDownloadCSV.setEnabled(True)
            self.dlg.koboDownloadCSV.repaint()

        if feature_collection["features"] and len(feature_collection["features"]) > 0:
            cleaned_asset_name = "".join(asset_name.split(" "))

            self.load_data_to_qgis(feature_collection, cleaned_asset_name, geo_field)
            self.dlg.koboPorgressBar.setValue(0)
            self.dlg.koboOkButton.setEnabled(True)
        else:
            self.dlg.app_logs.appendPlainText(
                "The selected geo field doesn't have geo data"
            )

            self.iface.messageBar().pushMessage(
                "Notice",
                f"The selected geo field doesn't have geo data",
                level=Qgis.Warning,
                duration=10,
            )
            self.dlg.koboPorgressBar.setValue(0)
            self.dlg.koboOkButton.setEnabled(True)

    def fetch_kobo_date_range_fields(self, api_url, username, password, asset_id):
        auth = HTTPBasicAuth(username, password)
        selected_form = self.dlg.comboKoboForms.currentData()
        asset_from_date = selected_form.get("date_created")
        url = f"https://{api_url}/api/v2/assets/{asset_id}/data.json"
        sort_param = json.dumps({"_submission_time": -1})
        params = {
            "sort": sort_param,
            "start": 0,
            "limit": 1,  # just a single submission is required
        }

        response = self.fetch_with_retries(url, auth, params)

        if response.status_code == 200:
            data = response.json()
            results = data.get("results")
            if results:
                latest_submission_date = results[0].get("_submission_time")

                from_dt = datetime.strptime(asset_from_date, "%Y-%m-%dT%H:%M:%S.%fZ")
                to_dt = datetime.strptime(latest_submission_date, "%Y-%m-%dT%H:%M:%S")

                # Adjust the times
                from_dt = from_dt.replace(
                    hour=0, minute=0, second=0, microsecond=0
                )  # 12:00 AM
                to_dt = to_dt.replace(hour=23, minute=59, second=59, microsecond=0)

                from_date = QDate(from_dt.year, from_dt.month, from_dt.day)
                to_date = QDate(to_dt.year, to_dt.month, to_dt.day)

                self.dlg.KoboDateTimeFrom.setDate(from_date)
                self.dlg.KoboDateTimeTo.setDate(to_date)

                self.dlg.KoboDateTimeFrom.setEnabled(True)
                self.dlg.KoboDateTimeTo.setEnabled(True)

                self.dlg.KoboDateTimeFrom.repaint()
                self.dlg.KoboDateTimeTo.repaint()
                self.dlg.koboOkButton.setEnabled(True)
                time.sleep(0.5)
            else:
                self.iface.messageBar().pushMessage(
                    "Notice", "No Data Found", level=Qgis.Warning
                )
        else:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Error fetching data: {response.status_code}",
                level=Qgis.Critical,
            )

        return

    def fetch_kobo_geo_fields(self, api_url, username, password, asset_id):
        auth = HTTPBasicAuth(username, password)
        url = f"https://{api_url}/api/v2/assets/{asset_id}.json"
        params = {"metadata": "on"}
        self.dlg.comboKoboGeoFields.setEnabled(False)
        response = self.fetch_with_retries(url, auth, params=params)
        if response.status_code == 200:
            self.dlg.comboKoboGeoFields.clear()
            data = response.json()
            self.asset_from_date = data.get("date_created")
            content = data.get("content")
            survey_arr = content.get("survey")
            geo_fields = [
                field.get("$autoname") or field.get("name")
                for field in survey_arr
                if field.get("type") in self.geo_types
            ]
            if geo_fields:
                self.dlg.comboKoboGeoFields.addItems(geo_fields)
                self.dlg.comboKoboGeoFields.setEnabled(True)
                self.dlg.comboKoboForms.setEnabled(True)
            else:
                self.dlg.comboKoboForms.setEnabled(True)
                self.iface.messageBar().pushMessage(
                    "Notice",
                    "No Geo Fields Present on Selected Form",
                    level=Qgis.Warning,
                    duration=10,
                )
        else:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Error fetching data: {response.status_code}",
                level=Qgis.Critical,
            )

    def on_combo_box_kobo_forms_change(self):
        api_url = self.dlg.kobo_api_url.text()
        username = self.dlg.kobo_username.text()
        password = self.dlg.koboMLineEdit.text()
        selected_form = self.dlg.comboKoboForms.currentData()

        self.dlg.comboKoboForms.setEnabled(False)
        self.dlg.koboOkButton.setEnabled(False)
        self.dlg.koboDownloadCSV.setEnabled(False)
        self.dlg.koboDownloadCSV.repaint()
        self.dlg.comboKoboForms.repaint()
        self.dlg.koboOkButton.repaint()

        self.dlg.comboKoboGeoFields.clear()

        self.dlg.comboKoboGeoFields.setEnabled(False)
        self.dlg.comboKoboGeoFields.repaint()
        self.dlg.app_logs.clear()

        # reset saved data
        self.kobo_reset_saved_data()

        if selected_form:
            asset_id = selected_form.get("asset_uid")
            self.fetch_kobo_geo_fields(api_url, username, password, asset_id)
            self.fetch_kobo_date_range_fields(api_url, username, password, asset_id)

    def fetch_kobo_assets_handler(self):
        api_url = self.dlg.kobo_api_url.text()
        username = self.dlg.kobo_username.text()
        password = self.dlg.koboMLineEdit.text()
        self.kobo_reset_saved_data()
        self.fetch_kobo_assets(api_url, username, password)

    def fetch_kobo_assets(self, api_url, username, password):
        auth = HTTPBasicAuth(username, password)

        self.dlg.comboKoboForms.clear()
        self.dlg.comboKoboGeoFields.clear()

        self.dlg.btnFetchKoboForms.setEnabled(False)
        self.dlg.btnFetchKoboForms.setText("Connecting...")
        self.dlg.btnFetchKoboForms.repaint()

        url = f"https://{api_url}/api/v2/assets.json"
        response = self.fetch_with_retries(url, auth)

        if response.status_code == 200:
            assets = response.json()
            assets_list = assets.get("results")
            if assets_list:
                assets_with_geo_data = [
                    asset
                    for asset in assets_list
                    if asset.get("summary", {}).get("geo")
                ]

                if assets_with_geo_data:
                    self.asset_from_date = assets_with_geo_data[0].get("date_created")
                    for geo_asset in assets_with_geo_data:
                        asset_uid = geo_asset.get("uid")
                        asset_label = geo_asset.get("name")
                        self.dlg.comboKoboForms.addItem(
                            asset_label,
                            {
                                "asset_uid": asset_uid,
                                "date_created": geo_asset.get("date_created"),
                                "asset_name": asset_label,
                            },
                        )
                    self.dlg.comboKoboForms.setEnabled(True)

            self.dlg.btnFetchKoboForms.setEnabled(True)
            self.dlg.btnFetchKoboForms.setText("Connect")
            self.dlg.btnFetchKoboForms.repaint()

        else:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Error fetching data: {response.status_code}",
                level=Qgis.Critical,
            )
            self.dlg.btnFetchKoboForms.setText("Connect")
            self.dlg.btnFetchKoboForms.setEnabled(True)

        return

    def on_combo_box_geo_fields_change(self, index):
        text = self.dlg.comboOnaGeoFields.currentText()
        self.curr_geo_field = text.split("-")[0].strip()
        self.dlg.onaProgressBar.setValue(0)

    def flatten_es_props(self, datum, parent_key="", sep="."):
        items = []

        if isinstance(datum, dict):
            for key, value in datum.items():
                new_key = f"{parent_key}{sep}{key}" if parent_key else key
                items.extend(self.flatten_es_props(value, new_key, sep=sep).items())
        elif isinstance(datum, list):
            for index, value in enumerate(datum):
                new_key = f"{parent_key}{sep}{index}" if parent_key else str(index)
                items.extend(self.flatten_es_props(value, new_key, sep=sep).items())
        else:
            items.append((parent_key, datum))

        return dict(items)

    def on_es_combo_topography_change(self):
        self.es_reset_saved_data()

    def fetch_es_data_clicked(self):
        api_url = self.dlg.es_api_url.text()
        es_api_version = self.dlg.esAPIVersion.text()
        topography = self.dlg.combESTopology.currentText()
        topography_param = topography.lower()
        site_admin_tokens = []
        lab_ids = []
        params = dict()

        export_url = None

        self.dlg.esOkButton.setEnabled(False)

        sites_feature_collection = {
            "type": "FeatureCollection",
            "features": [],
        }

        if topography_param == "sites":
            countries_url = (
                f"https://{api_url}/api/{es_api_version}-prod/admin/countries"
            )
            self.dlg.esProgressBar.setValue(20)
            response = self.fetch_with_retries(countries_url)
            if response.status_code == 200:
                data = response.json()
                if data:
                    features = data.get("features")
                    if features:
                        site_admin_tokens = [
                            f.get("properties").get("token") for f in features
                        ]
                        tokens_str = ",".join(site_admin_tokens)
                        params["export"] = "geojson"
                        params["admin"] = tokens_str
                        export_url = (
                            f"https://{api_url}/api/{es_api_version}-prod/sites"
                        )
                    else:
                        self.iface.messageBar().pushMessage(
                            "Notice",
                            "No Available Features",
                            level=Qgis.Warning,
                            duration=10,
                        )
            else:
                self.dlg.esProgressBar.setValue(0)
                self.dlg.esOkButton.setEnabled(True)
                self.iface.messageBar().pushMessage(
                    "Error",
                    f"Error fetching data: {response.status_code}",
                    level=Qgis.Critical,
                    duration=10,
                )

        if topography_param == "labs":
            labs_url = f"https://{api_url}/api/{es_api_version}-prod/{topography_param}"
            self.dlg.esProgressBar.setValue(20)
            response = self.fetch_with_retries(labs_url)
            if response.status_code == 200:
                data = response.json()
                if data:
                    lab_ids = [datum.get("id") for datum in data]
                    lab_ids_str = ",".join(lab_ids)

                    params["export"] = "geojson"
                    params["admin"] = lab_ids_str
                    export_url = f"https://{api_url}/api/{es_api_version}-prod/labs"
                else:
                    self.iface.messageBar().pushMessage(
                        "Notice", "No Available Data", level=Qgis.Warning, duration=10
                    )
            else:
                self.dlg.esProgressBar.setValue(0)
                self.dlg.esOkButton.setEnabled(True)
                self.iface.messageBar().pushMessage(
                    "Error",
                    f"Error fetching data: {response.status_code}",
                    level=Qgis.Critical,
                    duration=10,
                )

        if export_url:
            self.dlg.esProgressBar.setValue(50)

            if site_admin_tokens:
                quater = len(site_admin_tokens) // 4  # Find the midpoint
                first = site_admin_tokens[:quater]
                second = site_admin_tokens[quater : 2 * quater]
                third = site_admin_tokens[2 * quater : 3 * quater]
                fourth = site_admin_tokens[3 * quater :]

                for admin_tokens in [first, second, third, fourth]:
                    token_str = ",".join(admin_tokens)

                    response = self.fetch_with_retries(
                        export_url, params={"export": "geojson", "admin": token_str}
                    )

                    if response.status_code == 200:
                        data = response.json()
                        if data:
                            features = data.get("features")
                            sites_feature_collection["features"].extend(features)
                    else:
                        self.dlg.esProgressBar.setValue(0)
                        self.dlg.esOkButton.setEnabled(True)
                        self.iface.messageBar().pushMessage(
                            "Error",
                            f"Error fetching data: {response.status_code}",
                            level=Qgis.Critical,
                        )

                self.dlg.esProgressBar.setValue(100)

                if (
                    sites_feature_collection["features"]
                    and len(sites_feature_collection["features"]) > 0
                ):
                    self.es_json_data = [
                        feature.get("properties")
                        for feature in sites_feature_collection["features"]
                    ]
                    self.dlg.esDownloadCSV.setEnabled(True)

                    self.load_data_to_qgis(
                        sites_feature_collection, "es", topography_param
                    )
                    self.dlg.esProgressBar.setValue(0)
                    self.dlg.esOkButton.setEnabled(True)
                else:
                    self.dlg.esProgressBar.setValue(0)
                    self.iface.messageBar().pushMessage(
                        "Notice", "No Data Found", level=Qgis.Warning
                    )
                    self.dlg.esOkButton.setEnabled(True)

            else:
                response = self.fetch_with_retries(export_url, params=params)

                if response.status_code == 200:
                    self.dlg.esProgressBar.setValue(100)
                    feature_collection = response.json()
                    if (
                        feature_collection["features"]
                        and len(feature_collection["features"]) > 0
                    ):
                        self.es_json_data = [
                            feature.get("properties")
                            for feature in feature_collection["features"]
                        ]
                        self.dlg.esDownloadCSV.setEnabled(True)

                        self.load_data_to_qgis(
                            feature_collection, "es", topography_param
                        )
                        self.dlg.esProgressBar.setValue(0)
                        self.dlg.esOkButton.setEnabled(True)
                    else:
                        self.dlg.esProgressBar.setValue(0)
                        self.iface.messageBar().pushMessage(
                            "Notice", "No Data Found", level=Qgis.Warning
                        )
                        self.dlg.esOkButton.setEnabled(True)
                else:
                    self.dlg.esProgressBar.setValue(0)
                    self.dlg.esOkButton.setEnabled(True)
                    self.iface.messageBar().pushMessage(
                        "Error",
                        f"Error fetching data: {response.status_code}",
                        level=Qgis.Critical,
                    )

    def ona_reset_saved_data(self):
        self.json_data = list()
        self.dlg.onaDownloadCSV.setEnabled(False)
        self.dlg.onaDownloadCSV.repaint()

    def odk_reset_saved_data(self):
        self.odk_json_data = list()
        self.dlg.odkDownloadCSV.setEnabled(False)
        self.dlg.odkDownloadCSV.repaint()

    def kobo_reset_saved_data(self):
        self.kobo_json_data = list()
        self.dlg.koboDownloadCSV.setEnabled(False)
        self.dlg.koboDownloadCSV.repaint()

    def gts_reset_saved_data(self):
        self.gts_json_data = list()
        self.dlg.gtsDownloadCSV.setEnabled(False)
        self.dlg.gtsDownloadCSV.repaint()

    def es_reset_saved_data(self):
        self.es_json_data = list()
        self.dlg.esDownloadCSV.setEnabled(False)
        self.dlg.esDownloadCSV.repaint()

    def dhis_reset_saved_data(self):
        self.dhis_json_data = list()
        self.dlg.dhisDownloadCSV.setEnabled(False)
        self.dlg.dhisDownloadCSV.repaint()

    def polis_reset_saved_data(self):
        self.polis_json_data = list()
        self.dlg.polisDownloadCSV.setEnabled(False)
        self.dlg.polisDownloadCSV.repaint()

    def on_odk_forms_combo_box_change(self):

        self.dlg.comboODKForms.setEnabled(False)
        self.dlg.odkOkButton.setEnabled(False)
        self.dlg.odkDownloadCSV.setEnabled(False)
        self.dlg.odkDownloadCSV.repaint()
        self.dlg.comboODKForms.repaint()
        self.dlg.odkOkButton.repaint()

        self.dlg.comboODKGeoFields.clear()

        self.dlg.comboODKGeoFields.setEnabled(False)
        self.dlg.comboODKGeoFields.repaint()
        self.dlg.app_logs.clear()

        form_data = self.dlg.comboODKForms.currentData()

        # reset saved data
        self.odk_reset_saved_data()

        if form_data:
            form_id_str = form_data.get("form_id")
            project_id = form_data.get("project_id")
            api_url = self.dlg.odk_api_url.text()
            username = self.dlg.odk_username.text()
            password = self.dlg.odkmLineEdit.text()
            self.fetch_odk_geo_fields(
                api_url, username, password, project_id, form_id_str
            )
            self.fetch_odk_date_range_fields(
                api_url, username, password, project_id, form_id_str
            )

        # fetch time fields to activate date range filters

    def fetch_odk_date_range_fields(
        self, api_url, username, password, project_id, form_id_str
    ):
        auth = HTTPBasicAuth(username, password)
        url = f"https://{api_url}/v1/projects/{project_id}/forms/{form_id_str}"

        # headers for additional metadata
        headers = {"X-Extended-Metadata": "true"}
        response = self.fetch_with_retries(url, auth, params=None, headers=headers)
        if response.status_code == 200:
            data = response.json()
            from_timestamp = data.get("createdAt")
            to_timestamp = data.get("lastSubmission")

            from_dt = datetime.strptime(from_timestamp, "%Y-%m-%dT%H:%M:%S.%fZ")
            to_dt = None

            if to_timestamp:
                to_dt = datetime.strptime(to_timestamp, "%Y-%m-%dT%H:%M:%S.%fZ")
            else:
                to_dt = datetime.now()

            # Adjust the times
            from_dt = from_dt.replace(
                hour=0, minute=0, second=0, microsecond=0
            )  # 12:00 AM
            to_dt = to_dt.replace(
                hour=23, minute=59, second=59, microsecond=0
            )  # 11:59:59 PM

            from_date = QDate(from_dt.year, from_dt.month, from_dt.day)
            to_date = QDate(to_dt.year, to_dt.month, to_dt.day)

            self.dlg.ODKDateTimeFrom.setDate(from_date)
            self.dlg.ODKDateTimeTo.setDate(to_date)

            self.dlg.ODKDateTimeFrom.update()
            self.dlg.ODKDateTimeTo.update()
            self.dlg.ODKDateTimeFrom.setEnabled(True)
            self.dlg.ODKDateTimeTo.setEnabled(True)

            self.dlg.odkOkButton.setEnabled(True)

            self.dlg.ODKDateTimeFrom.setDisplayFormat("yyyy-MM-dd")
            self.dlg.ODKDateTimeTo.setDisplayFormat("yyyy-MM-dd")

            # set calendar range
            # self.dlg.ODKDateTimeFrom.setDateTimeRange(
            #     from_date, to_date)
            # self.dlg.ODKDateTimeTo.setDateTimeRange(
            #     from_date, to_date)

            self.dlg.ODKDateTimeFrom.update()
            self.dlg.ODKDateTimeTo.update()
        else:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Error fetching Date Ranges: {response.status_code}",
                level=Qgis.Critical,
            )
            self.dlg.ODKDateTimeFrom.setEnabled(False)
            self.dlg.ODKDateTimeTo.setEnabled(False)

    def fetch_odk_geo_fields(
        self, api_url, username, password, project_id, form_id_str
    ):
        auth = HTTPBasicAuth(username, password)
        url = f"https://{api_url}/v1/projects/{project_id}/forms/{form_id_str}/fields"
        params = {"odata": True}
        self.dlg.comboODKGeoFields.setEnabled(False)
        response = self.fetch_with_retries(url, auth, params=params)
        if response.status_code == 200:
            self.dlg.comboODKGeoFields.clear()
            data = response.json()
            geo_fields = [
                field.get("name")
                for field in data
                if field.get("type") in self.geo_types
            ]
            if geo_fields:
                self.dlg.comboODKGeoFields.addItems(geo_fields)
                self.dlg.comboODKGeoFields.setEnabled(True)
                self.dlg.comboODKForms.setEnabled(True)
            else:
                self.dlg.comboODKForms.setEnabled(True)
                self.iface.messageBar().pushMessage(
                    "Notice",
                    "No Geo Fields Present on Selected Form",
                    level=Qgis.Warning,
                    duration=10,
                )
        else:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Error fetching data: {response.status_code}",
                level=Qgis.Critical,
            )

    def fetch_and_save_geojson_fields(self, api_url, username, password, formID):
        self.fetchGeoFields(api_url, username, password, formID)

    def fetch_ona_forms_handler(self):
        self.stop_workers()
        api_url = self.dlg.onadata_api_url.text()
        username = self.dlg.onadata_username.text()
        password = self.dlg.onaMLineEdit.text()
        sync = self.dlg.onaSyncInterval.value()
        self.dlg.app_logs.appendPlainText(f"sync interval - {sync}")
        self.ona_reset_saved_data()
        self.fetch_ona_forms(api_url, username, password)

    def handle_ona_forms_data_fetched(self, data):
        if data:
            for datum in data:
                title = datum.get("title")
                form_id = datum.get("formid")
                self.dlg.comboOnaForms.addItem(title, form_id)

            self.dlg.btnFetchOnaForms.setEnabled(True)
            self.dlg.btnFetchOnaForms.setText("Connect")
            self.dlg.btnFetchOnaForms.repaint()

        self.dlg.comboOnaForms.setEnabled(True)

    def handle_ona_forms_fetch_error(self, msg):
        self.iface.messageBar().pushMessage(
            "Error", f"Error fetching data: {msg}", level=Qgis.Critical
        )

    def handle_ona_forms_no_data(self, msg):
        self.iface.messageBar().pushMessage("Notice", {msg}, level=Qgis.Warning)
        self.dlg.btnFetchOnaForms.setEnabled(True)
        self.dlg.btnFetchOnaForms.setText("Connect")
        self.dlg.btnFetchOnaForms.repaint()

    def handle_ona_forms_status_error(self, msg):
        self.iface.messageBar().pushMessage(
            "Error", f"Error fetching data: {msg}", level=Qgis.Critical
        )
        self.dlg.btnFetchOnaForms.setEnabled(True)
        self.dlg.btnFetchOnaForms.setText("Connect")
        self.dlg.btnFetchOnaForms.repaint()

    def fetch_ona_forms(self, api_url, username, password):
        auth = HTTPBasicAuth(username, password)
        self.dlg.comboOnaForms.clear()
        self.dlg.btnFetchOnaForms.setEnabled(False)
        self.dlg.btnFetchOnaForms.setText("Connecting...")
        self.dlg.btnFetchOnaForms.repaint()

        url = f"https://{api_url}/{username}/formList"

        self.fetch_ona_forms_worker = FetchOnaFormsThread(
            url, auth, params=None, headers=None
        )
        self.dlg.app_logs.appendPlainText(f"The URL: {url}")

        # Connect signals to the handler methods
        self.fetch_ona_forms_worker.data_fetched.connect(
            self.handle_ona_forms_data_fetched
        )
        self.fetch_ona_forms_worker.error_occurred.connect(
            self.handle_ona_forms_fetch_error
        )
        self.fetch_ona_forms_worker.no_data.connect(self.handle_ona_forms_no_data)
        self.fetch_ona_forms_worker.status_error.connect(
            self.handle_ona_forms_status_error
        )
        self.fetch_ona_forms_worker.start()

    def handle_geo_fields_fetched(self, data):
        if isinstance(data, dict):
            geo_fields_set = data.get("geo_fields_set")
            geo_fields_dict = data.get("geo_fields_dict")

            for i, gf in enumerate(geo_fields_set):
                cleaned_gf = gf.strip()
                self.dlg.comboOnaGeoFields.addItem(cleaned_gf)
                geo_label = geo_fields_dict[cleaned_gf]
                if geo_label:
                    self.dlg.comboOnaGeoFields.setItemText(
                        i, f"{cleaned_gf} - ({geo_label})"
                    )

            self.dlg.onaOkButton.setEnabled(True)

            self.dlg.comboOnaForms.setEnabled(True)
            self.dlg.comboOnaGeoFields.setEnabled(True)

            self.dlg.app_logs.appendPlainText(
                f"Number of Geo Fields Found: {len(geo_fields_set)} \n"
            )

    def handle_geo_fields_progress(self, data):
        if isinstance(data, str):
            self.dlg.app_logs.appendPlainText(data)
        elif isinstance(data, dict):
            page = data.get("curr_page")
            total_pages = data.get("total_pages")
            progress = (
                (int(page) / int(total_pages)) * 100 if int(total_pages) > 1 else 100
            )
            self.dlg.onaProgressBar.setValue(math.ceil(progress))

    def handle_geo_fields_no_data(self, msg):
        if isinstance(msg, str):
            self.dlg.app_logs.appendPlainText(msg)
            self.iface.messageBar().pushMessage(
                "Notice", msg, level=Qgis.Warning, duration=10
            )
            self.reset_inputs()

    def fetch_ona_form_geo_fields(self):
        # clear geo fields combo box

        self.dlg.comboOnaForms.setEnabled(False)
        self.dlg.onaOkButton.setEnabled(False)
        self.dlg.onaDownloadCSV.setEnabled(False)
        self.dlg.onaDownloadCSV.repaint()
        self.dlg.comboOnaForms.repaint()
        self.dlg.onaOkButton.repaint()

        self.dlg.comboOnaGeoFields.clear()

        self.dlg.comboOnaGeoFields.setEnabled(False)
        self.dlg.comboOnaGeoFields.repaint()
        self.dlg.app_logs.clear()
        # reset geo fields
        self.geo_fields = set()

        api_url = self.dlg.onadata_api_url.text()
        formID = self.dlg.comboOnaForms.currentData()
        username = self.dlg.onadata_username.text()
        password = self.dlg.onaMLineEdit.text()

        auth = HTTPBasicAuth(username, password)

        if formID:
            url = f"https://{api_url}/api/v1/forms/{formID}/versions"

            self.fetch_ona_geo_fields_worker = FetchOnaGeoFieldsThread(
                url, auth, params=None, headers=None, formID=formID
            )

            self.fetch_ona_geo_fields_worker.data_fetched.connect(
                self.handle_geo_fields_fetched
            )
            self.fetch_ona_geo_fields_worker.progress_updated.connect(
                self.handle_geo_fields_progress
            )
            self.fetch_ona_geo_fields_worker.no_data.connect(
                self.handle_geo_fields_no_data
            )
            self.fetch_ona_geo_fields_worker.count_and_date_fields_fetched.connect(
                self.handle_date_and_count_fields
            )
            self.fetch_ona_geo_fields_worker.count_and_date_fields_error_occurred.connect(
                self.handle_date_and_count_fields_error
            )
            self.fetch_ona_geo_fields_worker.error_occurred.connect(
                self.handle_fetch_error
            )
            self.fetch_ona_geo_fields_worker.status_error.connect(
                self.handle_status_error
            )
            self.fetch_ona_geo_fields_worker.start()

    def fetch_odk_forms_handler(self):
        api_url = self.dlg.odk_api_url.text()
        username = self.dlg.odk_username.text()
        password = self.dlg.odkmLineEdit.text()
        self.odk_reset_saved_data()
        self.fetch_odk_projects(api_url, username, password)

    def fetch_odk_forms_per_proj(self, api_url, username, password, project_ids):
        auth = HTTPBasicAuth(username, password)
        for proj_id in project_ids:
            url = f"https://{api_url}/v1/projects/{proj_id}/forms"
            response = self.fetch_with_retries(url, auth)
            if response.status_code == 200:
                forms = response.json()
                if forms:
                    for i, form in enumerate(forms):
                        form_id = form.get("xmlFormId")
                        form_name = form.get("name")
                        project_id = form.get("projectId")
                        self.dlg.comboODKForms.addItem(
                            form_name, {"form_id": form_id, "project_id": project_id}
                        )
                        if not self.odk_forms_to_projects_map.get(form_id):
                            self.odk_forms_to_projects_map[form_id] = proj_id
                    self.dlg.btnFetchODKForms.setEnabled(True)
                    self.dlg.btnFetchODKForms.setText("Connect")
                    self.dlg.btnFetchODKForms.repaint()
                    self.dlg.comboODKForms.setEnabled(True)
                else:
                    self.iface.messageBar().pushMessage(
                        "Notice", "No Forms Found", level=Qgis.Warning
                    )
            else:
                self.dlg.btnFetchODKForms.setEnabled(True)
                self.dlg.btnFetchODKForms.setText("Connect")
                self.dlg.btnFetchODKForms.repaint()
        return

    def fetch_odk_projects(self, api_url, username, password):
        auth = HTTPBasicAuth(username, password)
        self.dlg.comboODKForms.clear()
        self.dlg.comboODKGeoFields.clear()

        self.dlg.btnFetchODKForms.setEnabled(False)
        self.dlg.btnFetchODKForms.setText("Connecting...")
        self.dlg.btnFetchODKForms.repaint()

        url = f"https://{api_url}/v1/projects"
        response = self.fetch_with_retries(url, auth)

        if response.status_code == 200:
            odk_projects = response.json()
            project_ids = [project.get("id") for project in odk_projects]
            if project_ids:
                self.fetch_odk_forms_per_proj(api_url, username, password, project_ids)
            else:
                self.iface.messageBar().pushMessage(
                    "Notice", "No Projects Found", level=Qgis.Warning
                )
                self.dlg.btnFetchODKForms.setEnabled(True)
        else:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Error fetching data: {response.status_code}",
                level=Qgis.Critical,
            )
            self.dlg.btnFetchODKForms.setText("Connect")
            self.dlg.btnFetchODKForms.setEnabled(True)

    def on_odk_data_sync_enabled(self):
        api_url = self.dlg.odk_api_url.text()
        username = self.dlg.odk_username.text()
        password = self.dlg.odkmLineEdit.text()
        geo_field = self.dlg.comboODKGeoFields.currentText()
        odk_sync_interval = int(self.dlg.odkSyncInterval.value())

        form_data = self.dlg.comboODKForms.currentData()
        form_id_str = None
        project_id = None

        if form_data:
            form_id_str = form_data.get("form_id")
            project_id = form_data.get("project_id")

            self.fetch_odk_date_range_fields(
                api_url,
                username,
                password,
                project_id,
                form_id_str,
            )

            # extract date fields
            odk_from_date = self.dlg.ODKDateTimeFrom.date()
            odk_to_date = self.dlg.ODKDateTimeTo.date()

            from_dt = datetime(
                odk_from_date.year(),
                odk_from_date.month(),
                odk_from_date.day(),
                0,
                0,
                0,
            )  # 12:00 AM
            to_dt = datetime(
                odk_to_date.year(), odk_to_date.month(), odk_to_date.day(), 23, 59, 59
            )

            # Convert datetime to timestamp string
            odk_from_timestamp = (
                from_dt.strftime("%Y-%m-%dT%H:%M:%S.%f")[:-3] + "Z"
            )  # Adjust to match original format
            odk_to_timestamp = to_dt.strftime("%Y-%m-%dT%H:%M:%S.%f")[:-3] + "Z"

            if hasattr(self, "vlayers"):
                if self.vlayers.get(f"{form_id_str}_{geo_field}"):
                    self.vlayers[f"{form_id_str}_{geo_field}"]["syncData"] = True

            self.fetch_and_save_odk_data(
                api_url,
                username,
                password,
                form_id_str,
                geo_field,
                odk_from_timestamp,
                odk_to_timestamp,
            )

    def fetch_odk_form_data_clicked(self):
        # Extract parameters from the dialog

        # disable OK button
        self.dlg.odkOkButton.setEnabled(False)
        self.dlg.odkProgressBar.setValue(0)
        self.dlg.odkOkButton.repaint()

        # reset saved data
        self.odk_reset_saved_data()

        api_url = self.dlg.odk_api_url.text()
        form_data = self.dlg.comboODKForms.currentData()
        form_id_str = None
        if form_data:
            form_id_str = form_data.get("form_id")
        username = self.dlg.odk_username.text()
        password = self.dlg.odkmLineEdit.text()
        geo_field = self.dlg.comboODKGeoFields.currentText()
        odk_sync_interval = int(self.dlg.odkSyncInterval.value())
        # extract date fields
        odk_from_date = self.dlg.ODKDateTimeFrom.date()
        odk_to_date = self.dlg.ODKDateTimeTo.date()

        from_dt = datetime(
            odk_from_date.year(), odk_from_date.month(), odk_from_date.day(), 0, 0, 0
        )  # 12:00 AM
        to_dt = datetime(
            odk_to_date.year(), odk_to_date.month(), odk_to_date.day(), 23, 59, 59
        )  # 11:59:59 PM

        # Convert datetime to timestamp string
        odk_from_timestamp = (
            from_dt.strftime("%Y-%m-%dT%H:%M:%S.%f")[:-3] + "Z"
        )  # Adjust to match original format
        odk_to_timestamp = to_dt.strftime("%Y-%m-%dT%H:%M:%S.%f")[:-3] + "Z"

        page_size = 1000
        directory = True

        if directory:
            # Create a worker for background data fetching
            # self.odk_worker = FetchODKDataWorker(
            #     self,
            #     api_url,
            #     form_id_str,
            #     username,
            #     password,
            #     geo_field,
            #     page_size,
            #     directory,
            #     self.dlg,
            #     False,
            #     odk_from_timestamp,
            #     odk_to_timestamp
            # )

            # # # Connect signals to handle results in the main thread
            # self.odk_worker.odk_data_fetched.connect(self.on_data_fetched)
            # self.odk_worker.odk_fetch_error.connect(self.on_fetch_error)
            # self.odk_worker.odk_rogress_updated.connect(self.on_progress_update)

            # # Start the thread
            # self.odk_worker.start()
            # if odk_sync_interval > 0:
            #     self.odk_sync_timer.start(odk_sync_interval * 1000)

            if form_id_str:
                self.fetch_and_save_odk_data(
                    api_url,
                    username,
                    password,
                    form_id_str,
                    geo_field,
                    odk_from_timestamp,
                    odk_to_timestamp,
                )

                if odk_sync_interval > 0:
                    self.odk_sync_timer.start(odk_sync_interval * 1000)

    def is_valid_wkt(self, wkt_string):
        # Define basic geometry types
        valid_geometries = [
            "POINT",
            "LINESTRING",
            "POLYGON",
            "MULTIPOINT",
            "MULTILINESTRING",
            "MULTIPOLYGON",
        ]

        # Extract the geometry type
        match = re.match(r"^\s*(\w+)\s*\((.*)\)\s*$", wkt_string, re.IGNORECASE)
        if not match:
            return False  # Invalid format

        geometry_type, coordinates_part = match.groups()
        geometry_type = geometry_type.upper()

        # Check if geometry type is valid
        if geometry_type not in valid_geometries:
            return False
        else:
            return True

    def wkt_to_geometry_obj(self, wkt_string):
        # Identify the geometry type and coordinates
        wkt = wkt_string.strip()
        geometry_type, coords = wkt.split("(", 1)
        geometry_type = geometry_type.strip().upper()
        coords = coords.strip().rstrip(")").lstrip("(")

        if geometry_type == "POINT":
            # Take only the first two values (X and Y)
            coordinate_values = coords.split()
            coordinates = [float(coordinate_values[0]), float(coordinate_values[1])]
            geometry_obj = {"type": "Point", "coordinates": coordinates}

        elif geometry_type == "LINESTRING" or geometry_type == "GEOTRACE":
            # Extract all numeric values, including negative and decimal points
            coordinate_values = re.findall(r"-?\d+\.?\d*", coords)
            # Group into pairs (X, Y), ignoring the Z component if present
            coordinates = [
                [float(coordinate_values[i]), float(coordinate_values[i + 1])]
                for i in range(
                    0,
                    len(coordinate_values),
                    2 if len(coordinate_values) % 3 != 0 else 3,
                )
            ]
            geometry_obj = {"type": "LineString", "coordinates": coordinates}

        elif geometry_type == "POLYGON":
            rings = coords.split("), (")
            rings = [ring.replace("(", "").replace(")", "").strip() for ring in rings]
            coordinates = [
                [
                    [float(pair.split()[0]), float(pair.split()[1])]
                    for pair in ring.split(",")
                ]
                for ring in rings
            ]
            geometry_obj = {"type": "Polygon", "coordinates": coordinates}
        else:
            raise ValueError(f"Unsupported geometry type: {geometry_type}")

        return geometry_obj

    def get_odk_geo_data(self, datum, geom_field):
        field_keys = datum.keys()
        if geom_field in field_keys:
            # this means that the geo field is not inside a repeat
            # build the corresponding geometry/feature collection
            geom = datum.get(geom_field)
            # determine whether polygon or point
            if geom and self.is_valid_wkt(geom):
                geometry = self.wkt_to_geometry_obj(geom)
                # this means that it is a polygon
                # build the correspoing collection
                feature = {"type": "Feature", "geometry": geometry, "properties": datum}
                return feature
        else:
            # flatten the datum
            repeat_geo_arr = []
            for k in datum.keys():
                field_arr = k.split("/")
                if geom_field in field_arr:
                    repeat_geo_arr.append(k)
            if repeat_geo_arr:
                for f in repeat_geo_arr:
                    nested_geom = datum.get(f)
                    if nested_geom and self.is_valid_wkt(nested_geom):
                        geometry = self.wkt_to_geometry_obj(nested_geom)
                        # this means that it is a polygon
                        # build the correspoing collection
                        feature = {
                            "type": "Feature",
                            "geometry": geometry,
                            "properties": datum,
                        }
                        return feature

    def flatten_odk_json(self, json_obj, parent_key=""):
        """Recursively flattens a nested JSON object into a dictionary with XPath keys."""
        flattened = {}

        def _flatten(obj, key_prefix=""):
            if isinstance(obj, dict):
                for k, v in obj.items():
                    new_key = f"{key_prefix}/{k}" if key_prefix else k
                    _flatten(v, new_key)
            elif isinstance(obj, list):
                for i, item in enumerate(obj):
                    _flatten(item, f"{key_prefix}[{i + 1}]")
            else:
                flattened[key_prefix] = obj

        _flatten(json_obj)
        return flattened

    def fetch_and_save_odk_data(
        self,
        api_url,
        username,
        password,
        form_id_str,
        geo_field,
        odk_from_date,
        odk_to_date,
    ):
        auth = HTTPBasicAuth(username, password)
        self.dlg.odkOkButton.setEnabled(False)
        page_size = int(self.dlg.odkPageSize.value())

        params = {"$expand": "*", "$wkt": True, "$top": page_size, "$skip": 0}

        if odk_from_date and odk_to_date:
            filter_query = f"__system/submissionDate ge {odk_from_date} and __system/submissionDate le {odk_to_date}"
            params["$filter"] = filter_query

        project_id = self.odk_forms_to_projects_map.get(form_id_str)

        feature_collection = {
            "type": "FeatureCollection",
            "features": [],
        }

        url = f"https://{api_url}/v1/projects/{project_id}/forms/{form_id_str}.svc/Submissions"
        hasData = True

        while hasData:
            response = self.fetch_with_retries(url, auth, params)
            if response.status_code == 200:
                self.dlg.gtsProgressBar.setValue(50)
                data = response.json()
                data_list = data.get("value")
                if data_list:
                    self.dlg.odkProgressBar.setValue(100)
                    for datum in data_list:
                        flat_data = self.flatten_odk_json(datum)
                        self.odk_json_data.append(flat_data)
                        feature = self.get_odk_geo_data(flat_data, geo_field)
                        if feature:
                            feature_collection["features"].append(feature)
                else:
                    hasData = False
                    self.dlg.gtsProgressBar.setValue(100)
                    self.dlg.odkOkButton.setEnabled(True)
            else:
                hasData = False
                self.dlg.gtsProgressBar.setValue(0)
                self.iface.messageBar().pushMessage(
                    "Error",
                    f"Error fetching data: {response.status_code}",
                    level=Qgis.Critical,
                    duration=10,
                )
                self.dlg.odkOkButton.setEnabled(True)

            params["$skip"] += params["$top"]

        if self.odk_json_data:
            self.dlg.odkDownloadCSV.setEnabled(True)
            self.dlg.odkDownloadCSV.repaint()

        if feature_collection["features"] and len(feature_collection["features"]) > 0:
            self.load_data_to_qgis(feature_collection, form_id_str, geo_field)
            self.dlg.odkOkButton.setEnabled(True)
            self.dlg.odkProgressBar.setValue(0)
        else:
            self.iface.messageBar().pushMessage(
                "Notice",
                f"The selected geo field doesn't have geo data",
                level=Qgis.Warning,
                duration=10,
            )
            self.dlg.odkOkButton.setEnabled(True)
            self.dlg.odkProgressBar.setValue(0)

    # Slots to handle signals
    def on_data_fetched(self, data):
        self.iface.messageBar().pushMessage(
            "Success", "Data fetched successfully!", level=Qgis.Success
        )
        # Process and display data in QGIS as needed

    def on_fetch_error(self, error_message):
        self.iface.messageBar().pushMessage(
            "Error", f"Error fetching data: {error_message}", level=Qgis.Critical
        )

    def on_progress_update(self, message):
        self.iface.messageBar().pushMessage("Progress", message, level=Qgis.Info)

    def loop_cleanup(self):
        """Ensure the event loop is closed."""
        if self.loop and self.loop.is_running():
            logging.debug("Stopping event loop")
            self.loop.stop()
            self.loop.close()

    def handle_data_fetched(self, data):
        formID = self.dlg.comboOnaForms.currentData()
        form_str = self.dlg.comboOnaForms.currentText()
        cleaned_form_str = "_".join(form_str.split(" "))

        geo_field = self.curr_geo_field

        self.dlg.app_logs.appendPlainText("Data Fetch Complete, Building GeoJSON... \n")

        if isinstance(data, dict) or (isinstance(data, list) and len(data) == 0):
            self.dlg.app_logs.appendPlainText(
                "No Data Available For the selected date range"
            )

            if self.ona_sync_timer.isActive():
                self.ona_sync_timer.stop()
                self.dlg.onaOkButton.setEnabled(True)
        else:
            feature_collection = {
                "type": "FeatureCollection",
                "features": [],
            }

            if data:
                self.json_data = data
                self.dlg.onaDownloadCSV.setEnabled(True)

            for datum in data:
                self.get_geo_data(datum, geo_field, feature_collection)

            if (
                feature_collection["features"]
                and len(feature_collection["features"]) > 0
            ):
                # self.dlg.onaProgressBar.setValue(100)
                self.dlg.app_logs.appendPlainText(
                    "Building GeoJSON Complete. Adding Layer to Map...\n"
                )
                self.load_data_to_qgis(feature_collection, cleaned_form_str, geo_field)

                self.dlg.onaProgressBar.setValue(0)
                if not self.ona_sync_timer.isActive():
                    self.dlg.onaOkButton.setEnabled(True)
            else:
                self.dlg.app_logs.appendPlainText(
                    "The selected geo field doesn't have geo data"
                )

                self.iface.messageBar().pushMessage(
                    "Notice",
                    f"The selected geo field doesn't have geo data",
                    level=Qgis.Warning,
                    duration=10,
                )
                if not self.ona_sync_timer.isActive():
                    self.dlg.onaOkButton.setEnabled(True)

    def handle_fetch_error(self, message):
        self.dlg.app_logs.appendPlainText(f"Error - {message}")
        self.iface.messageBar().pushMessage(
            "Error", f"{message}", level=Qgis.Critical, duration=10
        )

    def handle_status_error(self, msg):
        self.dlg.app_logs.appendPlainText(f"Error - {msg}")
        self.iface.messageBar().pushMessage(
            "Error", f"{msg}", level=Qgis.Critical, duration=10
        )

    def handle_date_and_count_fields(self, data):
        if isinstance(data, dict):
            self.data_count = data.get("count")
            get_from_date = data.get("from_date")
            get_to_date = data.get("to_date")

            self.dlg.app_logs.appendPlainText(f"Submissions Count: {self.data_count}")

            from_dt = datetime.fromisoformat(get_from_date).astimezone(timezone.utc)
            to_dt = datetime.fromisoformat(get_to_date).astimezone(timezone.utc)

            from_dt = from_dt.replace(
                hour=0, minute=0, second=0, microsecond=0
            )  # 12:00 AM
            to_dt = to_dt.replace(hour=23, minute=59, second=59, microsecond=0)

            from_date = QDate(from_dt.year, from_dt.month, from_dt.day)
            to_date = QDate(to_dt.year, to_dt.month, to_dt.day)

            self.dlg.onaDateTimeFrom.setDate(from_date)
            self.dlg.onaDateTimeTo.setDate(to_date)

            # set Date ranges
            # self.dlg.onaDateTimeFrom.setMinimumDate(
            #     from_date
            # )
            # self.dlg.onaDateTimeFrom.setMaximumDate(
            #     to_date
            # )
            # self.dlg.onaDateTimeTo.setMinimumDate(
            #     from_date
            # )
            # self.dlg.onaDateTimeTo.setMaximumDate(
            #     to_date
            # )

            # UI updates
            self.dlg.onaDateTimeFrom.setEnabled(True)
            self.dlg.onaDateTimeTo.setEnabled(True)
            self.dlg.onaDateTimeFrom.repaint()
            self.dlg.onaDateTimeTo.repaint()

    def handle_date_and_count_fields_error(self, message):
        self.dlg.app_logs.appendPlainText(f"Error - {message}")
        self.iface.messageBar().pushMessage(
            "Error", f"{message}", level=Qgis.Critical, duration=10
        )

    def handle_ona_data_fetch_progress(self, data):
        if data:
            if isinstance(data, dict):
                page = data.get("curr_page")
                total_pages = data.get("total_pages")
                progress = (
                    (int(page) / int(total_pages)) * 100
                    if int(total_pages) > 1
                    else 100
                )
                self.dlg.onaProgressBar.setValue(math.ceil(progress))
            elif isinstance(data, str):
                self.dlg.app_logs.appendPlainText(data)

    def handle_no_json_data(self, msg):
        self.dlg.app_logs.appendPlainText(f"Warning - {msg}")
        self.iface.messageBar().pushMessage(
            "Notice", msg, level=Qgis.Warning, duration=10
        )
        self.dlg.onaOkButton.setEnabled(True)

    def ona_fetch_data_sync_enabled(self):
        # disable OK button during sync
        self.dlg.onaOkButton.setEnabled(False)
        api_url = self.dlg.onadata_api_url.text()
        formID = self.dlg.comboOnaForms.currentData()
        form_str = self.dlg.comboOnaForms.currentText()
        cleaned_form_str = "_".join(form_str.split(" "))
        username = self.dlg.onadata_username.text()
        password = self.dlg.onaMLineEdit.text()
        page_size = int(self.dlg.onaPageSize.value())

        geo_text = self.dlg.comboOnaGeoFields.currentText()
        geo_field = geo_text.split("-")[0].strip()

        # data count url
        auth = HTTPBasicAuth(username, password)
        url = f"https://{api_url}/api/v1/data/{formID}.json"
        params = dict()

        ona_from_date = self.dlg.onaDateTimeFrom.date()
        ona_to_date = self.dlg.onaDateTimeTo.date()

        from_dt = datetime(
            ona_from_date.year(), ona_from_date.month(), ona_from_date.day(), 0, 0, 0
        )  # 12:00 AM
        to_dt = datetime(
            ona_to_date.year(), ona_to_date.month(), ona_to_date.day(), 23, 59, 59
        )

        ona_from_timestamp = from_dt.strftime("%Y-%m-%dT%H:%M:%S")
        ona_to_timestamp = to_dt.strftime("%Y-%m-%dT%H:%M:%S")

        if ona_from_timestamp and ona_to_timestamp:
            params["query"] = json.dumps(
                {
                    "_submission_time": {
                        "$gte": ona_from_timestamp,
                        "$lte": ona_to_timestamp,
                    }
                }
            )

        if formID:
            if hasattr(self, "vlayers"):
                if self.vlayers.get(f"{cleaned_form_str}_{geo_field}"):
                    self.vlayers[f"{cleaned_form_str}_{geo_field}"] = {
                        "syncData": True,
                        "vlayer": self.vlayers.get(
                            f"{cleaned_form_str}_{geo_field}"
                        ).get("vlayer"),
                    }

            self.ona_worker = OnaRequestThread(
                url,
                auth,
                params,
                headers=None,
                total_records=self.data_count,
                records_per_page=page_size,
                formID=formID,
            )

            # Connect signals to the handler methods
            self.ona_worker.data_fetched.connect(self.handle_data_fetched)
            self.ona_worker.progress_updated.connect(
                self.handle_ona_data_fetch_progress
            )
            self.ona_worker.count_and_date_fields_fetched.connect(
                self.handle_date_and_count_fields
            )
            self.ona_worker.no_data.connect(self.handle_no_json_data)
            self.ona_worker.count_and_date_fields_error_occurred.connect(
                self.handle_date_and_count_fields_error
            )
            self.ona_worker.error_occurred.connect(self.handle_fetch_error)
            self.ona_worker.start()

    def fetch_button_clicked(self):
        """Handles the Fetch button click event."""
        # Extract parameters from the dialog

        # disable button
        self.dlg.onaOkButton.setEnabled(False)
        self.dlg.onaProgressBar.setValue(0)
        self.dlg.onaOkButton.repaint()

        api_url = self.dlg.onadata_api_url.text()
        formID = self.dlg.comboOnaForms.currentData()
        username = self.dlg.onadata_username.text()
        password = self.dlg.onaMLineEdit.text()
        geo_text = self.dlg.comboOnaGeoFields.currentText()
        geo_field = geo_text.split("-")[0].strip()
        ona_sync_interval = int(self.dlg.onaSyncInterval.value())
        page_size = int(self.dlg.onaPageSize.value())
        directory = True

        # data count url
        auth = HTTPBasicAuth(username, password)
        url = f"https://{api_url}/api/v1/data/{formID}.json"
        params = dict()

        ona_from_date = self.dlg.onaDateTimeFrom.date()
        ona_to_date = self.dlg.onaDateTimeTo.date()

        from_dt = datetime(
            ona_from_date.year(), ona_from_date.month(), ona_from_date.day(), 0, 0, 0
        )  # 12:00 AM
        to_dt = datetime(
            ona_to_date.year(), ona_to_date.month(), ona_to_date.day(), 23, 59, 59
        )

        ona_from_timestamp = from_dt.strftime("%Y-%m-%dT%H:%M:%S")
        ona_to_timestamp = to_dt.strftime("%Y-%m-%dT%H:%M:%S")

        if ona_from_timestamp and ona_to_timestamp:
            params["query"] = json.dumps(
                {
                    "_submission_time": {
                        "$gte": ona_from_timestamp,
                        "$lte": ona_to_timestamp,
                    }
                }
            )

        self.ona_worker = OnaRequestThread(
            url,
            auth,
            params,
            headers=None,
            total_records=self.data_count,
            records_per_page=page_size,
            formID=formID,
        )

        # Connect signals to the handler methods
        self.ona_worker.data_fetched.connect(self.handle_data_fetched)
        self.ona_worker.progress_updated.connect(self.handle_ona_data_fetch_progress)
        self.ona_worker.count_and_date_fields_fetched.connect(
            self.handle_date_and_count_fields
        )
        self.ona_worker.no_data.connect(self.handle_no_json_data)
        self.ona_worker.count_and_date_fields_error_occurred.connect(
            self.handle_date_and_count_fields_error
        )
        self.ona_worker.error_occurred.connect(self.handle_fetch_error)
        self.ona_worker.start()

        if ona_sync_interval > 0:
            self.ona_sync_timer.start(ona_sync_interval * 1000)

    def fetch_with_retries(
        self,
        url,
        auth=None,
        params=None,
        headers=None,
        max_retries=5,
        backoff_factor=0.2,
    ):
        with requests.Session() as session:  # Use a session
            if auth:
                session.auth = auth  # Set Basic Auth for the session

            # set headers
            if headers:
                session.headers.update(headers)

            for attempt in range(max_retries):
                try:
                    if params:
                        response = session.get(url, params=params, stream=True)
                    else:
                        response = session.get(url, stream=True)

                    if response.status_code == 404:
                        return response  # Return if the resource is not found
                    # response.raise_for_status()  # Raise HTTPError for bad responses (4xx and 5xx)
                    return response  # Successful request

                except (
                    requests.RequestException,
                    requests.ConnectionError,
                    requests.ConnectTimeout,
                    requests.ReadTimeout,
                ) as e:
                    print(f"Attempt {attempt + 1} failed: {e}")
                    if attempt < max_retries - 1:
                        time.sleep(backoff_factor * (2**attempt))  # Exponential backoff
                    else:
                        self.dlg.app_logs.appendPlainText(f"Failed to fetch data")
                        self.iface.messageBar().pushMessage(
                            "Error", f"{e}", level=Qgis.Critical, duration=10
                        )
                        self.dlg.accept()

    def retrieve_all_geofields(self, fields):
        for field in fields:
            if field.get("children"):
                self.retrieve_all_geofields(field.get("children"))
            else:
                if field.get("type") in self.geo_types:
                    cleaned_geo_field_name = field.get("name", "").strip()
                    cleaned_geo_field_label = ""
                    if isinstance(field.get("label", ""), dict):
                        labels = field.get("label", "")
                        cleaned_geo_field_label = (
                            labels.get("English (en)", "").strip()
                            or labels.get("English", "").strip()
                        )
                    elif isinstance(field.get("label", ""), str):
                        cleaned_geo_field_label = field.get("label", "").strip()
                    self.geo_fields.add(cleaned_geo_field_name)
                    if not self.geo_fields_dict.get(cleaned_geo_field_name):
                        self.geo_fields_dict[cleaned_geo_field_name] = (
                            cleaned_geo_field_label
                        )
        return self.geo_fields

    def fetch_time_fields(self, api_url, username, password, formID):
        auth = HTTPBasicAuth(username, password=password)
        url = f"https://{api_url}/api/v1/forms/{formID}.json"
        self.dlg.app_logs.appendPlainText(f"Fetching Form Metadata... \n")
        resp = self.fetch_with_retries(url, auth)
        if resp.status_code == 200:
            self.dlg.app_logs.appendPlainText(f"Done")
            data = resp.json()
            get_from_date = data.get("date_created")
            get_to_date = data.get("last_submission_time") or data.get("date_modified")

            from_dt = datetime.fromisoformat(get_from_date).astimezone(timezone.utc)
            to_dt = datetime.fromisoformat(get_to_date).astimezone(timezone.utc)

            from_dt = from_dt.replace(
                hour=0, minute=0, second=0, microsecond=0
            )  # 12:00 AM
            to_dt = to_dt.replace(hour=23, minute=59, second=59, microsecond=0)

            from_date = QDate(from_dt.year, from_dt.month, from_dt.day)
            to_date = QDate(to_dt.year, to_dt.month, to_dt.day)

            self.dlg.onaDateTimeFrom.setDate(from_date)
            self.dlg.onaDateTimeTo.setDate(to_date)
            # self.dlg.mDateTimeEditFrom.setDateTimeRange(
            #     datetime.fromisoformat(self.from_date), datetime.fromisoformat(self.to_date))
            # self.dlg.mDateTimeEditTo.setDateTimeRange(
            #     datetime.fromisoformat(self.from_date), datetime.fromisoformat(self.to_date))
            self.dlg.onaDateTimeFrom.setEnabled(True)
            self.dlg.onaDateTimeTo.setEnabled(True)
            self.dlg.onaDateTimeFrom.repaint()
            self.dlg.onaDateTimeTo.repaint()

    def fetchDataCount(self, api_url, username, password, formID):
        auth = HTTPBasicAuth(username, password=password)
        url = f"https://{api_url}/api/v1/forms/{formID}.json"
        self.dlg.app_logs.appendPlainText(f"Fetching Submissions Count...")
        resp = self.fetch_with_retries(url, auth)
        if resp.status_code == 200:
            data = resp.json()
            count = data.get("num_of_submissions")
            self.dlg.app_logs.appendPlainText(
                f"Done. Number Of Submissions - {count} \n"
            )
            return count

    def fetchGeoFields(self, api_url, username, password, formID):
        auth = HTTPBasicAuth(username, password)
        url = f"https://{api_url}/api/v1/forms/{formID}/versions"
        self.dlg.app_logs.appendPlainText(f"Fetching Form Versions...")
        resp = self.fetch_with_retries(url, auth)
        if resp.status_code == 200:
            versions = resp.json()
            self.dlg.app_logs.appendPlainText(
                f"Done. Number of form versions - {len(versions)}"
            )
            if versions:
                for v in versions:
                    version_str = v.get("version")
                    version_url = f"https://{api_url}/api/v1/forms/{formID}/versions/{version_str}"
                    self.dlg.app_logs.appendPlainText(
                        f"Fetching Form Schema for {version_str}..."
                    )
                    res = self.fetch_with_retries(version_url, auth)
                    if res.status_code == 200:
                        self.dlg.app_logs.appendPlainText(f"Done \n")
                        the_v = json.dumps(res.json())
                        fields = json.loads(the_v).get("children")
                        self.geo_fields = self.retrieve_all_geofields(fields)
                if self.geo_fields:
                    for i, gf in enumerate(self.geo_fields):
                        cleaned_gf = gf.strip()
                        self.dlg.comboOnaGeoFields.addItem(cleaned_gf)
                        geo_label = self.geo_fields_dict[cleaned_gf]
                        if geo_label:
                            self.dlg.comboOnaGeoFields.setItemText(
                                i, f"{cleaned_gf} - ({geo_label})"
                            )
                    self.dlg.onaOkButton.setEnabled(True)

                self.dlg.app_logs.appendPlainText(
                    f"Number of Geo Fields Found: {len(self.geo_fields)} \n"
                )
                self.dlg.comboOnaGeoFields.setEnabled(True)
        else:
            self.iface.messageBar().pushMessage(
                "Notice",
                f"Failed to fetch form versions: Status code {resp.status_code}",
                level=Qgis.Warning,
            )
        return

    def fetchFormFields(self, api_url, username, password, formID):
        auth = HTTPBasicAuth(username, password=password)
        url = f"https://{api_url}/api/v1/forms/{formID}/form.json"
        # clear any initial logs
        self.dlg.app_logs.clear()
        self.dlg.app_logs.appendPlainText(f"Fetching Form Schema...")
        resp = self.fetch_with_retries(url)
        if resp.status_code == 200:
            self.dlg.app_logs.appendPlainText(f"Done \n")
            data = resp.json()
            # get columns
            children = data.get("children")
            field_props = ",".join(
                map(
                    str,
                    [
                        c.get("name")
                        for c in children
                        if c.get("name") and c.get("type") not in self.excluded_types
                    ],
                )
            )
            return field_props

    def dataFetch(
        self, base_url, username, password, form_id, geo_field, fields, page, page_size
    ):
        auth = HTTPBasicAuth(username, password=password)
        url = f"https://{base_url}/api/v1/data/{form_id}.json"
        params = {"page": page, "page_size": page_size}
        if self.from_date and self.to_date:
            params["query"] = json.dumps(
                {"_submission_time": {"$gte": self.from_date, "$lte": self.to_date}}
            )
        self.dlg.app_logs.appendPlainText(f"Fetching Page {page} of geojson data...")
        response = self.fetch_with_retries(url, auth, params=params)
        if response.status_code == 200:
            # Calculate progress
            self.dlg.app_logs.appendPlainText(f"Done \n")
            total_pages = math.ceil((int(self.data_count) / int(page_size)))
            progress = (page / total_pages) * 100 if total_pages > 1 else 100
            # self.dlg.onaProgressBar.setValue(math.ceil(progress))
        elif response.status_code in [401, 500, 502, 503]:
            self.dlg.app_logs.appendPlainText(
                f"Fetch failed!!! Status Code - {response.status_code}"
            )
        return response

    def flatten_dict(self, data, parent_key="", sep="/"):
        flattened = {}

        for key, value in data.items():
            new_key = f"{parent_key}{sep}{key}" if parent_key else key

            if isinstance(value, dict):
                # Recursively flatten the dictionary
                flattened.update(self.flatten_dict(value, new_key, sep=sep))
            elif isinstance(value, list):
                for i, item in enumerate(value):
                    if isinstance(item, dict):
                        flattened.update(
                            self.flatten_dict(item, f"{new_key}[{i + 1}]", sep=sep)
                        )

                    else:
                        flattened[f"{new_key}[{i + 1}]"] = item
            else:
                flattened[new_key] = value

        return flattened

    def build_feature_collection(self, filtered_datum, geom, feature_collection):
        # determine whether polygon or point
        if geom and geom.__contains__(";") and len(geom.split(";")) > 1:
            coords_arr = geom.split(";")
            # this means that it is a polygon
            # build the correspoing collection
            coodinates = [
                [
                    float(x.strip().split(" ")[1]),
                    float(x.strip().split(" ")[0]),
                ]
                for x in coords_arr
            ]
            poly_feature = {
                "type": "Feature",
                "geometry": {
                    "type": "Polygon",
                    "coordinates": [coodinates],
                },
                "properties": filtered_datum,
            }
            feature_collection["features"].append(poly_feature)
        elif geom and not geom.__contains__(";"):
            # this means its a feature point
            point_arr = geom.strip().split(" ")
            point_feature = {
                "type": "Feature",
                "geometry": {
                    "type": "Point",
                    "coordinates": [
                        float(point_arr[1]),
                        float(point_arr[0]),
                    ],
                },
                "properties": filtered_datum,
            }
            feature_collection["features"].append(point_feature)

    def get_geo_data(self, datum, geom_field, feature_collection):
        field_keys = datum.keys()
        flattened_data = self.flatten_dict(datum)

        if geom_field in field_keys:
            # this means that the geo field is not inside a repeat
            # build the corresponding geometry/feature collection
            geom = datum.get(geom_field, "")
            filtered_datum = {
                key: value for key, value in flattened_data.items() if key != geom_field
            }
            # build feature collection
            self.build_feature_collection(filtered_datum, geom, feature_collection)
        else:
            # flatten the datum
            repeat_geo_arr = []
            flattened_data = self.flatten_dict(datum)
            for k in flattened_data.keys():
                field_arr = k.split("/")
                if geom_field in field_arr:
                    repeat_geo_arr.append(k)
            if repeat_geo_arr:
                for f in repeat_geo_arr:
                    nested_geom = flattened_data.get(f, "")
                    filtered_datum = {
                        key: value
                        for key, value in flattened_data.items()
                        if f in key.split("/")
                    }
                    self.build_feature_collection(
                        flattened_data, nested_geom, feature_collection
                    )

    def getTheGeoJson(
        self,
        api_url,
        username,
        password,
        formID,
        geo_field,
        page_size,
        page,
        response=None,
    ):
        self.dlg.onaOkButton.setEnabled(False)
        if page == 1 and not response:
            self.data_count = self.fetchDataCount(api_url, username, password, formID)
            response = self.dataFetch(
                api_url,
                username,
                password,
                formID,
                geo_field,
                self.fields,
                page,
                page_size,
            )
        if response and response.status_code == 200:
            data = response.json()
            if len(data) == 0:
                self.dlg.app_logs.appendPlainText(
                    "No Data Available For the selected date range"
                )
                self.dlg.onaProgressBar.setValue(0)
                self.dlg.onaOkButton.setEnabled(True)
                # self.dlg.progress_bar.update()
                return

            # features_list = data.get("features")
            # self.features.extend(features_list)
            self.json_data.extend(data)
            page = page + 1
            response = self.dataFetch(
                api_url,
                username,
                password,
                formID,
                geo_field,
                self.fields,
                page,
                page_size,
            )
            self.getTheGeoJson(
                api_url,
                username,
                password,
                formID,
                geo_field,
                page_size,
                page,
                response,
            )
        else:
            if self.json_data:
                feature_collection = {
                    "type": "FeatureCollection",
                    "features": [],
                }
                for datum in self.json_data:
                    self.get_geo_data(datum, geo_field, feature_collection)

                if (
                    feature_collection["features"]
                    and len(feature_collection["features"]) > 0
                ):
                    self.dlg.onaProgressBar.setValue(100)
                    self.load_data_to_qgis(
                        feature_collection,
                        api_url,
                        username,
                        password,
                        formID,
                        page_size,
                        geo_field,
                    )
                    self.dlg.onaOkButton.setEnabled(True)
                else:
                    self.dlg.app_logs.appendPlainText(
                        "The selected geo field doesn't have geo data"
                    )
                    if self.worker:
                        self.worker.stop()
                    self.dlg.onaOkButton.setEnabled(True)
            else:
                self.dlg.app_logs.appendPlainText(
                    "No Data Available For the selected date range"
                )
                return

        return self.json_data
        # with open(f"{formID}_{geo_field}", "w") as outfile:
        #     print("Writting geo data to json file...")
        #     json.dump(feature_collection, outfile)

    def fetch_and_save_data(
        self, api_url, formID, username, password, geo_field, page_size, directory
    ):
        self.getTheGeoJson(
            api_url,
            username,
            password,
            formID,
            geo_field,
            page_size,
            page=1,
            response=None,
        )

    def validate_geojson(self, geojson_data):
        """Validate if the fetched data is a valid GeoJSON."""
        if not isinstance(geojson_data, dict):
            return False

        if geojson_data.get("type") != "FeatureCollection":
            return False

        if not isinstance(geojson_data.get("features"), list):
            return False

        if len(geojson_data.get("features")) == 0:
            return False

        return True

    def geojson_to_wkt(self, geometry):
        geom_type = geometry["type"]
        coords = geometry["coordinates"]

        if geom_type == "Point":
            return f"POINT ({coords[0]} {coords[1]})"

        elif geom_type == "LineString":
            coord_str = ", ".join([f"{x} {y}" for x, y in coords])
            return f"LINESTRING ({coord_str})"

        elif geom_type == "Polygon":
            rings = []
            for ring in coords:
                ring_str = ", ".join([f"{x} {y}" for x, y in ring])
                rings.append(f"({ring_str})")
            return f"POLYGON ({', '.join(rings)})"

        elif geom_type == "MultiPolygon":
            polygons = []
            for polygon in coords:
                rings = []
                for ring in polygon:
                    ring_str = ", ".join([f"{x} {y}" for x, y in ring])
                    rings.append(f"({ring_str})")
                polygons.append(f"({', '.join(rings)})")
            return f"MULTIPOLYGON ({', '.join(polygons)})"
        else:
            raise ValueError(f"Unsupported geometry type: {geom_type}")

    def rename_dhis_row_entries(self, row, metadata_items, org_id, indicator_id):
        new_row = []
        for elem in row:
            if elem == org_id:
                new_row.append(metadata_items.get(org_id).get("name"))
            elif elem == indicator_id:
                new_row.append(metadata_items.get(indicator_id).get("name"))
            else:
                new_row.append(elem)

        return new_row

    def load_dhis_data_to_qgis(
        self, analytics_data, curr_org_unit_id, curr_indicator_id
    ):
        """
        Add DHIS2 analytics data as a non-spatial layer to QGIS.
        """
        if not analytics_data:
            print("No analytics data provided.")
            return

        # Extract headers and rows
        headers = []
        rows = analytics_data.get("rows", [])
        metadata = analytics_data.get("metaData")

        faux_headers = [header["name"] for header in analytics_data.get("headers", [])]
        meta_items = metadata.get("items")
        for faux_h in faux_headers:
            if faux_h == "dx":
                headers.append("Indicator")
            elif faux_h == "value":
                headers.append("Value")
            else:
                headers.append(meta_items.get(faux_h).get("name"))

        # Create a non-spatial memory layer
        indicator_text = self.dlg.comboDhisIndicators.currentText()
        period_text = self.dlg.comboDhisPeriod.currentText()
        layer = QgsVectorLayer("None", f"{indicator_text}_{period_text}", "memory")

        provider = layer.dataProvider()

        # Add fields (columns) based on headers
        fields = [QgsField(header, QVariant.String) for header in headers]
        provider.addAttributes(fields)
        layer.updateFields()

        # Add features (rows)
        features = []
        for row in rows:
            feature = QgsFeature()
            # new_row = self.rename_dhis_row_entries(
            #     row, meta_items, curr_org_unit_id, curr_indicator_id
            # )
            feature.setAttributes(row)
            features.append(feature)

        provider.addFeatures(features)
        layer.updateExtents()

        # Add the layer to the QGIS project
        QgsProject.instance().addMapLayer(layer)

    def load_data_to_qgis(self, geojson_data, formID, geo_field):
        """Load the fetched GeoJSON data into QGIS as a layer."""
        # validate fetched GeoJSON
        if not self.validate_geojson(geojson_data):
            self.iface.messageBar().pushMessage("Invalid GeoJSON data.")
            return  # Stop if the GeoJSON is invalid

        else:
            layer_name = f"{formID}_{geo_field}"
            features = geojson_data["features"]
            chunk_size = 10000  # make this configurable
            # Define the layer with the same geometry type and fields as the GeoJSON data
            feature_type = features[0].get("geometry").get("type")

            # Check for existing layer
            existing_layer = None
            for layer in QgsProject.instance().mapLayers().values():
                if layer.name() == layer_name:
                    existing_layer = True
                    if (
                        self.vlayers
                        and self.vlayers.get(layer_name)
                        and not self.vlayers.get(layer_name).get("syncData")
                        and existing_layer
                    ):
                        self.iface.messageBar().pushMessage(
                            "Notice", "Layer Exists", level=Qgis.Warning, duration=5
                        )

            vlayer = QgsVectorLayer(
                f"{feature_type}?crs=EPSG:4326", f"{layer_name}", "memory"
            )
            pr = vlayer.dataProvider()

            if not vlayer.isValid():
                self.iface.messageBar().pushMessage("Failed to load Layer")
                return
            else:
                if (
                    self.vlayers
                    and self.vlayers.get(layer_name)
                    and self.vlayers.get(layer_name).get("syncData")
                    and existing_layer
                ):
                    self.update_layer_data(layer_name, geojson_data, vlayer)
                elif (
                    not self.vlayers.get(layer_name)
                    or not self.vlayers.get(layer_name).get("syncData")
                ) and not existing_layer:
                    # Start editing to add fields and features
                    vlayer.startEditing()
                    prop_keys = [
                        QgsField(f"{prop}", QVariant.String)
                        for prop in features[0].get("properties").keys()
                    ]
                    pr.addAttributes(prop_keys)  # Add fields as needed
                    vlayer.updateFields()

                    # Add features to the layer
                    for feature in features:
                        new_feature = QgsFeature(vlayer.fields())
                        # Convert GeoJSON geometry to WKT
                        geometry_wkt = self.geojson_to_wkt(feature["geometry"])
                        geometry = QgsGeometry.fromWkt(geometry_wkt)
                        if geometry:
                            new_feature.setGeometry(geometry)
                        new_feature.setAttributes(
                            list(feature.get("properties").values())
                        )  # Adjust based on properties
                        pr.addFeatures([new_feature])

                    # Commit changes and add to project
                    vlayer.commitChanges()

                    QgsProject.instance().addMapLayer(vlayer)
                    canvas = self.iface.mapCanvas()
                    curr_layer = QgsProject.instance().mapLayersByName(f"{layer_name}")[
                        0
                    ]

                    # Set the extent of the canvas to the basemap layer's extent
                    canvas.setExtent(curr_layer.extent())

                    canvas.refresh()

                    self.dlg.app_logs.appendPlainText(
                        f"Layer {layer_name} Added Successfully!"
                    )

                    if not self.vlayers.get(layer_name):
                        self.vlayers[layer_name] = {"syncData": False, "vlayer": vlayer}

    def update_layer_data(self, layer_name, geojson_data, vlayer):
        """Fetch new data and update the existing layer in QGIS."""
        layers = QgsProject.instance().mapLayersByName(layer_name)

        if layers:
            vlayer = layers[0]
            vlayer.startEditing()

            # Collect all unique property keys from the new data
            all_property_keys = set()
            for feature_data in geojson_data["features"]:
                all_property_keys.update(feature_data.get("properties", {}).keys())

            # Ensure layer fields include all property keys
            layer_fields = [field.name() for field in vlayer.fields()]
            for key in all_property_keys:
                if key not in layer_fields:
                    # Add missing fields dynamically
                    vlayer.addAttribute(
                        QgsField(key, QVariant.String)
                    )  # Adjust type if needed
                    layer_fields.append(key)

            vlayer.updateFields()  # Refresh the field structure

            # Delete existing features
            vlayer.deleteFeatures([feature.id() for feature in vlayer.getFeatures()])

            # Load new features
            for feature_data in geojson_data["features"]:
                new_feature = QgsFeature(vlayer.fields())

                # Set geometry
                geometry_wkt = self.geojson_to_wkt(feature_data["geometry"])
                geometry = QgsGeometry.fromWkt(geometry_wkt)
                if geometry:
                    new_feature.setGeometry(geometry)

                # Align attributes with updated fields
                properties = feature_data.get("properties", {})
                attributes = [
                    properties.get(field_name, None) for field_name in layer_fields
                ]
                new_feature.setAttributes(attributes)

                vlayer.addFeature(new_feature)

            # Commit changes and refresh the layer
            vlayer.commitChanges()
            vlayer.triggerRepaint()

            # Zoom to the updated layer
            canvas = self.iface.mapCanvas()

            # Set the extent of the canvas to the layer's extent
            canvas.setExtent(vlayer.extent())

            canvas.refresh()

            self.dlg.app_logs.appendPlainText(
                f"Layer {layer_name} Updated Successfully!"
            )
        else:
            self.iface.messageBar().pushMessage(
                "Notice",
                f"No Available Layers to be Updated",
                level=Qgis.Warning,
                duration=10,
            )

    # stop workers if they are running
    def stop_workers(self):
        if hasattr(self, "ona_worker") and self.ona_worker.isRunning():
            self.ona_worker.quit()

        if (
            hasattr(self, "fetch_ona_forms_worker")
            and self.fetch_ona_forms_worker.isRunning()
        ):
            self.fetch_ona_forms_worker.quit()

        if (
            hasattr(self, "fetch_ona_geo_fields_worker")
            and self.fetch_ona_geo_fields_worker.isRunning()
        ):
            self.fetch_ona_geo_fields_worker.quit()

    def reset_inputs(self):
        """Reset all input fields in the dialog."""
        # self.dlg.api_url.setText("")
        self.features = []
        self.new_features = []
        self.fields = None
        # self.dlg.form_id.clear()
        self.dlg.app_logs.clear()
        self.dlg.onaProgressBar.setValue(0)
        if self.ona_sync_timer.isActive():
            self.ona_sync_timer.stop()

        self.stop_workers()

        self.dlg.btnFetchOnaForms.setText("Connect")
        self.dlg.onaOkButton.setEnabled(True)
        self.dlg.btnFetchOnaForms.setEnabled(True)

        self.dlg.comboOnaForms.setEnabled(True)

    def reset_odk_inputs(self):
        if self.odk_sync_timer.isActive():
            self.odk_sync_timer.stop()

        self.dlg.app_logs.clear()
        self.dlg.odkProgressBar.setValue(0)

        self.dlg.odkOkButton.setEnabled(True)
        self.dlg.btnFetchODKForms.setEnabled(True)
        self.dlg.comboODKForms.setEnabled(True)

    def reset_kobo_inputs(self):
        if self.kobo_sync_timer.isActive():
            self.kobo_sync_timer.stop()

        self.dlg.app_logs.clear()
        self.dlg.koboPorgressBar.setValue(0)

        self.dlg.koboOkButton.setEnabled(True)
        self.dlg.btnFetchKoboForms.setEnabled(True)
        self.dlg.comboKoboForms.setEnabled(True)

    def reset_gts_inputs(self):
        self.fetch_gts_tracking_rounds_data_handler()
