# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ZoomToPostcode
                                 A QGIS plugin
 Zoom the map extent to a UK postcode
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-03-16
        git sha              : $Format:%H$
        copyright            : (C) 2018 by Matthew Walsh
        email                : walsh.gis@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import pickle
import ssl
import tempfile
import xml.etree.ElementTree as ET
from datetime import datetime
from os import path, makedirs
from zipfile import ZipFile

import urllib3
from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication, Qt
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QLineEdit, QCompleter, QMessageBox, QProgressBar
from qgis.core import QgsCoordinateReferenceSystem, QgsCoordinateTransform, QgsRectangle, QgsProject, QgsPointXY
from qgis.gui import QgsVertexMarker

from .zoom_to_postcode_dialog import ZoomToPostcodeDialog
from .zoomtopostcodedialog import LicenceDialog

ssl._create_default_https_context = ssl._create_unverified_context


class ZoomToPostcode:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        self.canvas = iface.mapCanvas()

        # initialize plugin directory
        self.plugin_dir = path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = path.join(
            self.plugin_dir,
            'i18n',
            'ZoomToPostcode_{}.qm'.format(locale))

        if path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Create the dialog (after translation) and keep reference
        self.dlg = ZoomToPostcodeDialog()

        # Create licence dlg
        self.licence_dlg = LicenceDialog()

        # Create various class references
        self.marker = None
        self.completer = None

        self.previous_searches = []

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Zoom To Postcode')
        self.toolbar = self.iface.addToolBar(u'ZoomToPostcode')
        self.toolbar.setObjectName(u'ZoomToPostcode')

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ZoomToPostcode', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar."""

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/zoom_to_postcode/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Zoom to postcode'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # Configure toolbar widget
        self.toolbar = self.iface.addToolBar("Zoom To Postcode Toolbar")
        self.toolbar.setObjectName("Zoom To Postcode Toolbar")
        self.toolbar_search = QLineEdit()
        self.toolbar_search.setMaximumWidth(100)
        self.toolbar_search.setAlignment(Qt.AlignLeft)
        self.toolbar_search.setPlaceholderText("Enter postcode...")
        self.toolbar.addWidget(self.toolbar_search)
        self.toolbar_search.returnPressed.connect(self.check_pkl)

        self.search_btn = QAction(QIcon(":/plugins/zoomtopostcode/zoomicon.png"), "Search", self.iface.mainWindow())
        self.search_btn.triggered.connect(self.check_pkl)
        self.toolbar.addActions([self.search_btn])

        # Create action that will start plugin configuration
        self.action = QAction(QIcon(":/plugins/zoomtopostcode/zoomicon.png"), u"Zoom to Postcode",
                              self.iface.mainWindow())
        self.action.triggered.connect(self.toolbar.show)

        self.licence = QAction(u"OS Licence", self.iface.mainWindow())
        self.licence.triggered.connect(self.licence_dlg.show)

        # Add toolbar button and menu item
        self.iface.addPluginToMenu(u"&Zoom to Postcode", self.action)
        self.iface.addPluginToMenu(u"&Zoom to Postcode", self.licence)

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Zoom To Postcode'),
                action)
            self.iface.removeToolBarIcon(action)

        # remove the toolbar
        del self.toolbar

    def run(self):
        """Show the toolbar dialog."""
        self.dlg.show()

    def search_completer(self):
        self.completer = QCompleter(self.previous_searches, self.iface.mainWindow())
        self.completer.setCompletionMode(QCompleter.PopupCompletion)
        self.completer.setCaseSensitivity(Qt.CaseInsensitive)

    def check_crs(self):
        # Check if a transformation needs to take place
        srs = self.canvas.mapSettings().destinationCrs()
        current_crs = srs.authid()
        return current_crs

    def transform(self, cor):
        # Transforms point from british nation grid to map crs
        srs = self.canvas.mapSettings().destinationCrs()
        crs_src = QgsCoordinateReferenceSystem(27700)
        crs_dest = QgsCoordinateReferenceSystem(srs)
        xform = QgsCoordinateTransform(crs_src, crs_dest, QgsProject.instance())
        x = float(cor[0])
        y = float(cor[1])
        t_point = xform.transform(x, y)
        return t_point

    def check_pkl(self):
        # Check the Pickle postcode dir exists
        checkpkl = path.isdir(path.join(self.plugin_dir, 'UK_Postcodes'))
        if checkpkl:
            xml_path = path.join(self.plugin_dir, r'UK_Postcodes/metadata.xml')
            check_xml = path.isfile(xml_path)
            if check_xml:
                check_currency = self.check_pcode_date(xml_path)
                if check_currency:
                    self.postcode_dict()
                else:
                    self.update_pcode_option()
            else:
                self.update_pcode_option()
        else:
            msg = "Postcode files must be downloaded to use this plugin, do you wish to continue?"
            goahead = QMessageBox.question(self.iface.mainWindow(), "Download Message", msg, QMessageBox.Yes,
                                           QMessageBox.No)
            if goahead == QMessageBox.Yes:
                self.download_pkl()
            else:
                pass

    def check_pcode_date(self, xml_path):
        """Parses metadata xml to check currency of pcodes.

        :param xml_path:
        :return:
        """
        try:
            url = "http://qgis.locationcentre.co.uk/ZoomToPostcode_medata.xml"
            http = urllib3.PoolManager()
            r = http.request('GET', url)

            xml_as_string = ET.fromstring(r.data.decode('utf-8'))
            tree_web = ET.ElementTree(xml_as_string)
            root_web = tree_web.getroot()
            current_version = ""
            for child in root_web:
                if child.tag == "pcode_date":
                    current_version = child.text
            tree_plugin = ET.parse(xml_path)
            root_plugin = tree_plugin.getroot()
            last_update = ""
            for child in root_plugin:
                if child.tag == "pcode_date":
                    last_update = child.text
            last_up_datetime = datetime.strptime(last_update, '%Y-%m-%d')
            curr_ver_datetime = datetime.strptime(current_version, '%Y-%m-%d')
            if last_up_datetime.date() >= curr_ver_datetime.date():
                # Return True for up-to-date postcodes
                return True
            else:
                # False requires download to update
                return False

        except urllib3.exceptions.MaxRetryError:
            QMessageBox.information(self.iface.mainWindow(),
                                    "HTTP Error",
                                    "Unable to download file")

    def update_pcode_option(self):
        """Provide option to update postcodes.

        :return: None
        """
        msg = "Updated postcode files are available, do you wish to download?"
        goahead = QMessageBox.question(self.iface.mainWindow(), "Download Message", msg, QMessageBox.Yes,
                                       QMessageBox.No)
        if goahead == QMessageBox.Yes:
            self.download_pkl()
        if goahead == QMessageBox.No:
            self.postcode_dict()

    def download_pkl(self):
        """Download the Pickle postcode file to the plugin dir.

        :return: None
        """
        pcode_path = path.join(path.dirname(__file__), 'UK_Postcodes')
        if not path.exists(pcode_path):
            makedirs(pcode_path)

        try:
            url = "http://qgis.locationcentre.co.uk/UK_Postcodes.zip"
            http = urllib3.PoolManager()
            response = http.request('GET', url, preload_content=False)
            content_length = response.headers['Content-Length']
            total_size = int(content_length)
            downloaded = 0
            CHUNK = 256 * 10240
            dlbar = QProgressBar()
            dlbar.setMinimum(0)
            dlbar.setMaximum(total_size)
            zip_temp = tempfile.NamedTemporaryFile(mode='w+b', suffix='.zip', delete=False)
            zip_temp_n = zip_temp.name
            zip_temp.seek(0)

            with open(zip_temp_n, 'wb') as fp:
                while True:
                    dlbar.show()
                    chunk = response.read(CHUNK)
                    downloaded += len(chunk)
                    dlbar.setValue(downloaded)
                    if not chunk:
                        break
                    fp.write(chunk)
            response.release_conn()

            pcode_zip = ZipFile(zip_temp)
            pcode_zip.extractall(pcode_path)
            zip_temp.close()
            self.check_pkl()

        except urllib3.exceptions.MaxRetryError:
            QMessageBox.information(self.iface.mainWindow(),
                                    "HTTP Error",
                                    "Unable to download file")

    def postcode_dict(self):
        """Create dictionary of postcodes from correct Pickle file.

        :return: None
        """
        try:
            input_pcode = self.toolbar_search.text().replace(' ', '')
            if input_pcode[1].isdigit():
                find_pkl = str(r"UK_Postcodes/" + input_pcode[:1] + ".pkl")
            else:
                find_pkl = str(r"UK_Postcodes/" + input_pcode[:2] + ".pkl")
            pklfile = open(path.join(path.dirname(__file__), find_pkl), 'rb')
            pcode_dict = pickle.load(pklfile)
            pklfile.close()
            if input_pcode.upper() not in self.previous_searches:
                self.previous_searches.append(input_pcode.upper())
                self.search_completer()
                self.toolbar_search.setCompleter(self.completer)
            self.zoomto(pcode_dict)
        except (KeyError, IOError):
            QMessageBox.information(self.iface.mainWindow(),
                                    "Invalid postcode",
                                    "The postcode you entered was not found")
        except IndexError:
            pass

    def zoomto(self, pcode_dict):
        """Find the coordinates for postcode in the dictionary.

        :param pcode_dict:
        :return: None
        """
        current_crs = self.check_crs()
        input_pc = self.toolbar_search.text()
        input_pc_fmt = str(input_pc).replace(' ', '').upper()
        coords = pcode_dict[input_pc_fmt]
        x, y = coords
        if current_crs != "EPSG:27700":
            cor = (x, y)
            point = self.transform(cor)
            self.update_canvas(point)
        else:
            point = (x, y)
            self.update_canvas(point)

    def update_canvas(self, point):
        """Update the canvas and add vertex marker.

        :param point:
        :return: None
        """
        x, y = point
        scale = 120
        rect = QgsRectangle(float(x) - scale, float(y) - scale, float(x) + scale, float(y) + scale)
        self.canvas.setExtent(rect)
        self.marker = QgsVertexMarker(self.canvas)
        self.marker.setIconSize(15)
        self.marker.setPenWidth(2)
        self.marker.setCenter(QgsPointXY(int(x), int(y)))
        self.canvas.refresh()
        self.canvas.extentsChanged.connect(self.remove_marker)

    def remove_marker(self):
        """Remove vertex marker.

        :return: None
        """
        self.marker.hide()
        self.canvas.scene().removeItem(self.marker)
        self.canvas.extentsChanged.disconnect(self.remove_marker)
