import os
from qgis.PyQt.QtWidgets import QDialog, QMessageBox, QAction, QListWidgetItem
from qgis.core import (
    QgsProject,
    QgsPointXY,
    QgsGeometry,
    QgsFeature,
    QgsVectorLayer,
    QgsField,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsCoordinateTransformContext,
    QgsPalLayerSettings,
    QgsVectorLayerSimpleLabeling,
    QgsTextFormat,
    QgsTextBufferSettings,
    QgsSymbol,
    QgsSingleSymbolRenderer,
    QgsMarkerSymbol,
)
from qgis.PyQt.QtCore import QVariant, Qt
from .goto_xy_dialog import GoToXYDialog


class GoToXY:
    def __init__(self, iface):
        self.iface = iface
        self.canvas = iface.mapCanvas()
        self.plugin_dir = os.path.dirname(__file__)
        self.dlg = GoToXYDialog()
        self.coordinate_history = []

        # Connect UI
        self.dlg.btnZoom.clicked.connect(self.zoom_to_xy)
        self.dlg.btnAddPoint.clicked.connect(self.add_point)
        self.dlg.btnToggleLabels.clicked.connect(self.toggle_labels)
        self.dlg.comboLayer.currentIndexChanged.connect(self.update_crs_from_layer)
        self.dlg.crsComboBox.currentIndexChanged.connect(self.update_crs_label)
        self.dlg.listHistory.itemDoubleClicked.connect(self.navigate_to_history_item)

    def initGui(self):
        self.action = QAction("Go To XY", self.iface.mainWindow())
        self.action.triggered.connect(self.run)
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&Go To XY", self.action)

    def unload(self):
        self.iface.removePluginMenu("&Go To XY", self.action)
        self.iface.removeToolBarIcon(self.action)

    def run(self):
        self.populate_crs_combobox()
        self.populate_layer_combobox()
        self.update_crs_label()
        self.update_history_list()
        
        # Ensure dialog is properly sized
        self.dlg.resize(500, 550)
        
        # Initialize toggle button state
        self.dlg.btnToggleLabels.setText("Show Point Labels")
        self.dlg.btnToggleLabels.setChecked(False)
        
        self.dlg.show()
        self.dlg.raise_()
        self.dlg.activateWindow()

    def populate_crs_combobox(self):
        self.dlg.crsComboBox.clear()
        crs_list = ['EPSG:4326', 'EPSG:3857']

        for crs_authid in crs_list:
            crs = QgsCoordinateReferenceSystem(crs_authid)
            self.dlg.crsComboBox.addItem(f"{crs.description()} ({crs.authid()})", crs)

        # Add project CRS at the top
        project_crs = QgsProject.instance().crs()
        self.dlg.crsComboBox.insertItem(0, f"Project CRS: {project_crs.description()} ({project_crs.authid()})", project_crs)
        self.dlg.crsComboBox.setCurrentIndex(0)

    def populate_layer_combobox(self):
        self.dlg.comboLayer.clear()
        self.dlg.comboLayer.addItem("-- Select a layer --", None)
        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            if layer.type() == layer.VectorLayer:
                layer_crs = layer.crs().authid()
                self.dlg.comboLayer.addItem(f"{layer.name()} – {layer_crs}", layer)

    def update_crs_from_layer(self):
        index = self.dlg.comboLayer.currentIndex()
        if index > 0:  # Skip the first "-- Select a layer --" item
            layer = self.dlg.comboLayer.itemData(index)
            if layer:
                layer_crs = layer.crs()
                # Search and select CRS in crsComboBox
                for i in range(self.dlg.crsComboBox.count()):
                    crs_in_box = self.dlg.crsComboBox.itemData(i)
                    if crs_in_box.authid() == layer_crs.authid():
                        self.dlg.crsComboBox.setCurrentIndex(i)
                        break

    def update_crs_label(self):
        index = self.dlg.crsComboBox.currentIndex()
        if index >= 0:
            selected_crs = self.dlg.crsComboBox.itemData(index)
            if selected_crs:
                self.dlg.lblSelectedCRS.setText(
                    f"Coordinates will be interpreted using {selected_crs.authid()} - {selected_crs.description()}"
                )

    def get_input_point(self):
        try:
            x_text = self.dlg.lineEditX.text().strip()
            y_text = self.dlg.lineEditY.text().strip()
            
            if not x_text or not y_text:
                QMessageBox.warning(self.dlg, "Missing Input", "Please enter both X and Y coordinates.")
                return None
                
            x = float(x_text)
            y = float(y_text)
            return QgsPointXY(x, y)
        except ValueError:
            QMessageBox.critical(self.dlg, "Invalid Input", "Please enter valid numbers for X and Y coordinates.")
            return None

    def get_selected_crs(self):
        index = self.dlg.crsComboBox.currentIndex()
        if index >= 0:
            return self.dlg.crsComboBox.itemData(index)
        return QgsProject.instance().crs()

    def transform_point_to_project_crs(self, point, source_crs):
        dest_crs = QgsProject.instance().crs()
        if source_crs.authid() == dest_crs.authid():
            return point  # No transformation needed
        try:
            transform = QgsCoordinateTransform(source_crs, dest_crs, QgsProject.instance())
            return transform.transform(point)
        except Exception as e:
            QMessageBox.critical(self.dlg, "Transformation Error", f"Error transforming coordinates: {str(e)}")
            return None

    def add_to_history(self, point, crs):
        """Add coordinate to history list"""
        history_item = {
            'x': point.x(),
            'y': point.y(),
            'crs': crs.authid(),
            'description': crs.description()
        }
        
        # Remove duplicate if exists
        self.coordinate_history = [item for item in self.coordinate_history 
                                 if not (item['x'] == point.x() and item['y'] == point.y() and item['crs'] == crs.authid())]
        
        # Add to beginning of list
        self.coordinate_history.insert(0, history_item)
        
        # Keep only last 10 items
        self.coordinate_history = self.coordinate_history[:10]
        
        self.update_history_list()

    def update_history_list(self):
        """Update the history list widget"""
        self.dlg.listHistory.clear()
        for item in self.coordinate_history:
            display_text = f"X: {item['x']:.6f}, Y: {item['y']:.6f} ({item['crs']})"
            list_item = QListWidgetItem(display_text)
            list_item.setData(Qt.UserRole, item)
            self.dlg.listHistory.addItem(list_item)

    def navigate_to_history_item(self, item):
        """Navigate to a coordinate from history"""
        data = item.data(Qt.UserRole)
        if data:
            self.dlg.lineEditX.setText(str(data['x']))
            self.dlg.lineEditY.setText(str(data['y']))
            
            # Set the CRS
            for i in range(self.dlg.crsComboBox.count()):
                crs_in_box = self.dlg.crsComboBox.itemData(i)
                if crs_in_box.authid() == data['crs']:
                    self.dlg.crsComboBox.setCurrentIndex(i)
                    break
            
            # Automatically zoom to the location
            self.zoom_to_xy()

    def zoom_to_xy(self):
        point = self.get_input_point()
        if point is None:
            return
            
        source_crs = self.get_selected_crs()
        point_transformed = self.transform_point_to_project_crs(point, source_crs)
        
        if point_transformed:
            self.canvas.setCenter(point_transformed)
            self.canvas.zoomScale(1000)  # Adjust scale if needed
            self.canvas.refresh()
            
            # Add to history
            self.add_to_history(point, source_crs)
            
            QMessageBox.information(self.dlg, "Navigation Complete", 
                                  f"Successfully navigated to coordinates:\nX: {point.x():.6f}, Y: {point.y():.6f}\nCRS: {source_crs.authid()}")

    def add_point(self):
        point = self.get_input_point()
        if point is None:
            return

        source_crs = self.get_selected_crs()
        point_transformed = self.transform_point_to_project_crs(point, source_crs)
        if point_transformed is None:
            return

        layer_name = "GoToXY Points"
        existing_layer = next(
            (l for l in QgsProject.instance().mapLayers().values() if l.name() == layer_name),
            None
        )

        if existing_layer:
            layer = existing_layer
            # Check if the layer has the label field, if not add it
            field_names = [field.name() for field in layer.fields()]
            if 'label' not in field_names:
                layer.dataProvider().addAttributes([QgsField("label", QVariant.String)])
                layer.updateFields()
        else:
            project_crs = QgsProject.instance().crs()
            layer = QgsVectorLayer(f"Point?crs={project_crs.authid()}", layer_name, "memory")
            pr = layer.dataProvider()
            pr.addAttributes([
                QgsField("id", QVariant.Int),
                QgsField("x_original", QVariant.Double),
                QgsField("y_original", QVariant.Double),
                QgsField("crs_original", QVariant.String),
                QgsField("label", QVariant.String)
            ])
            layer.updateFields()
            
            # Set up point styling
            self.setup_point_styling(layer)
            
            QgsProject.instance().addMapLayer(layer)

        # Create label text
        label_text = f"({point.x():.6f}, {point.y():.6f})"
        
        feature = QgsFeature()
        feature.setGeometry(QgsGeometry.fromPointXY(point_transformed))
        
        # Set attributes based on available fields
        field_names = [field.name() for field in layer.fields()]
        attributes = []
        
        if 'id' in field_names:
            attributes.append(layer.featureCount() + 1)
        if 'x_original' in field_names:
            attributes.append(point.x())
        if 'y_original' in field_names:
            attributes.append(point.y())
        if 'crs_original' in field_names:
            attributes.append(source_crs.authid())
        if 'label' in field_names:
            attributes.append(label_text)
            
        feature.setAttributes(attributes)
        layer.dataProvider().addFeatures([feature])
        layer.updateExtents()
        layer.triggerRepaint()

        # Add to history
        self.add_to_history(point, source_crs)

        QMessageBox.information(self.dlg, "Point Added", 
                              f"Point successfully added at coordinates:\nX: {point.x():.6f}, Y: {point.y():.6f}\nCRS: {source_crs.authid()}")

    def toggle_labels(self):
        """Toggle labels on/off for the GoToXY Points layer"""
        layer_name = "GoToXY Points"
        layer = next(
            (l for l in QgsProject.instance().mapLayers().values() if l.name() == layer_name),
            None
        )
        
        if not layer:
            QMessageBox.warning(self.dlg, "No Points Layer", 
                              "No GoToXY Points layer found. Please add some points first.")
            self.dlg.btnToggleLabels.setChecked(False)
            return
        
        if self.dlg.btnToggleLabels.isChecked():
            # Enable labels
            self.setup_point_labeling(layer)
            self.dlg.btnToggleLabels.setText("Hide Point Labels")
            QMessageBox.information(self.dlg, "Labels Enabled", 
                                  "Point labels are now visible.")
        else:
            # Disable labels
            layer.setLabelsEnabled(False)
            layer.triggerRepaint()
            self.dlg.btnToggleLabels.setText("Show Point Labels")
            QMessageBox.information(self.dlg, "Labels Disabled", 
                                  "Point labels are now hidden.")

    def setup_point_styling(self, layer):
        """Set up custom styling for the points"""
        # Create a marker symbol
        symbol = QgsMarkerSymbol.createSimple({
            'name': 'circle',
            'color': '255,0,0',  # Red color
            'size': '4',
            'outline_color': '255,255,255',  # White outline
            'outline_width': '0.5'
        })
        
        # Apply the symbol to the layer
        renderer = QgsSingleSymbolRenderer(symbol)
        layer.setRenderer(renderer)

    def setup_point_labeling(self, layer):
        """Set up labeling for the points"""
        # Create label settings
        label_settings = QgsPalLayerSettings()
        
        # Set the field to use for labeling
        label_settings.fieldName = 'label'
        label_settings.enabled = True
        
        # Set text format
        text_format = QgsTextFormat()
        text_format.setSize(10)
        text_format.setColor(Qt.black)
        
        # Add text buffer (outline)
        buffer_settings = QgsTextBufferSettings()
        buffer_settings.setEnabled(True)
        buffer_settings.setSize(1)
        buffer_settings.setColor(Qt.white)
        text_format.setBuffer(buffer_settings)
        
        label_settings.setFormat(text_format)
        
        # Use simple placement without problematic enums
        try:
            # Try the newer enum first
            label_settings.placement = QgsPalLayerSettings.Placement.AroundPoint
        except:
            try:
                # Fallback to older enum
                label_settings.placement = QgsPalLayerSettings.AroundPoint
            except:
                # Final fallback - use numeric value
                label_settings.placement = 0  # Point placement
        
        # Set offset to position labels above points
        label_settings.yOffset = 3
        label_settings.xOffset = 0
        
        # Apply labeling to layer
        labeling = QgsVectorLayerSimpleLabeling(label_settings)
        layer.setLabelsEnabled(True)
        layer.setLabeling(labeling)

