# -*- coding: utf-8 -*-
"""
/***************************************************************************
 WAGeoData
                                 A QGIS plugin
 The WA Geo Data plugin provides quick access to DMPE publicly available map services.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-12-08
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Jesse Best
        email                : jessebest2@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QUrl, QUrlQuery
from qgis.PyQt.QtWidgets import QAction, QMenu
from qgis.PyQt.QtGui import QIcon, QDesktopServices
from qgis.core import Qgis
from functools import partial

from .resources import *
from .constants import PLUGIN_NAME, ABOUT_ICON, ABOUT_URL
from .helper import Helper
from .json_handler import WMS_LAYERS, EXCLUDED_SERVICES
from .load_layer import *
from .format_name import *

import os
import re

class WAGeoData:
    """QGIS Plugin Implementation."""
    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        self.icon_path = os.path.join(
            os.path.dirname(__file__), ABOUT_ICON
        )

        self.KEY_RE = re.compile(r"([A-Z]+)[\s\-]*?(\d{3})\)?$") # menu sort key regex

    def initGui(self):
        self.menu = QMenu(self.iface.mainWindow())
        self.menu.setObjectName(PLUGIN_NAME)
        self.menu.setTitle(PLUGIN_NAME)

        menu_bar = self.iface.mainWindow().menuBar()
        menu_bar.insertMenu(
            self.iface.firstRightStandardMenu().menuAction(), self.menu
        )
        self._menu_added = True

        self.menu_with_actions = []
        self.helpers = []
        self.services_remaining = len(WMS_LAYERS)

        for service in WMS_LAYERS:
            url = service["URL"]
            qurl = QUrl(url)
            service_format = service["Format"]
            
            query = QUrlQuery()
            query.addQueryItem("service", service_format)
            query.addQueryItem("request", "GetCapabilities")
            qurl.setQuery(query)
            
            full_url = qurl.toString()
            helper = Helper(full_url, service_format)
            helper.finished.connect(partial(self.create_menu,
                            service_name=service["Service"],
                            service_url=url,
                            service_format=service_format))
            helper.error.connect(self.show_error)
            self.helpers.append(helper)
            helper.fetch()
            
    def create_menu(self, layers, service_name, service_url, service_format):
        group_menu = QMenu()
        # check if menu already exists
        if self.menu_with_actions:
            for menu in self.menu_with_actions:
                if menu.title() == service_name:
                    group_menu = menu
                else:
                    group_menu.setTitle(service_name)
        else:
            group_menu.setTitle(service_name)

        for layer in layers:
            layer_exists = False
            layer_name = layer[0].replace("_"," ").strip()
            # check if new layer exists in menu preventing WMS duplicate
            for menu in self.menu_with_actions:
                if menu.title() == service_name:
                    for item in menu.actions():
                        if layer_name[-8:-1] == item.text()[-7:]:
                            layer_exists = True
            
            if not layer_exists and layer_name[-9:] not in EXCLUDED_SERVICES and layer_name[-10:-1] not in EXCLUDED_SERVICES:
                action = QAction(layer_name, self.iface.mainWindow())
                action.triggered.connect(partial(load_layer, layer=layer, service_url=service_url, iface=self.iface, layer_format = service_format))
                group_menu.addAction(action)

        self.menu.addMenu(group_menu)
        self.menu_with_actions.append(group_menu)
        
        self.services_remaining -= 1

        if self.services_remaining == 0:
            self.finish_menu()

    def finish_menu(self):
        if not hasattr(self, "_menu_added"):
            menu_bar = self.iface.mainWindow().menuBar()
            menu_bar.insertMenu(
                self.iface.firstRightStandardMenu().menuAction(), self.menu
            )
            self._menu_added = True

        self.reorder_menu()

        self.create_about_menu()
        if any(h.failed for h in self.helpers):
            self.create_reload_menu()

    def create_about_menu(self):
        self.about_menu = QAction(QIcon(self.icon_path), "About", self.iface.mainWindow())
        self.about_menu.triggered.connect(self.open_about_page)
        self.menu.addSeparator()
        self.menu.addAction(self.about_menu)

    def create_reload_menu(self):
        self.temp_menu = QAction(QIcon(self.icon_path),"Reload plugin", self.iface.mainWindow())
        self.temp_menu.triggered.connect(self.reload_menu)
        self.menu.addAction(self.temp_menu)

    def reorder_menu(self):
        for menu in self.menu_with_actions:
            actions_sorted = sorted(
                menu.actions(), key=self.sort_key
            )
            menu.clear()
            for action in actions_sorted:
                action.setText(format_name(action.text()))
                menu.addAction(action)
    
    def sort_key(self, action):
        text = action.text().upper()
        match = self.KEY_RE.search(text)

        if not match:
            return ("ZZZ", float("inf"))  # deterministic fallback

        prefix = match.group(1)
        number = int(match.group(2))
        return (prefix, number)
        
    def open_about_page(self):
        QDesktopServices.openUrl(QUrl(ABOUT_URL))

    def show_error(self, message):
        self.services_remaining -= 1
        self.iface.messageBar().pushMessage("WA Geo Data", message, level=Qgis.Warning, duration=15)
        if self.services_remaining == 0:
            self.finish_menu()
        
    def unload(self):
        self.clear_menu()

    def reload_menu(self):
        self.clear_menu()       
        self.initGui()

    def clear_menu(self):
        for child in self.menu_with_actions:
            if child:
                child.deleteLater()
        if self.menu:
            self.menu.clear()
            self.menu.deleteLater()
        self.menu_with_actions = []
        self.helpers = []

        if hasattr(self, "_menu_added"):
            delattr(self, "_menu_added")