# -*- coding: utf-8 -*-
"""
/***************************************************************************
 UmeMap
                                 A QGIS plugin
 UmeMap layer managment is a plugin which helps with styles for vector layers that come from a UmeMap server and form management
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-06-10
        git sha              : $Format:%H$
        copyright            : (C) 2022 by UmeMap
        email                : marcus.lindh@umea.se
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMenu
from qgis.PyQt.QtWidgets import QMessageBox

from qgis.PyQt.QtXml import QDomDocument
from dataclasses import dataclass
from typing import Optional, Dict

#from qgis.core import QgsProject
from qgis.core import *
from qgis.utils import iface


# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .UmeMap_dialog import UmeMapDialog
import os.path

import requests
import json
import tempfile

@dataclass
class ApiResponse:
    status: str
    data: Optional[Dict]
    message: str
    code: str

class UmeMap:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'UmeMap_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&UmeMap layer managment')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        QgsProject.instance().layerWasAdded.connect(self.repaintlayer)

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('UmeMap', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToWebMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        #icon_path = ':/plugins/UmeMap/icon.png'
        #self.add_action(
        #    icon_path,
        #    text=self.tr(u'UmeMap'),
        #    callback=self.run,
        #    parent=self.iface.mainWindow())

        my_action = QAction(self.tr("Save Style To UmeMap"), self.iface.mainWindow())
        my_action.triggered.connect(self.SaveStyleToUmeMap)


        # Add the menu item to the context menu
        # menu = QMenu()
        # menu.addAction(my_action)
        self.iface.addCustomActionForLayerType(my_action, "", QgsMapLayerType.VectorLayer, True)

        # will be set False in run()
        self.first_start = True

    def SaveStyleToUmeMap(self):
        print("SaveStyleToUmeMap")

        TempFolder = tempfile.gettempdir()
        currentLayer = self.iface.activeLayer()
        print(currentLayer)
        print(TempFolder)

        WfsUrl, WfsLayerName = self.GetWfs(currentLayer)
        Path = TempFolder + "\\" + WfsLayerName + ".qml"
        print(Path)

        if not self.IsUmeMap(WfsUrl):
            self.show_error_popup(self.tr("Save styles - Error"), self.tr("An error occurred. Could not save styles."))
            return

        currentLayer.saveNamedStyle(Path)

        with open(Path, mode="rb") as xml_file:
            xml_data = xml_file.read()

        url = WfsUrl + "?request=SaveVectorStyle&typename=" + WfsLayerName
        headers = self.GetHeadersFromLayer(currentLayer)
        response = requests.post(url, data=xml_data, headers=headers, verify=False, allow_redirects=False)

        # Handle redirect
        if response.status_code == 302:
            new_url = response.headers['Location']
            response = requests.post(new_url, data=xml_data, headers=headers, verify=False, allow_redirects=False)

        # Handle authentication error
        if response.status_code == 401:
            self.show_error_popup(self.tr("Authentication error"), self.tr("The API key is invalid or missing. Please check your authentication configuration."))
            return

        # Handle success
        if response.status_code == 200:
            try:
                response_data = response.json()
                api_response = ApiResponse(**response_data)

                if api_response.status == "success":
                    iface.messageBar().pushMessage(self.tr("Save"), self.tr("Saved styles"), level=Qgis.Success, duration=5)
                else:
                    self.show_error_popup(self.tr("Save styles - Error"), api_response.message)
            except Exception as e:
                self.show_error_popup(self.tr("Save styles - Error"), f"Error interpreting response: {str(e)}")
        else:
            # Handle other error status codes
            try:
                response_data = response.json()
                api_response = ApiResponse(**response_data)
                self.show_error_popup(self.tr("Save styles - Error"), api_response.message)
            except Exception as e:
                self.show_error_popup(self.tr("Save styles - Error"), f"An error occurred. HTTP status code: {response.status_code}")

    def GetHeadersFromLayer(self, layer):
        """Extract HTTP headers from the layer's authentication configuration"""
        headers = {}

        try:
            from qgis.core import QgsApplication, QgsAuthMethodConfig

            ds = QgsDataSourceUri(layer.dataProvider().dataSourceUri())
            auth_cfg = ds.authConfigId()

            if not auth_cfg:
                return headers

            auth_manager = QgsApplication.authManager()

            if not auth_manager:
                return headers

            # Get the authentication configuration
            auth_config = QgsAuthMethodConfig()
            auth_manager.loadAuthenticationConfig(auth_cfg, auth_config, True)

            # Extract all headers from the configuration map
            config_map = auth_config.configMap()
            for key, value in config_map.items():
                headers[key] = value

        except Exception as e:
            self.show_error_popup("Extrahera headers", f"Error extracting headers: {str(e)}")

        return headers


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginWebMenu(
                self.tr(u'&UmeMap layer managment'),
                action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = UmeMapDialog()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass

    def IsUmeMap(self, WfsUrl):
        """Check If WFS server is UmeMap"""
        try:
            resp = requests.get(WfsUrl + "?request=ServerInfo",verify=False)
            serverInfo = resp.json()
            return serverInfo.get("softwareName") == "UmeMap"
        except:
            return False


    def GetWfs(self, layer):
        ds = QgsDataSourceUri(layer.dataProvider().dataSourceUri())
        url = str(ds.encodedUri(), 'UTF-8')
        WfsUrl = ""
        WfsLayerName = ""

        for value in url.split("&"):
            vals = value.split("=")

            if vals[0] == "url":
                WfsUrl = vals[1]
            if vals[0] == "typename":
                WfsLayerName = vals[1]

        WfsUrlParts = WfsUrl.split("?")
        WfsUrl = WfsUrlParts[0]

        return WfsUrl, WfsLayerName

    def log(self, message: str, level=Qgis.Info):
        """Writes log messages to both the console and the QGIS log."""
        try:
            print(message)
            QgsMessageLog.logMessage(message, "UmeMap", level)
        except Exception:
            # Fallback om loggningssystemet inte är tillgängligt
            print(f"[LOGG-ERROR] {message}")

    def repaintlayer(self, layer):
        """Applies UmeMap style only if the layer does not already have a locally saved style in the project file."""

        try:
            if not layer:
                return

            # Read data source
            data_source = layer.dataProvider().dataSourceUri()
            ds = QgsDataSourceUri(data_source)
            url = str(ds.encodedUri(), 'UTF-8')

            # Extract URL and layer name
            WfsUrl, WfsLayerName = "", ""
            for value in url.split("&"):
                vals = value.split("=")
                if vals[0] == "url":
                    WfsUrl = vals[1]
                elif vals[0] == "typename":
                    WfsLayerName = vals[1]

            if not WfsUrl or not WfsLayerName:
                return

            # Remove possible query parameters
            WfsUrl = WfsUrl.split("?")[0]

            # Check if it is really a UmeMap-server
            if not self.IsUmeMap(WfsUrl):
                return

            # Skip if style already has been applied before
            if layer.customProperty("umemap_style_applied", False):
                return

            # If style is from project file, skip
            style_uri = layer.styleURI()
            if style_uri and style_uri.startswith("project:"):
                self.log(f"Local style used for '{layer.name()}'.", Qgis.Info)
                return

            # ----- Get and apply style from UmeMap -----
            res = requests.get(f"{WfsUrl}?REQUEST=GetVectorStyle&TYPENAME={WfsLayerName}", verify=False)
            if res.status_code != 200:
                return

            style_doc = QDomDocument("qgis")
            if not style_doc.setContent(res.content):
                return

            import_result = layer.importNamedStyle(style_doc)
            if import_result[0]:
                layer.setCustomProperty("umemap_style_applied", True)
                layer.triggerRepaint()
                self.log(f"UmeMap style applied on '{layer.name()}'.", Qgis.Info)
            else:
                self.log(f"Could not import style to '{layer.name()}'.", Qgis.Warning)

        except Exception as e:
            self.log(f"[ERROR] repaintlayer failed for '{layer.name()}': {e}", Qgis.Critical)


    def show_error_popup(self, title: str, message: str):
        msg_box = QMessageBox()
        msg_box.setIcon(QMessageBox.Critical)
        msg_box.setWindowTitle(title)
        msg_box.setText(message)
        msg_box.exec_()
