# -*- coding: utf-8 -*-
"""
/***************************************************************************
 TurnGray
                                 A QGIS plugin
 Turn black items in composer into gray (or any other color).
                              -------------------
        begin                : 2017-02-26
        git sha              : $Format:%H$
        copyright            : (C) 2017 by Raymond Nijssen / Terglobo
        email                : r.nijssen@terglobo.nl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QDialogButtonBox
from qgis.core import QgsMessageLog, QgsProject
# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .turn_gray_dialog import TurnGrayDialog
import os.path
from .composer_item_color_setters import setlayoutItemColor
import qgis


class TurnGray:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'TurnGray_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)
        self.do_log = False # set to True for debugging


        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Turn Gray')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'TurnGray')
        self.toolbar.setObjectName(u'TurnGray')

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('TurnGray', message)

    def log(self, message, tab='turn gray'):
        if self.do_log:
            QgsMessageLog.logMessage(str(message), 'turn gray', QgsMessageLog.INFO)
            #progress.setText('  '+str(message))


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        # Create the dialog (after translation) and keep reference
        self.dlg = TurnGrayDialog()

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/TurnGray/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Change color of composer items'),
            callback=self.run,
            parent=self.iface.mainWindow())
        # connections:
        self.dlg.comboBox_composer.currentIndexChanged.connect(self.updateInterface)
        self.dlg.checkBox_foreground.toggled.connect(self.updateInterface)
        self.dlg.checkBox_background.toggled.connect(self.updateInterface)
        #self.dlg.mColorButtonForeground.setAllowAlpha(True)
        #self.dlg.mColorButtonBackground.setAllowAlpha(True)
        self.updateInterface()


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Turn Gray'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar


    def updateInterface(self):
        self.dlg.mColorButtonForeground.setEnabled(self.dlg.checkBox_foreground.isChecked())
        self.dlg.mColorButtonBackground.setEnabled(self.dlg.checkBox_background.isChecked())
        if self.dlg.comboBox_composer.currentIndex() > 0 and (self.dlg.checkBox_foreground.isChecked() or self.dlg.checkBox_background.isChecked()):
            self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(True)
        else:
            self.dlg.button_box.button(QDialogButtonBox.Ok).setEnabled(False)


    def run(self):
        """Run method that performs all the real work"""
        # add composer names to dialog
        layouts = QgsProject.instance().layoutManager().layouts()
        self.dlg.comboBox_composer.clear()
        self.dlg.comboBox_composer.addItem('')
        for layout in layouts:
            self.dlg.comboBox_composer.addItem(layout.name())
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            self.log(layouts)

            if self.dlg.comboBox_composer.currentText() == '':
                self.log(u'no composer selected')
                return

            lo = None
            for layout in layouts:
                self.log(u'layout name: {}"'.format(layout.name()))
                self.log(u'combobox item: {}"'.format(self.dlg.comboBox_composer.currentText()))
                if layout.name() == self.dlg.comboBox_composer.currentText():
                    lo = layout
                    break
            if lo is None:
                self.log(u'layout "{}" does not exist'.format(self.dlg.comboBox_composer.currentText()))
                return

            if self.dlg.checkBox_foreground.isChecked():
                newForegroundColor = self.dlg.mColorButtonForeground.color()
                self.log(newForegroundColor.getRgb())
            else:
                newForegroundColor = None

            if self.dlg.checkBox_background.isChecked():
                newBackgroundColor = self.dlg.mColorButtonBackground.color()
                self.log(newBackgroundColor.getRgb())
            else:
                newBackgroundColor = None

            for item in lo.items():
                setlayoutItemColor(item, newForegroundColor, newBackgroundColor)

            # refresh
            lo.refresh()
