# -*- coding: utf-8 -*-

"""
/***************************************************************************
 TopoNodos
                                 A QGIS plugin
 Extrai automaticamente os pontos de inflexão do terreno sobre redes de distribuição baseando-se em um MDT.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-05-09
        copyright            : (C) 2025 by Evanderson H. Aguiar
        email                : evanderson.eng@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Evanderson H. Aguiar'
__date__ = '2025-05-09'
__copyright__ = '(C) 2025 by Evanderson H. Aguiar'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing,
    QgsFeatureSink,
    QgsProcessingAlgorithm,
    QgsProcessingParameterRasterLayer,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterNumber,
    QgsProcessingParameterFeatureSink,
    QgsFeature,
    QgsPointXY,
    QgsGeometry,
    QgsFields,
    QgsField,
    QgsWkbTypes,
    QgsCoordinateTransform,
    QgsProject,
    QgsCoordinateReferenceSystem,
    QgsUnitTypes
)
from PyQt5.QtCore import QVariant
from qgis.PyQt.QtGui import QIcon
from .resources import *
import os
from qgis.core import QgsPoint

class TopoNodosAlgorithm(QgsProcessingAlgorithm):

    def initAlgorithm(self, config=None):
        self.addParameter(QgsProcessingParameterRasterLayer('MDS', self.tr('Modelo Digital de Superfície (Raster)')))
        self.addParameter(QgsProcessingParameterFeatureSource('TUBULACOES', self.tr('Tubulações (linhas)'), types=[QgsProcessing.TypeVectorLine]))
        self.addParameter(QgsProcessingParameterNumber('SUAVIZACAO', self.tr('Fator de suavização (m)'), type=QgsProcessingParameterNumber.Double, defaultValue=1.0, minValue=0.1))
        self.addParameter(QgsProcessingParameterNumber('PASSO_INTERPOLACAO', self.tr('Passo de interpolação (m)'), type=QgsProcessingParameterNumber.Double, defaultValue=200.0, minValue=2.0))
        self.addParameter(QgsProcessingParameterFeatureSink('SAIDA_LINHAS', self.tr('Linhas quebradas nos pontos de inflexão')))
        self.addParameter(QgsProcessingParameterFeatureSink('SAIDA_PONTOS', self.tr('Pontos de inflexão (nós com elevação)')))

    def processAlgorithm(self, parameters, context, feedback):
        import math

        mds = self.parameterAsRasterLayer(parameters, 'MDS', context)
        tubulacoes = self.parameterAsSource(parameters, 'TUBULACOES', context)
        suavizacao = self.parameterAsDouble(parameters, 'SUAVIZACAO', context)
        passo = self.parameterAsDouble(parameters, 'PASSO_INTERPOLACAO', context)

        campos_pontos = QgsFields()
        campos_pontos.append(QgsField('elev', QVariant.Double))
        campos_pontos.append(QgsField('id_linha', QVariant.Int))

        campos_linhas = tubulacoes.fields()

        #sink_pontos, saida_pontos_id = self.parameterAsSink(parameters, 'SAIDA_PONTOS', context, campos_pontos, QgsWkbTypes.Point, mds.crs())

        sink_pontos, saida_pontos_id = self.parameterAsSink(     parameters,     'SAIDA_PONTOS',     context,     campos_pontos,     QgsWkbTypes.PointZ,     mds.crs() )
        sink_linhas, saida_linhas_id = self.parameterAsSink(parameters, 'SAIDA_LINHAS', context, campos_linhas, QgsWkbTypes.LineString, mds.crs())

        pontos_registrados = set()

        centro = mds.extent().center()
        lat = centro.y()
        utm_zone = int((centro.x() + 180) / 6) + 1
        epsg = 32700 + utm_zone if lat < 0 else 32600 + utm_zone
        crs_proj = QgsCoordinateReferenceSystem(f"EPSG:{epsg}")

        transform_to_proj = QgsCoordinateTransform(tubulacoes.sourceCrs(), crs_proj, QgsProject.instance())
        transform_to_mds = QgsCoordinateTransform(crs_proj, mds.crs(), QgsProject.instance())

        for i, feat in enumerate(tubulacoes.getFeatures()):
            if feedback.isCanceled():
                break

            geom = feat.geometry()
            if geom.isMultipart():
                linhas = geom.asMultiPolyline()
            else:
                linhas = [geom.asPolyline()]

            for linha in linhas:
                geom_linha = QgsGeometry.fromPolylineXY(linha)
                geom_linha.transform(transform_to_proj)
                coords_linha = geom_linha.asPolyline()

                inflexoes = self.extrair_inflexoes(geom_linha, mds, passo, suavizacao, transform_to_mds)

                inflexoes = [(coords_linha[0], self.amostrar_elevacao(coords_linha[0], mds, transform_to_mds))] + \
                            inflexoes + \
                            [(coords_linha[-1], self.amostrar_elevacao(coords_linha[-1], mds, transform_to_mds))]

                for pt, elev in inflexoes:
                    pt_mds = transform_to_mds.transform(pt)
                    key = (round(pt_mds.x(), 4), round(pt_mds.y(), 4))
                    if key not in pontos_registrados:
                        f = QgsFeature()
                        pt_z = QgsGeometry.fromPoint(QgsPoint(pt_mds.x(), pt_mds.y(), elev))
                        f.setGeometry(pt_z)
                        f.setFields(campos_pontos)
                        f.setAttribute('elev', elev)
                        f.setAttribute('id_linha', feat.id())                       
                        sink_pontos.addFeature(f, QgsFeatureSink.FastInsert)
                        pontos_registrados.add(key)

                if len(inflexoes) >= 2:
                    # Cortar a linha original entre os pontos de inflexão
                    distancias = [geom_linha.lineLocatePoint(QgsGeometry.fromPointXY(pt)) for pt, _ in inflexoes]
                    distancias = sorted(set(distancias))
                    comprimento = geom_linha.length()

                    for i in range(len(distancias) - 1):
                        d_start = distancias[i]
                        d_end = distancias[i + 1]
                        trecho = []

                        n = int((d_end - d_start) // 0.5) + 2  # cada 0.5m
                        for j in range(n):
                            dist = d_start + (d_end - d_start) * j / (n - 1)
                            if 0 <= dist <= comprimento:
                                trecho.append(geom_linha.interpolate(dist).asPoint())

                        if len(trecho) >= 2:
                            nova = QgsFeature(feat)
                            seg_geom = QgsGeometry.fromPolylineXY(trecho)
                            seg_geom.transform(transform_to_mds)
                            nova.setGeometry(seg_geom)
                            sink_linhas.addFeature(nova, QgsFeatureSink.FastInsert)
                else:
                    nova = QgsFeature(feat)
                    geom_original = QgsGeometry.fromPolylineXY(coords_linha)
                    geom_original.transform(transform_to_mds)
                    nova.setGeometry(geom_original)
                    sink_linhas.addFeature(nova, QgsFeatureSink.FastInsert)

            feedback.setProgress(int(100 * i / tubulacoes.featureCount()))

        return {
            'SAIDA_LINHAS': saida_linhas_id,
            'SAIDA_PONTOS': saida_pontos_id
        }

    def extrair_inflexoes(self, geom_linha, raster, passo, suavizacao, transform_to_mds):
        comprimento = geom_linha.length()
        n_pontos = int(comprimento // passo) + 1
        pontos = []
        elevacoes = []

        for i in range(n_pontos):
            dist = i * passo
            pt = geom_linha.interpolate(dist).asPoint()
            pt_proj = QgsPointXY(pt)
            pt_mds = transform_to_mds.transform(pt_proj)

            if not raster.extent().contains(pt_mds):
                continue

            resultado = raster.dataProvider().sample(pt_mds, 1)
            if resultado[1] is False or resultado[0] is None:
                continue

            pontos.append(QgsPointXY(pt))
            elevacoes.append(resultado[0])

        inflexoes = []
        for i in range(1, len(elevacoes) - 1):
            anterior = elevacoes[i - 1]
            atual = elevacoes[i]
            proximo = elevacoes[i + 1]

            delta1 = atual - anterior
            delta2 = proximo - atual

            mudou_direcao = delta1 * delta2 < 0
            variacao_total = abs(delta1) + abs(delta2)

            if mudou_direcao and variacao_total >= suavizacao:
                inflexoes.append((pontos[i], atual))

        return inflexoes

    def amostrar_elevacao(self, pt_proj, raster, transform_to_mds):
        pt_mds = transform_to_mds.transform(pt_proj)
        if not raster.extent().contains(pt_mds):
            return None
        resultado = raster.dataProvider().sample(pt_mds, 1)
        return resultado[0] if resultado[1] else None

    def name(self):
        return 'TopoNodos'

    def displayName(self):
        return self.tr('Análise Altimétrica')

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return TopoNodosAlgorithm()

    def icon(self):
        return QIcon(os.path.dirname(__file__) + '/icon.png')
