# -*- coding: utf-8 -*-

"""
/***************************************************************************
 AddGridRef
                                 A QGIS plugin
 Adds British or Irish grid references to layers
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-01-30
        copyright            : (C) 2019 by Field Studies Council
        email                : r.burkmar@field-studies-council.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Field Studies Council'
__date__ = '2019-01-30'
__copyright__ = '(C) 2019 by Field Studies Council'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from PyQt5.QtCore import (QCoreApplication, QVariant)
from qgis.core import (QgsProcessing,
                       QgsMessageLog,
                       QgsFeatureSink,
                       QgsFeature,
                       QgsProcessingAlgorithm,
                       QgsField,
                       QgsCoordinateReferenceSystem,
                       QgsCoordinateTransform,
                       QgsProject,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterString,
                       QgsProcessingParameterFeatureSink)
from . import osgr

class AddGridRefAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'
    GRTYPE = 'GRYPE'
    PRECISIONS = 'PRECISIONS'
    PREFIX = 'PREFIX'
    dGrType = [
            {"name": "British National Grid", "code": "os", "crs": QgsCoordinateReferenceSystem("EPSG:27700")},
            {"name": "Irish National Grid", "code": "irish", "crs": QgsCoordinateReferenceSystem("EPSG:29903")}
        ]
    aGrTypeName = list(map((lambda x: x["name"]), dGrType))
    dPrecisions = [
            {"text": "10 figure GR (1 m)", "num": 1, "field": "10figGR"},
            {"text": "8 figure GR (10 m)", "num": 10, "field": "8figGR"},
            {"text": "6 figure GR (100 m)", "num": 100, "field": "6figGR"},
            {"text": "Monad (1 km)", "num": 1000, "field": "monad"},
            {"text": "Tetrad (2 km)", "num": 2000, "field": "tetrad"},
            {"text": "Quadrant (5 km)", "num": 5000, "field": "quadrant"},
            {"text": "Hectad (10 km)", "num": 10000, "field": "hectad"},
            {"text": "100 km", "num": 100000, "field": "100km"},
            {"text": "Easting/northings", "num": -1, "field": "Easting Northing"},
            {"text": "Lat/lng (WGS84)", "num": -2, "field": "Lat Lng"}
        ]
    aPrecisionText = list(map((lambda x: x["text"]), dPrecisions))

    # Get a reference to an osgr object
    osgr = osgr.osgr()

    def logMessage(self, message):
        QgsMessageLog.logMessage(message, "Add Grid Ref Algorithm")

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input layer'),
                [QgsProcessing.TypeVectorAnyGeometry]
            )
        )

        # Parameter for selecting British or Irish Grid
        self.addParameter(
            QgsProcessingParameterEnum(
                self.GRTYPE,
                self.tr('Select British or Irish National Grid'),
                self.aGrTypeName,
                False # Single select
            )
        )

        # Parameter for selecting grid reference precisions
        # Note that the multi-select does not work in the Modeller - https://issues.qgis.org/issues/20406
        self.addParameter(
            QgsProcessingParameterEnum(
                self.PRECISIONS,
                self.tr('Select the type of Grid References to add'),
                self.aPrecisionText,
                True # Multi select
            )
        )

        # Parameter for prefixing field name
        self.addParameter(
            QgsProcessingParameterString(
                self.PREFIX,
                self.tr('Optionally, indicate a short prefix for output column names'),
                None, # No default value
                False, # Not multi-line
                True # Optional
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Output layer')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source.
        source = self.parameterAsSource(parameters, self.INPUT, context)
        fields = source.fields()

        # Get the field prefix
        prefix = self.parameterAsString (parameters, self.PREFIX, context)
        prefix = prefix.replace(" ", "")

        # Get the precision choices and make the appropriate output fields
        precisions = self.parameterAsEnums (parameters, self.PRECISIONS, context)
        for p in precisions:
            if self.dPrecisions[p]["num"] > 0:
                fieldName = prefix + self.dPrecisions[p]["field"]
                fields.append(QgsField(fieldName, QVariant.String))
            elif self.dPrecisions[p]["num"] < 0: #Easting/northing (-1) or Lat/lng (-2)
                if self.dPrecisions[p]["num"] == -1:
                    dataType =  QVariant.Int
                else:
                    dataType =  QVariant.Double
                fieldName = self.dPrecisions[p]["field"].split(" ")
                fieldNameX = prefix + fieldName[0]
                fields.append( QgsField(fieldNameX, dataType))
                fieldNameY = prefix + fieldName[1]
                fields.append(QgsField(fieldNameY, dataType))

        # Create the feature sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                context, fields, source.wkbType(), source.sourceCrs())
    
        # Get the other parameters
        grtype = self.parameterAsEnum (parameters, self.GRTYPE, context)       

        transform = None
        if self.dGrType[grtype]["crs"] != source.sourceCrs():
            transform = QgsCoordinateTransform(source.sourceCrs(), self.dGrType[grtype]["crs"], QgsProject.instance())
        transformWGS84 = QgsCoordinateTransform(self.dGrType[grtype]["crs"], QgsCoordinateReferenceSystem("EPSG:4326"), QgsProject.instance())
        
        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / source.featureCount() if source.featureCount() else 0
        features = source.getFeatures()

        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            # Add grid references in here
            geom = feature.geometry()
            if transform is not None:
                geom.transform(transform)
            pt = geom.centroid().asPoint()

            # Add a feature in the sink
            newFeature = QgsFeature()
            newFeature.setGeometry(feature.geometry())
            newFeature.setFields(fields)

            # Copy current field values
            for field in source.fields():
                newFeature[field.name()] = feature.attribute(field.name())
            
            # Generate new field values
            for p in precisions:
                if self.dPrecisions[p]["num"] > 0:
                    gr =  self.osgr.grFromEN(pt.x(), pt.y(), self.dPrecisions[p]["num"], self.dGrType[grtype]["code"])
                    if gr == "na":
                        gr = ""
                    newFeature[prefix + self.dPrecisions[p]["field"]] = gr
                elif self.dPrecisions[p]["num"] < 0: #Easting/northing (-1) or Lat/lng (-2)
                    fieldName = self.dPrecisions[p]["field"].split(" ")
                    if self.dPrecisions[p]["num"] == -1:
                        x = int(pt.x())
                        y = int(pt.y())
                        fieldNameX = prefix + fieldName[0]
                        fieldNameY = prefix + fieldName[1]
                    else:
                        wgs84Point = transformWGS84.transform(pt)
                        x = wgs84Point.x()
                        y = wgs84Point.y()
                        fieldNameX = prefix + fieldName[1]
                        fieldNameY = prefix + fieldName[0]

                    newFeature[fieldNameX] = x
                    newFeature[fieldNameY] = y

            sink.addFeature(newFeature, QgsFeatureSink.FastInsert)

            # Update the progress bar
            feedback.setProgress(int(current * total))

        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {self.OUTPUT: dest_id}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Add GRs to layers'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        #return 'Grid references'
        return None
    
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return AddGridRefAlgorithm()
