# -*- coding: utf-8 -*-
import math
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsFeature,
    QgsFeatureSink,
    QgsFields,
    QgsField,
    QgsGeometry,
    QgsPointXY,
    QgsProcessing,
    QgsProcessingAlgorithm,
    QgsProcessingException,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterNumber,
    QgsProcessingParameterString,
    QgsSpatialIndex,
    QgsWkbTypes,
)

def _norm(vx, vy):
    n = math.hypot(vx, vy)
    if n == 0:
        return (0.0, 0.0)
    return (vx / n, vy / n)

def _angle_deg(v1, v2):
    """Angle between unit vectors v1 and v2 in degrees (0..180)."""
    dot = max(-1.0, min(1.0, v1[0]*v2[0] + v1[1]*v2[1]))
    return math.degrees(math.acos(dot))

def _endpoint_and_dir(polyline, is_start):
    """Return (endpoint_point, outward_unit_vector) for a polyline part."""
    if is_start:
        p0 = polyline[0]
        p1 = polyline[1]
        vx, vy = (p1.x() - p0.x(), p1.y() - p0.y())
        return p0, _norm(vx, vy)
    else:
        p0 = polyline[-1]
        p1 = polyline[-2]
        vx, vy = (p1.x() - p0.x(), p1.y() - p0.y())
        return p0, _norm(vx, vy)

def _geom_to_parts(geom):
    """Convert QgsGeometry to list of polyline parts (list of QgsPointXY)."""
    if geom.isMultipart():
        parts = geom.asMultiPolyline()
        return [[QgsPointXY(p) for p in part] for part in parts]
    part = geom.asPolyline()
    return [[QgsPointXY(p) for p in part]]

def _parts_to_geom(parts):
    if len(parts) == 1:
        return QgsGeometry.fromPolylineXY(parts[0])
    return QgsGeometry.fromMultiPolylineXY(parts)

def _move_endpoint(parts, part_idx, is_start, new_pt):
    part = parts[part_idx]
    if is_start:
        part[0] = QgsPointXY(new_pt)
    else:
        part[-1] = QgsPointXY(new_pt)
    parts[part_idx] = part
    return parts

def _adjacent_segment_len(part, is_start):
    if len(part) < 2:
        return 0.0
    if is_start:
        a, b = part[0], part[1]
    else:
        a, b = part[-1], part[-2]
    return math.hypot(b.x()-a.x(), b.y()-a.y())

def _dedupe_endpoint(part, is_start, eps=1e-12):
    # remove duplicate endpoint if it equals its neighbor (zero-length segment)
    if len(part) < 2:
        return part
    if is_start:
        a, b = part[0], part[1]
        if math.hypot(b.x()-a.x(), b.y()-a.y()) <= eps:
            del part[0]
    else:
        a, b = part[-1], part[-2]
        if math.hypot(b.x()-a.x(), b.y()-a.y()) <= eps:
            del part[-1]
    return part

QGIS_GEOSTRANSFORM_ERR = "QGIS のジオメトリを GEOS のジオメトリに変換できません。"

class ConnectTileCutRoads(QgsProcessingAlgorithm):
    """Snap near endpoints to restore connectivity for tile-cut road lines."""

    INPUT = "INPUT"
    TOL = "TOL"
    ANGLE = "ANGLE"
    USE_ATTR = "USE_ATTR"
    ATTR_FIELDS = "ATTR_FIELDS"
    USE_LVORDER = "USE_LVORDER"
    LVORDER_FIELD = "LVORDER_FIELD"
    USE_EP2LINE = "USE_EP2LINE"
    SPLIT_TARGET = "SPLIT_TARGET"
    OUTPUT = "OUTPUT"
    ISSUES = "ISSUES"

    def tr(self, s):
        return QCoreApplication.translate("TileRoadConnector", s)

    def createInstance(self):
        return ConnectTileCutRoads()

    def name(self):
        return "connect_tile_cut_roads"

    def displayName(self):
        return self.tr("タイル境界で分断された道路を接続（近接端点をスナップ）")

    def group(self):
        return self.tr("タイル境界道路接続")

    def groupId(self):
        return "tileroadconnector"

    def shortHelpString(self):
        return self.tr(
            "タイル境界等で分断された道路ラインについて、許容距離内の端点ペア（ほぼ接続）を検出し、"
            "端点ペア（ほぼ接続）を検出し、片側端点を相手端点へスナップしてネットワーク接続性を回復します。"
            "必要に応じて、属性一致（例：motorway/rdCtg/tollSect/rnkWidth/medSect）や lvOrder 一致を条件にして、"
            "立体交差等で異なるレベルの道路を誤って接続しないようにできます。"
            "また、端点—端点で修正できない場合は端点—線分スナップも実行できます。"
        )

    def initAlgorithm(self, config=None):
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr("入力：道路ライン"),
                [QgsProcessing.TypeVectorLine],
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.TOL,
                self.tr("探索許容距離（地図単位。メートルCRS推奨）"),
                QgsProcessingParameterNumber.Double,
                defaultValue=1.0,
                minValue=0.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterNumber(
                self.ANGLE,
                self.tr("角度許容差（度）。0で角度フィルタ無効"),
                QgsProcessingParameterNumber.Double,
                defaultValue=0.0,
                minValue=0.0,
                maxValue=180.0,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.USE_ATTR,
                self.tr("端点マッチングに属性一致を必須とする"),
                defaultValue=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterString(
                self.ATTR_FIELDS,
                self.tr("一致判定に使う属性フィールド名（カンマ区切り）"),
                defaultValue="motorway,rdCtg,tollSect,rnkWidth,medSect",
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                self.USE_LVORDER,
                self.tr("lvOrder 一致を必須とする（推奨）"),
                defaultValue=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterString(
                self.LVORDER_FIELD,
                self.tr("lvOrder フィールド名（存在する場合）"),
                defaultValue="lvOrder",
                optional=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                "MAKE_VALID",
                self.tr("不正ジオメトリを可能な範囲で修復（makeValid）"),
                defaultValue=True,
            )
        )
        self.addParameter(
            QgsProcessingParameterBoolean(
                "SKIP_GEOS_FAIL",
                self.tr("GEOS変換できないフィーチャはスキップしてログ出力"),
                defaultValue=True,
            )
        )

        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr("出力：修正後の道路ライン"),
                QgsProcessing.TypeVectorLine,
            )
        )
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.ISSUES,
                self.tr("出力：接続した箇所（ポイント）"),
                QgsProcessing.TypeVectorPoint,
            )
        )
    @staticmethod
    def _safe_make_valid(g):
        """Try to make geometry GEOS-compatible. Returns (geom, ok, msg)."""
        if g is None or g.isEmpty():
            return g, True, ""
        try:
            # Touch GEOS-related call. Some geometries fail here.
            _ = g.isGeosValid()
            return g, True, ""
        except Exception:
            pass
        try:
            gv = g.makeValid()
            if gv is None or gv.isEmpty():
                return g, False, "makeValid returned empty"
            return gv, True, "makeValid applied"
        except Exception as e:
            return g, False, f"makeValid failed: {e}"

    def processAlgorithm(self, parameters, context, feedback):
        nodeize_intersections = True
        src = self.parameterAsSource(parameters, self.INPUT, context)
        if src is None:
            raise QgsProcessingException(self.tr("Invalid input layer"))

        make_valid = bool(self.parameterAsBool(parameters, "MAKE_VALID", context))
        skip_geos_fail = bool(self.parameterAsBool(parameters, "SKIP_GEOS_FAIL", context))


        tol = float(self.parameterAsDouble(parameters, self.TOL, context))
        ang_tol = float(self.parameterAsDouble(parameters, self.ANGLE, context))
        use_attr = bool(self.parameterAsBool(parameters, self.USE_ATTR, context))
        attr_fields_raw = self.parameterAsString(parameters, self.ATTR_FIELDS, context) or ""
        attr_fields = [f.strip() for f in attr_fields_raw.split(",") if f.strip()]
        use_lv = bool(self.parameterAsBool(parameters, self.USE_LVORDER, context))
        # Removed options (unused by design): endpoint-to-line snap and target nodeization
        use_ep2line = False
        split_target = False
        lv_field = (self.parameterAsString(parameters, self.LVORDER_FIELD, context) or "lvOrder").strip()

        src_fields = src.fields()
        existing = set([f.name() for f in src_fields])
        attr_fields = [f for f in attr_fields if f in existing]
        lv_exists = lv_field in existing

        (out_sink, out_id) = self.parameterAsSink(
            parameters, self.OUTPUT, context, src_fields, src.wkbType(), src.sourceCrs()
        )
        if out_sink is None:
            raise QgsProcessingException(self.tr("Could not create output sink"))

        issue_fields = QgsFields()
        issue_fields.append(QgsField("fid1", QVariant.LongLong))
        issue_fields.append(QgsField("fid2", QVariant.LongLong))
        issue_fields.append(QgsField("dist", QVariant.Double))
        issue_fields.append(QgsField("auto", QVariant.Int))
        issue_fields.append(QgsField("message", QVariant.String))
        (iss_sink, iss_id) = self.parameterAsSink(
            parameters, self.ISSUES, context, issue_fields, QgsWkbTypes.Point, src.sourceCrs()
        )
        if iss_sink is None:
            raise QgsProcessingException(self.tr("Could not create issues sink"))

        feats = list(src.getFeatures())
        geoms = {}
        # Optional simplification to prevent vertex explosions (disabled by default)
        simplify_tol = locals().get('simplify_tol', 0.0)
        do_simplify = simplify_tol is not None and float(simplify_tol) > 0.0
        attrs = {}
        for f in feats:
            g = QgsGeometry(f.geometry())
            if make_valid:
                try:
                    g = g.makeValid()
                except Exception:
                    pass
            if skip_geos_fail:
                try:
                    _ = g.isGeosValid()
                except Exception:
                    try:
                        ip = QgsFeature(issue_fields)
                        ip.setGeometry(QgsGeometry.fromPointXY(g.boundingBox().center()))
                        ip['fid1'] = int(f.id())
                        ip['fid2'] = -1
                        ip['auto'] = 99
                        ip['message'] = self.tr('QGISのジオメトリをGEOSのジオメトリに変換できません。')
                        iss_sink.addFeature(ip, QgsFeatureSink.FastInsert)
                    except Exception:
                        pass
                    continue
            geoms[f.id()] = g
            attrs[f.id()] = f.attributes()

        def attr_key(fid):
            if not use_attr or not attr_fields:
                return None
            return tuple(attrs[fid][src_fields.indexFromName(nm)] for nm in attr_fields)

        def lv_key(fid):
            if not (use_lv and lv_exists):
                return None
            return attrs[fid][src_fields.indexFromName(lv_field)]

        endpoints = []
        idx_map = {}
        sindex = QgsSpatialIndex()

        synth_id = 0
        for f in feats:
            if feedback.isCanceled():
                break
            g = geoms[f.id()]
            if g is None or g.isEmpty():
                continue
            parts = _geom_to_parts(g)
            for pi, part in enumerate(parts):
                if len(part) < 2:
                    continue
                for is_start in (True, False):
                    p, v = _endpoint_and_dir(part, is_start)
                    e = {
                        "idx": synth_id,
                        "fid": f.id(),
                        "part_idx": pi,
                        "is_start": is_start,
                        "pt": p,
                        "dir": v,
                        "akey": attr_key(f.id()),
                        "lv": lv_key(f.id()),
                    }
                    endpoints.append(e)
                    idx_map[synth_id] = e

                    pf = QgsFeature()
                    pf.setId(synth_id)
                    pf.setGeometry(QgsGeometry.fromPointXY(p))
                    sindex.addFeature(pf)
                    synth_id += 1

        if synth_id == 0:
            for f in feats:
                out_sink.addFeature(f, QgsFeatureSink.FastInsert)
            return {self.OUTPUT: out_id, self.ISSUES: iss_id}

        # line spatial index for endpoint-to-line snapping
        line_index = QgsSpatialIndex()
        for f in feats:
            g = geoms.get(f.id())
            if g is None or g.isEmpty():
                continue
            lf = QgsFeature()
            lf.setId(int(f.id()))
            lf.setGeometry(g)
            line_index.addFeature(lf)

        pairs = []
        for e in endpoints:
            if feedback.isCanceled():
                break
            p = e["pt"]
            rect = QgsGeometry.fromPointXY(p).buffer(tol, 8).boundingBox()
            cand_ids = sindex.intersects(rect)
            for cid in cand_ids:
                if cid <= e["idx"]:
                    continue
                e2 = idx_map.get(cid)
                if e2 is None:
                    continue
                if e2["fid"] == e["fid"]:
                    continue
                if use_attr and e["akey"] != e2["akey"]:
                    continue
                if use_lv and lv_exists and e["lv"] != e2["lv"]:
                    continue
                d = math.hypot(e2["pt"].x() - p.x(), e2["pt"].y() - p.y())
                if d > tol:
                    continue
                if ang_tol > 0:
                    v1 = e["dir"]
                    v2 = e2["dir"]
                    ang = _angle_deg(v1, (-v2[0], -v2[1]))
                    if ang > ang_tol:
                        continue
                pairs.append((d, e["idx"], e2["idx"]))

        pairs.sort(key=lambda x: x[0])

        used = set()
        matches = []
        for d, i1, i2 in pairs:
            if i1 in used or i2 in used:
                continue
            used.add(i1); used.add(i2)
            matches.append((d, i1, i2))

        fixed_count = 0
        for k, (d, i1, i2) in enumerate(matches):
            if feedback.isCanceled():
                break
            e1 = idx_map[i1]; e2 = idx_map[i2]
            p1 = e1["pt"]; p2 = e2["pt"]

            # Decide which endpoint to move to minimize distortion:
            # move the endpoint whose adjacent segment is shorter (less visual impact).
            g1 = geoms[e1["fid"]]
            parts1 = _geom_to_parts(g1)
            seglen1 = _adjacent_segment_len(parts1[e1["part_idx"]], e1["is_start"])

            g2 = geoms[e2["fid"]]
            parts2 = _geom_to_parts(g2)
            seglen2 = _adjacent_segment_len(parts2[e2["part_idx"]], e2["is_start"])

            # snap_pt is the endpoint we keep (target); we move the other endpoint onto it.
            if seglen1 <= seglen2:
                snap_pt = p2
                move_fid, move_parts, move_e = e1["fid"], parts1, e1
                keep_fid, keep_parts, keep_e = e2["fid"], parts2, e2
            else:
                snap_pt = p1
                move_fid, move_parts, move_e = e2["fid"], parts2, e2
                keep_fid, keep_parts, keep_e = e1["fid"], parts1, e1

            move_parts = _move_endpoint(move_parts, move_e["part_idx"], move_e["is_start"], snap_pt)
            # avoid zero-length segment created by snapping exactly onto neighbor vertex
            part = move_parts[move_e["part_idx"]]
            move_parts[move_e["part_idx"]] = _dedupe_endpoint(part, move_e["is_start"])
            geoms[move_fid] = _parts_to_geom(move_parts)

            # keep feature geometry unchanged (prevents shifting/shortening on both sides)
            geoms[keep_fid] = _parts_to_geom(keep_parts)

            ip = QgsFeature(issue_fields)
            ip.setGeometry(QgsGeometry.fromPointXY(QgsPointXY(snap_pt)))
            ip["fid1"] = int(e1["fid"])
            ip["fid2"] = int(e2["fid"])
            ip["dist"] = float(d)
            ip["auto"] = 1
            iss_sink.addFeature(ip, QgsFeatureSink.FastInsert)

            fixed_count += 1
            if len(matches) > 0:
                feedback.setProgress(int(100 * (k + 1) / len(matches)))
        # endpoint-to-line snapping for remaining dangling endpoints
        if use_ep2line and tol > 0:
            fixed_ep2line = 0
            for e in endpoints:
                if feedback.isCanceled():
                    break
                if e["idx"] in used:
                    continue
                p = e["pt"]
                rect = QgsGeometry.fromPointXY(p).buffer(tol, 8).boundingBox()
                cand_fids = line_index.intersects(rect)
                best = None  # (dist, target_fid, snap_pt, afterVertex)
                for tfid in cand_fids:
                    if int(tfid) == int(e["fid"]):
                        continue
                    # attribute constraints
                    if use_attr and attr_fields:
                        if attr_key(e["fid"]) != attr_key(tfid):
                            continue
                    if use_lv and lv_exists:
                        if lv_key(e["fid"]) != lv_key(tfid):
                            continue

                    tg = geoms.get(tfid)
                    if tg is None or tg.isEmpty():
                        continue
                    afterVertex = -1
                    try:
                        _minDist, minDistPoint, afterVertex, _leftOf = tg.closestSegmentWithContext(p)
                        sp = QgsPointXY(minDistPoint)
                    except Exception:
                        sp_geom = tg.nearestPoint(QgsGeometry.fromPointXY(p))
                        sp = sp_geom.asPoint()

                    d = math.hypot(sp.x() - p.x(), sp.y() - p.y())
                    if d > tol:
                        continue

                    # optional safety: only snap to the vicinity of the target line's endpoints (avoid snapping onto intersections/mid-segments)
                    e2line_endpoint_only = locals().get('e2line_endpoint_only', True)
                    e2line_endpoint_tol = locals().get('e2line_endpoint_tol', 0.0)
                    if e2line_endpoint_only:
                        tol2 = tol if (e2line_endpoint_tol is None or e2line_endpoint_tol <= 0.0) else float(e2line_endpoint_tol)
                        try:
                            tg_geom = geoms[tfid]
                            if tg_geom is not None and (not tg_geom.isEmpty()):
                                # Determine nearest endpoint distance of the target line
                                if tg_geom.isMultipart():
                                    mls = tg_geom.asMultiPolyline()
                                    tg_line = mls[0] if len(mls) > 0 else []
                                else:
                                    tg_line = tg_geom.asPolyline()
                                if len(tg_line) >= 2:
                                    epA = tg_line[0]; epB = tg_line[-1]
                                    dA = math.hypot(sp.x() - epA.x(), sp.y() - epA.y())
                                    dB = math.hypot(sp.x() - epB.x(), sp.y() - epB.y())
                                    if min(dA, dB) > tol2:
                                        continue
                        except Exception:
                            pass

                    # optional angle check (best-effort)
                    if ang_tol > 0 and afterVertex >= 0:
                        ok_angle = True
                        try:
                            pts = _geom_to_parts(tg)
                            flat = []
                            for part in pts:
                                flat.extend(part)
                            if 0 < afterVertex < len(flat):
                                a = flat[afterVertex - 1]
                                b = flat[afterVertex]
                                tv = _norm(b.x() - a.x(), b.y() - a.y())
                                ang = _angle_deg(e["dir"], tv)
                                # accept either same direction or opposite direction
                                if ang > ang_tol and abs(180 - ang) > ang_tol:
                                    ok_angle = False
                        except Exception:
                            pass
                        if not ok_angle:
                            continue

                    if best is None or d < best[0]:
                        best = (d, int(tfid), sp, int(afterVertex))

                if best is None:
                    continue

                d, tfid, sp, afterVertex = best

                # move endpoint of source feature to snapped point
                g1 = geoms[e["fid"]]
                parts1 = _geom_to_parts(g1)
                parts1 = _move_endpoint(parts1, e["part_idx"], e["is_start"], sp)
                geoms[e["fid"]] = _parts_to_geom(parts1)

                # ensure target has a vertex at snapped point (optional)
                if split_target and afterVertex >= 0:
                    tg = geoms[tfid]
                    # Safer nodeization: only attempt on singlepart line and only if the snap point is not already a vertex.
                    try:
                        if (not tg.isMultipart()) and (tg.type() == QgsWkbTypes.LineGeometry):
                            # guard: skip extremely detailed features
                            if max_vertices is not None and max_vertices > 0:
                                if tg.vertexCount() > int(max_vertices):
                                    raise Exception('too_many_vertices')
                            verts = tg.asPolyline()
                            if 0 <= afterVertex <= len(verts):
                                # check if already a vertex
                                already = any((abs(v.x()-sp.x()) < 1e-12 and abs(v.y()-sp.y()) < 1e-12) for v in verts)
                                if not already:
                                    tg.insertVertex(sp.x(), sp.y(), afterVertex)
                                    geoms[tfid] = tg
                    except Exception:
                        pass

                ip = QgsFeature(issue_fields)
                ip.setGeometry(QgsGeometry.fromPointXY(sp))
                ip["fid1"] = int(e["fid"])
                ip["fid2"] = int(tfid)
                ip["dist"] = float(d)
                ip["auto"] = 2
                iss_sink.addFeature(ip, QgsFeatureSink.FastInsert)

                fixed_ep2line += 1

            fixed_count += fixed_ep2line


# --- nodeize intersections (optional) ---
        nodeize_intersections = locals().get('nodeize_intersections', True)
        if nodeize_intersections:
            feedback.pushInfo(self.tr("Nodeizing intersections (splitting at crossing points)…"))
            try:
                idx_line = QgsSpatialIndex()
                for fid, g in geoms.items():
                    if g is None or g.isEmpty():
                        continue
                    ftmp = QgsFeature()
                    ftmp.setId(int(fid))
                    ftmp.setGeometry(g)
                    idx_line.addFeature(ftmp)

                def _vertex_count(geom):
                    try:
                        return geom.vertexCount()
                    except Exception:
                        # fallback: count polyline vertices
                        if geom is None or geom.isEmpty():
                            return 0
                        if geom.isMultipart():
                            return sum(len(p) for p in geom.asMultiPolyline())
                        return len(geom.asPolyline())

                def _near_endpoints(geom, pt, tol_m):
                    if geom is None or geom.isEmpty():
                        return False
                    try:
                        if geom.isMultipart():
                            mls = geom.asMultiPolyline()
                            if not mls:
                                return False
                            line = mls[0]
                        else:
                            line = geom.asPolyline()
                        if len(line) < 2:
                            return False
                        epA = line[0]; epB = line[-1]
                        dA = math.hypot(pt.x()-epA.x(), pt.y()-epA.y())
                        dB = math.hypot(pt.x()-epB.x(), pt.y()-epB.y())
                        return min(dA, dB) <= tol_m
                    except Exception:
                        return False


                def _is_vertex(poly, pt, eps=1e-12):
                    return any((abs(v.x()-pt.x())<eps and abs(v.y()-pt.y())<eps) for v in poly)

                def _nodeize_single(geom, pt):
                    if geom is None or geom.isEmpty():
                        return geom
                    if max_vertices is not None and max_vertices > 0:
                        if _vertex_count(geom) > int(max_vertices):
                            return geom
                    if nodeize_endpoint_only:
                        tol_m = tol if (nodeize_endpoint_tol is None or nodeize_endpoint_tol <= 0.0) else float(nodeize_endpoint_tol)
                        if not _near_endpoints(geom, pt, tol_m):
                            return geom
                    if geom.isMultipart():
                        return geom  # avoid risky edits on multipart
                    poly = geom.asPolyline()
                    if len(poly) < 2:
                        return geom
                    if _is_vertex(poly, pt):
                        return geom
                    # insert after nearest segment
                    mind = None
                    insert_after = None
                    px, py = pt.x(), pt.y()
                    for i in range(len(poly)-1):
                        a = poly[i]; b = poly[i+1]
                        ax, ay = a.x(), a.y(); bx, by = b.x(), b.y()
                        dx, dy = bx-ax, by-ay
                        if dx==0 and dy==0:
                            continue
                        t = ((px-ax)*dx + (py-ay)*dy) / (dx*dx + dy*dy)
                        t = 0 if t < 0 else (1 if t > 1 else t)
                        qx, qy = ax + t*dx, ay + t*dy
                        d = (px-qx)**2 + (py-qy)**2
                        if (mind is None) or (d < mind):
                            mind = d
                            insert_after = i+1
                    if insert_after is None:
                        return geom
                    try:
                        geom.insertVertex(px, py, insert_after)
                    except Exception:
                        return geom
                    return geom

                fids = list(geoms.keys())
                for fid in fids:
                    g = geoms.get(fid)
                    if g is None or g.isEmpty():
                        continue
                    cand = idx_line.intersects(g.boundingBox())
                    for ofid in cand:
                        if ofid <= fid:
                            continue
                        og = geoms.get(ofid)
                        if og is None or og.isEmpty():
                            continue
                        inter = g.intersection(og)
                        if inter is None or inter.isEmpty():
                            continue
                        if inter.type() != QgsWkbTypes.PointGeometry:
                            continue
                        pts = []
                        try:
                            if inter.isMultipart():
                                pts = [QgsPointXY(p) for p in inter.asMultiPoint()]
                            else:
                                pts = [QgsPointXY(inter.asPoint())]
                        except Exception:
                            pts = []
                        for pt in pts:
                            geoms[fid] = _nodeize_single(geoms[fid], pt)
                            geoms[ofid] = _nodeize_single(geoms[ofid], pt)
            except Exception as e:
                feedback.pushInfo(self.tr(f"Intersection nodeize error: {e}"))



        for f in feats:
            if feedback.isCanceled():
                break
            out_f = QgsFeature(src_fields)
            out_f.setAttributes(attrs[f.id()])
            out_f.setGeometry(geoms[f.id()])
            out_sink.addFeature(out_f, QgsFeatureSink.FastInsert)

        feedback.pushInfo(self.tr("Fixed connections: {}".format(fixed_count)))
        return {self.OUTPUT: out_id, self.ISSUES: iss_id}
