# -*- coding: utf-8 -*-
"""
/***************************************************************************
 TilePick
                                 A QGIS plugin
 Easy load raster tiles file from index vector
                             -------------------
        begin                : 2024-09-01
        copyright            : (C) 2024 by DDT67
        email                : ddt-inondation@bas-rhin.gouv.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt import QtCore, QtGui, QtWidgets
from qgis.gui import QgsProjectionSelectionWidget, QgsFileWidget
from qgis.core import QgsCoordinateReferenceSystem
from .picker import Picker, Param, TEXTS
from .patcher import Patcher
from functools import partial
import os, json, time, random, traceback

def classFactory(iface):  # pylint: disable=invalid-name
    return PickerPlugin(iface)

def makeUniqueIdent():
    t = hex(int(time.time()*1000))[2:]
    r = hex(random.randint(1000000,10000000))[2:]
    return str(t) + str(r)  

class PickerPlugin(QtCore.QObject):
    def __init__(self, iface):
        super().__init__()
        self.iface = iface
        self.canvas = iface.mapCanvas() 
        self.plugin_dir = os.path.dirname(__file__)
        self.imageFolder = os.path.join(self.plugin_dir,'images')
        self.tileFolder = os.path.join(self.plugin_dir,'tiles')
        self.picker = Picker(self)
        self.param = self.picker.param
        self.patcher = Patcher(self)
        self.prevTool = None
        self.getIniJson()
        
        try:
            locale = QtCore.QSettings().value("locale/userLocale", "en",  type=str)[0:2]
        except:
            locale = "en"
            
        localePath = os.path.join(self.plugin_dir, 'i18n', 'picker_'+locale+'.qm')
        self.translator = QtCore.QTranslator()
        self.translator.load(localePath)
        QtCore.QCoreApplication.installTranslator(self.translator)
        
    def tr(self, txt, disambiguation=None):
        return QtCore.QCoreApplication.translate("TilePick", txt, disambiguation)  
        
    def initGui(self):  
        
        self.pickerButton = QtWidgets.QToolButton()
        self.pickerButton.setMenu(QtWidgets.QMenu())
        self.pickerButton.setPopupMode(QtWidgets.QToolButton.ToolButtonPopupMode.MenuButtonPopup)
        self.pickerBtnAction = self.iface.addToolBarWidget(self.pickerButton)
        
        icon = QtGui.QIcon(os.path.join(self.imageFolder,'pick.png'))
        text = self.tr("Select tiles zone to load")
        self.action = QtWidgets.QAction(icon, text, self.iface.mainWindow())   
        self.action.triggered.connect(self.run)
        self.action.setCheckable( True )
        self.picker.setAction( self.action )
        self.pickerButton.setDefaultAction(self.action)
        
        # self.iface.addToolBarIcon(self.action)
        m = self.pickerButton.menu()
        m.addAction(QtGui.QIcon(os.path.join(self.imageFolder,'pick.png')), self.tr("Load selection"), self.loadSelected)
        m.addAction(QtGui.QIcon(os.path.join(self.imageFolder,'puzzle.png')), self.tr("Assemble"), self.assemble)
        m.addSeparator()
        m.addAction(QtGui.QIcon(os.path.join(self.imageFolder,'config.png')), self.tr("Configure"), self.configure)
        

    def run(self):
        if self.canvas.mapTool() != self.picker:
            self.prevTool = self.canvas.mapTool()
            self.canvas.setMapTool(self.picker)
        else:
            self.canvas.setMapTool(self.prevTool)
       
    def loadSelected(self):
        self.picker.loadFromSelected()
    
    def configure(self):
        dlg = PickerConfigure(self)
        dlg.exec()
        if dlg.result():
            self.param.set('targetLayers', dlg.targetLayers.currentData())
            self.param.set('noReload', dlg.noReload.isChecked())
            self.param.set('fromCoords', [it.file for it in dlg.tilesConfig.selectedItems()])
            self.param.set('mode', dlg.mode.value)
            self.param.set('maxTiles', dlg.maxTiles.value())
            self.param.set('maxZone', dlg.maxZone.value())
            self.param.set('groupActive', dlg.groupActive.isChecked())
            self.param.set('groupName', dlg.groupName.text())
        
    def assemble(self):
        self.patcher.launch()
    
    def unload(self):
        self.picker.unload()
        self.canvas.unsetMapTool(self.picker)
        # self.iface.removeToolBarIcon(self.action)
        self.iface.removeToolBarIcon(self.pickerBtnAction)
        
    def getIniJson(self):
        self.json = self.param.get('json')
        if not os.path.exists(self.tileFolder): os.mkdir(self.tileFolder)
        if self.json is None: self.json = {}
        for n,data in self.json.items():
            file = os.path.join(self.tileFolder, n)
            if not os.path.exists(file):
                f = open(file, 'w')
                f.write(json.dumps(data, indent=4))
                f.close()
        
    def setIniJson(self):
        self.param.set('json', self.json)
        
        
class PickerConfigure(QtWidgets.QDialog):
    def __init__(self, parent):
        super().__init__()
        self.plugin = parent
        self.setWindowTitle(self.plugin.tr('TilePick configuration'))
        QtWidgets.QVBoxLayout(self)
        h = QtWidgets.QHBoxLayout()
        self.layout().addLayout(h)
        self.param = self.plugin.picker.param
        self.mode = QtWidgets.QButtonGroup()
        
        modeDict = {
            'index': self.plugin.tr("From vector tile index"), 
            'coords': self.plugin.tr("From canvas position"),
        }
        modeToolTips = {
            'index': self.plugin.tr("Tiles will be load from the path (absolute or relative) contained in the attributes layer"), 
            'coords': self.plugin.tr("Tiles will be load from the calculate coordinates and the configuration choosen"),
        }
        targetDict = {
            'active': self.plugin.tr("Active layer"),
            'all': self.plugin.tr("All layers"),
        }
        self.buttonMode = {}
        
        for k,v in modeDict.items():
            h = QtWidgets.QHBoxLayout()
            self.layout().addLayout(h)
            b = QtWidgets.QRadioButton(v)
            self.buttonMode[k] = b
            b.code = k
            b.base = v
            b.setToolTip(modeToolTips[k])
            b.clicked.connect(partial(self.changeMode, b))
            if self.param.get('mode')==k: b.click()
            self.mode.addButton(b)
            h.addWidget(b)
            h2 = QtWidgets.QHBoxLayout()
            h.addLayout(h2)
            h2.setAlignment(QtCore.Qt.AlignmentFlag.AlignRight)
            v = QtWidgets.QVBoxLayout()
            v.setContentsMargins(30,0,0,0)
            self.layout().addLayout(v)
            
            if k=='index':
                w = QtWidgets.QComboBox()
                h2.addWidget(w)
                for c,(kk,v) in enumerate(targetDict.items()):
                    w.addItem(v,kk)
                    if self.param.get('targetLayers')==kk: w.setCurrentIndex(c)
                self.targetLayers = w
                
            if k=='coords':
                # w = QtWidgets.QPushButton(QtGui.QIcon(os.path.join(self.plugin.imageFolder,'import.png')), "")
                # w.setToolTip(self.plugin.tr("Import config"))
                # w.setMaximumWidth(30)
                # h2.addWidget(w)
                # w.clicked.connect(self.importJson)
                w = QtWidgets.QPushButton(QtGui.QIcon(os.path.join(self.plugin.imageFolder,'add.png')), "")
                w.setToolTip(self.plugin.tr("Add a new tile folder"))
                w.setMaximumWidth(30)
                h2.addWidget(w)
                w.clicked.connect(self.editTile)
                w = ListWidgetUpgrade()
                w.setSelectionMode(QtWidgets.QAbstractItemView.SelectionMode.MultiSelection)
                v.addWidget(w)
                self.tilesConfig = w
                w.itemDoubleClicked.connect(self.editTile)
                w.itemPressed.connect(self.menuTile)
                w.rightClick.connect(self.menuGlobal)
                w.afterClick.connect(self.actuCoords)

        w = QtWidgets.QCheckBox(self.plugin.tr("Don't reload the same tile twice"))
        w.setToolTip(self.plugin.tr("Prevent to reload the same tile when mutilple selection overlays"))
        w.setChecked(self.param.get('noReload'))
        self.layout().addWidget(w)
        self.noReload = w
        
        h = QtWidgets.QHBoxLayout()
        self.layout().addLayout(h)
        w = QtWidgets.QCheckBox(self.plugin.tr("Load in group"))
        h.addWidget(w)
        self.groupActive = w
        w = QtWidgets.QLineEdit(self.param.get('groupName'))
        w.setEnabled(False)
        h.addWidget(w)
        self.groupName = w
        self.groupActive.stateChanged.connect(self.actuGroupChoice)
        self.groupActive.setChecked(self.param.get('groupActive'))
        
        
        h = QtWidgets.QHBoxLayout()
        self.layout().addLayout(h)
        w = QtWidgets.QLabel(self.plugin.tr("Confirm if the number of tiles excede"))
        h.addWidget(w)
        w = QtWidgets.QSpinBox()
        w.setRange(1,99999)
        w.setValue(self.param.get('maxTiles'))
        h.addWidget(w)
        self.maxTiles = w
        h = QtWidgets.QHBoxLayout()
        self.layout().addLayout(h)
        w = QtWidgets.QLabel(self.plugin.tr("Do nothing if area is bigger than"))
        h.addWidget(w)
        w = QtWidgets.QSpinBox()
        w.setRange(1,99999999)
        w.setValue(self.param.get('maxZone'))
        h.addWidget(w)
        self.maxZone = w
        w = QtWidgets.QLabel("km²")
        h.addWidget(w)
        self.loadConfig()
        self.actuCoords()
        buttons = QtWidgets.QDialogButtonBox(QtWidgets.QDialogButtonBox.StandardButton.Ok | QtWidgets.QDialogButtonBox.StandardButton.Cancel)
        self.layout().addWidget(buttons)
        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)
        
    def changeMode(self, b):
        self.mode.value = b.code
        self.actuCoords()
    
    def loadConfig(self):
        self.tilesConfig.clear()
        for f in os.listdir(self.plugin.tileFolder):
            b,e = os.path.splitext(f)
            if e.lower()!='.json': continue
            try: data = json.load(open(os.path.join(self.plugin.tileFolder, f)))
            except: continue 
            self.plugin.json[f] = data     
            n = data.get('alias')
            if n in (None,""): n = b
            it = QtWidgets.QListWidgetItem(n)
            it.file = f
            it.data = data
            self.tilesConfig.addItem(it)
            if f in self.param.get('fromCoords'): 
                self.tilesConfig.setCurrentItem(it, QtCore.QItemSelectionModel.SelectionFlag.Select)
        self.plugin.setIniJson()
        
    def actuCoords(self):
        try: b = self.buttonMode['coords']
        except: return
        txt = b.base
        if self.mode.value=='coords':
            try: txt += f" ({len(self.tilesConfig.selectedItems())})"
            except: pass
        b.setText(txt)
    
    def actuGroupChoice(self):
        self.groupName.setEnabled(self.groupActive.isChecked())
    
    def editTile(self, tile=None):
        dlg = TileConfigure(self, tile)
        dlg.exec()
        if dlg.result():
            d = {
                'alias': dlg.alias.text(),
                'folder': dlg.folder.filePath(),
                'crs': dlg.crs.crs().authid(),
                'corner': (dlg.cornerY.currentData(),dlg.cornerX.currentData()),
                'size': dlg.size.value(),
                'prec': dlg.prec.value(),
                'digits': dlg.digits.value(),
                'order': dlg.order.currentData(),
                'parts': (dlg.parts[0].text(),dlg.parts[1].text(),dlg.parts[2].text()),
            }
            file = dlg.file
            if file is None: file = f"{makeUniqueIdent()}.json"
            self.saveJson(os.path.join(self.plugin.tileFolder, file), d)
            self.loadConfig()
            
    def menuTile(self, tile):
        if not self.tilesConfig.hasRightClicked: return
        self.tilesConfig.hasRightClicked = False
        name = tile.data.get('alias',tile.file)
        if name=="": name = tile.file
        w = QtWidgets.QMenu(self)
        w.addAction(QtGui.QIcon(os.path.join(self.plugin.imageFolder,'delete.png')), self.plugin.tr("Delete")+f" {name}", partial(self.delete, file=tile.file))
        w.addAction(QtGui.QIcon(os.path.join(self.plugin.imageFolder,'export.png')), self.plugin.tr("Export")+f" {name}", partial(self.export, tile=tile))
        w.popup(QtGui.QCursor.pos())
        
    def delete(self, file):
        try: os.remove(os.path.join(self.plugin.tileFolder, file))
        except: pass
        try: del self.plugin.json[file]
        except: pass
        self.loadConfig()
    
    def export(self, tile=None): 
        if tile is None:
            data = {'listJson':list(self.plugin.json.values())}
            d = os.path.join(self.getUserFolder(), "TilePick_config.json")
            file,t = QtWidgets.QFileDialog.getSaveFileName(None, self.plugin.tr("Save file"), directory=d, filter="JSON files (*.json)")
            if file not in ("",None):
                self.saveJson(file, data)
            return
        name = tile.data.get('alias',tile.file)
        if name=="": name = tile.file
        d = os.path.join(self.getUserFolder(), name)
        file,t = QtWidgets.QFileDialog.getSaveFileName(None, self.plugin.tr("Save file"), directory=d, filter="JSON files (*.json)")
        if file not in ("",None):
            self.saveJson(file, tile.data)
    
    def menuGlobal(self):
        if not self.tilesConfig.hasRightClicked: return
        self.tilesConfig.hasRightClicked = False
        w = QtWidgets.QMenu(self)
        w.addAction(QtGui.QIcon(os.path.join(self.plugin.imageFolder,'import.png')), self.plugin.tr("Import config"), self.importJson)
        w.addAction(QtGui.QIcon(os.path.join(self.plugin.imageFolder,'export.png')), self.plugin.tr("Export all"), self.export)
        w.popup(QtGui.QCursor.pos())
    
    def importJson(self):
        files,t = QtWidgets.QFileDialog.getOpenFileNames(None, self.plugin.tr("Open files"), directory=self.getUserFolder(), filter="JSON files (*.json)")
        for n in files:
            try: data = json.load(open(n))
            except: continue 
            if 'listJson' in data:
                for d in data['listJson']:
                    self.saveJson(os.path.join(self.plugin.tileFolder, f"{makeUniqueIdent()}.json"), d)
            else:
                self.saveJson(os.path.join(self.plugin.tileFolder, f"{makeUniqueIdent()}.json"), data)
        self.loadConfig()
        
    def saveJson(self, path, data):
        f = open(path, 'w')
        f.write(json.dumps(data, indent=4))
        f.close()
    
    def getUserFolder(self):
        d = os.path.join(os.environ['USERPROFILE'], 'Documents')
        if not os.path.exists(d): d = os.environ['HOME']
        return d
 
 
class ListWidgetUpgrade(QtWidgets.QListWidget):
    rightClick = QtCore.pyqtSignal()
    afterClick = QtCore.pyqtSignal()
    def __init__(self, *v, **kw):
        super().__init__(*v, **kw)
        self.hasRightClicked = False
        self.setSortingEnabled(True)
        
    def mousePressEvent(self, e):
        if e.button()==QtCore.Qt.MouseButton.RightButton:
            self.hasRightClicked = True
        else:
            self.hasRightClicked = False
        super().mousePressEvent(e)
        self.afterClick.emit()
        
    def mouseReleaseEvent(self, e):
        if e.button()==QtCore.Qt.MouseButton.RightButton:
            self.rightClick.emit()
        super().mouseReleaseEvent(e)
            
 
class TileConfigure(QtWidgets.QDialog):
    def __init__(self, parent, tile):
        super().__init__()
        self.config = parent
        self.setWindowTitle(self.config.plugin.tr('Folder Tile edit'))
        form = QtWidgets.QFormLayout(self)
        self.setMinimumWidth(200)
        self.c = {
            'xy': (self.config.plugin.tr("Lon"), self.config.plugin.tr("Lat")),
            'yx': (self.config.plugin.tr("Lat"), self.config.plugin.tr("Lon")),
        }
        values = {
            'alias': '',
            'folder': '',
            'crs': self.config.plugin.iface.mapCanvas().mapSettings().destinationCrs().authid(),
            'corner': ('upper','left'),
            'size': 1000,
            'prec': 1000,
            'digits': 4,
            'order': 'xy',
            'parts': ('','',''),
        }
        self.file = None
        if tile:
            self.file = tile.file
            values.update(tile.data)
        
        w = QtWidgets.QLineEdit(values['alias'])
        w.setToolTip(self.config.plugin.tr("Visible name in the list widget"))
        self.alias = w
        form.addRow(self.config.plugin.tr('Alias'), w)
        
        w = QgsFileWidget()
        w.setToolTip(self.config.plugin.tr("Directory containing the tiles"))
        w.setStorageMode(QgsFileWidget.StorageMode.GetDirectory)
        w.setFilePath(values['folder'])
        self.folder = w
        w.fileChanged.connect(self.valid)
        form.addRow(self.config.plugin.tr('Folder'), w)
        
        w = QgsProjectionSelectionWidget()
        w.setToolTip(self.config.plugin.tr("Native projection of the tiles"))
        w.setCrs(QgsCoordinateReferenceSystem(values['crs']))
        w.crsChanged.connect(self.valid)
        self.crs = w
        form.addRow(self.config.plugin.tr('Projection'), w)
        
        wg = QtWidgets.QWidget()
        wg.setToolTip(self.config.plugin.tr("Corner position for the ")+self.config.plugin.tr("Lon/Lat coordinates")+self.config.plugin.tr(" of the tile"))
        h = QtWidgets.QHBoxLayout(wg)
        h.setContentsMargins(0,0,0,0)
        h.setSpacing(0)
        w = QtWidgets.QComboBox()
        for e in ('lower','upper'):
            n = self.config.plugin.tr(e[:1].upper()+e[1:])
            w.addItem(n,e)
        w.setCurrentIndex(w.findData(values['corner'][0]))
        self.cornerY = w
        h.addWidget(w)
        w = QtWidgets.QComboBox()
        for e in ('left','right'):
            n = self.config.plugin.tr(e[:1].upper()+e[1:])
            w.addItem(n,e)
        # w.setCurrentText(values['corner'][1])
        w.setCurrentIndex(w.findData(values['corner'][1]))
        self.cornerX = w
        h.addWidget(w)
        form.addRow(self.config.plugin.tr('Corner'), wg)
        
        w = QtWidgets.QSpinBox()
        w.setToolTip(self.config.plugin.tr("Size of the tile (in the crs)"))
        w.setRange(1,1000000000)
        w.setValue(values['size'])
        self.size = w
        form.addRow(self.config.plugin.tr('Size'), w)
        w = QtWidgets.QSpinBox()
        w.setToolTip(self.config.plugin.tr("Precision of the ")+self.config.plugin.tr("Lon/Lat coordinates")+self.config.plugin.tr(" in the name of the tile"))
        w.setRange(1,1000000000)
        w.setValue(values['prec'])
        self.prec = w
        form.addRow(self.config.plugin.tr('Precision'), w)
        w = QtWidgets.QSpinBox()
        w.setToolTip(self.config.plugin.tr("Minimum number of digits for the ")+self.config.plugin.tr("Lon/Lat coordinates"))
        w.setRange(1,19)
        w.setValue(values['digits'])
        self.digits = w
        form.addRow(self.config.plugin.tr('Digits'), w)
        
        w = QtWidgets.QComboBox()
        w.setToolTip(self.config.plugin.tr("Coordinates order to compose the tile name"))
        d = {
            'xy': f"{self.config.plugin.tr('Lon')} - {self.config.plugin.tr('Lat')}", 
            'yx': f"{self.config.plugin.tr('Lat')} - {self.config.plugin.tr('Lon')}", 
        }
        for k,v in d.items():
            w.addItem(v,k)
        w.currentIndexChanged.connect(self.valid)
        w.setCurrentIndex(w.findData(values['order']))
        self.order = w
        form.addRow(self.config.plugin.tr('Order'), w)
        
        wg = QtWidgets.QWidget()
        wg.setToolTip(self.config.plugin.tr("Parts to compose the tile name before, between and after the ")+self.config.plugin.tr("Lon/Lat coordinates"))
        h = QtWidgets.QHBoxLayout(wg)
        h.setContentsMargins(0,0,0,0)
        h.setSpacing(0)
        
        self.parts = {}
        self.xy = {}
        c = self.c[values['order']]
        for i in range(3):
            w = QtWidgets.QLineEdit(values['parts'][i])
            w.textEdited.connect(self.valid)
            h.addWidget(w)
            self.parts[i] = w
            if i<2:
                w = QtWidgets.QLabel(f"{{{c[i]}}}")
                self.xy[i] = w
                h.addWidget(w)
        form.addRow(self.config.plugin.tr('Name parts'), wg)
        
        w = QtWidgets.QLabel()
        
        self.libelle = w
        font = QtGui.QFont()
        font.setItalic(True)
        w.setFont(font)
        form.addRow('', w)
        
        self.buttons = QtWidgets.QDialogButtonBox(QtWidgets.QDialogButtonBox.StandardButton.Ok | QtWidgets.QDialogButtonBox.StandardButton.Cancel)
        self.layout().addWidget(self.buttons)
        self.buttons.accepted.connect(self.accept)
        self.buttons.rejected.connect(self.reject)
        
        self.valid()
        
    def valid(self):
        b = True
        if self.folder.filePath() in (None, ''): b = False
        if not self.crs.crs().isValid(): b = False
        self.buttons.button(QtWidgets.QDialogButtonBox.StandardButton.Ok).setEnabled(b)
        txt = ""
        c = self.c[self.order.currentData()]
        for i in range(3):
            txt += self.parts[i].text()
            if i<2: 
                txt += f"{{{c[i]}}}"
                self.xy[i].setText(f"{{{c[i]}}}")
        self.libelle.setText(txt)
        