# -*- coding: utf-8 -*-

"""

/***************************************************************************
 DemShading
                                 A QGIS plugin
 This plugin simulates natural shadows over an elevation model (DEM)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-06-05
        copyright            : (C) 2019 by Zoran Čučković
        email                : cuckovic.zoran@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Zoran Čučković'
__date__ = '2019-06-05'
__copyright__ = '(C) 2019 by Zoran Čučković'

from os import sys, path

try: from PyQt5.QtCore import QCoreApplication
except :from PyQt6.QtCore import QCoreApplication
 
from qgis.core import (QgsProcessing,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsProcessingMultiStepFeedback,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterRasterDestination,
                        QgsProcessingParameterBoolean,
                      QgsProcessingParameterNumber,
                       QgsProcessingParameterEnum,

                       QgsProcessingUtils,
                       QgsRasterBandStats
                        )
from processing.core.ProcessingConfig import ProcessingConfig

try:
    from osgeo import gdal
except ImportError:
    import gdal

import numpy as np
from .modules import Raster as rs
from .modules.helpers import window_loop, filter3

class NodataAlgorithm(QgsProcessingAlgorithm):
    """
    Fill in the ugly nodata (wrapper for GDAL=. 
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    RADIUS= 'RADIUS'
    SMOOTH = 'SMOOTH'
    OUTPUT = 'OUTPUT'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                self.tr('Digital elevation model')
            ) )
        
        self.addParameter(QgsProcessingParameterNumber(
            self.RADIUS,
            self.tr('Radius (distance) for the fill (in pixels)'),
            QgsProcessingParameterNumber.Integer,
            defaultValue = 10, minValue= 1))
            
        self.addParameter(QgsProcessingParameterNumber(
            self.SMOOTH,
            self.tr('Number of smoothing iterations'),
            QgsProcessingParameterNumber.Integer,
            defaultValue = 0))
            

        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
            self.tr("Output")))
        
    def processAlgorithm(self, parameters, context, feedback):
        """
        Fills NoData values in a raster and saves to a new file using GDAL.
        
        Parameters:
            input_path (str): Path to input raster file with NoData.
            output_path (str): Path to output file (will be overwritten if exists).
            max_search_distance (int): Max pixel distance to search for valid data.
            smoothing_iterations (int): Smoothing steps after fill.
        """

        feedback = QgsProcessingMultiStepFeedback(4, feedback)
        
# 1) -------------- INPUT -----------------
        elevation_model= self.parameterAsRasterLayer(parameters,self.INPUT, context)

        output_model = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)

        radius = self.parameterAsInt(parameters,self.RADIUS, context)
         
        smoothing_iterations= self.parameterAsInt(parameters,self.SMOOTH, context)
        
        dem = rs.Raster(elevation_model)
        
        err, fatal = dem.verify_raster()
        if err: feedback.reportError(err, fatalError = fatal)
                        
        
        # Open source dataset
        src_ds = gdal.Open(elevation_model.source(), gdal.GA_ReadOnly)
        if src_ds is None:
            raise IOError(f"Cannot open input file: {elevation_model}")

        band = src_ds.GetRasterBand(1)
        nodata = band.GetNoDataValue()
        if nodata is None:
            raise ValueError("Input raster has no NoData value defined.")

        # Create output dataset
        driver = gdal.GetDriverByName("GTiff")
        if path.exists(output_model):  driver.Delete(output_model)
        dst_ds = driver.CreateCopy(output_model, src_ds, strict=0)
        dst_band = dst_ds.GetRasterBand(1)

        # Let GDAL handle the interpolation in-place
        gdal.FillNodata(targetBand=dst_band,
                    maskBand=None,
                    maxSearchDist=radius,
                    smoothingIterations=smoothing_iterations)

        # Finalize
        dst_band.FlushCache()
        dst_ds.FlushCache()
        dst_ds, src_ds = None, None
        
        return {self.OUTPUT: output_model}

   
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'fill_nodata'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return 'Fill NoData'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def shortHelpString(self):
        curr_dir = path.dirname(path.realpath(__file__))
        h = ( """
            
             Interpolates nasty no-data values. (Same algorithm as GDAL)
             To fill-in with a FLAT VALUE (eg. 0 = sea), use Raster tools - Fill Nodata.

	    If you find this plugin useful, consider to : 
             <a href='https://ko-fi.com/D1D41HYSW' target='_blank'><img height='30' style='border:0px;height:36px;' src='%s/help/kofi2.webp' /></a>
            """) % curr_dir
		
        return self.tr(h)

    def createInstance(self):
        return NodataAlgorithm()
