# -*- coding: utf-8 -*-

"""
/***************************************************************************
 DemShading - ambient occlusion
 This algorithm simulates ambiental occlusion effect over an elevation model (DEM)
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-02-20
        copyright            : (C) 2020 by Zoran Čučković
 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Zoran Čučković'
__date__ = '2020-02-05'
__copyright__ = '(C) 2020 by Zoran Čučković'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from PyQt5.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterRasterDestination,
                        QgsProcessingParameterBoolean,
                      QgsProcessingParameterNumber,
                       QgsProcessingParameterEnum,


                       QgsProcessingUtils,
                        QgsRasterBandStats,
                       QgsSingleBandGrayRenderer,
                       QgsContrastEnhancement
                        )

from processing.core.ProcessingConfig import ProcessingConfig

import gdal
import numpy as np

from qgis.core import QgsMessageLog # for testing

class OcclusionAlgorithm(QgsProcessingAlgorithm):
    """
    This algorithm simulates ambient lighting over a raster DEM (in input). 
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    #DIRECTION= 'DIRECTION'
    RADIUS= 'RADIUS'
    DENOISE = 'DENOISE'
    ANALYSIS_TYPE='ANALYSIS_TYPE'
    OUTPUT = 'OUTPUT'

    ANALYSIS_TYPES = ['Sky-view', 'Openness']

    output_model = None #for post-processing

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT,
                self.tr('Digital elevation model')
            ) )
        
##        self.addParameter(QgsProcessingParameterNumber(
##            self.DIRECTION,
##            self.tr('Direction (0 to 360°)'),
##            1, 315, False, 0, 360))
        self.addParameter(QgsProcessingParameterEnum (
            self.ANALYSIS_TYPE,
            self.tr('Analysis type'),
            self.ANALYSIS_TYPES,
            defaultValue=0))
                    
        self.addParameter(QgsProcessingParameterNumber(
            self.RADIUS,
            self.tr('Radius (pixels)'),
            0, # QgsProcessingParameterNumber.Integer = 0
            5, False, 0, 100))

        self.addParameter(QgsProcessingParameterBoolean(
            self.DENOISE,
            self.tr('Denoise'),
            False, False)) 
        
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT,
            self.tr("Output file")))
        
    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        def view (offset_y, offset_x, shape, step=1, edge=0):
            """            
            Function returning two matching numpy views for moving window routines.
            - offset_y and offset_x refer to the shift in relation to the analysed (central) cell 
            - size_y and size_x refer to the size of the data matrix (not of the window!)
            - view_in is the shifted view and view_out is the position of central cells
            """
            size_y, size_x = shape

            size_y -= edge; size_x -= edge
             
            x = abs(offset_x) 
            y = abs(offset_y) 
         
            x_in = slice(x + edge , size_x, step) 
            x_out = slice(edge, size_x - x, step)

         
            y_in = slice(y + edge, size_y, step)
            y_out = slice(edge, size_y - y, step)
         
            # the swapping trick    
            if offset_x < 0: x_in, x_out = x_out, x_in                                 
            if offset_y < 0: y_in, y_out = y_out, y_in
         
            # return window view (in) and main view (out)
            return np.s_[y_in, x_in], np.s_[y_out, x_out]


        def window_loop (shape, chunk, axis = 0, reverse = False, overlap = 0):
            """
            Construct a frame to extract chunks of data from gdal
            (and to insert them properly to a numpy matrix)
            """
            xsize, ysize = shape if axis==0 else shape[::-1]

            if reverse :
                steps = np.arange(xsize // chunk, -1, -1 )
                begin = xsize
            else: 
                steps = np.arange(1, xsize // chunk +2 )
                begin =0

            x, y, x_off, y_off = 0,0, xsize, ysize

            for step in steps:

                end = min(int(chunk * step), xsize)
                
                if reverse :  x, x_off = end, begin - end
                else:         x, x_off = begin, end - begin

                begin = end
                
               # ov = overlap * int(step)
                ov = overlap

                ov_left = ov if x > ov else x
                ov_right = ov if (x + x_off + ov < xsize) else (xsize -(x + x_off))

                x_in = x - ov_left
                #this is an offset from x_in !!, not coords
                x_in_off = x_off + ov_right + ov_left

               
                if not axis : gdal_take =(x_in, y, x_in_off, y_off)
                else: gdal_take = (y, x_in, y_off, x_in_off)
              
                    #AXIS SWAP : cannot be handled as transposition,
                    # we need precise coords for GDAL
                in_view = np.s_[:,: x_in_off] if not axis else np.s_[: x_in_off, :]

                x_out = x if ov_left == ov else 0

                x_out_off = x_off + (ov_left if ov_left < ov else 0) + (ov_right if ov_right < ov else 0) 

                if not axis : gdal_put =(x_out, y, x_out_off, y_off)
                else: gdal_put = (y, x_out, y_off, x_out_off)
                
                #todo !! off nije isti za view i za GDAL !!
                sx = slice(0 if ov_left < ov else ov , x_out_off + ov_left + (ov_right if ov_right < ov else 0))

                out_view = np.s_[:, sx] if not axis else np.s_[sx , :]
          
                yield in_view, gdal_take, out_view, gdal_put


            
        # 1) -------------- INPUT -----------------
        
        
        elevation_model= self.parameterAsRasterLayer(parameters,self.INPUT, context)

        if elevation_model.crs().mapUnits() != 0 :
            err= " \n ****** \n ERROR! \n Raster data has to be projected in a metric system!"
            feedback.reportError(err, fatalError = False)
           # raise QgsProcessingException(err)
        #could also use:
        #raise QgsProcessingException(self.invalidSourceError(parameters, self.INPUT))

        if  round(abs(elevation_model.rasterUnitsPerPixelX()),
                    2) !=  round(abs(elevation_model.rasterUnitsPerPixelY()),2):
            
            err= (" \n ****** \n ERROR! \n Raster pixels are irregular in shape " +
                  "(probably due to incorrect projection)!")
            feedback.reportError(err, fatalError = False)
            #raise QgsProcessingException(err)

        self.output_model = self.parameterAsOutputLayer(parameters,self.OUTPUT,context)

        #direction = self.parameterAsDouble(parameters,self.DIRECTION, context)
        denoise = self.parameterAsInt(parameters,self.DENOISE, context)        
       
        radius =self.parameterAsInt(parameters,self.RADIUS, context)

        openness = self.parameterAsInt(parameters,self.ANALYSIS_TYPE, context)
        
        overlap = radius if not denoise else radius +1

             
        dem = gdal.Open(elevation_model.source())
          
        # ! attention: x in gdal is y dimension un numpy (the first dimension)
        xsize, ysize = dem.RasterXSize,dem.RasterYSize
        #assuming one band dem !
        nodata = dem.GetRasterBand(1).GetNoDataValue()
        
        pixel_size = dem.GetGeoTransform()[1]
        
   
        chunk = int(ProcessingConfig.getSetting('DATA_CHUNK')) * 1000000
        chunk = min(chunk // xsize, xsize)

        
        # writing output to dump data chunks
        driver = gdal.GetDriverByName('GTiff')
        ds = driver.Create(self.output_model, xsize,ysize, 1, gdal.GDT_Float32)
        ds.SetProjection(dem.GetProjection())
        ds.SetGeoTransform(dem.GetGeoTransform())
        
             
        chunk_slice = (ysize, chunk + 2 * overlap) 
        
        mx_z = np.zeros( chunk_slice)
        mx_a = np.zeros(mx_z.shape)
        mx_b = np.zeros(mx_z.shape)
                           

        counter = 0
      
            
        for mx_view_in, gdal_take, mx_view_out, gdal_put in window_loop ( 
            shape = (xsize, ysize), 
            chunk = chunk,
            overlap = overlap) :

            
            mx_z[mx_view_in]= dem.ReadAsArray(*gdal_take).astype(float)

            if denoise :
                mx_a[:]=0
                for i in range(-1,2):
                    for j in range(-1,2):
                        view_in, view_out = view(i , j ,mx_z.shape)
                        mx_a[view_out] += mx_z[view_in]
                    
                mx_z = mx_a/9
                
                    
            mx_b[:]=0
            # 8 standard lines, for more : directions = cosines
            for dy in [-1,0,1]:
                for dx in [-1,0,1]:

                    if dx == dy == 0 : continue
                    mx_a [:]= -9999 if openness else 0
                    
                    for r in range (1, radius + 1):

                        view_in, view_out = view(r * dy, r * dx, mx_z.shape)

                        angles = (mx_z[view_in] - mx_z[view_out]) / (r * pixel_size)  

                        mx_a[view_out]= np.maximum(mx_a[view_out], angles)
                    
                    mx_b += mx_a

                    counter += 1
                    feedback.setProgress(100 * chunk * (counter/8) /  xsize)
        
            mx_b /= 8
                   
            if openness:
                out = 1 - (np.arctan(mx_b)/ np.pi + 0.5)
            else:
                out = 1 - (np.arctan(mx_b)* 2 / np.pi )
            
            ds.GetRasterBand(1).WriteArray(out[mx_view_out], * gdal_put[:2])

        ds = None
        
        return {self.OUTPUT: self.output_model}

    def postProcessAlgorithm(self, context, feedback):

        output = QgsProcessingUtils.mapLayerFromString(self.output_model, context)
        provider = output.dataProvider()

        stats = provider.bandStatistics(1,QgsRasterBandStats.All,output.extent(),0)
        mean, sd = stats.mean, stats.stdDev
        
        rnd = QgsSingleBandGrayRenderer(provider, 1)
        ce = QgsContrastEnhancement(provider.dataType(1))
        ce.setContrastEnhancementAlgorithm(QgsContrastEnhancement.StretchToMinimumMaximum)

      #  print (provider.hasStatistics(1, QgsRasterBandStats.All))
       

        ce.setMinimumValue(mean-3*sd)
        ce.setMaximumValue(min(1, mean+2*sd))

        rnd.setContrastEnhancement(ce)

        output.setRenderer(rnd)
        
        output.triggerRepaint()

        return {self.OUTPUT: self.output_model}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Ambient occlusion'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    # no need for groups
##    def group(self):
##        """
##        Returns the name of the group this algorithm belongs to. This string
##        should be localised.
##        """
##        return self.tr(self.groupId())
##
##    def groupId(self):
##        """
##        Returns the unique ID of the group this algorithm belongs to. This
##        string should be fixed for the algorithm, and must not be localised.
##        The group id should be unique within each provider. Group id should
##        contain lowercase alphanumeric characters only and no spaces or other
##        formatting characters.
##        """
##        return 'Shading'

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def shortHelpString(self):
        h = ( """
                Ambient occlusion of a locale is the proportion of the ambient light that it may recieve. This algorithm assumes equal light intensity from all directions (simple ambient lighting).
                Parameters:
                 - Analysis type: Sky-view option will model the light comming from directions above individual pixels (i.e. from the sky). Openness option allows for light sources situated below the horizontal plane.
                 - Radius: The ambient occlusion is caluclated within a defined radius for each raster pixel (computation time is directly dependant on the analysis radius)
                 - Denoise: Apply a simple smoothing filter.
                NB. This algorithm is made for terrain visualisation, it is not appropriate for precise calculation of solar exposition or of incident light.
                For more information see <a href="https://zoran-cuckovic.github.io/QGIS-terrain-shading/"> the manual </a> and the in-depth <a href=https://landscapearchaeology.org/2020/ambient-occlusion/"> blog post</a>.
            """)
		
        return self.tr(h)

    def createInstance(self):
        return OcclusionAlgorithm()
