set -euo pipefail

QGIS_PLUGINS_DIR="/Users/benbishop/Library/Application Support/QGIS/QGIS3/profiles/default/python/plugins"

ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)"
cd "$ROOT"

if [ ! -d "$ROOT/.git" ]; then
  echo "Not a git repo: $ROOT"
  exit 1
fi

if [ ! -d "$QGIS_PLUGINS_DIR" ]; then
  echo "QGIS plugins dir not found: $QGIS_PLUGINS_DIR"
  exit 1
fi

PATTERN="${1:-TerraLink*}"

LATEST_DIR="$(python3 - <<PY
import os, glob
base = r"""$QGIS_PLUGINS_DIR"""
pat  = r"""$PATTERN"""
candidates = [p for p in glob.glob(os.path.join(base, pat)) if os.path.isdir(p)]
if not candidates:
    raise SystemExit(2)
candidates.sort(key=lambda p: os.path.getmtime(p), reverse=True)
print(candidates[0])
PY
)"

if [ -z "$LATEST_DIR" ] || [ ! -d "$LATEST_DIR" ]; then
  echo "No plugin folder found matching pattern: $PATTERN"
  echo "In: $QGIS_PLUGINS_DIR"
  exit 1
fi

VER_OVERRIDE="${2:-}"

if [ -n "$VER_OVERRIDE" ]; then
  VER="$VER_OVERRIDE"
else
  VER="$(python3 - <<PY
import os, re
p = os.path.join(r"""$LATEST_DIR""", "metadata.txt")
if not os.path.exists(p):
    raise SystemExit(3)
txt = open(p, "r", encoding="utf-8").read().splitlines()
ver = None
for line in txt:
    if line.lower().startswith("version="):
        ver = line.split("=",1)[1].strip()
        break
if not ver:
    raise SystemExit(4)
print(ver)
PY
)"
fi

echo "Using QGIS plugin folder: $LATEST_DIR"
echo "Release version: v$VER"

if [ -n "$(git status --porcelain)" ]; then
  echo "Working tree not clean. Commit/stash changes before publishing."
  exit 1
fi

git checkout main
git pull

BR="release/v$VER"
git checkout -b "$BR"

rm -rf "$ROOT/TerraLink"
mkdir -p "$ROOT/TerraLink"

rsync -a --delete \
  --exclude ".git" \
  --exclude "__MACOSX" \
  --exclude "__pycache__" \
  --exclude ".DS_Store" \
  --exclude "TerraLink.zip" \
  --exclude ".venv" \
  --exclude "venv" \
  --exclude ".idea" \
  --exclude ".vscode" \
  "$LATEST_DIR/" "$ROOT/TerraLink/"

find "$ROOT/TerraLink" -name "__pycache__" -type d -prune -exec rm -rf {} +
find "$ROOT/TerraLink" -name ".DS_Store" -type f -delete

python3 - <<PY
p = "TerraLink/metadata.txt"
lines = open(p, "r", encoding="utf-8").read().splitlines(True)
out = []
changed = False
for line in lines:
    if line.lower().startswith("version="):
        out.append(f"version={VER}\n")
        changed = True
    else:
        out.append(line)
if not changed:
    raise SystemExit("metadata.txt missing version= line")
open(p, "w", encoding="utf-8").write("".join(out))
PY

bash "$ROOT/scripts/make_release_zip.sh"

git add -A
git commit -m "chore(release): v$VER"
git tag -a "v$VER" -m "v$VER"
git push -u origin "$BR"
git push origin "v$VER"

gh release create "v$VER" --title "v$VER" --generate-notes
gh release upload "v$VER" TerraLink.zip --clobber

gh pr create --fill

echo "Done: pushed branch $BR, tagged v$VER, created GitHub Release, uploaded TerraLink.zip, opened PR."
