# -*- coding: utf-8 -*-
import os, importlib
from qgis.PyQt.QtCore import QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox
from qgis.core import QgsProcessingProvider, QgsApplication
from .terraclimate_algorithm import TerraClimateClipByYear_GDAL

PLUGIN_PROVIDER_ID = "terraclimate_downloader"

REQUIRED_PKGS = ["xarray", "rioxarray", "numpy"]
OPTIONAL_PKGS = ["netCDF4", "dask"]  # optional, nice to have

class TerraClimateProvider(QgsProcessingProvider):
    def icon(self):
        return QIcon(os.path.join(os.path.dirname(__file__), 'icon.svg'))

    def loadAlgorithms(self):
        self.addAlgorithm(TerraClimateClipByYear_GDAL())

    def id(self):
        return PLUGIN_PROVIDER_ID

    def name(self):
        return self.tr("TerraClimateDownloader")

    def longName(self):
        return self.name()

    def tr(self, text):
        return QCoreApplication.translate("TerraClimateProvider", text)


class TerraClimateProviderPlugin:
    """Registers the Processing provider, adds a Plugins-menu action, and checks requirements."""
    def __init__(self, iface):
        self.iface = iface
        self.provider = None
        self.action = None
        self.menu_text = "TerraClimate Downloader"
        self.icon_path = os.path.join(os.path.dirname(__file__), "icon.svg")

    # --------- Requirements check helpers ---------
    def _check_python_requirements(self):
        missing = []
        optional_missing = []
        for mod in REQUIRED_PKGS:
            try:
                importlib.import_module(mod)
            except Exception:
                missing.append(mod)
        for mod in OPTIONAL_PKGS:
            try:
                importlib.import_module(mod)
            except Exception:
                optional_missing.append(mod)
        return missing, optional_missing

    def _check_gdal_clip_available(self):
        try:
            import processing
            alg = QgsApplication.processingRegistry().algorithmById("gdal:cliprasterbymasklayer")
            return alg is not None
        except Exception:
            return False

    def _show_requirements_dialog(self, *args, **kwargs):
        # Disabled: per v0.0.2 remove requirements check dialog
        return

    def initGui(self):
        # 1) Register processing provider
        self.provider = TerraClimateProvider()
        QgsApplication.processingRegistry().addProvider(self.provider)

        # 2) Requirements check
        missing, optional_missing = self._check_python_requirements()
        gdal_ok = self._check_gdal_clip_available()
        self._show_requirements_dialog(missing, optional_missing, gdal_ok)

        # 3) Add Plugins menu/toolbar action
        self.action = QAction(QIcon(self.icon_path) if os.path.exists(self.icon_path) else QIcon(), self.menu_text, self.iface.mainWindow())
        self.action.setToolTip("Open TerraClimate Downloader")
        self.action.triggered.connect(self.open_tool_dialog)
        self.iface.addPluginToMenu(self.menu_text, self.action)
        self.iface.addToolBarIcon(self.action)

    def unload(self):
        if self.action:
            try:
                self.iface.removePluginMenu(self.menu_text, self.action)
            except Exception:
                pass
            try:
                self.iface.removeToolBarIcon(self.action)
            except Exception:
                pass
            self.action = None
        if self.provider:
            QgsApplication.processingRegistry().removeProvider(self.provider)
            self.provider = None

    def open_tool_dialog(self):
        """Open the Processing dialog for our algorithm from the Plugins menu."""
        try:
            import processing
            alg_id = f"{PLUGIN_PROVIDER_ID}:terraclimate_clip_remote_to_layer_gdalclip"
            processing.execAlgorithmDialog(alg_id, {})
        except Exception as e:
            self.iface.messageBar().pushWarning("TerraClimateDownloader", f"Could not open dialog: {e}. Open Processing Toolbox and search for 'TerraClimateDownloader'.")
