import os
import pandas as pd
from qgis.core import (
    QgsVectorLayer, QgsProject, QgsField, QgsFeature,
    QgsPoint, QgsGeometry, QgsCoordinateReferenceSystem
)
from qgis.gui import QgsProjectionSelectionDialog
from PyQt5.QtWidgets import (
    QDialog, QVBoxLayout, QLabel, QComboBox,
    QPushButton, QFileDialog, QMessageBox, QLineEdit,
    QInputDialog, QAction
)
from PyQt5.QtCore import QVariant
from PyQt5.QtGui import QIcon


class TableToPointLayerConverter(QDialog):
    def __init__(self):
        super().__init__()
        self.setup_ui()
        
    def setup_ui(self):
        self.setWindowTitle("Table to Point Layer Converter")
        self.setMinimumWidth(500)
        layout = QVBoxLayout()
        
        # Input File Section
        layout.addWidget(QLabel("<b>Input File:</b>"))
        self.input_path = QLineEdit()
        self.input_path.setPlaceholderText("Select Excel/CSV file...")
        btn_browse = QPushButton("Browse")
        btn_browse.clicked.connect(self.browse_input)
        layout.addWidget(self.input_path)
        layout.addWidget(btn_browse)
        
        # Coordinate Fields
        layout.addWidget(QLabel("<b>Coordinate Fields:</b>"))
        
        layout.addWidget(QLabel("X Coordinate (Longitude/Easting):"))
        self.x_field = QComboBox()
        layout.addWidget(self.x_field)
        
        layout.addWidget(QLabel("Y Coordinate (Latitude/Northing):"))
        self.y_field = QComboBox()
        layout.addWidget(self.y_field)
        
        layout.addWidget(QLabel("Z Coordinate (Elevation - Optional):"))
        self.z_field = QComboBox()
        self.z_field.addItem("None", None)
        layout.addWidget(self.z_field)
        
        # CRS Selection
        layout.addWidget(QLabel("<b>Coordinate System:</b>"))
        self.crs_label = QLabel("Selected CRS: EPSG:4326 - WGS 84")
        self.crs = QgsCoordinateReferenceSystem("EPSG:4326")
        btn_crs = QPushButton("Select CRS")
        btn_crs.clicked.connect(self.select_crs)
        layout.addWidget(btn_crs)
        layout.addWidget(self.crs_label)
        
        # Convert Button
        btn_convert = QPushButton("Convert to Point Layer")
        btn_convert.clicked.connect(self.convert)
        btn_convert.setStyleSheet("background-color: #4CAF50; color: white; padding: 8px;")
        layout.addWidget(btn_convert)
        
        self.setLayout(layout)
        self.df = None
    
    def browse_input(self):
        """Select input file"""
        path, _ = QFileDialog.getOpenFileName(
            self, "Select Input File", "",
            "Excel Files (*.xlsx *.xls);;CSV Files (*.csv)"
        )
        if path:
            self.input_path.setText(path)
            self.load_fields(path)
    
    def load_fields(self, path):
        """Load fields from input file"""
        try:
            ext = os.path.splitext(path)[1].lower()
            if ext == '.csv':
                self.df = pd.read_csv(path)
            else:
                sheets = pd.ExcelFile(path).sheet_names
                sheet, ok = QInputDialog.getItem(
                    self, "Select Sheet", "Choose worksheet:", sheets, 0, False
                )
                if ok:
                    self.df = pd.read_excel(path, sheet_name=sheet)
            
            if self.df is not None:
                for combo in [self.x_field, self.y_field, self.z_field]:
                    combo.clear()
                    if combo == self.z_field:
                        combo.addItem("None", None)
                    
                    for col in self.df.columns:
                        combo.addItem(col, col)
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Failed to load file:\n{str(e)}")
    
    def select_crs(self):
        """Select coordinate system"""
        dialog = QgsProjectionSelectionDialog()
        if dialog.exec_():
            self.crs = dialog.crs()
            self.crs_label.setText(f"Selected CRS: {self.crs.description()}")
    
    def convert(self):
        """Convert to point layer"""
        try:
            # Validate inputs
            if not self.validate_inputs():
                return
                
            # Create vector layer in memory
            layer = self.create_vector_layer()
            
            if not layer.isValid():
                QMessageBox.critical(self, "Error", "Failed to create temporary layer!")
                return
                
            # Add to project
            QgsProject.instance().addMapLayer(layer)
            
            QMessageBox.information(self, "Success", "Point layer created successfully!")
            self.close()
            
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Conversion failed:\n{str(e)}")
    
    def validate_inputs(self):
        """Validate user inputs"""
        if not self.input_path.text() or not os.path.exists(self.input_path.text()):
            QMessageBox.warning(self, "Error", "Please select a valid input file")
            return False
            
        if self.x_field.currentIndex() < 0 or self.y_field.currentIndex() < 0:
            QMessageBox.warning(self, "Error", "Please select X and Y coordinate fields")
            return False
            
        return True
    
    def create_vector_layer(self):
        """Create vector layer with features"""
        has_z = self.z_field.currentData() is not None
        geometry_type = "PointZ" if has_z else "Point"
        
        layer = QgsVectorLayer(
            f"{geometry_type}?crs={self.crs.authid()}",
            "Temporary_Points_Layer",
            "memory"
        )
        
        provider = layer.dataProvider()
        
        # Add fields
        fields = []
        for col in self.df.columns:
            dtype = self.df[col].dtype
            if pd.api.types.is_integer_dtype(dtype):
                field_type = QVariant.Int
            elif pd.api.types.is_numeric_dtype(dtype):
                field_type = QVariant.Double
            else:
                field_type = QVariant.String
            fields.append(QgsField(col, field_type))
        
        provider.addAttributes(fields)
        layer.updateFields()
        
        # Add features
        x_col = self.x_field.currentData()
        y_col = self.y_field.currentData()
        z_col = self.z_field.currentData()
        
        features = []
        for _, row in self.df.iterrows():
            try:
                x = float(row[x_col])
                y = float(row[y_col])
                z = float(row[z_col]) if z_col else 0.0
                
                point = QgsPoint(x, y, z) if has_z else QgsPoint(x, y)
                feat = QgsFeature()
                feat.setGeometry(QgsGeometry(point))
                feat.setAttributes(list(row))
                features.append(feat)
            except Exception as e:
                print(f"Skipped row: {str(e)}")
                continue
        
        provider.addFeatures(features)
        layer.updateExtents()
        return layer


class TableToPointLayerPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.dlg = None
        
    def initGui(self):
        # Get the plugin directory
        plugin_dir = os.path.dirname(__file__)
        # Construct full path to icon
        icon_path = os.path.join(plugin_dir, 'icon.png')
        
        # Check if icon exists and load it
        if os.path.exists(icon_path):
            icon = QIcon(icon_path)
        else:
            # Use a default QGIS icon if custom icon not found
            icon = QIcon(":/images/themes/default/mActionAdd.svg")
        
        # Create action with icon
        self.action = QAction(
            icon,
            "Table to Point Layer",
            self.iface.mainWindow()
        )
        self.action.triggered.connect(self.run)
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&Table Tools", self.action)
        
    def unload(self):
        self.iface.removePluginMenu("&Table Tools", self.action)
        self.iface.removeToolBarIcon(self.action)
        
    def run(self):
        if self.dlg is None:
            self.dlg = TableToPointLayerConverter()
        self.dlg.show()
        self.dlg.raise_()
        self.dlg.activateWindow()