"""
Release Notes Dialog Module

This module handles the display of release notes when the plugin is first installed
or updated to a new version.
"""

from PyQt5.QtWidgets import (QDialog, QTextBrowser, QVBoxLayout, QPushButton, 
                            QDialogButtonBox, QLabel)
from PyQt5.QtCore import QSettings, Qt
import os
import logging

try:
    from .logger import Logger
except ImportError:
    from logger import Logger

logger = Logger(__name__).get()

class ReleaseNotesDialog(QDialog):
    """Dialog for displaying plugin release notes"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("StreetSmart Plugin - What's New")
        self.setMinimumSize(700, 500)
        self._setup_ui()
        self.load_release_notes()

    def _setup_ui(self):
        """Setup the dialog's user interface"""
        layout = QVBoxLayout(self)
        
        # Add header label
        header = QLabel("StreetSmart Plugin Updates")
        header.setStyleSheet("font-size: 14pt; font-weight: bold; margin-bottom: 10px;")
        layout.addWidget(header)
        
        # Create text browser for release notes
        self.text_browser = QTextBrowser()
        self.text_browser.setOpenExternalLinks(True)
        self.text_browser.setStyleSheet("""
            QTextBrowser {
                border: 1px solid #cccccc;
                background-color: #ffffff;
                padding: 10px;
            }
        """)
        layout.addWidget(self.text_browser)
        
        # Create button box with custom buttons
        button_box = QDialogButtonBox()
        ok_button = QPushButton("OK (Show again on next start)")
        dont_show = QPushButton("Don't show again")
        
        # Add tooltip for clarity
        ok_button.setToolTip("Close the release notes. They will appear again when QGIS is restarted.")
        dont_show.setToolTip("Never show the release notes again.")
        
        button_box.addButton(ok_button, QDialogButtonBox.AcceptRole)
        button_box.addButton(dont_show, QDialogButtonBox.ActionRole)
        layout.addWidget(button_box)
        
        # Connect buttons
        ok_button.clicked.connect(self.ok_clicked)
        dont_show.clicked.connect(self.dont_show_clicked)

    def load_release_notes(self):
        """Load and display release notes from markdown file"""
        try:
            release_notes_path = os.path.join(os.path.dirname(__file__), 'release_notes.md')
            logger.debug(f"Loading release notes from: {release_notes_path}")
            
            with open(release_notes_path, 'r', encoding='utf-8') as f:
                content = f.read()
                self.text_browser.setMarkdown(content)
                logger.debug("Release notes loaded successfully")
                
        except Exception as e:
            error_message = f"Error loading release notes: {str(e)}"
            logger.error(error_message)
            self.text_browser.setPlainText(error_message)

    def ok_clicked(self):
        """Handle 'OK' button click - will show again on next QGIS start"""
        try:
            settings = QSettings()
            # Remove any existing settings to ensure it shows on next start
            settings.remove("StreetSmart/permanently_hide_notes")
            settings.remove("StreetSmart/show_release_notes")
            logger.debug("Release notes settings cleared - will show on next start")
            self.accept()
        except Exception as e:
            logger.error(f"Error handling OK click: {str(e)}")

    def dont_show_clicked(self):
        """Handle 'Don't show again' button click - will never show again"""
        try:
            settings = QSettings()
            settings.setValue("StreetSmart/permanently_hide_notes", True)
            logger.debug("Release notes permanently disabled")
            self.accept()
        except Exception as e:
            logger.error(f"Error saving release notes preference: {str(e)}")


def should_show_release_notes():
    """Check if release notes should be shown
    
    Returns:
        bool: True if release notes should be shown, False otherwise
    """
    try:
        settings = QSettings()
        
        # Only check for permanent hide setting
        if settings.value("StreetSmart/permanently_hide_notes", False, type=bool):
            logger.debug("Release notes are permanently hidden")
            return False
        
        logger.debug("Release notes should be shown")
        return True
        
    except Exception as e:
        logger.error(f"Error checking release notes display status: {str(e)}")
        return False


def show_release_notes(parent_widget):
    """Show release notes dialog if appropriate
    
    Args:
        parent_widget: Parent widget for the dialog
    """
    try:
        if should_show_release_notes():
            dialog = ReleaseNotesDialog(parent_widget)
            dialog.exec_()
    except Exception as e:
        logger.error(f"Error showing release notes: {str(e)}")