# 2017 - Antonio Carlon

from PyQt4.QtCore import *
from PyQt4.QtGui import *
from math import sqrt

import resources
import threading
import qgis.utils

from qgis.core import *
from qgis.gui import *

name = "Split Features On Steroids"
areaUnits = {0 : "m<sup>2</sup>", 1 : "km<sup>2</sup>", 2 : "ft<sup>2</sup>", 3 : "sq yd", 4 : "sq mi", 5 : "ha", 6 : "ac", 7 : "M<sup>2</sup>", 8 : "deg<sup>2</sup>", 9 : ""}
maxDistanceHitTest = 5

class SplitFeaturesOnSteroidsPlugin:
	mapTool = None

	def __init__(self, iface):
		self.iface = iface

	def initGui(self):
		self.toolbar = self.iface.addToolBar(name)
		self.toolbar.setObjectName(name)
		
		icon = QIcon(":/plugins/SplitPolygonShowingAreas/icon.png")
		self.action = QAction(icon, name, self.iface.mainWindow())
		self.action.setCheckable(True)
		QObject.connect(self.action, SIGNAL("triggered()"), self.onClick)
		self.toolbar.addAction(self.action)

		self.actionMoveVertices = QAction(QIcon(":/plugins/SplitPolygonShowingAreas/moveVertices.png"), "Move Vertices", self.iface.mainWindow())
		self.actionMoveVertices.setCheckable(True)
		QObject.connect(self.actionMoveVertices, SIGNAL("triggered()"), self.onClickMoveVertices)
		self.toolbar.addAction(self.actionMoveVertices)

		self.iface.addPluginToMenu(name, self.action)
		self.iface.addPluginToMenu(name, self.actionMoveVertices)
		self.help_action = QAction("Help", self.iface.mainWindow())
		QObject.connect(self.help_action, SIGNAL("triggered()"), self.onHelp)
		self.iface.addPluginToMenu(name, self.help_action)

		self.iface.currentLayerChanged.connect(self.currentLayerChanged)

		self.enableTool()

	def disableAll(self):
		self.actionMoveVertices.setChecked(False)
		self.actionMoveVertices.setEnabled(False)

	def unload(self):
		self.iface.removePluginMenu(name, self.action)
		self.iface.removePluginMenu(name, self.actionMoveVertices)
		self.iface.removePluginMenu(name, self.help_action)
		self.iface.removeToolBarIcon(self.action)
		self.iface.removeToolBarIcon(self.actionMoveVertices)

	def onHelp(self):
		qgis.utils.showPluginHelp(filename="index")

	def onClick(self):
		self.disableAll()
		if not self.action.isChecked():
			if self.mapTool != None:
				self.mapTool.stopCapturing()
			self.iface.mapCanvas().unsetMapTool(self.mapTool)
			self.mapTool = None
			return
		layer = self.iface.activeLayer()
		if layer == None or not isinstance(layer, QgsVectorLayer) or (layer.wkbType() != QGis.WKBPolygon and layer.wkbType() != QGis.WKBMultiPolygon):
			self.iface.messageBar().pushMessage("No Polygon Vectorial Layer Selected", "Select a Polygon Vectorial Layer first", level=QgsMessageBar.WARNING)
			self.action.setChecked(False)
			return
		selectedFeatures = layer.selectedFeatures()
		if selectedFeatures == None or len(selectedFeatures) == 0:
			self.iface.messageBar().pushMessage("No Features Selected", "Select some features first", level=QgsMessageBar.WARNING)
			self.action.setChecked(False)
			return
		
		self.action.setChecked(True)
		self.mapTool = SplitMapTool(self.iface.mapCanvas(), layer, selectedFeatures, self.actionMoveVertices)
		self.mapTool.setAction(self.action)
		self.iface.mapCanvas().setMapTool(self.mapTool)
		self.actionMoveVertices.setEnabled(True)

	def onClickMoveVertices(self):
		if not self.actionMoveVertices.isChecked():
			if self.mapTool != None:
				self.mapTool.stopMovingVertices()
			return

		self.actionMoveVertices.setChecked(True)
		self.mapTool.startMovingVertices()

	def currentLayerChanged(self):
		if self.mapTool != None:
			self.mapTool.stopCapturing()

		layer = self.iface.activeLayer()
		if layer != None:
			try:
				layer.editingStarted.disconnect(self.layerEditingChanged)
			except:
				pass
			try:
				layer.editingStopped.disconnect(self.layerEditingChanged)
			except:
				pass
			try:
				layer.selectionChanged .disconnect(self.layerEditingChanged)
			except:
				pass
			
			if isinstance(layer, QgsVectorLayer):
				layer.editingStarted.connect(self.layerEditingChanged)
				layer.editingStopped.connect(self.layerEditingChanged)
				layer.selectionChanged.connect(self.layerSelectionChanged)
			
		self.enableTool()		

	def layerEditingChanged(self):
		self.enableTool()	

	def layerSelectionChanged(self):
		self.enableTool()

	def enableTool(self):
		self.disableAll()
		self.action.setEnabled(False)
		layer = self.iface.activeLayer()
		
		if layer != None and isinstance(layer, QgsVectorLayer):
			selectedFeatures = layer.selectedFeatures()
			if isinstance(layer, QgsVectorLayer) and (layer.wkbType() == QGis.WKBPolygon or layer.wkbType() == QGis.WKBMultiPolygon) and selectedFeatures != None and len(selectedFeatures) > 0 and layer.isEditable():
				self.action.setEnabled(True)

class SplitMapTool(QgsMapToolEdit):
	def __init__(self, canvas, layer, selectedFeatures, actionMoveVertices):
		super(SplitMapTool, self).__init__(canvas)
		self.canvas = canvas
		self.canvas.renderStarting.connect(self.mapCanvasChanged)
		self.canvas.keyReleased.connect(self.keyReleaseEvent)
		self.layer = layer
		self.layer.editingStopped.connect(self.stopCapturing)
		self.selectedFeatures = selectedFeatures
		self.actionMoveVertices = actionMoveVertices
		self.rubberBand = None
		self.tempRubberBand = None
		self.capturedPoints = []
		self.capturing = False
		self.setCursor(Qt.CrossCursor)
		self.proj = QgsProject.instance()
		self.scene = canvas.scene()
		self.labels = []
		self.vertices = []
		self.calculator = QgsDistanceArea()
		self.calculator.setSourceCrs(self.layer.dataProvider().crs())
		self.calculator.setEllipsoid(self.layer.dataProvider().crs().ellipsoidAcronym())
		self.calculator.setEllipsoidalMode(self.layer.dataProvider().crs().geographicFlag())
		self.drawingLine = False
		self.movingVertices = False
		self.showingVertices = False
		self.movingVertex = -1

	def mapCanvasChanged(self):
		self.redrawAreas()
		if self.showVertices:
			self.redrawVertices()

	def canvasMoveEvent(self, event):
		if self.drawingLine:
			if self.tempRubberBand != None and self.capturing:
				mapPoint = self.toMapCoordinates(event.pos())
				self.tempRubberBand.movePoint(mapPoint)
				self.redrawAreas(event.pos())

		if self.movingVertices and self.movingVertex >= 0:
			layerPoint = self.toLayerCoordinates(self.layer, event.pos())
			self.capturedPoints[self.movingVertex] = layerPoint
			self.redrawRubberBand()
			self.redrawVertices()
			self.redrawAreas()

	def redrawAreas(self, mousePos=None):
		self.deleteLabels()

		if self.capturing:
			for i in range(len(self.selectedFeatures)):
				geometry = QgsGeometry(self.selectedFeatures[i].geometry())
				movingPoints = list(self.capturedPoints)
				
				if mousePos != None:
					movingPoints.append(self.toLayerCoordinates(self.layer, mousePos))

				result, newGeometries, topoTestPoints = geometry.splitGeometry(movingPoints, self.proj.topologicalEditing())

				self.addLabel(geometry)
				if newGeometries != None and len(newGeometries) > 0:
					for i in range(len(newGeometries)):
						self.addLabel(newGeometries[i])

	def addLabel(self, geometry):
		area = self.calculator.measureArea(geometry)
		labelPoint = geometry.pointOnSurface().vertexAt(0)
		label = QGraphicsTextItem("%.2f" % round(area,2))
		label.setHtml("<div style=\"color:#ffffff;background:#111111;padding:5px\">"
			+ "%.2f" % round(area,2) + " "
			+ areaUnits[self.calculator.areaUnits()]
			+ "</div>")
		label.setPos(self.toCanvasCoordinates(self.toMapCoordinates(self.layer, labelPoint)))

		self.scene.addItem(label)
		self.labels.append(label)

	def deleteLabels(self):
		for i in range(len(self.labels)):
			self.scene.removeItem(self.labels[i])
		self.labels = []

	def canvasPressEvent(self, event):
		if self.movingVertices:
			for i in range(len(self.capturedPoints)):
				currentVertex = self.toCanvasCoordinates(self.toMapCoordinates(self.layer, self.capturedPoints[i]))
				if self.distance(event.pos(), currentVertex) <= maxDistanceHitTest:
					self.movingVertex = i
					break

	def distance(self, eventPos, vertexPos):
		return sqrt((eventPos.x() - vertexPos.x())**2 + (eventPos.y() - vertexPos.y())**2)

	def canvasReleaseEvent(self, event):
		if not self.movingVertices:
			if event.button() == Qt.LeftButton:
				if not self.capturing:
					self.startCapturing()
				self.addVertex(event.pos())
			elif event.button() == Qt.RightButton:
				self.doSplit()
				self.stopCapturing()
		else:
			if event.button() == Qt.RightButton:
				self.actionMoveVertices.setChecked(False)
				self.doSplit()
				self.stopCapturing()

		self.movingVertex = -1

	def keyReleaseEvent(self, event):
		if event.key() == Qt.Key_Escape:
			self.stopCapturing()
		if event.key() == Qt.Key_Backspace or event.key() == Qt.Key_Delete:
			self.removeLastVertex()
			event.ignore()
		if event.key() == Qt.Key_Return or event.key() == Qt.Key_Enter:
			self.stopCapturing()
			self.doSplit()

	def doSplit(self):
		if self.capturedPoints != None:
			self.layer.splitFeatures(self.capturedPoints, self.proj.topologicalEditing())

	def startCapturing(self):
		self.prepareRubberBand()
		self.prepareTempRubberBand()

		self.drawingLine = True
		self.capturing = True

	def prepareRubberBand(self):
		color = QColor("red")
		color.setAlphaF(0.78)

		self.rubberBand = QgsRubberBand(self.canvas, QGis.Line)
		self.rubberBand.setWidth(1)
		self.rubberBand.setColor(color)
		self.rubberBand.show()

	def prepareTempRubberBand(self):
		color = QColor("red")
		color.setAlphaF(0.78)

		self.tempRubberBand = QgsRubberBand(self.canvas, QGis.Line)
		self.tempRubberBand.setWidth(1)
		self.tempRubberBand.setColor(color)
		self.tempRubberBand.setLineStyle(Qt.DotLine)
		self.tempRubberBand.show()

	def redrawRubberBand(self):
		self.canvas.scene().removeItem(self.rubberBand)
		self.prepareRubberBand()
		for i in range(len(self.capturedPoints)):
			vertexCoord = self.toMapCoordinates(self.layer, self.capturedPoints[i])
			self.rubberBand.addPoint(vertexCoord)

	def redrawTempRubberBand(self):
		if self.tempRubberBand != None:
			self.tempRubberBand.reset(QGis.Line)
			self.tempRubberBand.addPoint(self.toMapCoordinates(self.layer, self.capturedPoints[len(self.capturedPoints) - 1]))

	def stopCapturing(self):
		self.deleteLabels()
		self.deleteVertices()
		if self.rubberBand:
			self.canvas.scene().removeItem(self.rubberBand)
			self.rubberBand = None
		if self.tempRubberBand:
			self.canvas.scene().removeItem(self.tempRubberBand)
			self.tempRubberBand = None
		self.drawingLine = False
		self.movingVertices = False
		self.showingVertices = False
		self.capturing = False
		self.capturedPoints = []
		self.canvas.refresh()

	def addVertex(self, canvasPoint):
		mapPoint = self.toMapCoordinates(canvasPoint)
		layerPoint = self.toLayerCoordinates(self.layer, canvasPoint)

		self.rubberBand.addPoint(mapPoint)
		self.capturedPoints.append(layerPoint)

		self.tempRubberBand.reset(QGis.Line)
		self.tempRubberBand.addPoint(mapPoint)

	def removeLastVertex(self):
		if not self.capturing: return

		rubberBandSize = self.rubberBand.numberOfVertices()
		tempRubberBandSize = self.tempRubberBand.numberOfVertices()
		numPoints = len(self.capturedPoints)

		if rubberBandSize < 1 or numPoints < 1:
			return

		self.rubberBand.removePoint(-1)

		if rubberBandSize > 1:
			if tempRubberBandSize > 1:
				point = self.rubberBand.getPoint(0, rubberBandSize-2)
				self.tempRubberBand.movePoint(tempRubberBandSize-2, point)
		else:
			self.tempRubberBand.reset(self.bandType())

		del self.capturedPoints[-1]

	def startMovingVertices(self):
		self.movingVertices = True
		self.showingVertices = True
		self.drawingLine = False
		self.canvas.scene().removeItem(self.tempRubberBand)
		self.redrawVertices()
		self.redrawAreas()

	def stopMovingVertices(self):
		self.movingVertices = False
		self.showingVertices = False
		self.drawingLine = True
		self.deleteVertices()
		self.redrawRubberBand()
		self.redrawTempRubberBand()
		self.canvas.scene().addItem(self.tempRubberBand)

	def showVertices(self):
		for i in range(len(self.capturedPoints)):
			vertexCoords = self.toCanvasCoordinates(self.toMapCoordinates(self.layer, self.capturedPoints[i]))
			if i == self.movingVertex:
				vertex = self.scene.addRect(vertexCoords.x() - 5, vertexCoords.y() - 5, 10, 10, QPen(QColor("green")), QBrush(QColor("green")))
				self.vertices.append(vertex)
			else:
				vertex = self.scene.addRect(vertexCoords.x() - 4, vertexCoords.y() - 4, 8, 8, QPen(QColor("red")), QBrush(QColor("red")))
				self.vertices.append(vertex)

	def deleteVertices(self):
		for i in range(len(self.vertices)):
			self.scene.removeItem(self.vertices[i])
		self.vertices = []

	def redrawVertices(self):
		self.deleteVertices()
		self.showVertices()