from qgis.PyQt.QtWidgets import (QAction, QFileDialog, QMessageBox, 
                                 QPushButton, QDialog, QVBoxLayout, 
                                 QProgressBar, QLabel)

from qgis.PyQt.QtGui import QIcon
from qgis.core import QgsProject, QgsCoordinateReferenceSystem
from qgis.gui import QgsProjectionSelectionDialog
import os

class SmartLoaderPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        
    def initGui(self):
        # Create action
        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        self.action = QAction(
            QIcon(icon_path),
            "Smart GIS Loader",
            self.iface.mainWindow()
        )
        
        # Connect action to run method
        self.action.triggered.connect(self.run)
        
        # Add toolbar button and menu item
        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&Smart GIS Tools", self.action)
        
    def unload(self):
        # Remove plugin menu item and icon
        self.iface.removePluginMenu("&Smart GIS Tools", self.action)
        self.iface.removeToolBarIcon(self.action)
        
    def run(self):
        # File selection dialog
        file_dialog = QFileDialog()
        file_dialog.setWindowTitle('Select GIS Files')
        file_dialog.setFileMode(QFileDialog.ExistingFiles)
        file_dialog.setNameFilter(
            "All GIS Formats (*.shp *.gpkg *.geojson *.tif *.tiff *.img *.kml *.kmz *.dxf);;"
            "Vector Files (*.shp *.gpkg *.geojson *.kml *.kmz);;"
            "Raster Files (*.tif *.tiff *.img);;"
            "CAD Files (*.dxf);;"
            "All Files (*)"
        )
        
        if not file_dialog.exec_():
            return
            
        selected_files = file_dialog.selectedFiles()
        if not selected_files:
            return
            
        # CRS selection dialog
        crs_dialog = QgsProjectionSelectionDialog()
        if not crs_dialog.exec_():
            QMessageBox.information(
                self.iface.mainWindow(),
                "Information",
                "No CRS selected, using files' native CRS"
            )
            selected_crs = None
        else:
            selected_crs = crs_dialog.crs()
            
        # Progress dialog
        progress_dialog = ProgressDialog(self.iface.mainWindow())
        progress_dialog.show()
        progress_dialog.setMaximum(len(selected_files))
        
        # Process files
        for i, file_path in enumerate(selected_files):
            progress_dialog.setValue(i + 1)
            progress_dialog.setLabelText(f"Loading {os.path.basename(file_path)}...")
            
            try:
                base_name = os.path.splitext(os.path.basename(file_path))[0]
                
                if file_path.lower().endswith('.dxf'):
                    uri = f"{file_path}|layername=entities"
                    layer = self.iface.addVectorLayer(uri, base_name, "ogr")
                elif file_path.lower().endswith(('.shp', '.gpkg', '.geojson', '.kml', '.kmz')):
                    layer = self.iface.addVectorLayer(file_path, base_name, "ogr")
                elif file_path.lower().endswith(('.tif', '.tiff', '.img')):
                    layer = self.iface.addRasterLayer(file_path, base_name)
                else:
                    continue
                    
                if layer and layer.isValid():
                    if selected_crs and selected_crs.isValid():
                        layer.setCrs(selected_crs)
                    QgsProject.instance().addMapLayer(layer)
                    
            except Exception as e:
                QMessageBox.warning(
                    self.iface.mainWindow(),
                    "Warning",
                    f"Failed to load {file_path}\nError: {str(e)}"
                )
                
        progress_dialog.close()
        QMessageBox.information(
            self.iface.mainWindow(),
            "Completed",
            f"Successfully loaded {len(selected_files)} files"
        )

class ProgressDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Loading Files")
        self.setModal(True)
        self.layout = QVBoxLayout()
        self.progress_bar = QProgressBar(self)
        self.label = QLabel("Preparing to load files...", self)
        self.layout.addWidget(self.label)
        self.layout.addWidget(self.progress_bar)
        self.setLayout(self.layout)
        
    def setMaximum(self, value):
        self.progress_bar.setMaximum(value)
        
    def setValue(self, value):
        self.progress_bar.setValue(value)
        
    def setLabelText(self, text):
        self.label.setText(text)