# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SmartLineLabeler
                                 A QGIS plugin
 create dynamic stacked labels for linestring layers
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-08-26
        git sha              : $Format:%H$
        copyright            : (C) 2020-2025 by Christoph Candido
        email                : christoph.candido@gmx.at
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os.path
import math
import collections
from qgis.PyQt.QtCore import ( QSettings, QTranslator, QCoreApplication, Qt, QVariant )
from qgis.PyQt.QtGui import ( QIcon, QCursor, QPixmap, QPainter )
from qgis.PyQt.QtWidgets import ( QAction, QComboBox, QLabel, QPushButton, QSizePolicy, QToolButton, QMessageBox )
from qgis.PyQt.QtSvg import QSvgRenderer

from qgis.core import (
    Qgis,
    QgsMapLayer,
    QgsProject,
    QgsRectangle,
    QgsVectorLayerJoinInfo,
    QgsCoordinateTransform,
    QgsGeometry,
    QgsPointXY,
    QgsFeatureRequest,
    QgsFeature,
    QgsExpressionContextUtils,
    QgsField,
    QgsProperty,
    QgsPalLayerSettings,
    QgsExpression,
    QgsExpressionContext,
    edit,
    QgsRenderContext,
    QgsApplication
)

from qgis.gui import QgsMapTool, QgsRubberBand

# Import the code for the dialog
from .SmartLineLabeler_dialog import SmartLineLabelerConfigDialog


class SmartLineLabeler:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "SmartLineLabeler_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&SmartLineLabeler")
        self.toolbar = self.iface.addToolBar("SmartLineLabeler")
        self.toolbar.setObjectName("SmartLineLabeler")

        self.alignLabel = "L"
        self.offset = 5
        self.ticklength = 5
        self.units = Qgis.RenderUnit.Millimeters
        self.nearest = True

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("SmartLineLabeler", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        checkable_flag=False,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)
        action.setCheckable(checkable_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            # self.iface.addToolBarIcon(action)
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        icon_path = os.path.join(os.path.dirname(__file__), "icons/icon.svg")
        self.action = self.add_action(
            icon_path,
            text=self.tr("Create dynamic stacked Linestring Labels"),
            checkable_flag=True,
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        icon_path = os.path.join(os.path.dirname(__file__), "icons/settings.svg")
        self.add_action(
            icon_path,
            text=self.tr("Label Settings"),
            checkable_flag=False,
            callback=self.configure,
            parent=self.iface.mainWindow(),
        )

    def configure(self):
        dlg = SmartLineLabelerConfigDialog()
        if dlg.exec():
            self.offset = float(
                QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable(
                    "sll_label_offset"
                )
            )
            self.ticklength = float(
                QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable(
                    "sll_label_ticklength"
                )
            )
            self.units = Qgis.RenderUnit[
                QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable(
                    "sll_label_units"
                )
            ]

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        self.uncheckToolBarButton()
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&SmartLineLabeler"), action)
            self.iface.removeToolBarIcon(action)
        del self.toolbar

    def updateLabels(self):
        layerID = self.cboLblLayer.itemData(self.cboLblLayer.currentIndex())
        lblLayer = self.getLayerByID(layerID)
        if not lblLayer:
            self.iface.messageBar().pushMessage(
                "Info", "No Labeling Layer found!", level=Qgis.Info, duration=3
            )
            return

        layerID = self.cboLayer.itemData(self.cboLayer.currentIndex())
        srcLayer = self.getLayerByID(layerID)
        if not srcLayer:
            self.iface.messageBar().pushMessage(
                "Info", "No Line Layer found!", level=Qgis.Info, duration=3
            )
            return

        lblLayerID = self.cboID.currentText()

        cursor = QCursor()
        cursor.setShape(Qt.CursorShape.WaitCursor)
        self.iface.mapCanvas().setCursor(cursor)
        # build a request to filter the features based on an attribute
        request = QgsFeatureRequest().setFilterExpression(lblLayerID + " is not NULL")

        # we don't need attributes or geometry, skip them to minimize overhead.
        # these lines are not strictly required but improve performance
        request.setSubsetOfAttributes([])
        request.setFlags(QgsFeatureRequest.NoGeometry)

        # loop over the features and delete
        ids = [f.id() for f in lblLayer.getFeatures(request)]
        lblLayer.dataProvider().deleteFeatures(ids)

        crsSrc = lblLayer.crs()
        crsDest = srcLayer.crs()
        xform = QgsCoordinateTransform(crsSrc, crsDest, QgsProject.instance())

        fieldList = list(lblLayer.dataProvider().fields())
        hasAlignment = False
        hasNearest = False
        hasAngle = False

        if "label_align" in [f.name() for f in fieldList]:
            hasAlignment = True
        if "nearest" in [f.name() for f in fieldList]:
            hasNearest = True
        if "angle" in [f.name() for f in fieldList]:
            hasAngle = True

        features = list(lblLayer.getFeatures())

        for feat in features:
            alignLabel = "L"
            nearest = True
            if hasAlignment:
                if feat["label_align"]:
                    alignLabel = feat["label_align"]
            if hasNearest:
                if not feat["nearest"]:
                    nearest = False

            az = None
            if not nearest and hasAngle:
                if feat["angle"] is not None:
                    az = feat["angle"]

            lgeom = feat.geometry()
            lgeom.transform(xform)
            self.createLabel(
                srcLayer,
                lblLayer,
                lgeom,
                float(
                    str(
                        QgsExpressionContextUtils.projectScope(
                            QgsProject.instance()
                        ).variable("sll_label_offset")
                    ).replace(",", ".")
                ),
                align=alignLabel,
                update=True,
                nearest=nearest,
                azim=az,
            )

        cursor.setShape(Qt.CursorShape.PointingHandCursor)
        self.iface.mapCanvas().setCursor(cursor)
        self.iface.mapCanvas().refresh()
        self.iface.messageBar().pushMessage(
            "Info", "Label update finished!", level=Qgis.Info, duration=3
        )

    def getPerpAngle(self, a):
        if a >= -90 and a < 90:
            b = -a
            c = a + 90
        else:
            b = 180 - a
            c = a - 90
        return (b, c)

    def evalFieldExpression(self, layer, feature, context):
        exp = QgsExpression('"label_offset"')
        context.setFeature(feature)
        val = exp.evaluate(context)
        return val

    def createLabel(
        self,
        srcLayer,
        lblLayer,
        lgeom,
        offset,
        align="L",
        update=False,
        nearest=True,
        azim=None,
    ):
        crsSrc = srcLayer.crs()
        crsDest = lblLayer.crs()
        xform = QgsCoordinateTransform(crsSrc, crsDest, QgsProject.instance())
        pt1 = QgsPointXY(lgeom.vertexAt(0))
        pt2 = QgsPointXY(lgeom.vertexAt(1))

        if azim is None:
            a = pt1.azimuth(pt2)
        else:
            a = azim
            pt3 = self.polar(pt2, a, 1)

        (b, c) = self.getPerpAngle(a)

        # if right aligned switch direction
        if align == "R":
            (b, c) = (b - 180, c - 180)

        pt1_dest = xform.transform(pt1)
        pt2_dest = xform.transform(pt2)
        if azim is None:
            a_dest = pt1_dest.azimuth(pt2_dest)
        else:
            pt3_dest = xform.transform(pt3)
            a_dest = pt2_dest.azimuth(pt3_dest)

        (b_dest, c_dest) = self.getPerpAngle(a_dest)

        targetFeatures = {}

        wkt = lgeom.asWkt()
        expr = QgsExpression("intersects( @geometry, geom_from_wkt('%s') )" % wkt)
        features = srcLayer.getFeatures(QgsFeatureRequest(expr))

        # set Expression context to srcLayer for label_offset evaluation
        context = QgsExpressionContext()
        context.appendScopes(
            QgsExpressionContextUtils.globalProjectLayerScopes(srcLayer)
        )
        context.clearCachedValues()

        hasSrcLayerOffsetAtt = False
        if "label_offset" in [f.name() for f in list(srcLayer.fields())]:
            hasSrcLayerOffsetAtt = True

        for feat in features:
            if lgeom.intersects(feat.geometry()):
                geometry = lgeom.intersection(feat.geometry())
                if geometry.isMultipart():
                    intersPoints = geometry.asMultiPoint()
                else:
                    intersPoints = [geometry.asPoint()]

                for pt in intersPoints:
                    dist = pt2.distance(pt)
                    id = feat[self.cboID.currentText()]
                    if hasSrcLayerOffsetAtt:
                        label_offset = self.evalFieldExpression(srcLayer, feat, context)
                    else:
                        label_offset = None

                    # create key for ordered dict
                    key = float(str(round(dist,5))+str(id))
                    targetFeatures[key] = [id, pt, label_offset]

        orderedTargetFeatures = collections.OrderedDict(
            reversed(sorted(targetFeatures.items()))
        )

        if len(list(orderedTargetFeatures.keys())) == 0:
            return

        pt0 = xform.transform(targetFeatures[list(orderedTargetFeatures.keys())[-1]][1])

        fieldlist = [f.name() for f in list(lblLayer.dataProvider().fields())]

        # add field "label_align" if not available
        if "label_align" not in fieldlist:
            with edit(lblLayer):
                lblLayer.dataProvider().addAttributes(
                    [QgsField("label_align", QVariant.String)]
                )
                lblLayer.updateFields()
                # update horizontal alignment settings
                layer_settings = lblLayer.labeling().settings()
                pc = layer_settings.dataDefinedProperties()
                pc.setProperty(
                    QgsPalLayerSettings.Hali,
                    QgsProperty.fromExpression("if( label_align, label_align, 'L')"),
                )
                layer_settings.setDataDefinedProperties(pc)
                lblLayer.labeling().setSettings(layer_settings)
                lblLayer.triggerRepaint()

        # add field "nearest" if not available
        if "nearest" not in fieldlist:
            with edit(lblLayer):
                lblLayer.dataProvider().addAttributes(
                    [QgsField("nearest", QVariant.Bool)]
                )
                lblLayer.updateFields()

        # add field "angle" if not available
        if "angle" not in fieldlist:
            with edit(lblLayer):
                lblLayer.dataProvider().addAttributes(
                    [QgsField("angle", QVariant.Double)]
                )
                lblLayer.updateFields()

        f = QgsFeature(lblLayer.dataProvider().fields())

        # create reference line for label updates
        if not update and len(orderedTargetFeatures) > 0:
            lgeom.transform(xform)
            f.setGeometry(lgeom)
            f["label_align"] = align
            if nearest:
                f["nearest"] = True
            else:
                f["nearest"] = False

            if azim is not None:
                f["angle"] = azim
                nearest = False
                f["nearest"] = False

            lblLayer.dataProvider().addFeature(f)

        label_offset = targetFeatures[list(orderedTargetFeatures.keys())[0]][2]
        if not label_offset:
            dist = offset
            curroff = offset / 2.0
        else:
            dist = offset + label_offset / 2.0
            curroff = label_offset / 2.0

        # if Ctrl mode => no offset for first label
        if azim is not None:
            dist = 0

        featList = list(orderedTargetFeatures.items())

        # for key in orderedTargetFeatures:
        for i, feat in enumerate(featList):
            if nearest:
                pt = self.polar(pt0, a, self.rc.convertToMapUnits(dist, self.units))
            else:
                pt = self.polar(pt2, a, self.rc.convertToMapUnits(dist, self.units))

            pt3 = self.polar(
                pt, c, self.rc.convertToMapUnits(self.ticklength, self.units)
            )

            if nearest:
                lgeom = QgsGeometry.fromPolylineXY(
                    [QgsPointXY(feat[1][1][0], feat[1][1][1]), pt, pt3]
                )
            else:
                lgeom = QgsGeometry.fromPolylineXY(
                    [QgsPointXY(feat[1][1][0], feat[1][1][1]), pt2, pt, pt3]
                )
            lgeom.transform(xform)

            # increment primary key if no memory layer
            if "memory" not in lblLayer.dataProvider().storageType().lower():
                f[list(lblLayer.dataProvider().fields())[0].name()] = None

            f[self.cboID.currentText()] = feat[1][0]

            if nearest:
                pt2_dest = xform.transform(QgsPointXY(pt0))
            else:
                pt2_dest = xform.transform(QgsPointXY(pt3))

            f["x"] = pt2_dest.x()
            f["y"] = pt2_dest.y()
            f["angle"] = b_dest
            f["label_align"] = align
            f["nearest"] = nearest
            f.setGeometry(lgeom)
            lblLayer.dataProvider().addFeature(f)
            if self.iface.mapCanvas().isCachingEnabled():
                lblLayer.triggerRepaint()
            else:
                self.iface.mapCanvas().refresh()

            if i < len(featList) - 1:
                off = featList[i + 1][1][2]

                if not off:
                    off = offset / 2.0
                else:
                    off = off / 2.0

                dist = dist + curroff + off

                curroff = off

    def polar(self, point, bearing, distance):
        # bearing in radians
        bearing = math.radians(bearing)
        # direction cosines
        cosa = math.sin(bearing)
        cosb = math.cos(bearing)
        xfinal, yfinal = (point.x() + (distance * cosa), point.y() + (distance * cosb))
        # print(xfinal,yfinal)
        return QgsPointXY(xfinal, yfinal)

    def uncheckToolBarButton(self):
        self.msgbar = False
        self.iface.mainWindow().statusBar().showMessage("")
        if hasattr(self.iface.mapCanvas(), "lastMapTool"):
            self.iface.mapCanvas().setMapTool(self.iface.mapCanvas().lastMapTool)

    def getLayerByID(self, layerID):
        layerList = [
            layer
            for layer in QgsProject.instance().mapLayers().values()
            if layer.id() == layerID
        ]
        if len(layerList) > 0:
            return layerList[0]
        else:
            return None

    def SourceLayerChanged(self):
        self.cboLblLayer.setEnabled(True)
        self.cboID.setEnabled(True)
        layerID = self.cboLayer.itemData(self.cboLayer.currentIndex())
        layer = self.getLayerByID(layerID)
        if layer:
            proj = QgsProject.instance()
            proj.writeEntry("SmartLineLabeler", "currentSrcLayer", layerID)
            self.setConnectedLabelLayer(layerID)

    def refreshComboBoxes(self, srcLayerID=None):
        self.cboLayer.blockSignals(True)
        self.cboLayer.clear()
        self.cboLblLayer.clear()
        self.cboID.clear()
        layers = [layer for layer in QgsProject.instance().mapLayers().values()]
        self.cboLblLayer.addItem("", "")
        i = 0

        for layer in layers:
            if (
                layer.type() == QgsMapLayer.VectorLayer
                and layer.isSpatial()
                and ('Line' in layer.wkbType().name or 'Curve' in layer.wkbType().name)
                and not layer.crs().isGeographic()
            ):
                if (
                    not layer.abstract().find("SmartLineLabeler") != -1
                    and layer.name().lower().find("_labels") == -1
                ):
                    i = i + 1
                    # fill layer combobox with names
                    self.cboLayer.addItem(
                        str(i).zfill(2) + ": " + layer.name(), layer.id()
                    )
                else:
                    # fill label layer combobox with names
                    self.cboLblLayer.addItem(layer.name(), layer.id())

        if srcLayerID:
            self.setConnectedLabelLayer(srcLayerID)

        self.cboLayer.blockSignals(False)

    def setConnectedLabelLayer(self, srcLayerID):
        self.cboLayer.blockSignals(True)
        indexList = [
            i
            for i in range(self.cboLayer.count())
            if self.cboLayer.itemData(i) == srcLayerID
        ]
        if len(indexList) > 0:
            self.cboLayer.setCurrentIndex(indexList[0])
        self.cboLayer.blockSignals(False)

        srcLayer = self.getLayerByID(srcLayerID)
        if not srcLayer:
            return
        self.cboLblLayer.setEnabled(True)
        self.cboID.setEnabled(True)
        self.cboID.clear()
        field_names = [field.name() for field in srcLayer.dataProvider().fields()]

        self.cboID.addItems(field_names)
        proj = QgsProject.instance()
        settings = proj.readEntry("SmartLineLabeler", srcLayerID)[0]
        if settings != "":
            lblLayerID = settings.split(":")[1]
            lblLayerList = [
                layer
                for layer in QgsProject.instance().mapLayers().values()
                if layer.id() == lblLayerID
            ]
            if len(lblLayerList) > 0:
                lblCboIndexList = [
                    i
                    for i in range(self.cboLblLayer.count())
                    if self.cboLblLayer.itemData(i) == lblLayerID
                ]
                if len(lblCboIndexList) > 0:
                    self.cboLblLayer.setCurrentIndex(lblCboIndexList[0])
                    self.cboLblLayer.setEnabled(False)

                lblLayerIDfield = settings.split(":")[0]
                if lblLayerIDfield:
                    self.cboID.setCurrentIndex(field_names.index(lblLayerIDfield))
                    self.cboID.setEnabled(False)

    def clearCurrentConnection(self):
        layerID = self.cboLayer.itemData(self.cboLayer.currentIndex())
        proj = QgsProject.instance()
        proj.removeEntry("SmartLineLabeler", layerID)
        self.refreshComboBoxes(layerID)

    def create_responsive_combobox(self):
        combo = QComboBox()
        combo.setMinimumContentsLength(5)
        combo.setSizePolicy(QSizePolicy.Policy.MinimumExpanding, 
                            QSizePolicy.Policy.Fixed)
        combo.setSizeAdjustPolicy(
          QComboBox.SizeAdjustPolicy.AdjustToMinimumContentsLengthWithIcon
        )
        combo.view().setTextElideMode(Qt.TextElideMode.ElideRight)
        return combo

    def show_info_dialog(self):
        msg = QMessageBox(self.iface.mainWindow())
        msg.setWindowTitle("Smart Line Labeler - Info")
        msg.setIcon(QMessageBox.Information)

        # Enable rich text and include a link
        msg.setTextFormat(Qt.RichText)
        msg.setText(
              '<style>kbd { color: blue; }</style>'
            + '<h3>Plugin Usage:</b></h3>'
            + 'Pick point in Canvas, then'
            + '<ul><li>press&amp;hold <kbd>Shift</kbd> key and pick a line feature on the active layer to adjust the label direction</li>'
            + '<li>or press <kbd>F5</kbd> key to switch the label direction and alignment (Left, Right, Center)</li>'
            + '<li>or press&amp;hold <kbd>Ctrl</kbd> key to fix the second insertion point and pick a third point</li>'
            + '<li>or press <kbd>Right Mouse Button</kbd> to cancel</li></ul>'
            + 'When a second (or third) point is picked, all line features of the source layer that intersect the temporary rubberband line are labeled.'
            + '<br/><br/>'
            + 'More info here: <a href="https://github.com/cxcandid/SmartLineLabeler/blob/main/README.md">Description</a>'
        )

        # Themed window icon (optional)
        msg.setWindowIcon(QgsApplication.getThemeIcon('mIconInfo.svg'))

        # Standard buttons or custom ones
        msg.setStandardButtons(QMessageBox.Close)
        msg.exec_()

    def run(self):
        self.startPoint = None

        # set render context for scale 1:1000
        s = self.iface.mapCanvas().scale()
        self.iface.mapCanvas().zoomScale(1000)
        self.rc = QgsRenderContext().fromMapSettings(
            self.iface.mapCanvas().mapSettings()
        )
        self.iface.mapCanvas().zoomScale(s)

        if not QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable(
            "sll_label_offset"
        ):
            QgsExpressionContextUtils.setProjectVariable(
                QgsProject.instance(), "sll_label_offset", 5
            )
            self.offset = 5
        else:
            self.offset = float(
                QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable(
                    "sll_label_offset"
                )
            )

        if not QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable(
            "sll_label_ticklength"
        ):
            QgsExpressionContextUtils.setProjectVariable(
                QgsProject.instance(), "sll_label_ticklength", 5
            )
            self.ticklength = 5
        else:
            self.ticklength = float(
                QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable(
                    "sll_label_ticklength"
                )
            )

        if not QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable(
            "sll_label_units"
        ):
            QgsExpressionContextUtils.setProjectVariable(
                QgsProject.instance(), "sll_label_units", "Millimeters"
            )
            self.units = Qgis.RenderUnit.Millimeters
        else:
            self.units = Qgis.RenderUnit[
                QgsExpressionContextUtils.projectScope(QgsProject.instance()).variable(
                    "sll_label_units"
                )
            ]

        if self.action.isChecked():
            self.cboLayer = self.create_responsive_combobox()
            self.cboLblLayer = self.create_responsive_combobox()
            self.cboID = self.create_responsive_combobox()

            self.widget = self.iface.messageBar().createMessage("")
            self.widget.children()[1].clear()
            pix = QPixmap(os.path.dirname(os.path.realpath(__file__)) + "/icons/icon.svg")
            pix = pix.scaled(22, 22, Qt.AspectRatioMode.KeepAspectRatio, 
                             Qt.TransformationMode.SmoothTransformation)
            self.widget.children()[1].setPixmap(pix)
            self.refreshComboBoxes()

            self.cboLayer.currentIndexChanged.connect(
                self.SourceLayerChanged
            )

            proj = QgsProject.instance()
            currentLayerID = proj.readEntry(
                "SmartLineLabeler", "currentSrcLayer", None
            )[0]
            if currentLayerID:
                indexList = [
                    i
                    for i in range(self.cboLayer.count())
                    if self.cboLayer.itemData(i) == currentLayerID
                ]
                if len(indexList) > 0:
                    self.cboLayer.setCurrentIndex(indexList[0])
                    self.setConnectedLabelLayer(currentLayerID)
            else:
                srcLayerID = self.cboLayer.itemData(self.cboLayer.currentIndex())
                self.setConnectedLabelLayer(srcLayerID)

            label1 = QLabel()
            label1.setText("Source Layer")
            label1.setAlignment(Qt.AlignmentFlag.AlignCenter)
            label2 = QLabel()
            label2.setText("Label Layer")
            label2.setAlignment(Qt.AlignmentFlag.AlignCenter)
            label3 = QLabel()
            label3.setText("ID Column")
            label3.setAlignment(Qt.AlignmentFlag.AlignCenter)

            layout = self.widget.layout() # QHBoxLayout
            layout.setContentsMargins(0, 0, 0, 0)
            info_btn = QToolButton()
            info_btn.setIcon(QIcon(":images/themes/default/propertyicons/metadata.svg"))
            info_btn.setToolTip("Show usage information")
            info_btn.setAutoRaise(True)

            #infoLabel = QLabel('<a href="https://github.com/cxcandid" title="ok"><img style="vertical-align:bottom" src="data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCA2MiA2MiIgd2lkdGg9IjIwIiBoZWlnaHQ9IjIwIiB2ZXJzaW9uPSIxLjEiPjxkZWZzPjxsaW5lYXJHcmFkaWVudCBpZD0iZmllbGRHcmFkaWVudCIgZ3JhZGllbnRVbml0cz0idXNlclNwYWNlT25Vc2UiIHgxPSI0Mi45ODYzIiB5MT0iNy4wMTI3MCIgeDI9IjIyLjAxNDQiIHkyPSI1MS45ODcxIj48c3RvcCBvZmZzZXQ9IjAuMCIgc3RvcC1jb2xvcj0iI0JDRDZGRSIgLz4gPHN0b3Agb2Zmc2V0PSIxLjAiIHN0b3AtY29sb3I9IiM2Nzg3RDMiIC8+IDwvbGluZWFyR3JhZGllbnQ+IDxsaW5lYXJHcmFkaWVudCBpZD0iZWRnZUdyYWRpZW50IiBncmFkaWVudFVuaXRzPSJ1c2VyU3BhY2VPblVzZSIgeDE9IjU1LjQ1NDEiIHkxPSI0Mi43NTI5IiB4Mj0iOS41NDcxMCIgeTI9IjE2LjI0ODUiPiA8c3RvcCBvZmZzZXQ9IjAuMCIgc3RvcC1jb2xvcj0iIzMwNTdBNyIgLz4gPHN0b3Agb2Zmc2V0PSIxLjAiIHN0b3AtY29sb3I9IiM1QTdBQzYiIC8+IDwvbGluZWFyR3JhZGllbnQ+IDxyYWRpYWxHcmFkaWVudCBpZD0ic2hhZG93R3JhZGllbnQiPiA8c3RvcCBvZmZzZXQ9IjAuMCIgc3RvcC1jb2xvcj0iI0MwQzBDMCIgLz4gPHN0b3Agb2Zmc2V0PSIwLjg4IiBzdG9wLWNvbG9yPSIjQzBDMEMwIiAvPiA8c3RvcCBvZmZzZXQ9IjEuMCIgc3RvcC1jb2xvcj0iI0MwQzBDMCIgc3RvcC1vcGFjaXR5PSIwLjAiIC8+IDwvcmFkaWFsR3JhZGllbnQ+IDwvZGVmcz4gPGNpcmNsZSBpZD0ic2hhZG93IiByPSIyNi41IiBjeD0iMzIuNSIgY3k9IjI5LjUiIGZpbGw9InVybCgjc2hhZG93R3JhZGllbnQpIiB0cmFuc2Zvcm09Im1hdHJpeCgxLjA2NDgsMC4wLDAuMCwxLjA2NDgyMiwtMi4xLDEuMDg2NCkiIC8+IDxjaXJjbGUgaWQ9ImZpZWxkIiByPSIyNS44IiBjeD0iMzEiIGN5PSIzMSIgZmlsbD0idXJsKCNmaWVsZEdyYWRpZW50KSIgc3Ryb2tlPSJ1cmwoI2VkZ2VHcmFkaWVudCkiIHN0cm9rZS13aWR0aD0iMiIgLz4gPGcgaWQ9ImluZm8iIGZpbGw9IndoaXRlIj4gPHBvbHlnb24gcG9pbnRzPSIyMywyNSAzNSwyNSAzNSw0NCAzOSw0NCAzOSw0OCAyMyw0OCAyMyw0NCAyNyw0NCAyNywyOCAyMywyOCAyMywyNSIgLz4gPGNpcmNsZSByPSI0IiBjeD0iMzEiIGN5PSIxNyIgLz4gPC9nPiA8L3N2Zz4="/></a>')
            #infoLabel.setOpenExternalLinks(True)
            info_btn.clicked.connect(self.show_info_dialog)
            layout.addWidget(info_btn)
            layout.addWidget(label1)
            layout.addWidget(self.cboLayer)
            layout.addWidget(label2)
            layout.addWidget(self.cboLblLayer)
            layout.addWidget(label3)
            layout.addWidget(self.cboID)
             
            if len(self.cboLayer) == 0:
                self.iface.messageBar().pushMessage(
                    "Info", "No Source Layer found!", level=Qgis.Info, duration=3
                )
                self.action.setChecked(False)
                return

            button = QPushButton(self.widget)
            button.setText("Clear Connection")
            icon = QIcon(":images/themes/default/mTaskCancel.svg")
            button.setIcon(icon)
            button.pressed.connect(self.clearCurrentConnection)
            layout.addWidget(button)
            
            button = QPushButton(self.widget)
            button.setText("Update Labels")
            icon = QIcon(":images/composer/refreshing_item.svg")
            button.setIcon(icon)
            button.pressed.connect(self.updateLabels)
            layout.addWidget(button)

            self.iface.messageBar().pushWidget(self.widget, Qgis.Info)
            self.msgbar = True
            # uncheck Pick Button if messageBar is removed
            self.widget.destroyed.connect(self.uncheckToolBarButton)
            self.iface.mapCanvas().lastMapTool = self.iface.mapCanvas().mapTool()
            self.iface.mapCanvas().setMapTool(LineMapTool(self.iface, self.action, 
                                                          self))
        else:
            if hasattr(self.iface.mapCanvas(), "lastMapTool"):
                self.iface.mapCanvas().setMapTool(self.iface.mapCanvas().lastMapTool)


class LineMapTool(QgsMapTool):
    def __init__(self, iface, action, parent):
        self.cursor = Qt.CursorShape.CrossCursor
        self.parent = parent
        self.iface = iface
        self.action = action
        self.canvas = iface.mapCanvas()
        QgsMapTool.__init__(self, self.canvas)
        self.rubberBand = QgsRubberBand(self.canvas)
        self.rubberBand.setColor(Qt.GlobalColor.red)
        self.rubberBand.setWidth(1)
        self.rubberBand2 = QgsRubberBand(self.canvas)
        self.rubberBand2.setColor(Qt.GlobalColor.red)
        self.rubberBand2.setWidth(1)
        self.oldStartPoint = None

        self.reset()

    def reset(self):
        self.startPoint = self.endPoint = None
        self.isFixedAngle = False
        self.isEmittingPoint = False
        self.rubberBand.reset()  # QgsWkbTypes.LineString
        cursor = QCursor()
        self.cursor = Qt.CursorShape.CrossCursor
        cursor.setShape(self.cursor)
        self.canvas.setCursor(cursor)

    def keyPressEvent(self, e):
        if e.modifiers() == Qt.KeyboardModifier.ShiftModifier:
            cursor = QCursor()
            self.cursor = Qt.CursorShape.PointingHandCursor
            cursor.setShape(self.cursor)
            self.canvas.setCursor(cursor)
            self.rubberBand.reset()  # QgsWkbTypes.LineString

    def keyReleaseEvent(self, e):
        if e.key() == Qt.Key.Key_F5:
            if self.parent.alignLabel == "L":
                self.parent.alignLabel = "C"
            elif self.parent.alignLabel == "C":
                self.parent.alignLabel = "R"
            else:
                self.parent.alignLabel = "L"

            self.rubberBand.reset()
            self.showLine(self.startPoint, self.endPoint, True, 100)

        cursor = QCursor()
        self.cursor = Qt.CursorShape.CrossCursor
        cursor.setShape(self.cursor)
        self.canvas.setCursor(cursor)

    def canvasPressEvent(self, e):
        if e.button() != Qt.MouseButton.LeftButton:
            self.reset()
            if self.oldStartPoint:
                self.rubberBand2.reset()
                self.oldStartPoint = None
            return

        self.endPoint = self.toMapCoordinates(e.pos())
        self.isEmittingPoint = True

        cursor = QCursor()
        if self.startPoint is not None:
            layerID = self.parent.cboLayer.itemData(self.parent.cboLayer.currentIndex())
            srcLayer = self.parent.getLayerByID(layerID)
            if not srcLayer:
                self.iface.messageBar().pushMessage(
                    "Info", "No Line Layer found!", level=Qgis.Info, duration=3
                )
                self.parent.action.setChecked(False)
                return

            if e.modifiers() == Qt.KeyboardModifier.ShiftModifier:
                pixTolerance = 6
                mapTolerance = pixTolerance * self.canvas.mapUnitsPerPixel()
                rect = QgsRectangle(
                    self.endPoint.x() - mapTolerance,
                    self.endPoint.y() - mapTolerance,
                    self.endPoint.x() + mapTolerance,
                    self.endPoint.y() + mapTolerance,
                )

                # transform rect coordinates
                crsSrc = QgsProject.instance().crs()
                crsDest = srcLayer.crs()
                xform = QgsCoordinateTransform(crsSrc, crsDest, QgsProject.instance())
                rect = xform.transform(rect)
                srcLayer.selectByRect(rect)
                if srcLayer.selectedFeatureCount() > 0:
                    feature = srcLayer.selectedFeatures()[0]
                    closeSegResult = feature.geometry().closestSegmentWithContext(
                        xform.transform(self.toMapCoordinates(e.pos()))
                    )
                    xform = QgsCoordinateTransform(
                        crsDest, crsSrc, QgsProject.instance()
                    )
                    vertex2 = xform.transform(
                        QgsPointXY(feature.geometry().vertexAt(closeSegResult[2]))
                    )
                    vertex1 = xform.transform(
                        QgsPointXY(feature.geometry().vertexAt(closeSegResult[2] - 1))
                    )
                    dx = vertex2.x() - vertex1.x()
                    dy = vertex2.y() - vertex1.y()
                    self.endPoint = self.droppedPerpFoot(
                        vertex1.x() - dx * 1000000,
                        vertex1.y() - dy * 1000000,
                        vertex2.x() + dx * 1000000,
                        vertex2.y() + dy * 1000000,
                        self.startPoint.x(),
                        self.startPoint.y(),
                    )
                    srcLayer.removeSelection()
                    self.isFixedAngle = True
                    return
                else:
                    self.reset()
                    return

            if e.modifiers() == Qt.KeyboardModifier.ControlModifier:
                self.parent.nearest = False
                if not self.oldStartPoint:
                    self.oldStartPoint = self.startPoint
                    self.startPoint = self.endPoint
                    self.rubberBand.reset()
                    self.showLine(self.oldStartPoint, self.startPoint, False, 0, True)
                    return
            else:
                if not self.oldStartPoint:
                    self.parent.nearest = True

            if self.endPoint is not None:
                self.isEmittingPoint = False

                lblLayer = None
                lblLayerID = self.parent.cboLblLayer.itemData(
                    self.parent.cboLblLayer.currentIndex()
                )
                if lblLayerID != "":
                    lblLayerList = [
                        layer
                        for layer in QgsProject.instance().mapLayers().values()
                        if layer.id() == lblLayerID
                    ]
                    if len(lblLayerList) > 0:
                        lblLayer = lblLayerList[0]

                try:
                    lblLayer.isValid()
                    lblLayer.dataProvider().isValid()
                    if self.parent.cboLblLayer.isEnabled():
                        proj = QgsProject.instance()
                        proj.writeEntry(
                            "SmartLineLabeler", "currentSrcLayer", srcLayer.id()
                        )
                        proj.writeEntry(
                            "SmartLineLabeler",
                            srcLayer.id(),
                            self.parent.cboID.currentText() + ":" + str(lblLayer.id()),
                        )
                        self.parent.cboLblLayer.setEnabled(False)
                        self.parent.cboID.setEnabled(False)

                except Exception:
                    epsgCode = srcLayer.crs().authid().split(":")[1]
                    lblLayer = self.iface.addVectorLayer(
                        "LineString?crs=epsg:"
                        + epsgCode
                        + "&field="
                        + self.parent.cboID.currentText()
                        + ":integer&field=x:double&field=y:double&field=angle:double"
                        + "&field=label_align:string(1,0)&field=nearest:bool",
                        srcLayer.name().lower() + "_labels (memory)",
                        "memory",
                    )
                    lblLayer.setAbstract(
                        'Layer created by "SmartLineLabeler" plugin.'
                    )

                    proj = QgsProject.instance()
                    proj.writeEntry(
                        "SmartLineLabeler", "currentSrcLayer", srcLayer.id()
                    )
                    proj.writeEntry(
                        "SmartLineLabeler",
                        srcLayer.id(),
                        self.parent.cboID.currentText() + ":" + str(lblLayer.id()),
                    )
                    self.parent.refreshComboBoxes(srcLayer.id())
                    self.parent.cboID.setEnabled(False)

                    info = QgsVectorLayerJoinInfo()
                    info.setJoinLayerId(srcLayer.id())
                    info.setJoinLayer(srcLayer)
                    info.setJoinFieldName(self.parent.cboID.currentText())
                    info.setTargetFieldName(self.parent.cboID.currentText())
                    info.setUsingMemoryCache(True)
                    lblLayer.addJoin(info)

                    lblLayer.loadNamedStyle(
                        os.path.dirname(__file__) + "/SmartLineLabeler.qml"
                    )

                crsSrc = QgsProject.instance().crs()
                crsDest = srcLayer.crs()
                xform = QgsCoordinateTransform(crsSrc, crsDest, QgsProject.instance())

                pt1 = xform.transform(self.point1)
                pt2 = xform.transform(self.point2)

                az = None
                if self.oldStartPoint:
                    self.rubberBand2.reset()
                    pt0 = xform.transform(self.oldStartPoint)
                    self.oldStartPoint = None
                    lgeom = QgsGeometry.fromPolylineXY([pt0, pt1])
                    az = pt1.azimuth(pt2)
                else:
                    lgeom = QgsGeometry.fromPolylineXY([pt1, pt2])

                self.parent.createLabel(
                    srcLayer,
                    lblLayer,
                    lgeom,
                    float(
                        str(
                            QgsExpressionContextUtils.projectScope(
                                QgsProject.instance()
                            ).variable("sll_label_offset")
                        ).replace(",", ".")
                    ),
                    align=self.parent.alignLabel,
                    nearest=self.parent.nearest,
                    azim=az,
                )
                self.parent.nearest = True

                try:
                    lblLayer.isValid()
                    lblLayer.updateExtents()
                except Exception:
                    pass

                self.canvas.refresh()

                self.reset()
                return

            if self.cursor == Qt.CursorShape.CrossCursor:
                cursor.setShape(self.cursor)
                self.canvas.setCursor(cursor)
        else:
            self.startPoint = self.toMapCoordinates(e.pos())
            self.isEmittingPoint = True
            self.cursor = Qt.CursorShape.CrossCursor
            cursor.setShape(self.cursor)
            self.canvas.setCursor(cursor)

    def canvasMoveEvent(self, e):
        lineLength = 100
        if not self.isEmittingPoint:
            return

        if self.isFixedAngle:
            if self.endPoint is not None:
                a = self.startPoint.azimuth(self.endPoint)
                currentPt = self.toMapCoordinates(e.pos())
                dist = currentPt.distance(self.startPoint)
                self.lastPoint = self.parent.polar(self.startPoint, a, dist)
                if currentPt.distance(self.lastPoint) > self.startPoint.distance(self.lastPoint):
                    self.lastPoint = self.parent.polar(self.startPoint, a + 180, dist)
                if self.cursor == Qt.CursorShape.CrossCursor:
                    self.showLine(self.startPoint, self.lastPoint, True, lineLength)
            else:
                self.lastPoint = self.toMapCoordinates(e.pos())
                if self.cursor == Qt.CursorShape.CrossCursor:
                    self.showLine(self.startPoint, self.lastPoint, True, lineLength)
        else:
            self.endPoint = self.toMapCoordinates(e.pos())
            if self.cursor == Qt.CursorShape.CrossCursor:
                self.showLine(self.startPoint, self.endPoint, True, lineLength)

    def droppedPerpFoot(self, x1, y1, x2, y2, x3, y3):  # x3,y3 is the point
        px = x2 - x1
        py = y2 - y1

        something = px * px + py * py

        u = ((x3 - x1) * px + (y3 - y1) * py) / float(something)

        if u > 1:
            u = 1
        elif u < 0:
            u = 0

        x = x1 + u * px
        y = y1 + u * py
        return QgsPointXY(x, y)

    def showLine(
        self, startPoint, endPoint, showCorner, lineLength, isHelperLine=False
    ):
        lineLength = lineLength * self.iface.mapCanvas().scale()

        if showCorner and self.oldStartPoint and not self.isFixedAngle:
            # Limit the selectable angle to 5-degree increments
            a = 5 * math.floor(startPoint.azimuth(endPoint) / 5)
        else:
            a = startPoint.azimuth(endPoint)

        if a >= -90 and a < 90:
            b = a + 90
        else:
            b = a - 90

        if self.parent.alignLabel == "R":
            b = b - 180

        self.point1 = QgsPointXY(startPoint.x(), startPoint.y())

        if showCorner and self.oldStartPoint:
            dist = startPoint.distance(endPoint)
            self.point2 = self.parent.polar(self.point1, a, dist)
        else:
            self.point2 = QgsPointXY(endPoint.x(), endPoint.y())

        if not isHelperLine:
            self.rubberBand.reset()  # QgsWkbTypes.LineString
            self.rubberBand.addPoint(self.point1, False)

            if showCorner:
                self.rubberBand.addPoint(self.point2, False)
                if self.parent.alignLabel != "C":
                    self.rubberBand.addPoint(
                        self.parent.polar(self.point2, b, lineLength), True
                    )
                else:
                    self.rubberBand.addPoint(
                        self.parent.polar(self.point2, b, lineLength / 2.0), True
                    )
                    self.rubberBand.addPoint(
                        self.parent.polar(self.point2, b - 180, lineLength / 2.0), True
                    )
            else:
                self.rubberBand.addPoint(self.point2, True)

            self.rubberBand.show()
        else:
            self.rubberBand2.addPoint(self.point1, False)
            self.rubberBand2.addPoint(self.point2, True)
            self.rubberBand2.show()

    def activate(self):
        pass

    def isZoomTool(self):
        return False

    def isTransient(self):
        return True

    def isEditTool(self):
        return False

    def deactivate(self):
        if self.parent.msgbar:
            self.iface.messageBar().popWidget(self.parent.widget)

        self.action.setChecked(False)
        self.rubberBand.reset()  # QgsWkbTypes.LineString

