# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ManageSkydeck
                                 A QGIS plugin
 This is a plugin to download files from Skydeck, process it in QGIS and upload the results back to Skydeck portal.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-03-10
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Asteria
        email                : neethu.narayanan@asteria.co.in
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os.path
import requests
import tempfile
import sys
import os
import platform
import subprocess
from urllib.parse import urlparse, parse_qs
import importlib
from PyQt5.QtWidgets import QMessageBox

current_path = os.getcwd()

def install_package(package):
    print(f"Installing {package}...")
    try:
        if platform.system() == 'Linux':
            subprocess.run(['pip', 'install', package], check=True)

        elif platform.system() == 'Darwin':
            current_path = sys.executable
            last_slash_index = current_path.rfind('/')
            install_path = current_path[:last_slash_index]
            subprocess.run([install_path+'/bin/pip3', 'install', package], check=True)
        else:
            current_path = os.getcwd()
            subprocess.run(["cd", current_path], shell=True)
            subprocess.run(['pip', 'install', package], shell=True, check=True)

    except subprocess.CalledProcessError as e:
        msg = QMessageBox()
        msg.setIcon(QMessageBox.Critical)
        msg.setWindowTitle("Installation Failed")
        msg.setText(f"Failed to install package: {package}")
        msg.setInformativeText(f"""Error: {str(e)}: Please install it manually using steps below:")
        On Windows:
            1. Open Start Menu and search for `OSGeo4W Shell`.
            2. Run the following command:
                pip install "{package}"
            3. Restart QGIS and try again.
                           """)
        msg.exec_()
        sys.exit(1)

required_packages = {
    'jwt': 'PyJWT==2.10.1',
    'PyQt5': 'PyQt5==5.15.11',
    'azure.storage.blob': 'azure-storage-blob==12.24.1',
    'PyQt5.QtWebEngineWidgets': 'PyQtWebEngine==5.15.7',
    'pip_system_certs': 'pip-system-certs==4.0'
}
restart_needed = False
for import_name, pip_name in required_packages.items():
    try:
        importlib.import_module(import_name)
    except ImportError:
        print(f"Module {import_name} not found. Installing {pip_name}...")
        install_package(pip_name)
        restart_needed = True

if restart_needed:
    msg = QMessageBox()
    msg.setIcon(QMessageBox.Information)
    msg.setWindowTitle("Restart Required")
    msg.setText("One or more Python packages were installed.")
    msg.setInformativeText("Please restart QGIS to complete plugin initialization.")
    msg.exec_()
    #sys.exit(1)

try:
    from PyQt5.QtCore import Qt, QCoreApplication
    QCoreApplication.setAttribute(Qt.AA_ShareOpenGLContexts)
    from PyQt5.QtWebEngineWidgets import QWebEngineView
    from azure.storage.blob import BlobServiceClient
    import jwt
except ImportError as e:
    print(f"Error importing required modules: {e}")
    #sys.exit(1)


from qgis.PyQt.QtCore import QSettings, QTranslator, QUrl
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QListWidgetItem
from qgis.core import QgsRasterLayer, QgsProject, QgsVectorLayer, QgsRasterFileWriter, QgsRasterPipe, QgsCoordinateReferenceSystem, QgsVectorFileWriter
from qgis.utils import iface

from .resources import *
from .manage_skydeck_dialog import ManageSkydeckDialog
from .import_export import ImportExportWindow

class ManageSkydeck:
    """QGIS Plugin Implementation."""

    endpoint = "https://app.skydeck.tech"
    blob_endpoint = "https://skydeckcorefilestrgprd.blob.core.windows.net"

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.dlg = ManageSkydeckDialog()

        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = str(QSettings().value('locale/userLocale'))[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ManageSkydeck_{}.qm'.format(locale))
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&SkyGIS')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        #self.open_web_page()

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ManageSkydeck', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/manage_skydeck/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u''),
            callback=self.run,
            parent=self.iface.mainWindow())
        
    

    def on_url_changed(self, url):
        url_string = url.toString()
        if "auth.skydeck.tech" not in url_string:
            self.web_view.loadFinished.connect(self.on_load_finished)

    def on_load_finished(self, ok):
        # This slot is called when the web page has finished loading.
        if ok:
            print("Page loaded successfully. Processing the page.")
            self.web_view.page().toHtml(self.process_page)
        else:
            print("Failed to load page")

    def process_page(self, html):
        url = self.web_view.url().toString()
        print(url)
        parsed_url = urlparse(url)
        query_params = parse_qs(parsed_url.query)
        token = query_params.get('token', [None])[0]
        print(f"Token: {token}")
        self.handle_token(token)


    def handle_token(self, token):
        if token:
            print("Token is valid")
            self.import_export_window = ImportExportWindow(token, self.web_view)
            #Verify RBAC
            print(f"Bearer {token}")
            if not isinstance(token, bytes):
                print("Token is not bytes")
                encoded_token = token.encode()
            try:
                decoded_token = jwt.decode(encoded_token, options={"verify_signature": False})
            except Exception as e:
                print(f"Failed to decode token : {e}")
            # Extract email and sub
            email = decoded_token.get('email')
            sub = decoded_token.get('sub')
            rbac_data = {"email": email, "sub": sub}
            print(f"Email: {email}, Sub: {sub}")
            headers = {"Authorization": f"Bearer {token}"}
            response = requests.post(url=f"{ManageSkydeck.endpoint}/api/gis/v1/qgis/rbac", headers=headers, json=rbac_data)
            print(f"Response message from rbac: {response}")
            print(f"Response : {response.json()}")
            self.dlg.close()
            if response.json() is not True:
                print("RBAC Failed")
                self.import_export_window.skydeckLogout()
                iface.messageBar().pushMessage(f"UNAUTHORISED USER", level=2, duration=5)
            else:
                print("RBAC Success")
                self.import_export_window.show()
        else:
            print("No token found")
            iface.messageBar().pushMessage(f"Error in validating the user. Please try after sometime", level=2, duration=5)
            self.dlg.close()

    def get_existing_token(self):
            url = self.web_view.url().toString()
            print(url)
            parsed_url = urlparse(url)
            query_params = parse_qs(parsed_url.query)
            token = query_params.get('token', [None])[0]
            print(f"Token: {token}")
            return token
    

    def open_web_page(self):
        try:
            print("Opening Skydeck login page")
            self.web_view = self.dlg.skydeckWebEngineView
            token = self.get_existing_token()
            if token:
                print("Token exists")
                self.handle_token(token)
            else:
                self.dlg.show()
                self.web_view.load(QUrl(f"https://app.skydeck.tech/auth/login"))
                self.web_view.urlChanged.connect(self.on_url_changed)
        except Exception as e:
            print(f"Error : {e}")


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&SkyGIS'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False

        self.open_web_page()
