"""
SHRIMP (Simple Hopefully Reliable Integrated Music Player) Plugin - Main plugin file
"""
import os
from qgis.PyQt.QtCore import QUrl, Qt, pyqtSignal, QObject, QTimer
from qgis.PyQt.QtGui import QIcon, QPixmap, QPainter, QFont, QDesktopServices, QKeySequence
from qgis.PyQt.QtMultimedia import QMediaPlayer, QMediaContent
from qgis.PyQt.QtWidgets import (
    QAction, QDialog, QVBoxLayout, QHBoxLayout, 
    QPushButton, QListWidget, QLabel, QSlider, QDialogButtonBox, QListWidgetItem, QMenu, QTextBrowser,
    QMessageBox
)
from qgis.gui import QgsGui


class MuzzPlayerSettingsDialog(QDialog):
    """Settings dialog for the music player"""
    
    def __init__(self, player_controller, parent=None):
        super().__init__(parent)
        self.player_controller = player_controller
        
        self.setWindowTitle('SHRIMP Settings')
        self.setMinimumWidth(320)
        self.setMinimumHeight(400)
        self.resize(320, 450)
        
        self.setup_ui()
        self.load_music_files()
        
    def setup_ui(self):
        """Setup the user interface"""
        layout = QVBoxLayout()
        
        # Title
        title = QLabel('SHRIMP')
        title.setAlignment(Qt.AlignmentFlag.AlignCenter)
        title.setStyleSheet('font-size: 18px; font-weight: bold; margin: 10px;')
        layout.addWidget(title)
        
        # Subtitle with full name
        subtitle = QLabel('Simple Hopefully Reliable Integrated Music Player')
        subtitle.setAlignment(Qt.AlignmentFlag.AlignCenter)
        subtitle.setStyleSheet('font-size: 10px; color: gray; padding: 2px; margin-bottom: 5px;')
        layout.addWidget(subtitle)
        
        # Current track info
        self.current_track_label = QLabel('No track selected')
        self.current_track_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.current_track_label.setStyleSheet('padding: 5px; font-weight: bold;')
        layout.addWidget(self.current_track_label)
        
        # HTML instructions text (moved to Help dialog) - uses simple HTML for nicer formatting
        music_dir_display = self.player_controller.music_dir if self.player_controller else ''
        self.instructions_html = f"""
            <div style="font-family:Calibri,Arial,sans-serif; font-size:12px; color:#222">
                <h2 style="margin:4px 0 8px 0;">SHRIMP Help</h2>
                <p style="margin:4px 0;"><strong>What is SHRIMP?</strong><br>
                Simple Hopefully Reliable Integrated Music Player — a small music player embedded in QGIS.</p>

                <h3 style="margin:8px 0 4px 0;">Quick actions</h3>
                <ul style="margin:4px 0 8px 18px;">
                    <li><strong>Previous:</strong> Ctrl+Shift+Left</li>
                    <li><strong>Play / Pause:</strong> Ctrl+Shift+Space</li>
                    <li><strong>Next:</strong> Ctrl+Shift+Right</li>
                </ul>

                <h3 style="margin:8px 0 4px 0;">Playlist</h3>
                <ul style="margin:4px 0 8px 18px;">
                    <li>Right-click a track and choose <em>Play Now</em> to play it immediately.</li>
                    <li>Drag tracks to reorder the playlist (changes persist for this session).</li>
                    <li>Use <em>Shuffle</em> to randomize the order, or <em>Refresh Playlist</em> to reload files from disk.</li>
                </ul>

                <h3 style="margin:8px 0 4px 0;">Files &amp; folder</h3>
                <p style="margin:4px 0;">Music folder: <code>{music_dir_display}</code></p>
                <p style="margin:6px 0 0 0; color:#444; font-size:11px;">Tip: Put your audio files here (mp3, wav, ogg, m4a, flac, aac). The plugin will list supported files automatically.</p>

                <hr style="margin:10px 0;" />
                <p style="margin:4px 0; color:#666; font-size:11px;">Contact with me: kk.at.work@pm.me</p>
                <p style="margin:2px 0 0 0; color:#666; font-size:11px;">For more information, edit the plugin files or contact the plugin author at the address above.</p>
            </div>
            """
        
        # Playlist label
        playlist_label = QLabel('Playlist:')
        playlist_label.setStyleSheet('font-weight: bold;')
        layout.addWidget(playlist_label)
        
        # Playlist (with drag and drop enabled)
        self.playlist_widget = QListWidget()
        self.playlist_widget.setDragDropMode(QListWidget.DragDropMode.InternalMove)
        self.playlist_widget.currentRowChanged.connect(self.on_playlist_selection_changed)
        self.playlist_widget.model().rowsMoved.connect(self.on_playlist_reordered)
        self.playlist_widget.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.playlist_widget.customContextMenuRequested.connect(self.show_context_menu)
        self.playlist_widget.setMaximumHeight(200)  # Limit playlist height
        layout.addWidget(self.playlist_widget)
        
        # Buttons layout
        buttons_layout = QHBoxLayout()
        
        # Open music folder button
        open_folder_button = QPushButton('📁 Open Music Folder')
        open_folder_button.clicked.connect(self.open_music_folder)
        buttons_layout.addWidget(open_folder_button)
        
        # Shuffle button
        shuffle_button = QPushButton('🔀 Shuffle')
        shuffle_button.clicked.connect(self.shuffle_playlist)
        buttons_layout.addWidget(shuffle_button)
        
        # Refresh button
        refresh_button = QPushButton('🔄 Refresh Playlist')
        refresh_button.clicked.connect(self.load_music_files)
        buttons_layout.addWidget(refresh_button)
        
        layout.addLayout(buttons_layout)
        
        # Help button (shows instructions) placed above the Close button
        help_layout = QHBoxLayout()
        help_button = QPushButton('❓ Help')
        help_button.clicked.connect(self.show_help_dialog)
        help_layout.addWidget(help_button)
        help_layout.addStretch()
        layout.addLayout(help_layout)

        # Close button
        button_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Close)
        button_box.rejected.connect(self.close)
        layout.addWidget(button_box)

        self.setLayout(layout)
        
        # Connect to player controller for updates
        if self.player_controller:
            self.player_controller.stateChanged.connect(self.on_state_changed)
            self.player_controller.trackChanged.connect(self.on_track_changed)
            # Update playlist selection when track changes
            self.player_controller.trackChanged.connect(self.update_playlist_selection)
        
    def load_music_files(self):
        """Load music files from the music directory and refresh UI"""
        if not self.player_controller:
            return
            
        # Get currently playing track path before refresh
        current_index = self.player_controller.get_current_index()
        old_playlist = self.player_controller.get_playlist()
        current_track_path = None
        if old_playlist and 0 <= current_index < len(old_playlist):
            current_track_path = old_playlist[current_index]
        
        # Reload files from disk
        self.player_controller.load_music_files()
        
        # Get new playlist
        new_playlist = self.player_controller.get_playlist()
        
        # Clear and rebuild UI
        self.playlist_widget.clear()
        
        # Store full paths in item data
        for file_path in new_playlist:
            item = QListWidgetItem(os.path.basename(file_path))
            item.setData(Qt.ItemDataRole.UserRole, file_path)
            self.playlist_widget.addItem(item)
        
        # Try to find the previously playing track in the new playlist
        new_index = 0
        if current_track_path and current_track_path in new_playlist:
            new_index = new_playlist.index(current_track_path)
        elif new_playlist:
            # If current track no longer exists, reset to first track
            new_index = 0
            # Stop playback if the track was removed
            if self.player_controller.player.state() == QMediaPlayer.PlayingState:
                self.player_controller.player.stop()
        
        # Update controller index
        if new_playlist:
            self.player_controller.set_current_index(new_index)
            # Block signals to prevent recursion
            self.playlist_widget.blockSignals(True)
            self.playlist_widget.setCurrentRow(new_index)
            self.playlist_widget.blockSignals(False)
                
    def on_playlist_reordered(self, parent, start, end, destination, row):
        """Handle playlist reordering via drag and drop"""
        if not self.player_controller:
            return
            
        # Rebuild playlist from widget order
        new_playlist = []
        for i in range(self.playlist_widget.count()):
            item = self.playlist_widget.item(i)
            file_path = item.data(Qt.ItemDataRole.UserRole)
            if file_path:
                new_playlist.append(file_path)
        
        # Update controller playlist
        old_index = self.player_controller.get_current_index()
        self.player_controller.set_playlist(new_playlist)
        
        # Find the new index of the currently playing track
        if old_index < len(self.player_controller.get_playlist()):
            old_track = self.player_controller.get_playlist()[old_index] if old_index < len(self.player_controller.get_playlist()) else None
            if old_track and old_track in new_playlist:
                new_index = new_playlist.index(old_track)
                self.player_controller.set_current_index(new_index)
                
    def shuffle_playlist(self):
        """Shuffle the playlist"""
        if not self.player_controller:
            return
            
        import random
        
        # Get current playlist
        playlist = self.player_controller.get_playlist()
        if len(playlist) < 2:
            return
            
        # Get currently playing track
        current_index = self.player_controller.get_current_index()
        current_track = playlist[current_index] if 0 <= current_index < len(playlist) else None
        
        # Shuffle all tracks
        shuffled = playlist.copy()
        random.shuffle(shuffled)
        
        # Find the new index of the currently playing track
        if current_track and current_track in shuffled:
            new_index = shuffled.index(current_track)
        else:
            new_index = 0
            
        # Update controller
        self.player_controller.set_playlist(shuffled)
        self.player_controller.set_current_index(new_index)
        
        # Reload UI to show shuffled order
        self.playlist_widget.clear()
        for file_path in shuffled:
            item = QListWidgetItem(os.path.basename(file_path))
            item.setData(Qt.ItemDataRole.UserRole, file_path)
            self.playlist_widget.addItem(item)
        
        # Highlight current track
        if 0 <= new_index < self.playlist_widget.count():
            self.playlist_widget.blockSignals(True)
            self.playlist_widget.setCurrentRow(new_index)
            self.playlist_widget.blockSignals(False)
                
    def on_playlist_selection_changed(self, current_row):
        """Handle selection change in playlist"""
        pass  # Just visual feedback
        
    def show_context_menu(self, position):
        """Show context menu for playlist item"""
        item = self.playlist_widget.itemAt(position)
        if not item or not self.player_controller:
            return
            
        # Get the file path from item data
        file_path = item.data(Qt.ItemDataRole.UserRole)
        if not file_path:
            return
            
        # Find the index in the controller's playlist
        playlist = self.player_controller.get_playlist()
        try:
            track_index = playlist.index(file_path)
        except ValueError:
            return  # Track not found
            
        # Create context menu
        menu = QMenu(self)
        
        # Play Now
        play_now_action = QAction('▶ Play Now', self)
        play_now_action.triggered.connect(lambda: self.player_controller.play_track(track_index))
        menu.addAction(play_now_action)
        
        menu.addSeparator()
        
        # Remove from Playlist
        remove_action = QAction('🗑 Remove from Playlist', self)
        remove_action.triggered.connect(lambda: self.remove_track_from_playlist(track_index))
        menu.addAction(remove_action)
        
        # Show menu
        menu.exec_(self.playlist_widget.mapToGlobal(position))
        
    def play_next_after_current(self, track_index):
        """Play track after current track finishes"""
        if not self.player_controller:
            return
            
        playlist = self.player_controller.get_playlist()
        current_index = self.player_controller.get_current_index()
        
        # If track is already next, just play it
        next_index = (current_index + 1) % len(playlist)
        if track_index == next_index:
            self.player_controller.play_track(track_index)
        else:
            # Move track to position after current
            # Remove from current position
            track_path = playlist.pop(track_index)
            
            # Insert after current position
            new_position = current_index + 1
            if new_position > len(playlist):
                playlist.append(track_path)
            else:
                playlist.insert(new_position, track_path)
            
            # Update controller
            self.player_controller.set_playlist(playlist)
            self.player_controller.set_current_index(current_index)
            
            # Reload UI
            self.load_music_files()
            self.playlist_widget.setCurrentRow(new_position)
            
    def remove_track_from_playlist(self, track_index):
        """Remove track from playlist"""
        if not self.player_controller:
            return
            
        playlist = self.player_controller.get_playlist()
        current_index = self.player_controller.get_current_index()
        
        # Don't remove if it's the only track
        if len(playlist) <= 1:
            return
            
        # Remove the track
        removed_track = playlist.pop(track_index)
        
        # Adjust current index if needed
        if track_index < current_index:
            new_current_index = current_index - 1
        elif track_index == current_index:
            # If removing current track, play next or previous
            if current_index >= len(playlist):
                new_current_index = len(playlist) - 1
            else:
                new_current_index = current_index
            # Stop current playback
            self.player_controller.player.stop()
        else:
            new_current_index = current_index
            
        # Update controller
        self.player_controller.set_playlist(playlist)
        self.player_controller.set_current_index(new_current_index)
        
        # Reload UI
        self.load_music_files()
        if 0 <= new_current_index < self.playlist_widget.count():
            self.playlist_widget.setCurrentRow(new_current_index)
            
    def open_music_folder(self):
        """Open the music folder in file explorer"""
        if not self.player_controller:
            return
            
        music_dir = self.player_controller.music_dir
        if not os.path.exists(music_dir):
            os.makedirs(music_dir, exist_ok=True)
        
        # Open folder using QDesktopServices
        QDesktopServices.openUrl(QUrl.fromLocalFile(music_dir))
            
    def on_state_changed(self, state):
        """Handle player state changes"""
        pass  # Could update UI if needed
        
    def on_track_changed(self, track_name, track_index):
        """Update current track label"""
        self.current_track_label.setText(f'Now Playing: {track_name}')
        
    def update_playlist_selection(self, track_name, track_index):
        """Update playlist widget to highlight current track"""
        if 0 <= track_index < self.playlist_widget.count():
            # Block signals to prevent recursion
            self.playlist_widget.blockSignals(True)
            self.playlist_widget.setCurrentRow(track_index)
            self.playlist_widget.blockSignals(False)
        
    def show_help_dialog(self):
        """Show a help dialog with instructions moved from the settings window."""
        dlg = QDialog(self)
        dlg.setWindowTitle('SHRIMP Help')
        dlg.setMinimumSize(420, 300)
        vlayout = QVBoxLayout()
        # Use QTextBrowser for HTML content with built-in scrolling and link support
        browser = QTextBrowser()
        browser.setHtml(self.instructions_html)
        browser.setOpenExternalLinks(True)
        browser.setReadOnly(True)
        vlayout.addWidget(browser)
        btn_box = QDialogButtonBox(QDialogButtonBox.StandardButton.Close)
        btn_box.rejected.connect(dlg.accept)
        vlayout.addWidget(btn_box)
        dlg.setLayout(vlayout)
        dlg.exec_()
        
    def format_time(self, milliseconds):
        """Format time in milliseconds to MM:SS"""
        total_seconds = milliseconds // 1000
        minutes = total_seconds // 60
        seconds = total_seconds % 60
        return f'{minutes}:{seconds:02d}'


class MuzzPlayerController(QObject):
    """Controller for the music player functionality"""
    
    # Signals for dialog updates
    positionChanged = pyqtSignal(int)
    durationChanged = pyqtSignal(int)
    stateChanged = pyqtSignal(int)
    trackChanged = pyqtSignal(str, int)  # track_name, track_index
    
    def __init__(self, plugin_dir):
        super().__init__()
        self.plugin_dir = plugin_dir
        self.music_dir = os.path.join(plugin_dir, 'music')
        self.current_index = 0
        self.playlist = []
        self._is_manual_change = False  # Flag to prevent auto-play conflicts
        
        # Initialize media player
        self.player = QMediaPlayer()
        self.player.setVolume(70)
        self.player.stateChanged.connect(self._on_state_changed)
        self.player.positionChanged.connect(self._on_position_changed)
        self.player.durationChanged.connect(self._on_duration_changed)
        
        self.load_music_files()
        
    def load_music_files(self):
        """Load music files from the music directory"""
        self.playlist = []
        
        if not os.path.exists(self.music_dir):
            os.makedirs(self.music_dir, exist_ok=True)
            return
        
        # Supported audio formats
        audio_extensions = ['.mp3', '.wav', '.ogg', '.m4a', '.flac', '.aac']
        
        for file in os.listdir(self.music_dir):
            file_path = os.path.join(self.music_dir, file)
            if os.path.isfile(file_path) and any(file.lower().endswith(ext) for ext in audio_extensions):
                self.playlist.append(file_path)
        
        # Sort playlist alphabetically
        self.playlist.sort()
        
    def get_playlist(self):
        """Get the current playlist"""
        return self.playlist.copy()
        
    def get_current_index(self):
        """Get current track index"""
        return self.current_index
        
    def set_playlist(self, new_playlist):
        """Set a new playlist order"""
        self.playlist = new_playlist.copy()
        
    def set_current_index(self, index):
        """Set current track index"""
        if 0 <= index < len(self.playlist):
            self.current_index = index
        
    def play_track(self, index):
        """Play a track by index"""
        if not self.playlist or index < 0 or index >= len(self.playlist):
            return
            
        # Set flag to prevent auto-play conflicts
        self._is_manual_change = True
        
        # Stop current playback first
        if self.player.state() != QMediaPlayer.StoppedState:
            self.player.stop()
        
        self.current_index = index
        track_path = self.playlist[index]
        track_name = os.path.basename(track_path)
        
        # Set media and play
        media_content = QMediaContent(QUrl.fromLocalFile(track_path))
        self.player.setMedia(media_content)
        self.player.play()
        
        # Emit track changed signal with both name and index
        self.trackChanged.emit(track_name, index)
        
        # Reset flag after a short delay
        QTimer.singleShot(100, lambda: setattr(self, '_is_manual_change', False))
            
    def toggle_play_pause(self):
        """Toggle between play and pause"""
        # Check if playlist is empty
        if not self.playlist:
            self.show_no_tracks_warning()
            return
            
        if self.player.state() == QMediaPlayer.PlayingState:
            self.player.pause()
        else:
            if self.player.media().isNull():
                # No track loaded, play current selection
                if self.playlist:
                    self.play_track(self.current_index)
            else:
                self.player.play()
                
    def play_next(self):
        """Play next track"""
        if not self.playlist:
            return
            
        # Loop to first track if at the end
        if self.current_index >= len(self.playlist) - 1:
            next_index = 0
        else:
            next_index = self.current_index + 1
            
        self.play_track(next_index)
            
    def play_previous(self):
        """Play previous track"""
        if self.playlist:
            prev_index = (self.current_index - 1) % len(self.playlist)
            self.play_track(prev_index)
            
    def set_volume(self, value):
        """Set volume (0-100)"""
        self.player.setVolume(value)
        
    def set_position(self, position):
        """Seek to position"""
        self.player.setPosition(position)
        
    def pause(self):
        """Pause playback"""
        self.player.pause()
        
    def play(self):
        """Start playback"""
        self.player.play()
        
    def show_no_tracks_warning(self):
        """Show warning dialog when no tracks are available"""
        # First, reload music files to ensure we have the latest
        self.load_music_files()
        
        # Check again if playlist is still empty
        if self.playlist:
            # Tracks found after reload, proceed with playback
            self.play_track(self.current_index)
            return
        
        msg_box = QMessageBox()
        msg_box.setWindowTitle('No Tracks Found')
        msg_box.setText('No track files found in the track directory.')
        msg_box.setIcon(QMessageBox.Icon.Warning)
        
        # Add two buttons: OK and Open Track Directory
        ok_button = msg_box.addButton(QMessageBox.StandardButton.Ok)
        open_dir_button = msg_box.addButton('Open Track Directory', QMessageBox.ButtonRole.ActionRole)
        
        # Show dialog and get result
        msg_box.exec()
        
        # Check which button was clicked
        if msg_box.clickedButton() == open_dir_button:
            # Open the music directory
            QDesktopServices.openUrl(QUrl.fromLocalFile(self.music_dir))
        
    def _on_state_changed(self, state):
        """Handle player state changes"""
        self.stateChanged.emit(state)
        
        # Auto-play next track when current finishes (only if not manually changed)
        if state == QMediaPlayer.StoppedState and not self._is_manual_change:
            if self.playlist:
                # Always play next, looping to first if at end
                if self.current_index >= len(self.playlist) - 1:
                    next_index = 0
                else:
                    next_index = self.current_index + 1
                self.play_track(next_index)
                
    def _on_position_changed(self, position):
        """Handle position changes"""
        self.positionChanged.emit(position)
        
    def _on_duration_changed(self, duration):
        """Handle duration changes"""
        self.durationChanged.emit(duration)


class MuzzPlayer:
    """QGIS Plugin Implementation"""
    
    def __init__(self, iface):
        """Constructor.
        
        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        
        # Create player controller
        self.controller = MuzzPlayerController(self.plugin_dir)
        
        # Actions
        self.prev_action = None
        self.play_action = None
        self.next_action = None
        self.settings_action = None
        
        # Settings dialog
        self.settings_dialog = None
        
    def get_icon(self, icon_name):
        """Get icon from icons folder"""
        icon_path = os.path.join(self.plugin_dir, 'icons', icon_name)
        if os.path.exists(icon_path):
            return QIcon(icon_path)
        # Fallback to text icon if file not found
        return self.create_icon_from_text('♪')
    
    def create_icon_from_text(self, text, size=24):
        """Create an icon from Unicode text (fallback)"""
        pixmap = QPixmap(size, size)
        pixmap.fill(Qt.GlobalColor.transparent)
        painter = QPainter(pixmap)
        painter.setRenderHint(QPainter.RenderHint.Antialiasing)
        font = QFont()
        font.setPixelSize(size - 4)
        painter.setFont(font)
        painter.drawText(pixmap.rect(), Qt.AlignmentFlag.AlignCenter, text)
        painter.end()
        return QIcon(pixmap)
        
    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""
        
        # Previous track button (icon only)
        prev_icon = self.get_icon('previoustrack.ico')
        self.prev_action = QAction(prev_icon, 'SHRIMP: Previous Track', self.iface.mainWindow())
        self.prev_action.setToolTip('Previous Track')
        self.prev_action.triggered.connect(self.controller.play_previous)
        self.iface.addToolBarIcon(self.prev_action)
        self.iface.addPluginToMenu('&SHRIMP', self.prev_action)
        # Register with QGIS shortcuts manager
        QgsGui.shortcutsManager().registerAction(self.prev_action, 'Ctrl+Shift+Left', 'SHRIMP')
        
        # Play/Pause button (icon only)
        play_icon = self.get_icon('playpause.ico')
        self.play_action = QAction(play_icon, 'SHRIMP: Play/Pause', self.iface.mainWindow())
        self.play_action.setToolTip('Play/Pause')
        self.play_action.triggered.connect(self.controller.toggle_play_pause)
        self.iface.addToolBarIcon(self.play_action)
        self.iface.addPluginToMenu('&SHRIMP', self.play_action)
        # Register with QGIS shortcuts manager
        QgsGui.shortcutsManager().registerAction(self.play_action, 'Ctrl+Shift+Space', 'SHRIMP')
        
        # Next track button (icon only)
        next_icon = self.get_icon('nexttrack.ico')
        self.next_action = QAction(next_icon, 'SHRIMP: Next Track', self.iface.mainWindow())
        self.next_action.setToolTip('Next Track')
        self.next_action.triggered.connect(self.controller.play_next)
        self.iface.addToolBarIcon(self.next_action)
        self.iface.addPluginToMenu('&SHRIMP', self.next_action)
        # Register with QGIS shortcuts manager
        QgsGui.shortcutsManager().registerAction(self.next_action, 'Ctrl+Shift+Right', 'SHRIMP')
        
        # Settings button
        settings_icon = self.get_icon('shrimpsettings.ico')
        self.settings_action = QAction(settings_icon, 'SHRIMP: Settings', self.iface.mainWindow())
        self.settings_action.setToolTip('SHRIMP Settings')
        self.settings_action.triggered.connect(self.show_settings)
        self.iface.addToolBarIcon(self.settings_action)
        self.iface.addPluginToMenu('&SHRIMP', self.settings_action)
        
        # Update play button state
        self.controller.stateChanged.connect(self.update_play_button)
        
    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        if self.settings_dialog:
            self.settings_dialog.close()
            
        # Unregister shortcuts
        QgsGui.shortcutsManager().unregisterAction(self.prev_action)
        QgsGui.shortcutsManager().unregisterAction(self.play_action)
        QgsGui.shortcutsManager().unregisterAction(self.next_action)
        
        self.iface.removeToolBarIcon(self.prev_action)
        self.iface.removeToolBarIcon(self.play_action)
        self.iface.removeToolBarIcon(self.next_action)
        self.iface.removeToolBarIcon(self.settings_action)
        self.iface.removePluginMenu('&SHRIMP', self.prev_action)
        self.iface.removePluginMenu('&SHRIMP', self.play_action)
        self.iface.removePluginMenu('&SHRIMP', self.next_action)
        self.iface.removePluginMenu('&SHRIMP', self.settings_action)
        
    def show_settings(self):
        """Show settings dialog"""
        if self.settings_dialog is None:
            self.settings_dialog = MuzzPlayerSettingsDialog(self.controller, self.iface.mainWindow())
        self.settings_dialog.load_music_files()
        self.settings_dialog.show()
        self.settings_dialog.raise_()
        self.settings_dialog.activateWindow()
        
    def update_play_button(self, state):
        """Update play button icon based on player state"""
        # Use the same playpause icon for both states (the icon should handle the visual state)
        # If you want different icons for play/pause, you could add play.ico and pause.ico
        play_icon = self.get_icon('playpause.ico')
        self.play_action.setIcon(play_icon)
        if state == QMediaPlayer.PlayingState:
            self.play_action.setToolTip('Pause')
        else:
            self.play_action.setToolTip('Play')
