# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SVG2ColoR
                                 A QGIS plugin
 Generates color-ramp styles from SVG files. It also compatible with CPT-CITY styles. SVG2ColoR improves your color-ramp library, by the way your maps look better.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-09-25
        git sha              : $Format:%H$
        copyright            : (C) 2018 by Mehmet Selim BILGIN
        email                : mselimbilgin@yahoo.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import *
from PyQt5.QtGui import *
from PyQt5.QtWidgets import *
from PyQt5.QtWebKitWidgets import QGraphicsWebView
from qgis.core import QgsStyle

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .svg2color_dialog import SVG2ColoRDialog
from xml.dom import minidom
import codecs
import os
from urllib.request import Request, urlopen
import tempfile


class SVG2ColoR:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'SVG2ColoR_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Create the dialog (after translation) and keep reference

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&SVG2ColoR')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'SVG2ColoR')
        self.toolbar.setObjectName(u'SVG2ColoR')

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('SVG2ColoR', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/svg2color/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u''),
            callback=self.run,
            parent=self.iface.mainWindow())

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&SVG2ColoR'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    def loadFromFile(self):
        # File browsing
        browseDlg, _ = QFileDialog.getOpenFileName(self.dlg, 'Choose SVG File...', self.dlg.lineEdit.text(),
                                                   'SVG file (*.svg)')
        if browseDlg:
            self.dlg.lineEdit.setText(browseDlg)
            try:
                with codecs.open(browseDlg, encoding='utf-8', mode='r') as readFile:
                    self.isLinearGrad(readFile.read())
            except Exception as readError:
                QMessageBox.critical(None, "Information", ("An error has occured: " + str(readError)))

    def loadFromURL(self):
        ##URl reading.
        try:
            urlSVG = urlopen(self.dlg.lineEdit.text()).read().decode("utf-8")
            self.isLinearGrad(urlSVG)
        except Exception as urlError:
            QMessageBox.critical(None, "Information", 'Cannot load from the given URL: ' + str(urlError))
            self.graphicRenderer(self.errorSVG)

    def optionHandler(self):
        ##RadioButton detection.
        if self.dlg.radioButton.isChecked():
            self.dlg.pushButton.setText('Browse...')
            try:
                self.dlg.pushButton.clicked.disconnect()
            except:
                pass
            self.dlg.pushButton.clicked.connect(self.loadFromFile)
            self.dlg.lineEdit.clear()
            self.dlg.lineEdit.setReadOnly(True)
            self.graphicRenderer(self.openingSVG)
        else:
            self.dlg.pushButton.setText('Get URL...')
            try:
                self.dlg.pushButton.clicked.disconnect()
            except:
                pass
            self.dlg.pushButton.clicked.connect(self.loadFromURL)
            self.dlg.lineEdit.clear()
            self.dlg.lineEdit.setReadOnly(False)
            self.dlg.lineEdit.setFocus()
            self.graphicRenderer(self.errorSVG)

    def isLinearGrad(self, svgDocument):
        ##Checking SVG document for XML syntax.
        try:
            svgDoc = minidom.parseString(svgDocument)
            self.firstNode = svgDoc.getElementsByTagName('linearGradient')
            # Checking SVG document for containing linearGradient tag.
            if len(self.firstNode) == 0:
                QMessageBox.critical(None, "Information",
                                     'The SVG file does not contain <b>linearGradient</b> tag. Please choose a proper SVG document. You can find lots of samples in <a href="http://soliton.vm.bytemark.co.uk/pub/cpt-city/">CPT-CITY webpage</a>')
                self.graphicRenderer(self.errorSVG)
                self.dlg.pushButton_2.setEnabled(False)
                self.dlg.pushButton_3.setEnabled(False)
                self.dlg.label.setEnabled(False)
            else:
                try:
                    self.colorList = list()
                    for i in self.firstNode[0].getElementsByTagName('stop'):
                        # Some SVG files contain hex values for defining colors. It must be converted to values. This issue is handled in here.
                        if i.attributes['stop-color'].value[0] == '#':
                            self.colorList.append([self.hexToRgb(i.attributes['stop-color'].value),
                                                   float(i.attributes['offset'].value[:-1]) / 100])
                        else:
                            self.colorList.append([i.attributes['stop-color'].value[4:-1],
                                                   float(i.attributes['offset'].value[:-1]) / 100])
                    self.sampleSvg(self.firstNode[0])
                    self.dlg.pushButton_2.setEnabled(True)
                    self.dlg.pushButton_3.setEnabled(True)
                    self.dlg.label.setEnabled(True)

                except Exception as linearGradError:
                    QMessageBox.critical(None, "Information",
                                         'Cannot read the color values. Please choose a proper SVG document. You can find lots of samples in <a href="http://soliton.vm.bytemark.co.uk/pub/cpt-city/">CPT-CITY webpage</a>')
                    self.graphicRenderer(self.errorSVG)
                    self.dlg.pushButton_2.setEnabled(False)
                    self.dlg.pushButton_3.setEnabled(False)
                    self.dlg.label.setEnabled(False)

        except Exception as xmlError:
            QMessageBox.critical(None, "Information", 'The SVG file is not valid (XML syntax error).')
            self.graphicRenderer(self.errorSVG)
            self.dlg.pushButton_2.setEnabled(False)
            self.dlg.pushButton_3.setEnabled(False)
            self.dlg.label.setEnabled(False)

    # Thanks for dan_waterworth from http://stackoverflow.com/questions/4296249/how-do-i-convert-a-hex-triplet-to-an-rgb-tuple-and-back
    def hexToRgb(self, hexademical):
        num = str(hexademical[1:])
        # Sometimes hex codes can be shortand (3 digits). Here is the solution...
        if len(num) == 3:
            num = num[:1] * 2 + num[1:2] * 2 + num[2:3] * 2

        return (str(int(num[:2], 16)) + ',' + str(int(num[2:4], 16)) + ',' + str(int(num[4:6], 16)))

    def styleMaker(self, colorList, name):
        ##Generating QGIS style XML document.
        # Adding main elements.
        styleDoc = minidom.Document()
        qgisBase = styleDoc.createElement('qgis_style')
        qgisBase.setAttribute('version', '1')
        styleDoc.appendChild(qgisBase)

        symbols = styleDoc.createElement('symbols')
        qgisBase.appendChild(symbols)

        colorRamps = styleDoc.createElement('colorramps')
        qgisBase.appendChild(colorRamps)

        colorRamp = styleDoc.createElement('colorramp')
        colorRamp.setAttribute('type', 'gradient')
        colorRamp.setAttribute('name', name)
        colorRamps.appendChild(colorRamp)

        # Adding prop elements.
        prop1 = styleDoc.createElement('prop')
        prop1.setAttribute('k', 'color1')
        prop1.setAttribute('v', colorList[0][0])
        colorRamp.appendChild(prop1)

        prop2 = styleDoc.createElement('prop')
        prop2.setAttribute('k', 'color2')
        prop2.setAttribute('v', colorList[-1][0])
        colorRamp.appendChild(prop2)

        prop3 = styleDoc.createElement('prop')
        prop3.setAttribute('k', 'discrete')
        prop3.setAttribute('v', '0')
        colorRamp.appendChild(prop3)

        stopList = list()
        for j in colorList[1:-1]:
            stopList.append(str(j[1]) + ';' + j[0])

        prop4 = styleDoc.createElement('prop')
        prop4.setAttribute('k', 'stops')
        prop4.setAttribute('v', ':'.join(stopList))
        colorRamp.appendChild(prop4)
        return styleDoc.toprettyxml(indent="    ", encoding="utf-8").decode("utf-8")

    def sampleSvg(self, linearGradientTag):
        # Loading sample SVG documents for visualisation.
        sampleSVG = minidom.Document()
        svgBase = sampleSVG.createElement('svg')
        svgBase.setAttribute('xmlns', 'http://www.w3.org/2000/svg')
        svgBase.setAttribute('version', '1.1')
        svgBase.setAttribute('width', '600px')
        svgBase.setAttribute('height', '110px')
        sampleSVG.appendChild(svgBase)

        defs = sampleSVG.createElement('defs')
        svgBase.appendChild(defs)
        defs.appendChild(linearGradientTag)

        rectangle = sampleSVG.createElement('rect')
        rectangle.setAttribute('x', '1')
        rectangle.setAttribute('y', '1')
        rectangle.setAttribute('width', '493')
        rectangle.setAttribute('height', '93')
        rectangle.setAttribute('stroke-width', '1')
        rectangle.setAttribute('stroke', 'black')
        rectangle.setAttribute('fill', 'url(#' + linearGradientTag.attributes['id'].value + ')')
        svgBase.appendChild(rectangle)
        self.graphicRenderer(sampleSVG.toprettyxml(indent="    ", encoding="utf-8"))

    def graphicRenderer(self, svgInput):
        # Graphics rendering.
        try:
            scene = QGraphicsScene()
            self.dlg.graphicsView.setScene(scene)
            webview = QGraphicsWebView()
            webview.setContent(svgInput, 'image/svg+xml')
            scene.addItem(webview)
        except Exception as e:
            print(e)

    def exportStyle(self):
        browseDlg, _ = QFileDialog.getSaveFileName(self.dlg, 'Save Style File...',
                                                   os.path.dirname(self.dlg.lineEdit.text()) + os.sep +
                                                   os.path.splitext(os.path.basename(self.dlg.lineEdit.text()))[0],
                                                   'XML file (*.xml)')

        styleName = os.path.splitext(os.path.basename(self.dlg.lineEdit.text()))[0]  # name of the style

        if browseDlg:
            if browseDlg[:3] != "xml":
                browseDlg = browseDlg + ".xml"

            try:
                with codecs.open(browseDlg, encoding='utf-8', mode='w') as saveFile:
                    saveFile.write(self.styleMaker(self.colorList, styleName))
                    QMessageBox.information(None, "Information",
                                            'The style file succesfully saved in <b>%s</b>. \nNow you can import it from <b>Settings -> Style Manager</b> menu.' % (
                                                saveFile.name))

            except Exception as saveError:
                QMessageBox.critical(None, "Information", ("An error has occured: " + str(saveError)))

    def importStyle(self):
        styleName = os.path.splitext(os.path.basename(self.dlg.lineEdit.text()))[0]  # name of the style.
        tempStyle = self.tempDir + os.sep + styleName + '.xml'

        try:
            with codecs.open(tempStyle, encoding='utf-8', mode='w') as saveFile:
                saveFile.write(self.styleMaker(self.colorList, styleName))

            styleMan = QgsStyle.defaultStyle()  # QgsStyle style.
            styleMan.importXml(tempStyle)  # Importing the style file. It does not save. So this handled in below line.
            styleMan.saveColorRamp(styleName, styleMan.colorRamp(styleName), 7,
                                   ['SVG2ColoR'])  # Saving the imported style.

            QMessageBox.information(None, "Information", 'The style has been succesfully imported.')

        except Exception as importError:
            QMessageBox.critical(None, "Information", ("An error has occured: " + str(importError)))

    def run(self):
        self.errorSVG = open(os.path.dirname(os.path.realpath(__file__)) + os.sep + 'error.svg', 'r').read().encode(
            "utf-8")
        self.openingSVG = open(os.path.dirname(os.path.realpath(__file__)) + os.sep + 'opening.svg', 'r').read().encode(
            "utf-8")

        # show the dialog
        self.dlg = SVG2ColoRDialog()
        self.dlg.setFixedSize(560, 330)  # disables resizing the window.

        self.optionHandler()
        self.tempDir = tempfile.mkdtemp('svg2color')  # Generating temporary directory for importing styles.
        # I didnt use tempfile.TemporaryFile() because i got errors when working on Linux. But works without error on Windows.
        self.dlg.radioButton.toggled.connect(self.optionHandler)
        self.dlg.radioButton_2.toggled.connect(self.optionHandler)
        self.dlg.pushButton_2.clicked.connect(self.exportStyle)
        self.dlg.pushButton_3.clicked.connect(self.importStyle)
        self.dlg.show()


