# -*- coding: utf-8 -*-

"""
/***************************************************************************
 SpragueMultipliers
                                 A QGIS plugin
 This plugin generates school-aged population for any polygon layer
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-04-01
        copyright            : (C) 2021 by IIEP-UNESCO
        email                : development@iiep.unesco.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'IIEP-UNESCO'
__date__ = '2021-04-01'
__copyright__ = '(C) 2021 by IIEP-UNESCO'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'
import os
from qgis.PyQt.QtGui import QIcon
import processing
from qgis.core import Qgis
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsProcessingMultiStepFeedback,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFile,
                       QgsProcessingParameterVectorLayer,
                       QgsProcessingParameterString,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterDefinition,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsExpression,
                       QgsProcessingParameterFileDestination)


class SpragueMultipliersAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # Folder containing the raster files
        self.addParameter(
            QgsProcessingParameterFile(
                'foldercontainingtherasterfiles',
                'Folder containing the raster files',
                behavior=QgsProcessingParameterFile.Folder,
                fileFilter='All files (*.*)',
                defaultValue=None
            )
        )

        # Vector Polygon Layer containing the administrative boundaries
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                'Administrativeboundaries',
                'Administrative boundaries',
                types=[QgsProcessing.TypeVectorPolygon],
                defaultValue=None
            )
        )

        # 3-letter ISO country code
        self.addParameter(
            QgsProcessingParameterString(
                'isoCountryCode',
                'ISO country code',
                multiLine=False,
                defaultValue='COL'
            )
        )

        # year of analysis
        self.addParameter(
            QgsProcessingParameterNumber(
                'Year',
                'Year',
                type=QgsProcessingParameterNumber.Integer,
                minValue=1900,
                maxValue=2100,
                defaultValue=2020
            )
        )

        # Use constrained population estimates boolean
        self.addParameter(
            QgsProcessingParameterBoolean(
                'useConstrainedPopulationEstimates',
                'Use constrained population estimates',
                defaultValue=False
            )
        )

        # Use UN-adjusted constrained population estimates boolean
        self.addParameter(
            QgsProcessingParameterBoolean(
                'UseUNadjustedconstrainedestimates',
                'Use UN adjusted constrained estimates',
                optional=True,
                defaultValue=False
            )
        )

        # Create custom school age groups boolean
        self.addParameter(
            QgsProcessingParameterBoolean(
                'Createcustomschoolagegroups',
                'Create custom school age groups',
                defaultValue=True
            )
        )

        #Pre-primary starting age
        param = QgsProcessingParameterNumber(
            'preprimarystartingage',
            'Group 1 starting age / Pre-primary starting age',
            optional=True,
            type=QgsProcessingParameterNumber.Integer,
            minValue=2,
            maxValue=6,
            defaultValue=None
        )

        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        # Pre-primary duration
        param = QgsProcessingParameterNumber(
            'preprimaryduration',
            'Group 1 duration / Pre-primary duration',
            optional=True,
            type=QgsProcessingParameterNumber.Integer,
            minValue=1,
            maxValue=4,
            defaultValue=None
        )

        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        # Primary starting age
        param = QgsProcessingParameterNumber(
            'primarystartingage',
            'Group 2 starting age / Primary starting age',
            optional=True,
            type=QgsProcessingParameterNumber.Integer,
            minValue=5,
            maxValue=7,
            defaultValue=None
        )
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        # Primary duration
        param = QgsProcessingParameterNumber(
            'primaryduration',
            'Group 2 duration / Primary duration',
            optional=True,
            type=QgsProcessingParameterNumber.Integer,
            minValue=4,
            maxValue=8,
            defaultValue=None
        )
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        # Lower secondary starting age
        param = QgsProcessingParameterNumber(
            'lowersecondarystartingage',
            'Group 3 starting age / Lower secondary starting age',
            optional=True,
            type=QgsProcessingParameterNumber.Integer,
            minValue=10,
            maxValue=14,
            defaultValue=None
        )
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        # Lower secondary duration
        param = QgsProcessingParameterNumber(
            'lowersecondaryduration',
            'Group 3 duration / Lower secondary duration',
            optional=True,
            type=QgsProcessingParameterNumber.Integer,
            minValue=1,
            maxValue=8,
            defaultValue=None
        )
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        # Upper secondary starting age
        param = QgsProcessingParameterNumber(
            'uppersecondarystartingage',
            'Group 4 starting age / Upper secondary starting age',
            optional=True,
            type=QgsProcessingParameterNumber.Integer,
            minValue=13,
            maxValue=18,
            defaultValue=None
        )
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        # Upper secondary starting age
        param = QgsProcessingParameterNumber(
            'uppersecondaryduration',
            'Group 4 duration / Upper secondary duration',
            optional=True,
            type=QgsProcessingParameterNumber.Integer,
            minValue=1,
            maxValue=8,
            defaultValue=None
        )
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        # Exporting the results
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                'Results',
                'Results',
                type=QgsProcessing.TypeVectorAnyGeometry,
                createByDefault=True,
                supportsAppend=True,
                defaultValue=None
            )
        )

    def processAlgorithm(self, parameters, context, model_feedback):
        # Use a multi-step feedback, so that individual child algorithm progress reports are adjusted for the
        # overall progress through the model
        feedback = QgsProcessingMultiStepFeedback(45, model_feedback)
        results = {}
        outputs = {}

        isoCountryCode = self.parameterAsString(parameters,
                                                'isoCountryCode',
                                                context)
        year = self.parameterAsInt(parameters,
                                   'Year',
                                   context)
        useConstrainedPopulationEstimates = self.parameterAsBool(parameters,
                                                                 'useConstrainedPopulationEstimates',
                                                                 context)
        useUNAdjustedConstrainedPopulationEstimates = self.parameterAsBool(parameters,
                                                                           'UseUNadjustedconstrainedestimates',
                                                                           context)
        createCustomAgeGroups = self.parameterAsBool(parameters,
                                                     'Createcustomschoolagegroups',
                                                     context)
        preprimarystartingage = self.parameterAsInt(parameters,
                                                    'preprimarystartingage',
                                                    context)
        preprimaryduration = self.parameterAsInt(parameters,
                                                 'preprimaryduration',
                                                 context)
        primarystartingage = self.parameterAsInt(parameters,
                                                 'primarystartingage',
                                                 context)
        primaryduration = self.parameterAsInt(parameters,
                                              'primaryduration',
                                              context)
        lowersecondarystartingage = self.parameterAsInt(parameters,
                                                        'Lowersecondarystartingage',
                                                        context)
        lowersecondaryduration = self.parameterAsInt(parameters,
                                                     'Lowersecondaryduration',
                                                     context)
        uppersecondarystartingage = self.parameterAsInt(parameters,
                                                        'Uppersecondarystartingage',
                                                        context)
        uppersecondaryduration = self.parameterAsInt(parameters,
                                                     'Uppersecondaryduration',
                                                     context)
        foldercontainingtherasterfiles = self.parameterAsString(parameters,
                                                                'foldercontainingtherasterfiles',
                                                                context)

        # Providing warnings in case some of the optional values was not specified
        if parameters['Createcustomschoolagegroups'] and \
           (parameters['preprimarystartingage'] is None or \
            parameters['preprimaryduration'] is None or \
            parameters['primarystartingage'] is None or \
            parameters['primaryduration'] is None or \
            parameters['lowersecondarystartingage'] is None or \
            parameters['lowersecondaryduration'] is None or \
            parameters['uppersecondarystartingage'] is None or \
            parameters['uppersecondaryduration'] is None):
            feedback.pushWarning(f'You have selected the option "Create custom school age groups". \
            However, it looks like you failed to provide starting age or duration for at least one of the levels. \
            The code might fail to create the desired columns')



        if not parameters['useConstrainedPopulationEstimates']:
            fn = parameters['isoCountryCode'].lower() + "_f_0_" + str(parameters['Year']) + ".tif"
        elif parameters['useConstrainedPopulationEstimates'] and not parameters['UseUNadjustedconstrainedestimates']:
            fn = parameters['isoCountryCode'].lower() + "_f_0_" + str(parameters['Year']) + "_constrained.tif"
        elif parameters[    'useConstrainedPopulationEstimates'] and parameters['UseUNadjustedconstrainedestimates']:
            fn = parameters['isoCountryCode'].lower() + "_f_0_" + str(parameters['Year']) + "_constrained_UNadj.tif"

        alg_params = {
            'COLUMN_PREFIX': 'F_0_',
            'INPUT': parameters['Administrativeboundaries'],
            'INPUT_RASTER': os.path.join(foldercontainingtherasterfiles, fn),
            'RASTER_BAND': 1,
            'STATISTICS': [1],  # Sum
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['1'] = processing.run('native:zonalstatisticsfb',
                                               alg_params,
                                               context=context,
                                               feedback=feedback,
                                               is_child_algorithm=True
                                               )


        feedback.setCurrentStep(1)

        genders = ['f', 'm']
        ages = [['0', '1'], ['1', '4'], ['5', '9'], ['10', '14'], ['15', '19'], ['20', '24'], ['25', '29'], ['30', '34'], ['35', '39']]

        count = 2
        for gender in genders:
            for age in ages:
                if gender=='f' and age[0]=='0':
                    continue
                else:
                    inputvector = str(count-1)
                    fn = '_{}_{}_'.format(gender, age[0]) + str(parameters['Year'])
                    out = '{}{}To{}'.format(gender.upper(), age[0], age[1])
                    if not parameters['useConstrainedPopulationEstimates']:
                        fn = parameters['isoCountryCode'].lower() + fn + ".tif"
                    elif parameters['useConstrainedPopulationEstimates'] and not parameters['UseUNadjustedconstrainedestimates']:
                        fn = parameters['isoCountryCode'].lower() + fn + "_constrained.tif"
                    elif parameters['useConstrainedPopulationEstimates'] and parameters['UseUNadjustedconstrainedestimates']:
                        fn = parameters['isoCountryCode'].lower() + fn + "_constrained_UNadj.tif"
                    alg_params = {
                        'COLUMN_PREFIX': '{}_{}_'.format(gender.upper(), age[0]),
                        'INPUT': outputs[inputvector]['OUTPUT'],
                        'INPUT_RASTER': os.path.join(foldercontainingtherasterfiles, fn),
                        'RASTER_BAND': 1,
                        'STATISTICS': [1],  # Sum
                        'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
                    }
                    outputs[str(count)] = processing.run('native:zonalstatisticsfb',
                                                           alg_params,
                                                           context=context,
                                                           feedback=feedback,
                                                           is_child_algorithm=True
                                                           )

                    feedback.setCurrentStep(count)
                    count += 1
                    if feedback.isCanceled():
                        return {}


        # Creating the 0 to 4 age groups

        genders = ["F", "M"]
        field_mapping = []

        for gender in genders:
            for i in range(5,40,5):

                expression_expression = f'\"{gender}_{i}_sum\"'
                expression_name = f'{gender}_{i}_sum'

                expression = {}
                expression['expression']= expression_expression
                expression['length']=0
                expression['name']=expression_name
                expression['precision']=0
                expression['type']=6

                field_mapping.append(expression)

        extra_expression = [{'expression': '\"F_0_sum\" + \"F_1_sum\"','length': 0,'name': 'F_0_sum','precision': 0,'type': 6},
                    {'expression': '\"M_0_sum\" + \"M_1_sum\"','length': 0,'name': 'M_0_sum','precision': 0,'type': 6}]

        field_mapping = field_mapping + extra_expression


        alg_params = {
            'FIELDS_MAPPING': field_mapping,
            'INPUT': outputs['18']['OUTPUT'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['CreatingThe0To4AgeGroups'] = processing.run('native:refactorfields',
                                                             alg_params,
                                                             context=context,
                                                             feedback=feedback,
                                                             is_child_algorithm=True
                                                             )



        feedback.setCurrentStep(21)
        if feedback.isCanceled():
            return {}

        # Create spatial index - Section 0
        alg_params = {
            'INPUT': outputs['CreatingThe0To4AgeGroups']['OUTPUT']
        }
        outputs['CreateSpatialIndexSection0'] = processing.run('native:createspatialindex',
                                                               alg_params,
                                                               context=context,
                                                               feedback=feedback,
                                                               is_child_algorithm=True
                                                               )


        feedback.setCurrentStep(22)
        if feedback.isCanceled():
            return {}

        # Calculating single years of age - Section 1
        alg_params = {
            'FIELDS_MAPPING': [{'expression': '$area','length': 0,'name': 'Area','precision': 0,'type': 6},
                               {'expression': ' 0.3616*\"M_0_sum\" - 0.2768*\"M_5_sum\" + 0.1488*\"M_10_sum\" - 0.0336*\"M_15_sum\"  ','length': 0,'name': 'Y_M_0','precision': 0,'type': 6},
                               {'expression': ' 0.2640*\"M_0_sum\" - 0.0960*\"M_5_sum\" + 0.0400*\"M_10_sum\" - 0.0080*\"M_15_sum\"  ','length': 0,'name': 'Y_M_1','precision': 0,'type': 6},
                               {'expression': ' 0.1840*\"M_0_sum\" + 0.0400*\"M_5_sum\" - 0.0320*\"M_10_sum\" + 0.0080*\"M_15_sum\" ','length': 0,'name': 'Y_M_2','precision': 0,'type': 6},
                               {'expression': ' 0.1200*\"M_0_sum\" + 0.1360*\"M_5_sum\" - 0.0720*\"M_10_sum\" + 0.0160*\"M_15_sum\"  ','length': 0,'name': 'Y_M_3','precision': 0,'type': 6},
                               {'expression': ' 0.0336*\"M_0_sum\" + 0.2272*\"M_5_sum\" - 0.0752*\"M_10_sum\" + 0.0144*\"M_15_sum\" ','length': 0,'name': 'Y_M_5','precision': 0,'type': 6},
                               {'expression': ' 0.0080*\"M_0_sum\" + 0.2320*\"M_5_sum\" - 0.0480*\"M_10_sum\" + 0.0080*\"M_15_sum\" ','length': 0,'name': 'Y_M_6','precision': 0,'type': 6},
                               {'expression': ' -0.0080*\"M_0_sum\" + 0.2160*\"M_5_sum\" - 0.0080*\"M_10_sum\" + 0.0000*\"M_15_sum\"','length': 0,'name': 'Y_M_7','precision': 0,'type': 6},
                               {'expression': ' -0.0160*\"M_0_sum\" + 0.1840*\"M_5_sum\" + 0.0400*\"M_10_sum\" - 0.0080*\"M_15_sum\"','length': 0,'name': 'Y_M_8','precision': 0,'type': 6},
                               {'expression': ' -0.0128*\"M_0_sum\" + 0.0848*\"M_5_sum\" + 0.1504*\"M_10_sum\" - 0.0240*\"M_15_sum\" + 0.0016*\"M_20_sum\"','length': 0,'name': 'Y_M_10','precision': 0,'type': 6},
                               {'expression': ' -0.0016*\"M_0_sum\" + 0.0144*\"M_5_sum\" + 0.2224*\"M_10_sum\" - 0.0416*\"M_15_sum\" + 0.0064*\"M_20_sum\"','length': 0,'name': 'Y_M_11','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"M_0_sum\" - 0.0336*\"M_5_sum\" + 0.2544*\"M_10_sum\" - 0.0336*\"M_15_sum\" + 0.0064*\"M_20_sum\"','length': 0,'name': 'Y_M_12','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"M_0_sum\" - 0.0416*\"M_5_sum\" + 0.2224*\"M_10_sum\" + 0.0144*\"M_15_sum\" - 0.0016*\"M_20_sum\"','length': 0,'name': 'Y_M_13','precision': 0,'type': 6},
                               {'expression': ' -0.0128*\"M_5_sum\" + 0.0848*\"M_10_sum\" + 0.1504*\"M_15_sum\" - 0.0240*\"M_20_sum\" + 0.0016*\"M_25_sum\"','length': 0,'name': 'Y_M_15','precision': 0,'type': 6},
                               {'expression': ' -0.0016*\"M_5_sum\" + 0.0144*\"M_10_sum\" + 0.2224*\"M_15_sum\" - 0.0416*\"M_20_sum\" + 0.0064*\"M_25_sum\"','length': 0,'name': 'Y_M_16','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"M_5_sum\" - 0.0336*\"M_10_sum\" + 0.2544*\"M_15_sum\" - 0.0336*\"M_20_sum\" + 0.0064*\"M_25_sum\"','length': 0,'name': 'Y_M_17','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"M_5_sum\" - 0.0416*\"M_10_sum\" + 0.2224*\"M_15_sum\" + 0.0144*\"M_20_sum\" - 0.0016*\"M_25_sum\"','length': 0,'name': 'Y_M_18','precision': 0,'type': 6},
                               {'expression': ' -0.0128*\"M_10_sum\" + 0.0848*\"M_15_sum\" + 0.1504*\"M_20_sum\" - 0.0240*\"M_25_sum\" + 0.0016*\"M_30_sum\"','length': 0,'name': 'Y_M_20','precision': 0,'type': 6},
                               {'expression': ' -0.0016*\"M_10_sum\" + 0.0144*\"M_15_sum\" + 0.2224*\"M_20_sum\" - 0.0416*\"M_25_sum\" + 0.0064*\"M_30_sum\"','length': 0,'name': 'Y_M_21','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"M_10_sum\" - 0.0336*\"M_15_sum\" + 0.2544*\"M_20_sum\" - 0.0336*\"M_25_sum\" + 0.0064*\"M_30_sum\"','length': 0,'name': 'Y_M_22','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"M_10_sum\" - 0.0416*\"M_15_sum\" + 0.2224*\"M_20_sum\" + 0.0144*\"M_25_sum\" - 0.0016*\"M_30_sum\"','length': 0,'name': 'Y_M_23','precision': 0,'type': 6},
                               {'expression': ' -0.0128*\"M_15_sum\" + 0.0848*\"M_20_sum\" + 0.1504*\"M_25_sum\" - 0.0240*\"M_30_sum\" + 0.0016*\"M_35_sum\"','length': 0,'name': 'Y_M_25','precision': 0,'type': 6},
                               {'expression': ' -0.0016*\"M_15_sum\" + 0.0144*\"M_20_sum\" + 0.2224*\"M_25_sum\" - 0.0416*\"M_30_sum\" + 0.0064*\"M_35_sum\"','length': 0,'name': 'Y_M_26','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"M_15_sum\" - 0.0336*\"M_20_sum\" + 0.2544*\"M_25_sum\" - 0.0336*\"M_30_sum\" + 0.0064*\"M_35_sum\"','length': 0,'name': 'Y_M_27','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"M_15_sum\" - 0.0416*\"M_20_sum\" + 0.2224*\"M_25_sum\" + 0.0144*\"M_30_sum\" - 0.0016*\"M_35_sum\"','length': 0,'name': 'Y_M_28','precision': 0,'type': 6},
                               {'expression': ' 0.3616*\"F_0_sum\" - 0.2768*\"F_5_sum\" + 0.1488*\"F_10_sum\" - 0.0336*\"F_15_sum\" ','length': 0,'name': 'Y_F_0','precision': 0,'type': 6},
                               {'expression': ' 0.2640*\"F_0_sum\" - 0.0960*\"F_5_sum\" + 0.0400*\"F_10_sum\" - 0.0080*\"F_15_sum\" ','length': 0,'name': 'Y_F_1','precision': 0,'type': 6},
                               {'expression': ' 0.1840*\"F_0_sum\" + 0.0400*\"F_5_sum\" - 0.0320*\"F_10_sum\" + 0.0080*\"F_15_sum\" ','length': 0,'name': 'Y_F_2','precision': 0,'type': 6},
                               {'expression': ' 0.1200*\"F_0_sum\" + 0.1360*\"F_5_sum\" - 0.0720*\"F_10_sum\" + 0.0160*\"F_15_sum\" ','length': 0,'name': 'Y_F_3','precision': 0,'type': 6},
                               {'expression': ' 0.0336*\"F_0_sum\" + 0.2272*\"F_5_sum\" - 0.0752*\"F_10_sum\" + 0.0144*\"F_15_sum\"','length': 0,'name': 'Y_F_5','precision': 0,'type': 6},
                               {'expression': ' 0.0080*\"F_0_sum\" + 0.2320*\"F_5_sum\" - 0.0480*\"F_10_sum\" + 0.0080*\"F_15_sum\" ','length': 0,'name': 'Y_F_6','precision': 0,'type': 6},
                               {'expression': ' -0.0080*\"F_0_sum\" + 0.2160*\"F_5_sum\" - 0.0080*\"F_10_sum\" + 0.0000*\"F_15_sum\"','length': 0,'name': 'Y_F_7','precision': 0,'type': 6},
                               {'expression': ' -0.0160*\"F_0_sum\" + 0.1840*\"F_5_sum\" + 0.0400*\"F_10_sum\" - 0.0080*\"F_15_sum\"','length': 0,'name': 'Y_F_8','precision': 0,'type': 6},
                               {'expression': ' -0.0128*\"F_0_sum\" + 0.0848*\"F_5_sum\" + 0.1504*\"F_10_sum\" - 0.0240*\"F_15_sum\" + 0.0016*\"F_20_sum\"','length': 0,'name': 'Y_F_10','precision': 0,'type': 6},
                               {'expression': ' -0.0016*\"F_0_sum\" + 0.0144*\"F_5_sum\" + 0.2224*\"F_10_sum\" - 0.0416*\"F_15_sum\" + 0.0064*\"F_20_sum\"','length': 0,'name': 'Y_F_11','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"F_0_sum\" - 0.0336*\"F_5_sum\" + 0.2544*\"F_10_sum\" - 0.0336*\"F_15_sum\" + 0.0064*\"F_20_sum\"','length': 0,'name': 'Y_F_12','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"F_0_sum\" - 0.0416*\"F_5_sum\" + 0.2224*\"F_10_sum\" + 0.0144*\"F_15_sum\" - 0.0016*\"F_20_sum\"','length': 0,'name': 'Y_F_13','precision': 0,'type': 6},
                               {'expression': ' -0.0128*\"F_5_sum\" + 0.0848*\"F_10_sum\" + 0.1504*\"F_15_sum\" - 0.0240*\"F_20_sum\" + 0.0016*\"F_25_sum\"','length': 0,'name': 'Y_F_15','precision': 0,'type': 6},
                               {'expression': ' -0.0016*\"F_5_sum\" + 0.0144*\"F_10_sum\" + 0.2224*\"F_15_sum\" - 0.0416*\"F_20_sum\" + 0.0064*\"F_25_sum\"','length': 0,'name': 'Y_F_16','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"F_5_sum\" - 0.0336*\"F_10_sum\" + 0.2544*\"F_15_sum\" - 0.0336*\"F_20_sum\" + 0.0064*\"F_25_sum\"','length': 0,'name': 'Y_F_17','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"F_5_sum\" - 0.0416*\"F_10_sum\" + 0.2224*\"F_15_sum\" + 0.0144*\"F_20_sum\" - 0.0016*\"F_25_sum\"','length': 0,'name': 'Y_F_18','precision': 0,'type': 6},
                               {'expression': ' -0.0128*\"F_10_sum\" + 0.0848*\"F_15_sum\" + 0.1504*\"F_20_sum\" - 0.0240*\"F_25_sum\" + 0.0016*\"F_30_sum\"','length': 0,'name': 'Y_F_20','precision': 0,'type': 6},
                               {'expression': ' -0.0016*\"F_10_sum\" + 0.0144*\"F_15_sum\" + 0.2224*\"F_20_sum\" - 0.0416*\"F_25_sum\" + 0.0064*\"F_30_sum\"','length': 0,'name': 'Y_F_21','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"F_10_sum\" - 0.0336*\"F_15_sum\" + 0.2544*\"F_20_sum\" - 0.0336*\"F_25_sum\" + 0.0064*\"F_30_sum\"','length': 0,'name': 'Y_F_22','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"F_10_sum\" - 0.0416*\"F_15_sum\" + 0.2224*\"F_20_sum\" + 0.0144*\"F_25_sum\" - 0.0016*\"F_30_sum\"','length': 0,'name': 'Y_F_23','precision': 0,'type': 6},
                               {'expression': ' -0.0128*\"F_15_sum\" + 0.0848*\"F_20_sum\" + 0.1504*\"F_25_sum\" - 0.0240*\"F_30_sum\" + 0.0016*\"F_35_sum\"','length': 0,'name': 'Y_F_25','precision': 0,'type': 6},
                               {'expression': ' -0.0016*\"F_15_sum\" + 0.0144*\"F_20_sum\" + 0.2224*\"F_25_sum\" - 0.0416*\"F_30_sum\" + 0.0064*\"F_35_sum\"','length': 0,'name': 'Y_F_26','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"F_15_sum\" - 0.0336*\"F_20_sum\" + 0.2544*\"F_25_sum\" - 0.0336*\"F_30_sum\" + 0.0064*\"F_35_sum\"','length': 0,'name': 'Y_F_27','precision': 0,'type': 6},
                               {'expression': ' 0.0064*\"F_15_sum\" - 0.0416*\"F_20_sum\" + 0.2224*\"F_25_sum\" + 0.0144*\"F_30_sum\" - 0.0016*\"F_35_sum\"','length': 0,'name': 'Y_F_28','precision': 0,'type': 6},
                               {'expression': '\"M_0_sum\"','length': 0,'name': 'M_0_sum','precision': 0,'type': 6},
                               {'expression': '\"M_5_sum\"','length': 0,'name': 'M_5_sum','precision': 0,'type': 6},
                               {'expression': '\"M_10_sum\"','length': 0,'name': 'M_10_sum','precision': 0,'type': 6},
                               {'expression': '\"M_15_sum\"','length': 0,'name': 'M_15_sum','precision': 0,'type': 6},
                               {'expression': '\"M_20_sum\"','length': 0,'name': 'M_20_sum','precision': 0,'type': 6},
                               {'expression': '\"M_25_sum\"','length': 0,'name': 'M_25_sum','precision': 0,'type': 6},
                               {'expression': '\"F_0_sum\"','length': 0,'name': 'F_0_sum','precision': 0,'type': 6},
                               {'expression': '\"F_5_sum\"','length': 0,'name': 'F_5_sum','precision': 0,'type': 6},
                               {'expression': '\"F_10_sum\"','length': 0,'name': 'F_10_sum','precision': 0,'type': 6},
                               {'expression': '\"F_15_sum\"','length': 0,'name': 'F_15_sum','precision': 0,'type': 6},
                               {'expression': '\"F_20_sum\"','length': 0,'name': 'F_20_sum','precision': 0,'type': 6},
                               {'expression': '\"F_25_sum\"','length': 0,'name': 'F_25_sum','precision': 0,'type': 6}],
            'INPUT': outputs['CreateSpatialIndexSection0']['OUTPUT'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['CalculatingSingleyearsOfAgeSection1'] = processing.run('native:refactorfields',
                                                                        alg_params,
                                                                        context=context,
                                                                        feedback=feedback,
                                                                        is_child_algorithm=True
                                                                        )

        feedback.setCurrentStep(23)
        if feedback.isCanceled():
            return {}

        # Create spatial index - Section 1
        alg_params = {
            'INPUT': outputs['CalculatingSingleyearsOfAgeSection1']['OUTPUT']
        }
        outputs['CreateSpatialIndexSection1'] = processing.run('native:createspatialindex',
                                                               alg_params,
                                                               context=context,
                                                               feedback=feedback,
                                                               is_child_algorithm=True
                                                               )

        feedback.setCurrentStep(24)
        if feedback.isCanceled():
            return {}

        # Calculating single years of age - Section 2

        field_mapping = []
        genders = ["M", "F"]

        for gender in genders:
            for i in range(4,34,5):

                expression_expression = f'\"{gender}_{i - 4}_sum\" - \"Y_{gender}_{i - 4}\" - \"Y_{gender}_{i - 3}\" - \"Y_{gender}_{i - 2}\" - \"Y_{gender}_{i - 1}\"'
                expression_name = f'Y_{gender}_{i}'

                expression = {}
                expression['expression']= expression_expression
                expression['length']=0
                expression['name']=expression_name
                expression['precision']=0
                expression['type']=6

                field_mapping.append(expression)

        pass_line = 4
        for i in range(29):
            if i == pass_line:
                pass_line = pass_line + 5
                continue

            expression_expression = f'\"Y_M_{i}\" + \"Y_F_{i}\"'
            expression_name = f'Y_T_{i}'

            expression = {}
            expression['expression']= expression_expression
            expression['length']=0
            expression['name']=expression_name
            expression['precision']=0
            expression['type']=6

            field_mapping.append(expression)


        alg_params = {
            'FIELDS_MAPPING': field_mapping,
            'INPUT': outputs['CreateSpatialIndexSection1']['OUTPUT'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['CalculatingSingleyearsOfAgeSection2'] = processing.run('native:refactorfields',
                                                                        alg_params,
                                                                        context=context,
                                                                        feedback=feedback,
                                                                        is_child_algorithm=True
                                                                        )

        feedback.setCurrentStep(25)
        if feedback.isCanceled():
            return {}

        # Calculating single years of age - Section 3

        field_mapping = []

        for i in range(4,34,5):

            expression_expression = f'\"Y_M_{i}\" + \"Y_F_{i}\"'
            expression_name = f'Y_T_{i}'

            expression = {}
            expression['expression']= expression_expression
            expression['length']=0
            expression['name']=expression_name
            expression['precision']=0
            expression['type']=6

            field_mapping.append(expression)

        alg_params = {
            'FIELDS_MAPPING': field_mapping,
            'INPUT': outputs['CalculatingSingleyearsOfAgeSection2']['OUTPUT'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['CalculatingSingleyearsOfAgeSection3'] = processing.run('native:refactorfields',
                                                                        alg_params,
                                                                        context=context,
                                                                        feedback=feedback,
                                                                        is_child_algorithm=True
                                                                        )

        feedback.setCurrentStep(26)
        if feedback.isCanceled():
            return {}

        # Create spatial index - Section 2
        alg_params = {
            'INPUT': outputs['CalculatingSingleyearsOfAgeSection2']['OUTPUT']
        }
        outputs['CreateSpatialIndexSection2'] = processing.run('native:createspatialindex',
                                                               alg_params,
                                                               context=context,
                                                               feedback=feedback,
                                                               is_child_algorithm=True
                                                               )

        feedback.setCurrentStep(27)
        if feedback.isCanceled():
            return {}

        # Create spatial index - Section 3
        alg_params = {
            'INPUT': outputs['CalculatingSingleyearsOfAgeSection3']['OUTPUT']
        }
        outputs['CreateSpatialIndexSection3'] = processing.run('native:createspatialindex',
                                                               alg_params,
                                                               context=context,
                                                               feedback=feedback,
                                                               is_child_algorithm=True
                                                               )

        feedback.setCurrentStep(28)
        if feedback.isCanceled():
            return {}

        # Join attributes by location - Section 1 and 2
        alg_params = {
            'DISCARD_NONMATCHING': True,
            'INPUT': outputs['CreateSpatialIndexSection1']['OUTPUT'],
            'JOIN': outputs['CreateSpatialIndexSection2']['OUTPUT'],
            'JOIN_FIELDS': [''],
            'METHOD': 1,
            'PREDICATE': [2],
            'PREFIX': '',
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['JoinAttributesByLocationSection1And2'] = processing.run('native:joinattributesbylocation',
                                                                         alg_params,
                                                                         context=context,
                                                                         feedback=feedback,
                                                                         is_child_algorithm=True
                                                                         )

        feedback.setCurrentStep(29)
        if feedback.isCanceled():
            return {}

        # Create spatial index - Section 1 and 2
        alg_params = {
            'INPUT': outputs['JoinAttributesByLocationSection1And2']['OUTPUT']
        }
        outputs['CreateSpatialIndexSection1and2'] = processing.run('native:createspatialindex',
                                                                   alg_params,
                                                                   context=context,
                                                                   feedback=feedback,
                                                                   is_child_algorithm=True
                                                                   )

        feedback.setCurrentStep(30)
        if feedback.isCanceled():
            return {}

        # Join attributes by location - Sections 1 through 3
        alg_params = {
            'DISCARD_NONMATCHING': True,
            'INPUT': outputs['CreateSpatialIndexSection1and2']['OUTPUT'],
            'JOIN': outputs['CreateSpatialIndexSection3']['OUTPUT'],
            'JOIN_FIELDS': [''],
            'METHOD': 1,
            'PREDICATE': [2],
            'PREFIX': '',
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['JoinAttributesByLocationSections1Through3'] = processing.run('native:joinattributesbylocation',
                                                                              alg_params,
                                                                              context=context,
                                                                              feedback=feedback,
                                                                              is_child_algorithm=True
                                                                              )

        feedback.setCurrentStep(31)
        if feedback.isCanceled():
            return {}

        # Reorganizing the results

        genders = ["M", "F", "T"]
        field_mapping = []
        for gender in genders:
            for age in range(30):

                expression_level = '"Y_' + gender + '_' + str(age) + '"'
                expression_precision = 'Y_' + gender + '_' + str(age)

                expression = {}
                expression['expression']= expression_level
                expression['length']=0
                expression['name']=expression_precision
                expression['precision']=0
                expression['type']=6

                field_mapping.append(expression)

        alg_params = {
            'FIELDS_MAPPING': field_mapping,
            'INPUT': outputs['JoinAttributesByLocationSections1Through3']['OUTPUT'],
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['ReorganizingTheResults'] = processing.run('native:refactorfields',
                                                           alg_params,
                                                           context=context,
                                                           feedback=feedback,
                                                           is_child_algorithm=True
                                                           )
        feedback.setCurrentStep(32)
        if feedback.isCanceled():
            return {}

        # Create spatial index for Administrative boundaries
        alg_params = {
            'INPUT': parameters['Administrativeboundaries']
        }
        outputs['IndexedAdministrativeboundaries'] = processing.run('native:createspatialindex',
                                                                    alg_params,
                                                                    context=context,
                                                                    feedback=feedback,
                                                                    is_child_algorithm=True
                                                                    )

        feedback.setCurrentStep(33)
        if feedback.isCanceled():
            return {}


        # Random points in extent
        alg_params = {
            'EXTENT': '1.000000000,1.000000000,1.000000000,1.000000000',
            'MAX_ATTEMPTS': 200,
            'MIN_DISTANCE': 0,
            'POINTS_NUMBER': 1,
            'TARGET_CRS': 'ProjectCrs',
            'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
        }
        outputs['RandomPointsInExtent'] = processing.run('native:randompointsinextent',
                                                         alg_params,
                                                         context=context,
                                                         feedback=feedback,
                                                         is_child_algorithm=True
                                                         )

        feedback.setCurrentStep(5)
        if feedback.isCanceled():
            return {}

        country_code = parameters['isoCountryCode']
        year = parameters['Year']
        fnParameters = f'Population_estimates_{country_code}{year}Parameters.xlsx'
        fnSchoolAge = f'Population_estimates_{country_code}{year}SchoolAge.xlsx'
        exportParametersToExcel = os.path.join(parameters['foldercontainingtherasterfiles'], fnParameters)
        exportSchoolAge = os.path.join(parameters['foldercontainingtherasterfiles'], fnSchoolAge)

        if parameters['Createcustomschoolagegroups']:

            # Calculating school ages with Lower and Upper secondary
            genders = ["M", "F", "T"]
            levels = ["preprimary", "primary", "lowersecondary", "uppersecondary"]
            field_mapping = []
            for gender in genders:
                for age in range(0,30):

                    expression_expression = '"Y_' + gender + '_' + str(age) + '"'
                    expression_name = 'Y_' + gender + '_' + str(age)

                    expression = {}
                    expression['expression']= expression_expression
                    expression['length']=0
                    expression['name']=expression_name
                    expression['precision']=0
                    expression['type']=6

                    field_mapping.append(expression)


            for level in levels:

                starting = vars()[level + 'startingage']
                duration = vars()[level + 'duration']

                for gender in genders:

                    expression_expression = f'eval(array_to_string(array_foreach(generate_series( {starting} ,  {starting} + {duration} - 1),concat(\'Y_{gender}_\',@element)),\'+\'))'
                    expression_name = level + '_' + gender

                    expression = {}
                    expression['expression']= expression_expression
                    expression['length']=0
                    expression['name']=expression_name
                    expression['precision']=0
                    expression['type']=6


                    field_mapping.append(expression)

            alg_params = {
                'FIELDS_MAPPING': field_mapping,
                'INPUT': outputs['ReorganizingTheResults']['OUTPUT'],
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }
            outputs['CalculatingSchoolAges'] = processing.run('native:refactorfields',
                                                                                        alg_params,
                                                                                        context=context,
                                                                                        feedback=feedback,
                                                                                        is_child_algorithm=True
                                                                                        )

            feedback.setCurrentStep(34)
            if feedback.isCanceled():
                return {}

            # Preparing the table to export the parameters - Lower and Upper secondary
            alg_params = {
                'FIELDS_MAPPING': [{'expression': parameters['isoCountryCode'].lower(),'length': 10,'name': 'ISO country code','precision': 0,'type': 10},
                                   {'expression': year,'length': 0,'name': 'year','precision': 0,'type': 2},
                                   {'expression': preprimarystartingage,'length': 0,'name': 'Pre-primary starting age','precision': 0,'type': 2},
                                   {'expression': preprimaryduration,'length': 0,'name': 'Pre-primary duration','precision': 0,'type': 2},
                                   {'expression': primarystartingage,'length': 0,'name': 'Primary starting age','precision': 0,'type': 2},
                                   {'expression': primaryduration,'length': 0,'name': 'Primary duration','precision': 0,'type': 2},
                                   {'expression': lowersecondarystartingage,'length': 0,'name': 'Lower secondary starting age','precision': 0,'type': 2},
                                   {'expression': lowersecondaryduration,'length': 0,'name': 'Lower secondary duration','precision': 0,'type': 2},
                                   {'expression': uppersecondarystartingage,'length': 0,'name': 'Upper secondary starting age','precision': 0,'type': 2},
                                   {'expression': uppersecondaryduration,'length': 0,'name': 'Upper secondary duration','precision': 0,'type': 2}],
                'INPUT': outputs['RandomPointsInExtent']['OUTPUT'],
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }
            outputs['PreparingTheTableToExportTheParameters'] = processing.run('native:refactorfields',
                                                                                                     alg_params,
                                                                                                     context=context,
                                                                                                     feedback=feedback,
                                                                                                     is_child_algorithm=True
                                                                                                     )

            feedback.setCurrentStep(38)
            if feedback.isCanceled():
                return {}

        if not parameters['Createcustomschoolagegroups']:


            # Calculating school ages
            genders = ["M", "F", "T"]
            field_mapping = []
            for gender in genders:
                for age in range(0,30):

                    expression_level = '"Y_' + gender + '_' + str(age) + '"'
                    expression_precision = 'Y_' + gender + '_' + str(age)

                    expression = {}
                    expression['expression']= expression_level
                    expression['length']=0
                    expression['name']=expression_precision
                    expression['precision']=0
                    expression['type']=6

                    field_mapping.append(expression)

            alg_params = {
                'FIELDS_MAPPING': field_mapping,
                'INPUT': outputs['ReorganizingTheResults']['OUTPUT'],
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }
            outputs['CalculatingSchoolAges'] = processing.run('native:refactorfields',
                                                                                        alg_params,
                                                                                        context=context,
                                                                                        feedback=feedback,
                                                                                        is_child_algorithm=True
                                                                                        )

            feedback.setCurrentStep(34)
            if feedback.isCanceled():
                return {}


            # Preparing the table to export the parameters
            alg_params = {
                'FIELDS_MAPPING': [{'expression': isoCountryCode,'length': 50,'name': 'ISO country code','precision': 0,'type': 10},
                                   {'expression': year,'length': 0,'name': 'year','precision': 0,'type': 2}],
                'INPUT': outputs['RandomPointsInExtent']['OUTPUT'],
                'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT
            }
            outputs['PreparingTheTableToExportTheParameters'] = processing.run('native:refactorfields',
                                                                               alg_params,
                                                                               context=context,
                                                                               feedback=feedback,
                                                                               is_child_algorithm=True
                                                                               )

            feedback.setCurrentStep(37)
            if feedback.isCanceled():
                return {}


        # Create spatial index
        alg_params = {
            'INPUT': outputs['CalculatingSchoolAges']['OUTPUT']
        }
        outputs['CreateSpatialIndex'] = processing.run('native:createspatialindex',
                                                                   alg_params,
                                                                   context=context,
                                                                   feedback=feedback,
                                                                   is_child_algorithm=True
                                                                   )

        feedback.setCurrentStep(35)
        if feedback.isCanceled():
            return {}

        # Creating the file shapefile
        alg_params = {
            'DISCARD_NONMATCHING': True,
            'INPUT': outputs['IndexedAdministrativeboundaries']['OUTPUT'],
            'JOIN': outputs['CreateSpatialIndex']['OUTPUT'],
            'JOIN_FIELDS': [''],
            'METHOD': 1,
            'PREDICATE': [2],
            'PREFIX': '',
            'OUTPUT': parameters['Results']
        }
        outputs['CreatingTheFileShapefile'] = processing.run('native:joinattributesbylocation',
                                                                      alg_params,
                                                                      context=context,
                                                                      feedback=feedback,
                                                                      is_child_algorithm=True
                                                                      )
        results['Results'] = outputs['CreatingTheFileShapefile']['OUTPUT']

        feedback.setCurrentStep(36)
        if feedback.isCanceled():
            return {}
        # Exporting the results to Excel
        alg_params = {
            'DATASOURCE_OPTIONS': '',
            'INPUT': outputs['CreatingTheFileShapefile']['OUTPUT'],
            'LAYER_NAME': '',
            'LAYER_OPTIONS': '',
            'OUTPUT': exportSchoolAge
        }
        outputs['ExportingTheResultsToExcel'] = processing.run('native:savefeatures',
                                                               alg_params,
                                                               context=context,
                                                               feedback=feedback,
                                                               is_child_algorithm=True
                                                               )

        results['Results'] = outputs['ExportingTheResultsToExcel']['OUTPUT']

        feedback.setCurrentStep(37)
        if feedback.isCanceled():
            return {}



        # Export to spreadsheet
        alg_params = {
            'FORMATTED_VALUES': False,
            'LAYERS': outputs['PreparingTheTableToExportTheParameters']['OUTPUT'],
            'OUTPUT': exportParametersToExcel,
            'OVERWRITE': True,
            'USE_ALIAS': False
        }
        outputs['ExportToSpreadsheet'] = processing.run('native:exporttospreadsheet',
                                                        alg_params,
                                                        context=context,
                                                        feedback=feedback,
                                                        is_child_algorithm=True
                                                        )

        feedback.setCurrentStep(40)
        if feedback.isCanceled():
            return {}


        return results




    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Sprague for school age populations'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def shortHelpString(self):
        return """<html><body><h2>Algorithm description</h2>
<p>This algorithm aims at creating single years of age for any level of administrative boundaries or any other polygon layer, based on clustered 5-year age group raster files, by applying Sprague multipliers. It also allows the user to reconstruct the different school age groups for a particular country, regardless of whether the system divides lower and upper secondary or not. </p>
<h2>Input parameters</h2>
<h3>Folder containing the raster files</h3>
<p>Select the folder containing the raster files with the 5-year age groups.</p>
<h3>Administrative boundaries</h3>
<p>Polygon layer containing the administrative boundaries on which the single years of age' variables will be calculated. It can also be any other arrangement (e.g. Voronoi polygons).</p>
<h3>ISO country code</h3>
<p>3-letter ISO country code</p>
<h3>Year</h3>
<p>Year of analysis (e.g. 2014)</p>
<h3>Use unconstrained population estimates</h3>
<p>As defined by the classification produced by WorldPop</p>
<h3>Use UN-adjusted constrained estimates</h3>
<p>As defined by the classification produced by WorldPop</p>
<h3>Create custom school age groups</h3>
<p>Select to create custom school age groups for the country or region.</p>
<h3>Pre-primary starting age</h3>
<p>This must be a number.</p>
<h3>Pre-primary duration</h3>
<p>This must be a number.</p>
<h3>Primary starting age</h3>
<p>This must be a number.</p>
<h3>Primary duration</h3>
<p>This must be a number.</p>
<h3>Lower secondary starting age</h3>
<p>This must be a number.</p>
<h3>Lower secondary duration</h3>
<p>This must be a number.</p>
<h3>Upper secondary starting age</h3>
<p>This must be a number.</p>
<h3>Upper secondary duration</h3>
<p>This must be a number.</p>
<h2>Outputs</h2>
<h3>Results</h3>
<p>It will contain the same information as the Administrative boundaries' polygon layer, plus additional columns for the single years of age by sex. If the option to reconstruct school age groups has been chosen and the parameters added, it should contain additional columns for pre-primary, primary, lower, and upper secondary, for female, male, and total.</p>
<br><p>Algorithm author: Development unit, IIEP-UNESCO (development@iiep.unesco.org)</p><p>Help author: Development unit, IIEP-UNESCO (development@iiep.unesco.org)</p><p>Algorithm version: 1.0</p></body></html>"""



    def icon(self):
        plugin_dir = os.path.dirname(__file__)
        icon = QIcon(os.path.join(plugin_dir, 'img', 'sprague_logo.svg'))
        return icon


    def createInstance(self):
        return SpragueMultipliersAlgorithm()
