# -*- coding: utf-8 -*-
"""
/***************************************************************************
 SLD4raster
                                 A QGIS plugin
 Generates SLD (Styled Layer Descriptor) for raster layers.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-06-15
        git sha              : $Format:%H$
        copyright            : (C) 2018 by Mehmet Selim BILGIN
        email                : mselimbilgin@yahoo.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import *
from PyQt5.QtGui import *
from PyQt5.QtWidgets import *
from PyQt5.QtXml import QDomDocument
from PyQt5.Qsci import QsciScintilla, QsciLexerXML
from qgis.core import *

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .dialogs import SLD4rasterDialog, GeoserverUploadDialog

import codecs
import base64
import os.path
from xml.etree.ElementTree import Element, SubElement, tostring
from xml.dom import minidom
from urllib.request import Request, urlopen


class SLD4raster:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'SLD4raster_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&SLD4raster')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'SLD4raster')
        self.toolbar.setObjectName(u'SLD4raster')

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('SLD4raster', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/sld4raster/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'SLD4raster'),
            callback=self.run,
            parent=self.iface.mainWindow())

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&SLD4raster'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    def uiSettings(self):
        self.dlg.sldText1.setMarginLineNumbers(1, True)
        self.dlg.sldText1.setMarginsBackgroundColor(QColor("#98AFC7"))
        self.dlg.sldText1.setMarginWidth(1, 30)
        self.dlg.sldText1.setUtf8(True)  # Enabling non-Ascii chars
        self.dlg.sldText1.setLexer(QsciLexerXML())

        self.dlg.sldText2.setMarginLineNumbers(1, True)
        self.dlg.sldText2.setMarginsBackgroundColor(QColor("#98AFC7"))
        self.dlg.sldText2.setMarginWidth(1, 30)
        self.dlg.sldText2.setUtf8(True)  # Enabling non-Ascii chars
        self.dlg.sldText2.setLexer(QsciLexerXML())

        self.dlg.setFixedSize(600, 550)

    def validation(self, inputXML):
        try:
            minidom.parseString(inputXML.encode("utf-8"))  # UTF encoding for non-ASCII chars.
            QMessageBox.information(None, "Information", "The XML document is well-formed.")
        except Exception as SLDerror:
            QMessageBox.critical(None, "Invalid XML", str(SLDerror))

    def browseFile(self):
        browseDlg, _ = QFileDialog.getOpenFileName(self.dlg, "Choose SLD File...", ".",
                                                   "SLD file (*.sld);; XML file (*.xml);; All files (*.*)")
        if browseDlg:
            self.dlg.lineEdit.setText(browseDlg)
            try:
                with codecs.open(browseDlg, encoding='utf-8', mode='r') as readFile:
                    self.dlg.sldText2.setText(readFile.read())
            except Exception as readError:
                QMessageBox.critical(None, "Information", ("An error has occured: " + str(readError)))

    def saveFile(self, document, type):
        # Saving sld or qml documents to files.
        try:
            # Firstly checking xml syntax.
            minidom.parseString(document.encode("utf-8"))
            if type == 'SLD':
                saveDlg, _ = QFileDialog.getSaveFileName(self.dlg, 'Save SLD File...', '.',
                                                         'SLD file (*.sld);; XML file (*.xml);; All files (*.*)')
                saveMessage = 'The SLD file has been succesfully saved.'
            else:
                saveDlg, _ = QFileDialog.getSaveFileName(self.dlg, 'Save QML File...', '.',
                                                         'QGIS Layer Style File (*.qml)')
                saveMessage = 'The QML file has been succesfully saved.'

            if saveDlg:
                try:
                    with open(saveDlg, 'w') as file:
                        file.write(document)
                    QMessageBox.information(None, "Information", saveMessage)
                except Exception as saveError:
                    QMessageBox.critical(None, "Information", ("An error has occured: " + str(saveError)))

        except Exception as error:
            QMessageBox.critical(None, "Invalid XML", str(error))

    def showUploadDialog(self):
        try:
            minidom.parseString(self.dlg.sldText1.text().encode('utf-8'))

            self.upDlg.sldText.setText(self.dlg.comboBox.currentText())
            self.upDlg.sldText.setFocus()
            self.checkLogs()
            self.upDlg.exec_()
        except Exception as error:
            QMessageBox.critical(None, "Invalid XML", str(error))

    def rememberMe(self):
        # Remember me option operation.
        try:
            if self.upDlg.checkBox.isChecked():
                flatData = self.upDlg.urlText.text() + ';' + self.upDlg.userText.text() + ';' + self.upDlg.passText.text()
                with open(os.path.join(os.path.dirname(__file__), 'int.dat'), 'w') as file:
                    base64Data = base64.b64encode(str.encode(flatData)).decode("utf-8")
                    file.write(base64Data)

            else:
                os.remove(os.path.join(os.path.dirname(__file__), 'int.dat'))
        except Exception as error:
            QMessageBox.critical(None, "Information", str(error))

    def upload(self):
        ###Everything about uploding data to GeoServer is handling in here.
        status = 1  # I use it for managing operations. If status = 0 nothing to do.
        if not self.upDlg.urlText.text() or not self.upDlg.userText.text() \
                or not self.upDlg.passText.text() or not self.upDlg.sldText.text():
            QMessageBox.critical(None, 'Information', 'Please fill in all fields.')

        else:
            username = self.upDlg.userText.text()
            password = self.upDlg.passText.text()
            sldName = self.upDlg.sldText.text()
            login = base64.b64encode(str.encode(username + ':' + password)).decode("utf-8")
            basicLogin = 'Basic ' + login  # Base64 auth for REST service.

            # Deleting the SLD if exists on GeoServer.
            if self.upDlg.urlText.text()[-1] == '/':
                usableUrl = self.upDlg.urlText.text()[:len(
                    self.upDlg.urlText.text()) - 1]  # Maybe user add slash '/' in the end of url, so this problem is solved in this way.
            else:  # I didnt use urlparse lib. Because this solution is simplier than it.
                usableUrl = self.upDlg.urlText.text()

            try:
                url = usableUrl + '/styles/' + sldName + ".sld"
                requestPut = Request(url)
                requestPut.add_header("Authorization", basicLogin)
                requestPut.add_header("Content-type", "application/vnd.ogc.sld+xml")
                requestPut.add_header("Accept", "*/*")
                requestPut.data = str.encode(self.dlg.sldText1.text())
                requestPut.get_method = lambda: 'PUT'
                urlopen(requestPut)
                QMessageBox.information(None, 'Information', 'The style was succesfully uploaded.')
                self.upDlg.close()
            except:
                try:
                    url = usableUrl + '/styles'
                    requestPrepare = Request(url)
                    requestPrepare.add_header("Authorization", basicLogin)
                    requestPrepare.add_header("Content-type", "text/xml")
                    requestPrepare.add_header("Accept", "*/*")
                    requestPrepare.data = str.encode(
                        '<style><name>' + sldName + '</name><filename>' + (sldName + '.sld') + '</filename></style>')  #

                    urlopen(requestPrepare)

                    url = usableUrl + '/styles/' + sldName + ".sld"
                    requestPut = Request(url)
                    requestPut.add_header("Authorization", basicLogin)
                    requestPut.add_header("Content-type", "application/vnd.ogc.sld+xml")
                    requestPut.add_header("Accept", "*/*")
                    requestPut.data = str.encode(self.dlg.sldText1.text())
                    requestPut.get_method = lambda: 'PUT'
                    urlopen(requestPut)
                    QMessageBox.information(None, 'Information', 'The style was succesfully uploaded.')
                    self.upDlg.close()
                except Exception as error:
                    QMessageBox.critical(None, 'Information', str(error))

    def checkLogs(self):
        # Saved login data is handled in here.
        iRememberYou = os.path.isfile(os.path.join(os.path.dirname(__file__), 'int.dat'))
        if iRememberYou:

            with open(os.path.join(os.path.dirname(__file__), 'int.dat'), 'r') as file:
                logInData = base64.b64decode(file.read())

            try:
                self.upDlg.urlText.setText(logInData.split(';')[0])
                self.upDlg.userText.setText(logInData.split(';')[1])
                self.upDlg.passText.setText(logInData.split(';')[2])
            except:
                pass

            self.upDlg.checkBox.setChecked(True)

        else:
            self.upDlg.urlText.setText('http://localhost:8080/geoserver/rest')
            self.upDlg.userText.setText('admin')
            self.upDlg.passText.setText('geoserver')

    def qml2Sld(self):
        try:
            layer = self.allMapLayers[self.dlg.comboBox.currentIndex()]
        except:
            return

        qmlDocument = QDomDocument()
        root = qmlDocument.createElement('SLD4raster')
        qmlDocument.appendChild(root)
        qgisNode = qmlDocument.createElement('qgis')
        root.appendChild(qgisNode)

        errorMessage = None
        context = QgsReadWriteContext()
        layer.writeSymbology(qgisNode, qmlDocument, errorMessage, context)

        qmlString = minidom.parseString(qmlDocument.toString().encode('utf-8'))  # for non ASCII labels.

        sldRoot = Element('sld:StyledLayerDescriptor')
        sldRoot.attrib['xmlns'] = 'http://www.opengis.net/sld'
        sldRoot.attrib['xmlns:sld'] = 'http://www.opengis.net/sld'
        sldRoot.attrib['xmlns:ogc'] = 'http://www.opengis.net/ogc'
        sldRoot.attrib['xmlns:gml'] = 'http://www.opengis.net/gml'
        sldRoot.attrib['version'] = '1.0.0'

        userLayer = SubElement(sldRoot, 'sld:UserLayer')
        layerFeatureConstraints = SubElement(userLayer, 'sld:LayerFeatureConstraints')
        featureTypeConstraint = SubElement(layerFeatureConstraints, 'sld:FeatureTypeConstraint')
        userStyle = SubElement(userLayer, 'sld:UserStyle')

        styleName = SubElement(userStyle, 'sld:Name')
        styleName.text = self.dlg.comboBox.currentText()

        styleDescription = SubElement(userStyle, 'sld:Description')
        styleDescription.text = 'Generated by SLD4raster - https://cbsuygulama.wordpress.com/sld4raster'

        styleTitle = SubElement(userStyle, 'sld:Title')
        featureTypeStyle = SubElement(userStyle, 'sld:FeatureTypeStyle')
        featureName = SubElement(featureTypeStyle, 'sld:Name')
        featureRule = SubElement(featureTypeStyle, 'sld:Rule')
        rasterSymbolizer = SubElement(featureRule, 'sld:RasterSymbolizer')
        geometry = SubElement(rasterSymbolizer, 'sld:Geometry')
        ogcPropertyName = SubElement(geometry, 'ogc:PropertyName')
        ogcPropertyName.text = 'grid'
        opacity = SubElement(rasterSymbolizer, 'sld:Opacity')

        ###Getting raster type parameters
        rasterType = str(qmlString.getElementsByTagName('rasterrenderer')[0].attributes['type'].value)
        isGradient = 'gradient' in qmlString.getElementsByTagName('rasterrenderer')[0].attributes.keys()

        ###SLD for multiband raster
        if rasterType == 'multibandcolor':
            ###Getting RGB band order.
            redBand = str(qmlString.getElementsByTagName('rasterrenderer')[0].attributes['redBand'].value)
            greenBand = str(qmlString.getElementsByTagName('rasterrenderer')[0].attributes['greenBand'].value)
            blueBand = str(qmlString.getElementsByTagName('rasterrenderer')[0].attributes['blueBand'].value)

            channelSelection = SubElement(rasterSymbolizer, 'sld:ChannelSelection')

            redChannel = SubElement(channelSelection, 'sld:RedChannel')
            redSourceChannel = SubElement(redChannel, 'sld:SourceChannelName')
            redSourceChannel.text = redBand

            greenChannel = SubElement(channelSelection, 'sld:GreenChannel')
            greenSourceChannel = SubElement(greenChannel, 'sld:SourceChannelName')
            greenSourceChannel.text = greenBand

            blueChannel = SubElement(channelSelection, 'sld:BlueChannel')
            blueSourceChannel = SubElement(blueChannel, 'sld:SourceChannelName')
            blueSourceChannel.text = blueBand

        ###SLD for gradiented (black to white) raster
        elif isGradient:
            blackWhiteColor = ['#000000', '#FFFFFF']
            colorMap = SubElement(rasterSymbolizer, 'sld:ColorMap')
            gradientType = qmlString.getElementsByTagName('rasterrenderer')[0].attributes['gradient'].value
            blackWhiteValue = [qmlString.getElementsByTagName('minValue')[0].firstChild.nodeValue,
                               qmlString.getElementsByTagName('maxValue')[0].firstChild.nodeValue]

            ###Getting gradient color type
            if gradientType == 'WhiteToBlack':
                blackWhiteColor.reverse()

            for i in range(len(blackWhiteColor)):
                colorMapEntry = SubElement(colorMap, 'sld:ColorMapEntry')
                colorMapEntry.attrib['color'] = blackWhiteColor[i]
                colorMapEntry.attrib['opacity'] = '1.0'
                colorMapEntry.attrib['quantity'] = blackWhiteValue[i]

        ###SLD for singleband raster
        else:
            colorMap = SubElement(rasterSymbolizer, 'sld:ColorMap')

            ###Getting color ramp type
            colorType = str(qmlString.getElementsByTagName('colorrampshader')[0].attributes['colorRampType'].value)
            if colorType == 'DISCRETE':
                colorMap.attrib['type'] = "intervals"

            ###Getting color values
            colorValue = list()
            itemlist = qmlString.getElementsByTagName('item')
            for n in itemlist:
                colorValue.append(
                    [n.attributes['color'].value, n.attributes['value'].value, n.attributes['label'].value])

            ###Color values posting to SLD document
            for i in range(len(colorValue)):
                colorMapEntry = SubElement(colorMap, 'sld:ColorMapEntry')
                colorMapEntry.attrib['color'] = colorValue[i][0]
                colorMapEntry.attrib['quantity'] = colorValue[i][1]
                colorMapEntry.attrib['label'] = colorValue[i][2]
                colorMapEntry.attrib['opacity'] = '1.0'

        rasterOpacity = str(qmlString.getElementsByTagName('rasterrenderer')[0].attributes['opacity'].value)
        opacity.text = rasterOpacity
        textSLD = minidom.parseString(tostring(sldRoot))
        self.dlg.sldText1.setText(textSLD.toprettyxml(indent="    "))

    def sld2qml(self):
        try:
            sldDocument = minidom.parseString(self.dlg.sldText2.text().encode('utf-8'))

            # Namespaces (sld, se, none) are most important part. Firstly they must be handled.
            if len(sldDocument.getElementsByTagName('sld:RasterSymbolizer')) > 0:
                nameSpace = 'sld:'

            elif len(sldDocument.getElementsByTagName('se:RasterSymbolizer')) > 0:
                nameSpace = 'se:'

            else:
                nameSpace = ''  # no namespace.

            # main QML structure.
            qmlRoot = Element('qgis')
            pipe = SubElement(qmlRoot, 'pipe')

            rasterRenderer = SubElement(pipe, 'rasterrenderer')
            rasterTransparency = SubElement(rasterRenderer, 'rasterTransparency')

            # checking document for containing ColorMap tag. If does, it is a single-band raster.
            sldRasterType = sldDocument.getElementsByTagName(nameSpace + 'ColorMap')

            if len(sldRasterType) == 0:
                sldRedBand = sldDocument.getElementsByTagName(nameSpace + 'RedChannel')[0].getElementsByTagName(
                    nameSpace + 'SourceChannelName')[0].firstChild.nodeValue
                sldGreenBand = sldDocument.getElementsByTagName(nameSpace + 'GreenChannel')[0].getElementsByTagName(
                    nameSpace + 'SourceChannelName')[0].firstChild.nodeValue
                sldBlueBand = sldDocument.getElementsByTagName(nameSpace + 'BlueChannel')[0].getElementsByTagName(
                    nameSpace + 'SourceChannelName')[0].firstChild.nodeValue

                rasterRenderer.attrib['type'] = 'multibandcolor'
                rasterRenderer.attrib['redBand'] = sldRedBand
                rasterRenderer.attrib['greenBand'] = sldGreenBand
                rasterRenderer.attrib['blueBand'] = sldBlueBand

            else:
                rasterRenderer.attrib['type'] = 'singlebandpseudocolor'
                rasterRenderer.attrib['band'] = '1'
                rasterShader = SubElement(rasterRenderer, 'rastershader')

                colorRampShader = SubElement(rasterShader, 'colorrampshader')

                sldColorMapType = None
                sldColorMapType2 = None
                try:
                    sldColorMapType = sldDocument.getElementsByTagName(nameSpace + 'ColorMap')[0].attributes.has_key(
                        'type')  # sometimes "ColorMap" tag does not contain "type" attribute. This means it is a ramp color.
                    sldColorMapType2 = sldDocument.getElementsByTagName(nameSpace + 'ColorMap')[0].attributes[
                        'type'].value  # or getting raster map colortype by "type" atribute.
                except:
                    pass

                if sldColorMapType and sldColorMapType2 == 'intervals':
                    colorRampShader.attrib['colorRampType'] = 'DISCRETE'
                else:
                    colorRampShader.attrib['colorRampType'] = 'INTERPOLATED'

                sldColorValues = list()
                sldItemList = sldDocument.getElementsByTagName(nameSpace + 'ColorMapEntry')

                # some SLD documents don't have 'label' attribute so the problem is handled by this way.
                try:
                    for m in sldItemList:
                        sldColorValues.append(
                            [m.attributes['quantity'].value, m.attributes['label'].value, m.attributes['color'].value])
                except:
                    for m in sldItemList:
                        sldColorValues.append([m.attributes['quantity'].value, m.attributes['quantity'].value,
                                               m.attributes['color'].value])

                for s in range(len(sldColorValues)):
                    item = SubElement(colorRampShader, 'item')
                    item.attrib['alpha'] = '255'
                    item.attrib['value'] = sldColorValues[s][0]
                    item.attrib['label'] = sldColorValues[s][1]
                    item.attrib['color'] = sldColorValues[s][2]

            # some SLD documents don't have 'Opacity' tag so the problem is handled by this way.
            try:
                sldOpacity = sldDocument.getElementsByTagName(nameSpace + 'Opacity')[0].firstChild.nodeValue
            except:
                sldOpacity = '1'

            rasterRenderer.attrib['opacity'] = sldOpacity

            brightnessContrast = SubElement(pipe, 'brightnesscontrast')
            hueSaturation = SubElement(pipe, 'huesaturation')
            rasterResampler = SubElement(pipe, 'rasterresampler')
            blendMode = SubElement(qmlRoot, 'blendMode')
            blendMode.text = '0'
            textQML = minidom.parseString(tostring(qmlRoot)).toprettyxml(indent="    ")
            self.saveFile(textQML, 'QML')

        except Exception as sldTransformError:
            QMessageBox.critical(None, "Invalid XML", str(sldTransformError))

    def listLayers(self):
        self.allMapLayers = []
        self.dlg.comboBox.clear()

        for (key, value) in QgsProject.instance().mapLayers().items():
            if isinstance(value, QgsRasterLayer):
                self.allMapLayers.append(value)
                self.dlg.comboBox.addItem(value.name())

                # layer name change handling
                try:
                    value.nameChanged.disconnect(self.listLayers)
                except:
                    pass
                value.nameChanged.connect(self.listLayers)

    def run(self):
        self.allMapLayers = []  # stores  all raster layers
        self.project = QgsProject.instance()  # QGIS project object

        # the dialog instance
        self.dlg = SLD4rasterDialog()
        self.upDlg = GeoserverUploadDialog()

        # show the dialog
        self.dlg.show()

        # UI elements' initial configurations
        self.uiSettings()

        # layer remove/add event handling
        self.project.layersRemoved.connect(self.listLayers)
        self.project.layersAdded.connect(self.listLayers)
        self.project.readProject.connect(self.listLayers)

        # listing the layers
        self.listLayers()

        # UI binding
        self.dlg.generateBtn.clicked.connect(self.qml2Sld)
        self.dlg.translateBtn.clicked.connect(self.sld2qml)
        self.dlg.saveBtn.clicked.connect(lambda: self.saveFile(self.dlg.sldText1.text(), 'SLD'))
        self.dlg.comboBox.currentIndexChanged.connect(lambda: self.dlg.sldText1.clear())

        self.dlg.browseBtn.clicked.connect(self.browseFile)
        self.dlg.validateBtn.clicked.connect(lambda: self.validation(self.dlg.sldText1.text()))
        self.dlg.validateBtn_2.clicked.connect(lambda: self.validation(self.dlg.sldText2.text()))

        self.dlg.UpDlgBtn.clicked.connect(self.showUploadDialog)
        self.upDlg.uploadBtn.clicked.connect(self.upload)
        self.upDlg.checkBox.stateChanged.connect(self.rememberMe)

        self.dlg.exec_()
