# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Siose2Manning
                                 A QGIS plugin
 Extrae el Manning n para HEC-RAS desde SIOSE AR (ID_COBERTURA_MAX)
 ***************************************************************************/
"""

import os
import csv
from pathlib import Path

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

import processing

from qgis.core import (
    QgsProject,
    QgsVectorLayer,
    QgsField,
    QgsFeature,
    QgsGeometry,
    QgsWkbTypes
)

from qgis.PyQt.QtCore import QVariant

from .Mod2_dialog import Mod2Dialog


class Siose2Manning:

    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(self.plugin_dir, "i18n", f"Siose2Manning_{locale}.qm")
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        self.actions = []
        self.menu = self.tr("&SIOSE2Manning")

        self.dlg = None
        self._is_dialog_open = False
        self._action_main = None

    def tr(self, message):
        return QCoreApplication.translate("Siose2Manning", message)

    # ======================================================
    # GUI
    # ======================================================

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        parent=None
    ):
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if add_to_toolbar:
            self.iface.addToolBarIcon(action)
        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)
        return action

    def initGui(self):
        icon_path = os.path.join(self.plugin_dir, "icon.png")

        self._action_main = self.add_action(
            icon_path,
            text=self.tr("SIOSE AR -> Manning"),
            callback=self.run,
            parent=self.iface.mainWindow()
        )

    def unload(self):
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&SIOSE2Manning"), action)
            self.iface.removeToolBarIcon(action)

        self.actions = []
        self._action_main = None
        self.dlg = None
        self._is_dialog_open = False

    # ======================================================
    # CSV
    # ======================================================

    def _csv_path(self) -> Path:
        return Path(self.plugin_dir) / "data" / "siose_ar_id_to_manning_es.csv"

    def _load_manning_map(self) -> dict:

        csv_path = self._csv_path()

        if not csv_path.exists():
            raise Exception(f"No se encontro el CSV interno en: {csv_path}")

        mapping = {}

        # Intentar UTF-8 primero, luego latin-1
        try:
            f = csv_path.open("r", encoding="utf-8-sig", newline="")
            reader = csv.DictReader(f)
            rows = list(reader)
        except UnicodeDecodeError:
            f = csv_path.open("r", encoding="latin-1", newline="")
            reader = csv.DictReader(f)
            rows = list(reader)

        fields = reader.fieldnames or []

        if "S20" not in fields or "MANNING" not in fields:
            raise Exception("El CSV debe tener columnas 'S20' y 'MANNING'.")

        for row in rows:
            code_raw = (row.get("S20") or "").strip()
            man_raw = (row.get("MANNING") or "").strip()

            if not code_raw or not man_raw:
                continue

            try:
                code = int(code_raw)
                man = float(man_raw.replace(",", "."))
            except Exception:
                continue

            mapping[code] = man

        if not mapping:
            raise Exception("El CSV no cargo ningun mapeo S20 -> MANNING.")

        return mapping

    # ======================================================
    # Buscar capa SIOSE
    # ======================================================

    def _find_siose_layer_in_project(self) -> QgsVectorLayer:

        for lyr in QgsProject.instance().mapLayers().values():
            if not isinstance(lyr, QgsVectorLayer) or not lyr.isValid():
                continue
            if lyr.geometryType() != QgsWkbTypes.PolygonGeometry:
                continue
            if "ID_COBERTURA_MAX" in [f.name() for f in lyr.fields()]:
                return lyr

        raise Exception(
            "No se encontro una capa SIOSE AR (T_COMBINADA) con el campo ID_COBERTURA_MAX en el proyecto."
        )

    # ======================================================
    # Crear AOI temporal
    # ======================================================

    def _make_aoi_layer(self, geom: QgsGeometry) -> QgsVectorLayer:

        crs_authid = self.iface.mapCanvas().mapSettings().destinationCrs().authid()
        aoi_layer = QgsVectorLayer(f"Polygon?crs={crs_authid}", "AOI_TMP", "memory")

        pr = aoi_layer.dataProvider()
        pr.addAttributes([QgsField("id", QVariant.Int)])
        aoi_layer.updateFields()

        feat = QgsFeature(aoi_layer.fields())
        feat["id"] = 1
        feat.setGeometry(geom)
        pr.addFeature(feat)

        aoi_layer.updateExtents()
        return aoi_layer

    # ======================================================
    # Proceso principal
    # ======================================================

    def _run_step3(self, aoi_geom: QgsGeometry):

        manning_map = self._load_manning_map()
        siose_layer = self._find_siose_layer_in_project()

        # Guardar estilo para heredarlo al resultado
        src_renderer = None
        try:
            if siose_layer.renderer() is not None:
                src_renderer = siose_layer.renderer().clone()
        except Exception:
            src_renderer = None

        aoi_layer = self._make_aoi_layer(aoi_geom)

        # Clip por AOI
        result_clip = processing.run(
            "native:clip",
            {
                "INPUT": siose_layer,
                "OVERLAY": aoi_layer,
                "OUTPUT": "memory:"
            }
        )
        tmp_layer = result_clip["OUTPUT"]

        # Retener solo campos utiles (si existen)
        keep_candidates = ["id", "REFCAT", "ID_COBERTURA_MAX"]
        tmp_fields = [f.name() for f in tmp_layer.fields()]
        keep_fields = [f for f in keep_candidates if f in tmp_fields]

        # Si por alguna razon no encuentra ninguno, al menos mantener ID_COBERTURA_MAX
        if "ID_COBERTURA_MAX" not in keep_fields and "ID_COBERTURA_MAX" in tmp_fields:
            keep_fields.append("ID_COBERTURA_MAX")

        result_keep = processing.run(
            "native:retainfields",
            {
                "INPUT": tmp_layer,
                "FIELDS": keep_fields,
                "OUTPUT": "memory:"
            }
        )

        out_layer = result_keep["OUTPUT"]
        out_layer.setName("SIOSE_Manning_AOI")

        # Campos salida
        pr = out_layer.dataProvider()
        pr.addAttributes([
            QgsField("MANNING_N", QVariant.Double),
            QgsField("AREA_M2", QVariant.Double),
            QgsField("AREA_HA", QVariant.Double),
        ])
        out_layer.updateFields()

        idx_code = out_layer.fields().indexOf("ID_COBERTURA_MAX")
        idx_man = out_layer.fields().indexOf("MANNING_N")
        idx_m2 = out_layer.fields().indexOf("AREA_M2")
        idx_ha = out_layer.fields().indexOf("AREA_HA")

        if idx_code == -1:
            raise Exception("La capa resultante no tiene el campo ID_COBERTURA_MAX. Revise que esta usando T_COMBINADA.")

        missing = set()

        out_layer.startEditing()

        for f in out_layer.getFeatures():

            fid = f.id()

            try:
                code_int = int(f[idx_code])
            except Exception:
                code_int = None

            man = None
            if code_int is not None:
                man = manning_map.get(code_int)
                if man is None:
                    missing.add(code_int)

            area_m2 = f.geometry().area()

            out_layer.changeAttributeValue(fid, idx_man, man)
            out_layer.changeAttributeValue(fid, idx_m2, area_m2)
            out_layer.changeAttributeValue(fid, idx_ha, area_m2 / 10000.0)

        out_layer.commitChanges()

        # Heredar estilo del SIOSE original (si se pudo clonar)
        if src_renderer is not None:
            try:
                out_layer.setRenderer(src_renderer)
                out_layer.triggerRepaint()
            except Exception:
                pass

        QgsProject.instance().addMapLayer(out_layer)

        if missing:
            self.iface.messageBar().pushWarning(
                "SIOSE -> Manning",
                f"Capa creada, pero faltan codigos en el CSV para: {sorted(missing)}"
            )
        else:
            self.iface.messageBar().pushInfo(
                "SIOSE -> Manning",
                "Capa resultado creada correctamente."
            )

    # ======================================================
    # Ejecutar
    # ======================================================

    def run(self):

        if self._is_dialog_open and self.dlg is not None:
            self.dlg.show()
            self.dlg.raise_()
            self.dlg.activateWindow()
            return

        if self.dlg is None:
            self.dlg = Mod2Dialog(self.iface)
            self.dlg.finished.connect(self._on_dialog_finished)

        self._is_dialog_open = True

        if self._action_main is not None:
            self._action_main.setEnabled(False)

        self.dlg.show()
        result = self.dlg.exec_()

        if result:
            try:
                self._run_step3(self.dlg.aoi_geom)
            except Exception as e:
                self.iface.messageBar().pushCritical("SIOSE -> Manning", str(e))

    def _on_dialog_finished(self, _code):
        self._is_dialog_open = False
        if self._action_main is not None:
            self._action_main.setEnabled(True)

