from typing import Any, Optional, Union

from qgis.core import QgsSettings
from qgis.PyQt.QtCore import QVariant

from .resources import plugin_name


def setting_key(*args: str) -> str:
    """
    Get QGIS setting key

    :param args List of path elements e.g. ['img', 'logos', 'image.png']
    """
    return "/" + "/".join((plugin_name(), *map(str, args)))


def get_setting(
    key: str,
    default: Optional[Any] = None,
    typehint: type = None,
    internal: bool = True,
    section: int = QgsSettings.NoSection,
) -> Union[QVariant, str]:
    """
    Get QGIS setting value plugin

    :param key: Key for the setting
    :param default: Optional default value
    :param typehint: Type hint
    :param internal: Whether to search from only plugin settings or all
    :param section: Section argument can be used to get a value from
    a specific Section.
    """
    s = QgsSettings()
    kwargs = {"defaultValue": default, "section": section}

    if typehint is not None:
        kwargs["type"] = typehint
    return s.value(setting_key(key) if internal else key, **kwargs)


def set_setting(
    key: str,
    value: Union[str, int, float, bool],
    internal: bool = True,
    section: int = QgsSettings.NoSection,
) -> bool:
    """
    Set a value in the QgsSetting

    :param key: Key for the setting
    :param value: Value for the setting
    :param internal: Whether to search from only plugin settings or all
    :param section: Section argument can be used to set a value to a specific Section
    """
    qs = QgsSettings()
    return qs.setValue(setting_key(key) if internal else key, value, section)
