"""Tools to work with resource files."""

import configparser
from os.path import abspath, dirname, join, pardir

try:
    from qgis.core import Qgis
except ImportError:
    from qgis.core import QGis as Qgis

PLUGIN_NAME: str = ""
PLUGIN_VERSION: str = ""

def plugin_path(*args: str) -> str:
    """Get the path to plugin root folder.

    :param args List of path elements e.g. ['img', 'logos', 'image.png']
    :type args: str

    :return: Absolute path to the resource.
    :rtype: str
    """
    path = dirname(__file__)
    path = abspath(abspath(join(path, pardir)))
    for item in args:
        path = abspath(join(path, item))

    return path

def plugin_version() -> str:
    """Return the plugin name according to metadata.txt.

    :return: The plugin name.
    :rtype: basestring
    """
    global PLUGIN_VERSION
    if PLUGIN_VERSION == "":
        try:
            metadata = metadata_config()
            version: str = metadata["general"]["version"]
            PLUGIN_VERSION = version
        except KeyError:
            PLUGIN_VERSION = "0.0.0"
    return PLUGIN_VERSION

def plugin_name() -> str:
    """Return the plugin name according to metadata.txt.

    :return: The plugin name.
    :rtype: basestring
    """
    global PLUGIN_NAME
    if PLUGIN_NAME == "":
        try:
            metadata = metadata_config()
            name: str = metadata["general"]["name"]
            name = name.replace(" ", "").strip()
            PLUGIN_NAME = name
        except KeyError:
            PLUGIN_NAME = "stratopo_plugin"
    return PLUGIN_NAME


def metadata_config() -> configparser.ConfigParser:
    """Get the INI config parser for the metadata file.

    :return: The config parser object.
    :rtype: ConfigParser
    """
    path = plugin_path("metadata.txt")
    config = configparser.ConfigParser()
    config.read(path)
    return config


def resources_path() -> str:
    """Get the path to our resources folder.

    :param args List of path elements e.g. ['img', 'logos', 'image.png']
    :type args: str

    :return: Absolute path to the resources folder.
    :rtype: str
    """
    path = abspath(abspath(join(plugin_path(), "resources")))
    return path


def qgis_version_sanitized() -> str:
    return str(Qgis.QGIS_VERSION_INT)

