# -*- coding: utf-8 -*-
"""
/**************************************************************************
        SDATool - A QGIS plugin to do spatial data analysis 
                                using R Statistical package
                             -------------------
    begin                : 2008-01-15
    copyright            : (C) 2008 by Volkan Kepoglu
    email                : vkepoglu at gmail.com
 *************************************************************************/
/**************************************************************************
 *   This program is free software; you can redistribute it and/or modify *
 *   it under the terms of the GNU General Public License as published by *
 *   the Free Software Foundation; either version 2 of the License, or    *
 *   (at your option) any later version.                                  *
 *************************************************************************/
"""
from PyQt4 import QtCore
from PyQt4 import QtGui
from qgis.core import *
# import the code for the dialog
from ui_density import Ui_densityDialog
#handling outputFile: path name, file name and extension
import os
# run R code
import densityRpart
# write to logFile.log
import wl
# dialog for open help file
import sda4ppDisplayHelpFile
# reading qgis layer
import sda4ppReadQGISLayer
# R options
import sda4ppRoptions

class DensityTool(QtGui.QDialog, Ui_densityDialog):
  def __init__(self, iface):
    QtGui.QDialog.__init__(self)
    # save reference to the QGIS interface
    self.iface = iface
    # Set up the user interface from Designer.
    self.setupUi(self)
    self.setFixedSize(self.size())
    # init variables
    self.index = {"inputFilePath":"","inputFileName":"","fieldName":"",
                  "projText":"","readBandwidth":0,"extent":1,
                  "outputFilePath":"","edgeCor":"TRUE","rstfile":"GTiff",
                  "bandWidthValue":"","noWeight":True,
                  "pixelX":100,"pixelY":100,"rstExtension":"tif",
                  "maxBandWidth":"","nearDist":"","xmin":0,"xmax":0,
                  "ymin":0,"ymax":0,"plot":True,"save":False,
                  "nonumericfieldError":"","readogrError":"",
                  "intersectionError":""}
    # read the name of qgis layer
    self.readInputFile()
    self.progressBar.setValue(0)
    QtCore.QObject.connect(self.chkSave, QtCore.SIGNAL("clicked()"),
                           self.chkSave_changed)
    QtCore.QObject.connect(self.chkPixels, QtCore.SIGNAL("clicked()"),
                           self.chkPixels_changed)
    QtCore.QObject.connect(self.chkEdgeCorrection,
                           QtCore.SIGNAL("clicked()"),
                           self.chkEdgeCorrection_changed)
    QtCore.QObject.connect(self.cmbPointLayer,
                           QtCore.SIGNAL("activated(QString )"),
                           self.cmbPointLayer_changed)
    QtCore.QObject.connect(self.cmbFieldName,
                           QtCore.SIGNAL("activated(QString )"),
                           self.cmbFieldName_changed)
    QtCore.QObject.connect(self.cmbRstFormat,
                           QtCore.SIGNAL("activated(QString )"),
                           self.cmbRstFormat_changed)
    QtCore.QObject.connect(self.cmbNewBoundary,
                           QtCore.SIGNAL("activated(QString )"),
                           self.cmbNewBoundary_changed)
    QtCore.QObject.connect(self.btnSave, QtCore.SIGNAL("clicked()"),
                           self.btnSave_changed)
    QtCore.QObject.connect(self.btnApply, QtCore.SIGNAL("clicked()"),
                           self.runApply)
    QtCore.QObject.connect(self.btnHelp, QtCore.SIGNAL("clicked()"),
                           self.runHelp)
    QtCore.QObject.connect(self.btnCancel, QtCore.SIGNAL("clicked()"),
                           self.runClose)

  def runClose(self):
    sda4ppRoptions.closeRplot(self.R)
    self.reject()

  def runHelp(self):
    sda4ppDisplayHelpFile.DisplayHelpFile("Density")

  def getUniqueOutputFileName(self, file=""):
    notFileExist = 0
    counter = 0
    if file == "":
      file = self.ReadPolyLayer.createOutputDir() + "Density"
    while (notFileExist == 0):
      outputFilename = file + "_" + str(counter) + "." + \
                       self.index["rstExtension"]
      if not(os.path.exists(outputFilename)):
        self.index["outputFilePath"] = outputFilename
        self.txtPath.setText(self.index["outputFilePath"])
        notFileExist = 1
      else:
        counter = counter + 1

  def changeExtension(self):
    if self.index["outputFilePath"] != "":
      f = QtCore.QFileInfo(self.index["outputFilePath"])
      if f.suffix() != self.index["rstExtension"]:
        self.index["outputFilePath"] = f.path() + "/" + f.completeBaseName() +\
                                   "." + self.index["rstExtension"]
      if not(os.path.exists(self.index["outputFilePath"])):
        self.txtPath.setText(self.index["outputFilePath"])
      else:
        file =f.path() + "/" + f.completeBaseName()
        self.getUniqueOutputFileName(file)

  def btnSave_changed(self):
    (self.index["outputFilePath"], foo) = self.ReadPointLayer.saveDialog(self,
                                    extension = self.index["rstExtension"])
    if self.index["outputFilePath"] is None:
      self.index["outputFilePath"] = str(self.txtPath.text())
      return
    self.changeExtension()

  def cmbRstFormat_changed(self):
    # raster file format: the driver name of GDAL Library in R
    if self.cmbRstFormat.currentIndex() == 0:
      self.index["rstfile"] = "GTiff"
      self.index["rstExtension"] = "tif"
    elif self.cmbRstFormat.currentIndex() == 1:
      self.index["rstfile"] = "HFA"
      self.index["rstExtension"] = "img"
    self.changeExtension()

  def chkSave_changed(self):
    dt = "density-chkSave"
    # inputFile
    if self.cmbPointLayer.currentIndex() == -1:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error",
            "Please add point typed vector layer in QGIS.")
      return
    if self.chkSave.checkState() == 2:
      self.index["save"] = True
      self.cmbRstFormat.setEnabled(True)
      self.btnSave.setEnabled(True)
      self.txtPath.setEnabled(True)
      self.chkPixels.setEnabled(True)
      self.getUniqueOutputFileName()
    elif self.chkSave.checkState() == 0:
      self.txtPath.clear()
      self.index["save"] = False
      self.cmbRstFormat.setEnabled(False)
      self.btnSave.setEnabled(False)
      self.txtPath.setEnabled(False)
      self.chkPixels.setEnabled(False)
      self.chkPixels.setCheckState(0)
      self.spinBoxX.setEnabled(False)
      self.spinBoxY.setEnabled(False)
    wl.wl("index: " + str(self.index),dt)

  def chkPixels_changed(self):
    if self.chkPixels.isChecked():
      self.spinBoxX.setEnabled(True)
      self.spinBoxY.setEnabled(True)
    else:
      self.spinBoxX.setEnabled(False)
      self.spinBoxY.setEnabled(False)

  def chkEdgeCorrection_changed(self):
    if self.chkEdgeCorrection.checkState() == 2:
      self.index["edgeCor"] = "TRUE" # in the R format
    elif self.chkEdgeCorrection.checkState() == 0:
      self.index["edgeCor"] = "FALSE" # in the R format

  def cmbNewBoundary_changed(self):
    dt = "density-cmbNewBoundary"
    self.index["intersectionError"] = ""
    if self.cmbNewBoundary.currentText()==self.cmbPointLayer.currentText():
      # Extent of Vector Layer
      self.index["extent"] = 1
    else:
      self.index["extent"] = 0
      # re-prepare sda4ppReadQGISLayer class to do analysis
      # as the layer is changed in the combobox
      self.ReadPolyLayer.updateLayer(str(self.cmbNewBoundary.currentText()))
      extent = self.ReadPolyLayer.getExtentObj()
      self.index["xmin"] = extent.xMinimum()
      self.index["xmax"] = extent.xMaximum()
      self.index["ymin"] = extent.yMinimum()
      self.index["ymax"] = extent.yMaximum()
    wl.wl("index: " +str(self.index),dt)
    # bandwidth
    self.computeBandwidth()

  def computeBandwidth(self):
    dt = "density-computeBandwidth"
    # nonumericfieldError
    if self.index["nonumericfieldError"] != "":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["nonumericfieldError"])
      return
    # readOGR
    error = sda4ppRoptions.readOGR(self.R,self.index["inputFilePath"],self.index["inputFileName"])
    if error != 0:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", error)
      return
    # R part of bandwidth
    wl.wl("index: " +str(self.index),dt)
    self.btnApply.setEnabled(False)
    self.index["readBandwidth"]=1
    (self.index["bandWidthValue"], self.index["maxBandWidth"],
         self.index["nearDist"]) = \
         densityRpart.doSpatialAnalysis(self.R, index = self.index)
    self.btnApply.setEnabled(True)
    # intersection
    # the error is point layer does not intersect with the study area layer
    if self.index["bandWidthValue"] == 0 and \
           self.index["maxBandWidth"] == 0 and \
           self.index["nearDist"] == "intersectionError":
      self.index["intersectionError"] = "The point layer, '" + \
                                        str(self.index["inputFileName"]) + \
                                        "', does not intersect with\n" + \
                                        "the selected new study area layer, '" + \
                                        str(self.cmbNewBoundary.currentText()) + \
                                         "'.\n\nPlease, select another study area."
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["intersectionError"])
      return
    # labeling
    self.txtBandwidth.setText(str(self.index["bandWidthValue"]))
    self.lblMaxBandwidth.setText("0 < bandwidth < " + \
                                 str(self.index["maxBandWidth"]))
    self.lblNearDist.setText(self.index["nearDist"])
    wl.wl("index: " +str(self.index),dt)

  def cmbPointLayer_changed(self):
    dt = "density-cmbPointLayer"
    self.index["readogrError"] = ""
    #update source of layer
    (self.index["inputFilePath"], self.index["inputFileName"]) = \
       self.ReadPointLayer.getFileSource (self.cmbPointLayer.currentText())
    # readOGR
    error = sda4ppRoptions.readOGR(self.R,self.index["inputFilePath"],self.index["inputFileName"])
    if error != 0:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", error)
      return
    # get projection
    self.index["projText"] = self.ReadPointLayer.getProj()
    # field names
    self.cmbFieldName.clear()
    self.cmbFieldName.addItem("No Weight")
    fields = self.ReadPointLayer.getFieldNameList()
    wl.wl("fields: " +str(fields),dt)
    if fields == []:
      self.index["nonumericfieldError"] = \
              "This tool requires at least one numeric field.\n\n" +\
              "Please, add one numeric field to the layer, '" +\
              self.index["inputFileName"] + \
              "',nor do not use this layer for this analysis."
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["nonumericfieldError"])
      self.index["firstNumericField"] = ""
      self.index["fieldName"] = ""
    else:
      for field in fields:
        self.cmbFieldName.addItem(field)
      self.index["nonumericfieldError"] = ""
      self.index["firstNumericField"] = fields[0]
      self.index["fieldName"] = fields[0]
    wl.wl("index: " +str(self.index),dt)
    #update bandwidth of new selected vector layer
    self.cmbNewBoundary_changed()

  def cmbFieldName_changed(self):
    self.index["nonumericfieldError"] = ""
    # No Weight
    if self.cmbFieldName.currentIndex() == 0:
      self.index["fieldName"] = self.index["firstNumericField"]
      self.index["noWeight"] = True
    else:
      self.index["fieldName"] = str(self.cmbFieldName.currentText())
      self.index["noWeight"] = False

  def readInputFile(self):
    dt = "density-readInputFile"
    # init rpy2
    import rpy2.robjects as robjects
    self.R = robjects.r
    # SECTION: POINT
    self.ReadPointLayer = sda4ppReadQGISLayer.ReadQgisLayer(self.iface)
    layers = self.ReadPointLayer.getLayerList(checkNumericField="YES")
    wl.wl("layers: " +str(layers),dt)
    if layers == []:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
                        "SDA4PP Plugin Error",
                        "Please add point typed vector layers in QGIS.")
      return
    for layer in layers:
      self.cmbPointLayer.addItem(layer)
    # inputFile
    (self.index["inputFilePath"], self.index["inputFileName"]) = \
       self.ReadPointLayer.getFileSource (self.cmbPointLayer.currentText())
    # field names
    self.cmbFieldName.clear()
    self.cmbFieldName.addItem("No Weight")
    fields = self.ReadPointLayer.getFieldNameList()
    wl.wl("fields: " +str(fields),dt)
    if fields == []:
      self.index["nonumericfieldError"] = \
              "This tool requires at least one numeric field.\n\n" +\
              "Please, add one numeric field to the layer, '" +\
              self.index["inputFileName"] + \
              "',nor do not use this layer for this analysis."
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["nonumericfieldError"])
      self.index["firstNumericField"] = ""
      self.index["fieldName"] = ""
    else:
      for field in fields:
        self.cmbFieldName.addItem(field)
      self.index["nonumericfieldError"] = ""
      self.index["firstNumericField"] = fields[0]
      self.index["fieldName"] = fields[0]
    # projection
    self.index["projText"] = self.ReadPointLayer.getProj()
    # SECTION: ALL for study area
    self.ReadPolyLayer = sda4ppReadQGISLayer.ReadQgisLayer(self.iface,
                                                           "all")
    polylayers = self.ReadPolyLayer.getLayerList()
    for polylayer in polylayers:
      self.cmbNewBoundary.addItem(polylayer)
    self.index["extent"] = 1
    # readOGR
    error = sda4ppRoptions.readOGR(self.R,self.index["inputFilePath"],self.index["inputFileName"])
    if error != 0:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", error)
      return
    wl.wl("index: " +str(self.index),dt)
    # bandwidth
    self.computeBandwidth()
    # unique output filename
    self.getUniqueOutputFileName()
    return

  def runApply(self):
    dt = "density-runApply"
    self.progressBar.setValue(5)
    #BEGIN SECTION: inputFile
    if self.cmbPointLayer.currentIndex() == -1:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error",
            "Please add point typed vector layer in QGIS.")
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(10)
    # could not save postgis layer as shp file
    if self.index["inputFilePath"] == 'error':
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["inputFileName"])
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(15)
    # nonumericfieldError
    if self.index["nonumericfieldError"] != "":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["nonumericfieldError"])
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(20)
    # readogrError
    if self.index["readogrError"] != "":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["readogrError"])
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(25)
    # intersectionError
    if self.index["intersectionError"] != "":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["intersectionError"])
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(30)
    # getting bandwidth
    userBandwidth = str(self.txtBandwidth.text())
    (value, errorText) = wl.readIntTxtBox(userBandwidth,
                                                         mode="decimal")
    if value == "error":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
                                    "SDA4PP Plugin Error", errorText)
      self.txtBandwidth.setText(str(self.index["bandWidthValue"]))
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(35)
    # user entered new bandwidth value
    if self.index["bandWidthValue"] != value:
      if value > self.index["maxBandWidth"]:
        QtGui.QMessageBox.information(self.iface.mainWindow(),
             "SDA4PP Plugin Error",
             "Bandwidth is too large for this study region.\n" + \
             "The bandwidth should be between 0 and " + \
             str (self.index["maxBandWidth"]) + ".")
        self.txtBandwidth.setText(str(self.index["bandWidthValue"]))
        self.progressBar.setValue(0)
        return
      else:
        self.index["bandWidthValue"] = value
        wl.wl("index: " +str(self.index),dt)
    self.progressBar.setValue(40)
    # reading outputFile
    if self.index["outputFilePath"] == "":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error",
            "New raster filename should be defined.\n" + \
            "Please, use \"Save As\" button.")
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(45)
    # plot option
    self.index["plot"] = self.chkPlot.isChecked()
    # at least one of them should be checked
    if not(self.index["save"]) and not(self.index["plot"]):
      QtGui.QMessageBox.information(self.iface.mainWindow(),
        "SDA4PP Plugin Error",
        "At least one of them 'Save' or 'Plot' option should be checked.")
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(50)
    # pixel number:
    if self.chkPixels.isChecked():
      self.index["pixelX"] = self.spinBoxX.value()
      self.index["pixelY"] = self.spinBoxY.value()
    #BEGIN SECTION: R part
    self.progressBar.setValue(100)
    self.index["readBandwidth"]=0
    wl.wl("index: " +str(self.index),dt)
    # apply analysis part to R
    self.btnApply.setEnabled(False)
    densityRpart.doSpatialAnalysis(self.R,index= self.index)
    self.btnApply.setEnabled(True)
    # save
    if self.index["save"]:
      resSave = self.ReadPointLayer.saveRaster(self.index["outputFilePath"])
      if resSave == "error":
        self.progressBar.setValue(0)
        return
      self.getUniqueOutputFileName()
    self.progressBar.setValue(0)
