# -*- coding: utf-8 -*-
"""
/**************************************************************************
        SDATool - A QGIS plugin to do spatial data analysis 
                                using R Statistical package
                             -------------------
    begin                : 2008-01-15
    copyright            : (C) 2008 by Volkan Kepoglu
    email                : vkepoglu at gmail.com
 *************************************************************************/
/**************************************************************************
 *   This program is free software; you can redistribute it and/or modify *
 *   it under the terms of the GNU General Public License as published by *
 *   the Free Software Foundation; either version 2 of the License, or    *
 *   (at your option) any later version.                                  *
 *************************************************************************/
"""
from PyQt4 import QtCore
from PyQt4 import QtGui
from qgis.core import *
# import the code for the dialog
from ui_poisson import Ui_poissonDialog
#handling outputFile: path name, file name and extension
import os
# run R code
import poissonRpart
# write to logFile.log
import wl
# dialog for open help file
import sda4ppDisplayHelpFile
# reading qgis layer
import sda4ppReadQGISLayer
# R options
import sda4ppRoptions

class PoissonTool(QtGui.QDialog, Ui_poissonDialog):
  def __init__(self, iface):
    QtGui.QDialog.__init__(self)
    # save reference to the QGIS interface
    self.iface = iface
    # Set up the user interface from Designer.
    self.setupUi(self)
    self.setFixedSize(self.size())
    # init variables
    self.index = {"inputFilePath":"","inputFileName":"","method":"~1",
                  "userSideX":1.0,"userCustom":"","plot":True,
                  "plotType":"trend","save":False,
                  "outputFilePath":"","rstfile":"GTiff","rstExtension":"tif",
                  "changePixel":False,"pixelX":100,"pixelY":100}
    # read the name of qgis layer
    self.readInputFile()
    self.progressBar.setValue(0)
    QtCore.QObject.connect(self.cmbPointLayer,
        QtCore.SIGNAL("activated(QString )"), self.cmbPointLayer_changed)
    QtCore.QObject.connect(self.cmbRstFormat,
        QtCore.SIGNAL("activated(QString )"), self.cmbRstFormat_changed)
    QtCore.QObject.connect(self.btnSave,
        QtCore.SIGNAL("clicked()"), self.btnSave_changed)
    QtCore.QObject.connect(self.btnApply,
        QtCore.SIGNAL("clicked()"), self.runApply)
    QtCore.QObject.connect(self.btnHelp,
        QtCore.SIGNAL("clicked()"), self.runHelp)
    QtCore.QObject.connect(self.btnCancel,
        QtCore.SIGNAL("clicked()"), self.runClose)
    QtCore.QObject.connect(self.chkSave,
        QtCore.SIGNAL("clicked()"), self.chkSave_changed)
    QtCore.QObject.connect(self.chkPixels,
        QtCore.SIGNAL("clicked()"), self.chkPixels_changed)
    QtCore.QObject.connect(self.chkPlot,
        QtCore.SIGNAL("clicked()"), self.chkPlot_changed)

  def runHelp(self):
    sda4ppDisplayHelpFile.DisplayHelpFile("poisson")

  def runClose(self):
    sda4ppRoptions.closeRplot(self.R)
    self.reject()

  def getUniqueOutputFileName(self, file=""):
    notFileExist = 0
    counter = 0
    if file == "":
      file = self.ReadPointLayer.createOutputDir() + "poisson"
    while (notFileExist == 0):
      outputFilename = file + "_" + str(counter) + "." + \
                       self.index["rstExtension"]
      if not(os.path.exists(outputFilename)):
        self.index["outputFilePath"] = outputFilename
        self.txtPath.setText(self.index["outputFilePath"])
        notFileExist = 1
      else:
        counter = counter + 1

  def changeExtension(self):
    if self.index["outputFilePath"] != "":
      f = QtCore.QFileInfo(self.index["outputFilePath"])
      if f.suffix() != self.index["rstExtension"]:
        self.index["outputFilePath"] = f.path() + "/" + f.completeBaseName() +\
                                   "." + self.index["rstExtension"]
      if not(os.path.exists(self.index["outputFilePath"])):
        self.txtPath.setText(self.index["outputFilePath"])
      else:
        file =f.path() + "/" + f.completeBaseName()
        self.getUniqueOutputFileName(file)

  def btnSave_changed(self):
    (self.index["outputFilePath"], foo) = self.ReadPointLayer.saveDialog(self,
                                    extension = self.index["rstExtension"])
    if self.index["outputFilePath"] is None:
      self.index["outputFilePath"] = str(self.txtPath.text())
      return
    self.changeExtension()

  def cmbRstFormat_changed(self):
    # raster file format: the driver name of GDAL Library in R
    if self.cmbRstFormat.currentIndex() == 0:
      self.index["rstfile"] = "GTiff"
      self.index["rstExtension"] = "tif"
    elif self.cmbRstFormat.currentIndex() == 1:
      self.index["rstfile"] = "HFA"
      self.index["rstExtension"] = "img"
    self.changeExtension()

  def chkSave_changed(self):
    dt = "poisson-chkSave_changed"
    if self.chkSave.checkState() == 2:
      self.index["save"] = True
      self.cmbRstFormat.setEnabled(True)
      self.btnSave.setEnabled(True)
      self.txtPath.setEnabled(True)
      self.chkPixels.setEnabled(True)
      self.getUniqueOutputFileName()
    elif self.chkSave.checkState() == 0:
      self.txtPath.clear()
      self.index["save"] = False
      self.cmbRstFormat.setEnabled(False)
      self.btnSave.setEnabled(False)
      self.txtPath.setEnabled(False)
      self.chkPixels.setEnabled(False)
      self.chkPixels.setCheckState(0)
      self.spinBoxX.setEnabled(False)
      self.spinBoxY.setEnabled(False)
    wl.wl("index: " + str(self.index),dt)

  def chkPixels_changed(self):
    if self.chkPixels.isChecked():
      self.spinBoxX.setEnabled(True)
      self.spinBoxY.setEnabled(True)
    else:
      self.spinBoxX.setEnabled(False)
      self.spinBoxY.setEnabled(False)

  def chkPlot_changed(self):
    if self.chkPlot.isChecked():
      self.rbTrend.setEnabled(True)
      self.rbSE.setEnabled(True)
    else:
      self.rbTrend.setEnabled(False)
      self.rbSE.setEnabled(False)

  def cmbPointLayer_changed(self):
    dt = "poisson-cmbPointLayer_changed"
    #update source of layer
    (self.index["inputFilePath"], self.index["inputFileName"]) = \
      self.ReadPointLayer.getFileSource (self.cmbPointLayer.currentText())
    wl.wl("index: " +str(self.index),dt)
    # readOGR
    error = sda4ppRoptions.readOGR(self.R,self.index["inputFilePath"],self.index["inputFileName"])
    if error != 0:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", error)
      return

  def readInputFile(self):
    dt = "poisson-readInputFile"
    # init rpy2
    import rpy2.robjects as robjects
    self.R = robjects.r
    # SECTION: POINT
    self.ReadPointLayer = sda4ppReadQGISLayer.ReadQgisLayer(self.iface)
    layers = self.ReadPointLayer.getLayerList()
    wl.wl("layers: " +str(layers),dt)
    if layers == []:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error",
            "Please add point typed vector layers in QGIS.")
      return
    for layer in layers:
      self.cmbPointLayer.addItem(layer)
    # inputFile
    (self.index["inputFilePath"], self.index["inputFileName"]) = \
      self.ReadPointLayer.getFileSource (self.cmbPointLayer.currentText())
    # readOGR
    error = sda4ppRoptions.readOGR(self.R,self.index["inputFilePath"],self.index["inputFileName"])
    if error != 0:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", error)
      return
    wl.wl("index: " +str(self.index),dt)
    return

  def runApply(self):
    dt = "poisson-runApply"
    self.progressBar.setValue(5)
    #BEGIN SECTION: inputFile
    if self.cmbPointLayer.currentIndex() == -1:
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error",
            "Please add point typed vector layer in QGIS.")
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(10)
    # could not save postgis layer as shp file
    if self.index["inputFilePath"] == 'error':
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error", self.index["inputFileName"])
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(15)
    # methods
    if self.rbHomo.isChecked():
      self.index["method"] = "~1"
    elif self.rbLinear.isChecked():
      self.index["method"] = "~x + y"
    elif self.rbPolynom.isChecked():
      self.index["method"] = "~polynom(x, y, 2)"
    elif self.rbSide.isChecked():
      # getting x value for side
      self.index["userSideX"] = str(self.txtSideX.text())
      (value, errorText) = wl.readIntTxtBox(\
          self.index["userSideX"], mode="decimal")
      if value == "error":
        QtGui.QMessageBox.information(self.iface.mainWindow(),
              "SDA4PP Plugin Error", errorText)
        self.txtSideX.setText("")
        self.progressBar.setValue(0)
        return
      self.index["method"] = "~side(x)"
    elif self.rbCustom.isChecked():
      self.index["method"] = "~custom(x)"
      self.index["userCustom"] = str(self.txtCustom.text())
    self.progressBar.setValue(25)
    # reading outputFile
    if self.index["save"] and self.index["outputFilePath"] == "":
      QtGui.QMessageBox.information(self.iface.mainWindow(),
            "SDA4PP Plugin Error",
            "New raster filename should be defined.\n" + \
            "Please, use \"Save As\" button.")
      self.progressBar.setValue(0)
      return
    self.progressBar.setValue(30)
    # pixel number:
    if self.chkPixels.isChecked():
      self.index["pixelX"] = self.spinBoxX.value()
      self.index["pixelY"] = self.spinBoxY.value()
      self.index["changePixel"] = True
    self.progressBar.setValue(100)
    #BEGIN SECTION: R part
    self.btnApply.setEnabled(False)
    # plotting options
    self.index["plot"] = self.chkPlot.isChecked()
    if self.rbTrend.isChecked():
      self.index["plotType"] = "trend"
    else:
      self.index["plotType"] = "se"
    wl.wl("index: " +str(self.index),dt)
    # apply analysis part to R
    resText = poissonRpart.doSpatialAnalysis(self.R, index= self.index)
    self.btnApply.setEnabled(True)
    # save
    if self.index["save"]:
      resSave = self.ReadPointLayer.saveRaster(self.index["outputFilePath"])
      if resSave == "error":
        self.progressBar.setValue(0)
        return
      self.getUniqueOutputFileName()
    # output list
    self.lblResult.setText(resText)
    self.btnApply.setEnabled(True)
    self.progressBar.setValue(0)
